<?php
/**
 * update_apply.php – Update installieren (INV)
 *
 * POST-Endpoint. Erwartet:
 *   csrf_token     – CSRF-Token
 *   target_version – Zielversion (z.B. "2.0")
 *
 * Gibt JSON zurück:
 *   success  – bool
 *   steps    – array mit Fortschrittsschritten
 *   error    – Fehlermeldung (nur bei Misserfolg)
 */

ini_set('display_errors', 0);
ob_start();

header('Content-Type: application/json; charset=utf-8');
set_time_limit(300);
ini_set('memory_limit', '256M');

// Session starten
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Auth-Prüfung: Hub-Superuser ODER eingeloggter Instanz-Admin
$inv_authorized = isset($_SESSION['hub_superuser']);
if (!$inv_authorized) {
    foreach ($_SESSION as $key => $val) {
        if (strpos($key, 'auth_') === 0 && is_array($val)) {
            $inv_authorized = true;
            break;
        }
    }
}
if (!$inv_authorized) {
    ob_end_clean();
    http_response_code(403);
    echo json_encode(['error' => 'Nicht autorisiert']);
    exit;
}

// Nur POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ob_end_clean();
    http_response_code(405);
    echo json_encode(['error' => 'Nur POST erlaubt']);
    exit;
}

try {
    require_once __DIR__ . '/../core/functions.php';
    require_once __DIR__ . '/../hub_config.php';
    require_once __DIR__ . '/Updater.php';

    // CSRF prüfen
    if (!validateCSRF($_POST['csrf_token'] ?? '')) {
        ob_end_clean();
        http_response_code(400);
        echo json_encode(['error' => 'Sicherheitsfehler (CSRF). Bitte Seite neu laden.']);
        exit;
    }

    // Zielversion validieren
    $target_version = trim($_POST['target_version'] ?? '');
    if (!preg_match('/^\d+\.\d+(\.\d+)?$/', $target_version)) {
        ob_end_clean();
        http_response_code(400);
        echo json_encode(['error' => 'Ungültige Version angegeben']);
        exit;
    }

    $pdo = new PDO(
        'mysql:host=' . HUB_DB_HOST . ';dbname=' . HUB_DB_NAME . ';charset=utf8mb4',
        HUB_DB_USER,
        HUB_DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    // Versions-Eintrag im Manifest suchen
    $manifest = Updater::fetchManifest();
    if (!$manifest) {
        ob_end_clean();
        echo json_encode(['success' => false, 'steps' => [], 'error' => 'Manifest nicht abrufbar']);
        exit;
    }

    $entry = null;
    foreach ($manifest['versions'] as $v) {
        if (($v['version'] ?? '') === $target_version) {
            $entry = $v;
            break;
        }
    }

    if (!$entry) {
        ob_end_clean();
        echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version nicht im Manifest gefunden"]);
        exit;
    }

    // Sicherstellen, dass die Zielversion wirklich neuer ist
    $installed = Updater::getInstalledVersion($pdo);
    if (version_compare($target_version, $installed, '<=')) {
        ob_end_clean();
        echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version ist nicht neuer als die installierte Version $installed"]);
        exit;
    }

    $result = Updater::applyUpdate($entry, $pdo);
    ob_end_clean();
    echo json_encode($result);

} catch (Throwable $e) {
    $msg = $e->getMessage();
    if (class_exists('Updater')) {
        Updater::log('FEHLER in update_apply.php: ' . $msg);
    }
    ob_end_clean();
    http_response_code(500);
    echo json_encode(['success' => false, 'steps' => [], 'error' => 'Interner Fehler: ' . $msg]);
}
