<?php
/**
 * Updater.php – Auto-Update Engine für INV (Inventarverwaltung)
 *
 * Prüft auf neue Versionen unter update.it-wil.at/inv,
 * lädt das Update-ZIP herunter und spielt es ein.
 *
 * Bewahrte Dateien (werden NICHT überschrieben):
 *   - hub_config.php
 *   - inventar/{instanz}/config.php
 */

define('INV_APP_ID',       'inv');
define('INV_UPDATE_SERVER', 'https://update.it-wil.at/inv');
define('INV_APP_ROOT',      dirname(__DIR__));         // /inv
define('INV_UPDATE_DIR',    __DIR__);                  // /inv/update
define('INV_UPDATE_LOG',    __DIR__ . '/update.log');
define('INV_UPDATE_LOCK',   __DIR__ . '/update.lock');
define('INV_CACHE_TTL',     3600);                     // Manifest 1 Std. cachen

class Updater
{
    // ── Öffentliche API ───────────────────────────────────────────────────────

    /**
     * Manifest (versions.json) vom Update-Server holen.
     * Ergebnis wird bis INV_CACHE_TTL Sekunden gecacht.
     */
    public static function fetchManifest(): ?array
    {
        $cache = INV_UPDATE_DIR . '/manifest_cache.json';
        if (file_exists($cache) && (time() - filemtime($cache)) < INV_CACHE_TTL) {
            $data = json_decode(file_get_contents($cache), true);
            if (is_array($data)) return $data;
        }

        $url  = INV_UPDATE_SERVER . '/versions.json';
        $json = @file_get_contents($url, false, self::httpContext());
        if ($json === false) return null;

        $data = json_decode($json, true);
        if (!is_array($data)) return null;

        file_put_contents($cache, $json, LOCK_EX);
        return $data;
    }

    /**
     * Neueste verfügbare Version ermitteln, die neuer als $current ist.
     * Verwendet version_compare().
     */
    public static function getAvailableUpdate(string $current): ?array
    {
        $manifest = self::fetchManifest();
        if (!$manifest || empty($manifest['versions'])) return null;

        $best = null;
        foreach ($manifest['versions'] as $entry) {
            if (!isset($entry['version'])) continue;
            if (!preg_match('/^\d+\.\d+(\.\d+)?$/', $entry['version'])) continue;

            if (version_compare($entry['version'], $current, '>')) {
                if ($best === null || version_compare($entry['version'], $best['version'], '>')) {
                    // Min-PHP prüfen
                    if (!empty($entry['min_php']) && version_compare(PHP_VERSION, $entry['min_php'], '<')) {
                        continue;
                    }
                    $best = $entry;
                }
            }
        }
        return $best;
    }

    /**
     * Update-Prozess ausführen.
     *
     * Schritte:
     *  1. Lock-Datei belegen
     *  2. ZIP herunterladen
     *  3. SHA-256 prüfen
     *  4. In Temp-Dir entpacken
     *  5. Dateien kopieren (bewahrte Pfade überspringen)
     *  6. Migration-SQL laden + ausführen (optional)
     *  7. Neue Version speichern
     *  8. Lock freigeben + Cache löschen
     *
     * @param array $version_entry Versions-Eintrag aus dem Manifest
     * @param PDO   $hub_pdo       Verbindung zur Hub-Datenbank
     * @return array ['success'=>bool, 'steps'=>string[], 'error'=>?string]
     */
    public static function applyUpdate(array $version_entry, PDO $hub_pdo): array
    {
        $steps = [];
        $lock  = null;

        // 1. Lock belegen
        $lock = fopen(INV_UPDATE_LOCK, 'w');
        if (!$lock || !flock($lock, LOCK_EX | LOCK_NB)) {
            return ['success' => false, 'steps' => $steps, 'error' => 'Update läuft bereits (Lock aktiv)'];
        }

        try {
            $ver = $version_entry['version'];
            self::log("=== Update auf v{$ver} gestartet ===");

            // 2. ZIP herunterladen
            $steps[] = "Lade ZIP herunter…";
            self::log("Lade ZIP: " . $version_entry['zip_url']);
            $zip_data = @file_get_contents($version_entry['zip_url'], false, self::httpContext());
            if ($zip_data === false) {
                throw new RuntimeException('ZIP konnte nicht heruntergeladen werden');
            }

            // 3. SHA-256 prüfen
            $actual_hash = hash('sha256', $zip_data);
            if (!hash_equals(strtolower($version_entry['zip_sha256']), $actual_hash)) {
                throw new RuntimeException("ZIP-Prüfsumme ungültig (erwartet: {$version_entry['zip_sha256']}, erhalten: {$actual_hash})");
            }
            $steps[] = "ZIP-Integrität OK (" . strlen($zip_data) . " Bytes)";
            self::log("ZIP-Prüfsumme OK");

            // ZIP in Temp-Datei schreiben
            $tmp_zip = sys_get_temp_dir() . '/inv_update_' . time() . '.zip';
            file_put_contents($tmp_zip, $zip_data);
            unset($zip_data);

            // 4. In Temp-Dir entpacken
            $tmp_dir = sys_get_temp_dir() . '/inv_update_extract_' . time();
            mkdir($tmp_dir, 0755, true);

            $zip = new ZipArchive();
            if ($zip->open($tmp_zip) !== true) {
                throw new RuntimeException('ZIP konnte nicht geöffnet werden');
            }
            $zip->extractTo($tmp_dir);
            $zip->close();
            @unlink($tmp_zip);
            $steps[] = "ZIP entpackt";
            self::log("ZIP entpackt nach: $tmp_dir");

            // 5. Dateien kopieren
            $preserved = self::getPreservedPaths();
            $copied    = self::copyFiles($tmp_dir, INV_APP_ROOT, $preserved);
            $steps[] = "Dateien kopiert ($copied Dateien)";
            self::log("$copied Dateien kopiert");

            // Temp-Dir aufräumen
            self::rmdir_recursive($tmp_dir);

            // 6. Migration ausführen (falls nötig)
            if (!empty($version_entry['requires_db_migration']) && !empty($version_entry['migration_url'])) {
                $steps[] = "Lade Migration-SQL…";
                self::log("Lade Migration: " . $version_entry['migration_url']);
                $sql_data = @file_get_contents($version_entry['migration_url'], false, self::httpContext());
                if ($sql_data === false) {
                    throw new RuntimeException('Migration-SQL konnte nicht heruntergeladen werden');
                }

                // Migration-Checksum prüfen (falls vorhanden)
                if (!empty($version_entry['migration_sha256'])) {
                    $sql_hash = hash('sha256', $sql_data);
                    if (!hash_equals(strtolower($version_entry['migration_sha256']), $sql_hash)) {
                        throw new RuntimeException('Migration-SQL Prüfsumme ungültig');
                    }
                }

                // Hub-DB Migration (ohne Prefix)
                $hub_result = self::runMigration($sql_data, '', $hub_pdo);
                self::log("Hub-Migration: {$hub_result['executed']} Statements");
                if (!empty($hub_result['errors'])) {
                    self::log("Hub-Migration Fehler: " . implode('; ', $hub_result['errors']));
                }

                // Pro-Instanz Migration
                $instance_dirs = glob(INV_APP_ROOT . '/inventar/*', GLOB_ONLYDIR);
                foreach ($instance_dirs as $inst_dir) {
                    $inst_cfg = $inst_dir . '/config.php';
                    if (!file_exists($inst_cfg)) continue;

                    $table_prefix = null;
                    // Prefix aus Config extrahieren (ohne ganze Config zu laden)
                    $cfg_content = file_get_contents($inst_cfg);
                    if (preg_match('/\$table_prefix\s*=\s*[\'"]([^\'"]+)[\'"]/', $cfg_content, $m)) {
                        $table_prefix = $m[1];
                    }
                    if (!$table_prefix) continue;

                    $inst_result = self::runMigration($sql_data, $table_prefix, $hub_pdo);
                    $inst_name   = basename($inst_dir);
                    self::log("Instanz '$inst_name' Migration: {$inst_result['executed']} Statements");
                }

                $steps[] = "Migrationen ausgeführt";
            }

            // 7. Neue Version speichern
            self::saveInstalledVersion($ver, $hub_pdo);
            $steps[] = "Version v{$ver} gespeichert";
            self::log("Version gespeichert: v{$ver}");

            // 8. Cache löschen (damit nächster Check aktuell ist)
            @unlink(INV_UPDATE_DIR . '/manifest_cache.json');

            self::log("=== Update auf v{$ver} erfolgreich ===");
            return ['success' => true, 'steps' => $steps, 'error' => null];

        } catch (Throwable $e) {
            $msg = $e->getMessage();
            self::log("FEHLER: $msg");
            return ['success' => false, 'steps' => $steps, 'error' => $msg];
        } finally {
            if ($lock) {
                flock($lock, LOCK_UN);
                fclose($lock);
                @unlink(INV_UPDATE_LOCK);
            }
        }
    }

    /**
     * Liste der Pfade, die beim Update NICHT überschrieben werden.
     * @return string[] Absolute Pfade
     */
    public static function getPreservedPaths(): array
    {
        $preserved = [
            INV_APP_ROOT . '/hub_config.php',
        ];

        // Alle Instanz-Configs
        foreach (glob(INV_APP_ROOT . '/inventar/*/config.php') ?: [] as $cfg) {
            $preserved[] = realpath($cfg) ?: $cfg;
        }

        return $preserved;
    }

    /**
     * Installierte Version aus der Hub-DB lesen.
     * Erstellt die hub_update_meta Tabelle falls nötig.
     */
    public static function getInstalledVersion(PDO $pdo): string
    {
        self::ensureMetaTable($pdo);
        $val = $pdo->query("SELECT meta_value FROM hub_update_meta WHERE meta_key = 'installed_version'")->fetchColumn();
        return $val ?: '0.0';
    }

    /**
     * Neue Version nach erfolgreichem Update in der Hub-DB speichern.
     */
    public static function saveInstalledVersion(string $version, PDO $pdo): void
    {
        self::ensureMetaTable($pdo);
        $stmt = $pdo->prepare(
            "INSERT INTO hub_update_meta (meta_key, meta_value) VALUES ('installed_version', ?)
             ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)"
        );
        $stmt->execute([$version]);

        $stmt2 = $pdo->prepare(
            "INSERT INTO hub_update_meta (meta_key, meta_value) VALUES ('last_update_at', ?)
             ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)"
        );
        $stmt2->execute([date('Y-m-d H:i:s')]);
    }

    /**
     * Zeile ans Update-Log anhängen.
     * Auto-Rotation: >256 KB → letzte 300 Zeilen behalten.
     */
    public static function log(string $msg): void
    {
        $line = '[' . date('Y-m-d H:i:s') . '] ' . $msg . PHP_EOL;

        if (file_exists(INV_UPDATE_LOG) && filesize(INV_UPDATE_LOG) > 262144) {
            $lines = file(INV_UPDATE_LOG, FILE_IGNORE_NEW_LINES);
            if ($lines !== false) {
                file_put_contents(INV_UPDATE_LOG, implode(PHP_EOL, array_slice($lines, -300)) . PHP_EOL);
            }
        }

        file_put_contents(INV_UPDATE_LOG, $line, FILE_APPEND | LOCK_EX);
    }

    /**
     * Letzte $n Zeilen des Update-Logs lesen.
     */
    public static function readLog(int $n = 50): array
    {
        if (!file_exists(INV_UPDATE_LOG)) return [];
        $lines = file(INV_UPDATE_LOG, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!$lines) return [];
        return array_slice($lines, -$n);
    }

    // ── Private Hilfsmethoden ─────────────────────────────────────────────────

    /**
     * Dateien rekursiv von $src nach $dst kopieren.
     * Pfade in $preserved werden übersprungen.
     * @return int Anzahl kopierter Dateien
     */
    private static function copyFiles(string $src, string $dst, array $preserved): int
    {
        $count   = 0;
        $root_re = realpath($dst);

        $iter = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($src, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iter as $item) {
            $rel  = ltrim(substr($item->getPathname(), strlen($src)), DIRECTORY_SEPARATOR);
            $dest = $dst . DIRECTORY_SEPARATOR . $rel;

            // Path-Traversal-Schutz
            $real_dest_dir = realpath(dirname($dest)) ?: dirname($dest);
            if ($root_re && strpos($real_dest_dir, $root_re) !== 0) {
                continue;
            }

            if ($item->isDir()) {
                if (!is_dir($dest)) mkdir($dest, 0755, true);
                continue;
            }

            // Bewahrt?
            $real_dest = realpath($dest) ?: $dest;
            $skip = false;
            foreach ($preserved as $p) {
                if ($real_dest === $p || strpos($real_dest, rtrim($p, '/\\')) === 0) {
                    $skip = true;
                    break;
                }
            }
            if ($skip) continue;

            if (!is_dir(dirname($dest))) mkdir(dirname($dest), 0755, true);
            copy($item->getPathname(), $dest);
            $count++;
        }

        return $count;
    }

    /**
     * SQL-Migration ausführen.
     * $prefix ersetzt {prefix} Platzhalter im SQL.
     */
    private static function runMigration(string $sql, string $prefix, PDO $pdo): array
    {
        if ($prefix) {
            $sql = str_replace('{prefix}', $prefix, $sql);
        }

        // Statements splitten (wie restore_sql_file in bib)
        $stmts = []; $current = ''; $in_str = false; $str_chr = '';
        $len = strlen($sql);
        for ($i = 0; $i < $len; $i++) {
            $c = $sql[$i]; $prev = $i > 0 ? $sql[$i-1] : '';
            if (!$in_str && ($c === "'" || $c === '"' || $c === '`')) { $in_str = true; $str_chr = $c; }
            elseif ($in_str && $c === $str_chr && $prev !== '\\')      { $in_str = false; }
            if (!$in_str && $c === ';') {
                $s = trim($current);
                if ($s !== '' && !str_starts_with($s, '--') && !str_starts_with($s, '/*')) {
                    $stmts[] = $s;
                }
                $current = '';
            } else { $current .= $c; }
        }

        $executed = 0; $errors = [];
        $pdo->exec('SET FOREIGN_KEY_CHECKS = 0');
        foreach ($stmts as $stmt) {
            if (preg_match('/^\s*(--|\/\*)/s', $stmt)) continue;
            try { $pdo->exec($stmt); $executed++; }
            catch (PDOException $e) {
                $errors[] = substr($stmt, 0, 60) . '… → ' . $e->getMessage();
                if (count($errors) >= 10) { $errors[] = '(weitere Fehler abgeschnitten)'; break; }
            }
        }
        $pdo->exec('SET FOREIGN_KEY_CHECKS = 1');

        return ['executed' => $executed, 'errors' => $errors];
    }

    /**
     * Hub-Update-Meta Tabelle anlegen falls nicht vorhanden.
     */
    private static function ensureMetaTable(PDO $pdo): void
    {
        $pdo->exec("CREATE TABLE IF NOT EXISTS `hub_update_meta` (
            `meta_key`   VARCHAR(100) NOT NULL,
            `meta_value` TEXT,
            `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`meta_key`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    }

    /**
     * HTTP-Stream-Context für Downloads.
     * @return resource
     */
    private static function httpContext()
    {
        return stream_context_create([
            'http' => [
                'timeout'       => 60,
                'user_agent'    => 'it-wil-updater/1.0 (inv)',
                'ignore_errors' => false,
            ],
            'ssl' => [
                'verify_peer'      => true,
                'verify_peer_name' => true,
            ],
        ]);
    }

    /**
     * Verzeichnis rekursiv löschen.
     */
    private static function rmdir_recursive(string $dir): void
    {
        if (!is_dir($dir)) return;
        $items = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($items as $item) {
            $item->isDir() ? rmdir($item->getPathname()) : unlink($item->getPathname());
        }
        rmdir($dir);
    }
}
