<?php
// core/scan.php
ob_start();
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

$msgSuccess = [];
$msgError = [];

// Stammdaten laden
$stmtCat = $pdo->query("SELECT name FROM `{$table_prefix}categories` ORDER BY name ASC");
$categories = $stmtCat->fetchAll(PDO::FETCH_COLUMN);

$stmtLoc = $pdo->query("SELECT name FROM `{$table_prefix}locations` ORDER BY name ASC");
$locations = $stmtLoc->fetchAll(PDO::FETCH_COLUMN);

// Formular Verarbeitung
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF-Schutz
    if (!validateCSRF($_POST['csrf_token'] ?? '')) {
        $msgError[] = "Ungültige Anfrage. Bitte laden Sie die Seite neu.";
    } else {
    $rawInput = trim($_POST['barcode']);
    $rawInput = preg_replace('/[\t\n\r\s]+/', ',', $rawInput);
    
    $itemName = trim($_POST['item_name']);
    $category = trim($_POST['category']);
    $location = trim($_POST['location']);
    $notes    = trim($_POST['notes']);
    $customDate = $_POST['custom_date'] ?? date('Y-m-d H:i:s');
    
    $dbDate = str_replace('T', ' ', $customDate);
    if(strlen($dbDate) == 16) $dbDate .= ':00';

    if (empty($rawInput) || empty($itemName)) {
        $msgError[] = "Barcode und Bezeichnung sind Pflichtfelder.";
    } else {
        if (!empty($category) && !in_array($category, $categories)) {
            $msgError[] = "Die Kategorie '<b>" . h($category) . "</b>' ist ungültig.";
        }
        if (!empty($location) && !in_array($location, $locations)) {
            $msgError[] = "Der Standort '<b>" . h($location) . "</b>' ist ungültig.";
        }

        if (empty($msgError)) {
            $barcodesToProcess = [];
            $parts = explode(',', $rawInput);
            
            foreach ($parts as $part) {
                $part = trim($part);
                if (empty($part)) continue;
                
                if (($_POST['remove_zeros'] ?? '1') === '1') {
                    $part = preg_replace('/0000$/', '', $part);
                }

                if (strpos($part, '-') !== false) {
                    $range = explode('-', $part);
                    if (count($range) == 2 && is_numeric($range[0]) && is_numeric($range[1])) {
                        $start = (int)$range[0];
                        $end = (int)$range[1];
                        for ($i = min($start, $end); $i <= max($start, $end); $i++) $barcodesToProcess[] = (string)$i;
                    } else { $barcodesToProcess[] = $part; }
                } else { $barcodesToProcess[] = $part; }
            }

            $pdo->beginTransaction();
            try {
                $insertStmt = $pdo->prepare("INSERT INTO `{$table_prefix}inventory` (barcode, item_name, category, location, notes, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $historyStmt = $pdo->prepare("INSERT INTO `{$table_prefix}history` (inventory_id, user_id, username, action, new_value, created_at) VALUES (?, ?, ?, 'create', ?, ?)");
                $checkStmt = $pdo->prepare("SELECT id FROM `{$table_prefix}inventory` WHERE barcode = ?");
                
                foreach ($barcodesToProcess as $code) {
                    $checkStmt->execute([$code]);
                    if ($checkStmt->fetch()) {
                        $msgError[] = "Code <b>$code</b> existiert bereits.";
                        continue; 
                    }

                    $insertStmt->execute([$code, $itemName, $category, $location, $notes, $currentUser, $dbDate]);
                    $newId = $pdo->lastInsertId();

                    $stateSnapshot = json_encode(['barcode' => $code, 'item_name' => $itemName, 'location' => $location]);
                    $historyStmt->execute([$newId, $currentUserId, $currentUser, $stateSnapshot, $dbDate]);
                    $msgSuccess[] = "<b>$code</b> erfolgreich angelegt.";
                }
                $pdo->commit();
            } catch (Exception $e) {
                $pdo->rollBack();
                $msgError[] = "Datenbankfehler: " . $e->getMessage();
            }
        }
    }
    } // Ende CSRF-Schutz
}

$pageTitle = "Erfassen";
require_once __DIR__ . '/header.php';
?>

<div class="container container-form">
    <?php if (!empty($msgSuccess)) echo '<div class="alert alert-success">'.implode('<br>', $msgSuccess).'</div>'; ?>
    <?php if (!empty($msgError)) echo '<div class="alert alert-error">'.implode('<br>', $msgError).'</div>'; ?>

    <div class="glass-panel" style="border-top: 4px solid var(--primary-hover);">
        <h2 class="text-center text-primary" style="margin-bottom: 2rem; border-bottom: 1px solid var(--glass-border); padding-bottom: 1rem;">Neues Inventar erfassen</h2>    
        
        <form method="POST" id="scanForm">
            <?= csrfField() ?>
            <input type="hidden" name="remove_zeros" id="removeZerosInput" value="1">
            <div class="form-group">
                <label class="text-primary font-bold uppercase">Barcode / Anlagennummer</label>
                <textarea name="barcode" id="barcodeInput" required autofocus autocomplete="off" rows="2"
                       placeholder="Scan..." 
                       class="font-mono" style="font-size: 1.2rem; padding: 1rem;"></textarea>
                <p class="text-muted" id="barcodeHint" style="font-size: 0.75rem; margin-top: 0.25rem;"></p>
            </div>

            <div class="form-group">
                <label>Bezeichnung</label>
                <input type="text" name="item_name" id="bezeichnungInput" required
                       placeholder="z.B. iPad Pro 12.9" style="font-size: 1.1rem;"
                       value="<?= isset($_POST['item_name']) ? h($_POST['item_name']) : '' ?>">
            </div>

            <div class="grid-2">
                <div class="form-group">
                    <label>Kategorie</label>
                    <select name="category" required style="width: 100%;">
                        <option value="">-- Bitte wählen --</option>
                        <?php foreach($categories as $c): ?>
                            <option value="<?= h($c) ?>" <?= (isset($_POST['category']) && $_POST['category'] == $c) ? 'selected' : '' ?>><?= h($c) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label>Standort</label>
                    <select name="location" required style="width: 100%;">
                        <option value="">-- Bitte wählen --</option>
                        <?php foreach($locations as $l): ?>
                            <option value="<?= h($l) ?>" <?= (isset($_POST['location']) && $_POST['location'] == $l) ? 'selected' : '' ?>><?= h($l) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="form-group">
                <label>Datum (Optional)</label>
                <input type="datetime-local" name="custom_date" value="<?= date('Y-m-d\TH:i') ?>">
            </div>

            <div class="form-group">
                <label>Anmerkung</label>
                <textarea name="notes" rows="3"><?= isset($_POST['notes']) ? h($_POST['notes']) : '' ?></textarea>
            </div>

            <button type="submit" class="btn btn-block" style="padding: 1rem; font-size: 1.1rem; margin-top: 1.5rem;">Speichern</button>
        </form>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', () => {
        const removeZeros = getBarcodeRemoveZeros();
        document.getElementById('removeZerosInput').value = removeZeros ? '1' : '0';
        const hint = document.getElementById('barcodeHint');
        if (hint) hint.textContent = removeZeros
            ? '(Scanner-Modus: 0000 am Ende werden automatisch entfernt.)'
            : '(Scanner-Modus: aktiv – Nullen werden NICHT entfernt.)';
        const barcodeInput = document.getElementById('barcodeInput');
        
        barcodeInput.addEventListener('keydown', (e) => {
            if (e.key === 'Tab' || e.key === 'Enter') {
                e.preventDefault();
                let start = barcodeInput.selectionStart; 
                let end = barcodeInput.selectionEnd; 
                let val = barcodeInput.value;
                
                // Text VOR dem Cursor holen
                let before = val.substring(0, start);
                let after = val.substring(end);

                before = cleanBarcode(before);
                
                // Füge Komma ein mit bereinigtem Wert
                barcodeInput.value = before + "," + after;
                
                // Cursor hinter das Komma setzen
                barcodeInput.selectionStart = barcodeInput.selectionEnd = before.length + 1;
            }
        });
    });
</script>
<?php require_once __DIR__ . '/footer.php'; ?>