<?php
// core/report_export.php
ob_start();
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

// Export-Format
$format = $_GET['format'] ?? 'excel';
if (!in_array($format, ['excel', 'pdf'])) {
    die('Ungültiges Format');
}

// Parameter von report.php übernehmen
$startDate  = $_GET['start'] ?? '2000-01-01';
$endDate    = $_GET['end']   ?? date('Y-m-d');
$reportType = $_GET['type']  ?? 'all';
$search     = trim($_GET['q'] ?? '');
$sort       = $_GET['sort'] ?? 'timestamp';
$order      = strtoupper($_GET['order'] ?? 'DESC');

$sortMap = [
    'timestamp' => 'created_at',
    'barcode' => 'barcode',
    'bezeichnung' => 'item_name',
    'username' => 'username',
    'status' => 'status',
    'last_inventur' => 'last_stocktake',
    'standort' => 'location'
];

if (!array_key_exists($sort, $sortMap)) $sort = 'timestamp';
if (!in_array($order, ['ASC', 'DESC'])) $order = 'DESC';

$dbSort = $sortMap[$sort];
$sqlStart = "$startDate 00:00:00";
$sqlEnd   = "$endDate 23:59:59";

$whereClauses = [];
$params = [];
$reportTitle = "System-Protokoll";
$isInventoryMode = false;
$isDisposalReport = false;

// Report-Typ Logic (identisch zu report.php)
switch ($reportType) {
    case 'active':
        $isInventoryMode = true;
        $whereClauses[] = "status = 'active'";
        $reportTitle = "Aktiver Bestand";
        break;
    case 'disposed':
        $isDisposalReport = true;
        $whereClauses[] = "h.created_at BETWEEN ? AND ?";
        $params[] = $sqlStart;
        $params[] = $sqlEnd;
        $whereClauses[] = "h.action = 'dispose'";
        $reportTitle = "Ausscheidungsprotokoll";
        break;
    case 'inventur_old':
        $isInventoryMode = true;
        $reportTitle = "Inventur überfällig";
        $whereClauses[] = "status = 'active' AND (last_stocktake < ? OR last_stocktake IS NULL)";
        $params[] = date('Y-m-d', strtotime('-1 year'));
        break;
    case 'no_location':
        $isInventoryMode = true;
        $reportTitle = "Ohne Standort";
        $whereClauses[] = "status = 'active' AND (location IS NULL OR location = '')";
        break;
    case 'unknown_loc':
        $isInventoryMode = true;
        $reportTitle = "Unbekannter Standort";
        $whereClauses[] = "status = 'active' AND location = 'Unbekannter Standort'";
        break;
    case 'move':
        $whereClauses[] = "h.created_at BETWEEN ? AND ?";
        $params[] = $sqlStart;
        $params[] = $sqlEnd;
        $whereClauses[] = "h.action = 'move'";
        $reportTitle = "Bewegungen";
        break;
    default:
        $whereClauses[] = "h.created_at BETWEEN ? AND ?";
        $params[] = $sqlStart;
        $params[] = $sqlEnd;
        $reportTitle = "Gesamte Historie";
        break;
}

// Suche
if (!empty($search)) {
    $prefixSearch = $isInventoryMode ? "" : "i.";
    $whereClauses[] = "({$prefixSearch}barcode LIKE ? OR {$prefixSearch}item_name LIKE ? OR {$prefixSearch}location LIKE ?)";
    $term = "%$search%";
    $params[] = $term;
    $params[] = $term;
    $params[] = $term;
}

$whereSQL = $whereClauses ? "WHERE " . implode(" AND ", $whereClauses) : "";

// Daten abrufen (ALLE, kein LIMIT)
try {
    if ($isInventoryMode) {
        $dataSql = "SELECT *, id as inventory_id, 'snapshot' as action, created_at as created_at, created_by as username 
                    FROM `{$table_prefix}inventory` 
                    $whereSQL 
                    ORDER BY $dbSort $order";
    } else {
        $sortPrefix = (in_array($sort, ['barcode', 'bezeichnung', 'status', 'last_inventur', 'standort'])) ? 'i.' : 'h.';
        $realSortCol = ($sort == 'bezeichnung') ? 'item_name' : (($sort == 'standort') ? 'location' : $dbSort);
        $dataSql = "SELECT h.*, i.barcode, i.item_name, i.status, i.last_stocktake 
                    FROM `{$table_prefix}history` h 
                    LEFT JOIN `{$table_prefix}inventory` i ON h.inventory_id = i.id 
                    $whereSQL 
                    ORDER BY {$sortPrefix}{$realSortCol} $order";
    }
    
    $stmt = $pdo->prepare($dataSql);
    $stmt->execute($params);
    $results = $stmt->fetchAll();
} catch (PDOException $e) {
    die("DB Fehler: " . $e->getMessage());
}

// Hilfsfunktion für Detail-Text
function getDetailText($row, $isInvMode) {
    if ($isInvMode) {
        if ($row['status'] == 'disposed') {
            return 'Grund: ' . str_replace('Ausgeschieden. Grund: ', '', $row['notes']);
        }
        return empty($row['location']) ? 'Kein Standort!' : $row['location'];
    } else {
        if ($row['action'] == 'dispose') {
            return str_replace('Ausgeschieden. Grund: ', '', $row['new_value']);
        }
        if ($row['action'] == 'move') {
            return "Von " . $row['old_value'] . " nach " . $row['new_value'];
        }
        if ($row['action'] == 'create') {
            $d = json_decode($row['new_value'], true);
            return "Initial: " . (is_array($d) ? ($d['location'] ?? '-') : $row['new_value']);
        }
        return $row['new_value'];
    }
}

// ===================================
// EXCEL EXPORT (CSV)
// ===================================
if ($format === 'excel') {
    $filename = 'Report_' . $reportTitle . '_' . date('Y-m-d') . '.csv';
    
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // UTF-8 BOM für Excel
    echo "\xEF\xBB\xBF";
    
    $output = fopen('php://output', 'w');
    
    // Header-Zeile
    $headers = ['Datum', 'Barcode', 'Gegenstand', 'Detail/Ort', 'User', 'Status', 'Letzte Inventur'];
    fputcsv($output, $headers, ';');
    
    // Daten-Zeilen
    foreach ($results as $row) {
        $datum = $row['created_at'] ? date('d.m.Y', strtotime($row['created_at'])) : '-';
        if (!$isInventoryMode) {
            $datum .= ' ' . date('H:i', strtotime($row['created_at']));
        }
        
        $barcode = $row['barcode'] ?? '-';
        $itemName = $row['item_name'] ?? '-';
        $detail = getDetailText($row, $isInventoryMode);
        $username = $row['username'] ?? '-';
        $status = ($row['status'] ?? '') === 'active' ? 'Aktiv' : 'Ausgeschieden';
        $inventur = $row['last_stocktake'] ? date('d.m.Y', strtotime($row['last_stocktake'])) : '-';
        
        fputcsv($output, [
            $datum,
            $barcode,
            $itemName,
            $detail,
            $username,
            $status,
            $inventur
        ], ';');
    }
    
    fclose($output);
    exit;
}

// ===================================
// PDF EXPORT
// ===================================
if ($format === 'pdf') {
    // HTML für PDF erstellen (Browser-Print-Funktion)
    $filename = 'Report_' . $reportTitle . '_' . date('Y-m-d') . '.pdf';
    
    ob_clean();
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title><?= htmlspecialchars($reportTitle) ?></title>
        <style>
            @media print {
                @page { margin: 1cm; }
            }
            body {
                font-family: Arial, sans-serif;
                font-size: 10pt;
                margin: 20px;
            }
            h1 {
                font-size: 16pt;
                margin-bottom: 5px;
            }
            .meta {
                font-size: 9pt;
                color: #666;
                margin-bottom: 20px;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                font-size: 9pt;
            }
            th {
                background: #f0f0f0;
                padding: 6px;
                text-align: left;
                border: 1px solid #ddd;
                font-weight: bold;
            }
            td {
                padding: 5px 6px;
                border: 1px solid #ddd;
            }
            tr:nth-child(even) {
                background: #f9f9f9;
            }
            .barcode {
                font-family: 'Courier New', monospace;
                color: #0066cc;
            }
        </style>
    </head>
    <body>
        <h1><?= htmlspecialchars($reportTitle) ?></h1>
        <div class="meta">
            Erstellt am: <?= date('d.m.Y H:i') ?> Uhr<br>
            Zeitraum: <?= date('d.m.Y', strtotime($startDate)) ?> - <?= date('d.m.Y', strtotime($endDate)) ?><br>
            Anzahl Einträge: <?= count($results) ?>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th width="12%">Datum</th>
                    <th width="15%">Barcode</th>
                    <th width="25%">Gegenstand</th>
                    <th width="25%">Detail/Ort</th>
                    <th width="12%">User</th>
                    <th width="11%">Status</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($results as $row): ?>
                <tr>
                    <td>
                        <?= $row['created_at'] ? date('d.m.Y', strtotime($row['created_at'])) : '-' ?>
                        <?php if (!$isInventoryMode): ?>
                            <br><small style="color: #999;"><?= date('H:i', strtotime($row['created_at'])) ?></small>
                        <?php endif; ?>
                    </td>
                    <td class="barcode"><?= htmlspecialchars($row['barcode'] ?? '-') ?></td>
                    <td><?= htmlspecialchars($row['item_name'] ?? '-') ?></td>
                    <td><?= htmlspecialchars(getDetailText($row, $isInventoryMode)) ?></td>
                    <td><?= htmlspecialchars($row['username'] ?? '-') ?></td>
                    <td><?= ($row['status'] ?? '') === 'active' ? 'Aktiv' : 'Weg' ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <script>
            // Automatisches Drucken beim Laden
            window.onload = function() {
                window.print();
            };
        </script>
    </body>
    </html>
    <?php
    exit;
}
