<?php
// core/report.php
ob_start();
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

if (!function_exists('renderSortLink')) {
    function renderSortLink($col, $label, $curSort, $curOrder) {
        if (isset($_GET['print_mode'])) return $label;
        $newOrder = ($curSort === $col && $curOrder === 'ASC') ? 'DESC' : 'ASC';
        $arrow = ($curSort === $col) ? ($curOrder === 'ASC' ? '▲' : '▼') : '';
        $cls = ($curSort === $col) ? 'text-primary' : 'text-muted'; 
        $url = buildUrl(['sort' => $col, 'order' => $newOrder, 'page' => 1]);
        return "<a href='$url' class='$cls'>$label $arrow</a>";
    }
}

$isPrintMode = isset($_GET['print_mode']);
$itemsPerPage = $isPrintMode ? 999999 : 50; 

$startDate  = $_GET['start'] ?? '2000-01-01';
$endDate    = $_GET['end']   ?? date('Y-m-d');
$reportType = $_GET['type']  ?? 'all'; 
$search     = trim($_GET['q'] ?? '');
$page       = max(1, intval($_GET['page'] ?? 1));

$sort       = $_GET['sort'] ?? 'timestamp';
$order      = strtoupper($_GET['order'] ?? 'DESC');

$sortMap = [
    'timestamp' => 'created_at',
    'barcode' => 'barcode',
    'bezeichnung' => 'item_name',
    'username' => 'username', 
    'status' => 'status',
    'last_inventur' => 'last_stocktake',
    'standort' => 'location'
];

if (!array_key_exists($sort, $sortMap)) $sort = 'timestamp';
if (!in_array($order, ['ASC', 'DESC'])) $order = 'DESC';

$dbSort = $sortMap[$sort];
$offset = ($page - 1) * $itemsPerPage;
$sqlStart = "$startDate 00:00:00";
$sqlEnd   = "$endDate 23:59:59";

$whereClauses = [];
$params = [];
$reportTitle = "System-Protokoll";
$isInventoryMode = false;
$isDisposalReport = false; 

switch ($reportType) {
    case 'active':   
        $isInventoryMode = true; $whereClauses[] = "status = 'active'"; $reportTitle = "Bericht: Aktiver Bestand"; break;
    case 'disposed': 
        $isDisposalReport = true; 
        $whereClauses[] = "h.created_at BETWEEN ? AND ?"; $params[] = $sqlStart; $params[] = $sqlEnd;
        $whereClauses[] = "h.action = 'dispose'"; $reportTitle = "Ausscheidungsprotokoll"; break;
    case 'inventur_old':
        $isInventoryMode = true; $reportTitle = "Bericht: Inventur überfällig";
        $whereClauses[] = "status = 'active' AND (last_stocktake < ? OR last_stocktake IS NULL)";
        $params[] = date('Y-m-d', strtotime('-1 year')); break;
    case 'no_location':
        $isInventoryMode = true; $reportTitle = "Bericht: Ohne Standort";
        $whereClauses[] = "status = 'active' AND (location IS NULL OR location = '')"; break;
    case 'unknown_loc':
        $isInventoryMode = true; $reportTitle = "Bericht: Unbekannter Standort";
        $whereClauses[] = "status = 'active' AND location = 'Unbekannter Standort'"; break;
    case 'move':      
        $whereClauses[] = "h.created_at BETWEEN ? AND ?"; $params[] = $sqlStart; $params[] = $sqlEnd;
        $whereClauses[] = "h.action = 'move'"; $reportTitle = "Bericht: Bewegungen"; break;
    default:
        $whereClauses[] = "h.created_at BETWEEN ? AND ?"; $params[] = $sqlStart; $params[] = $sqlEnd;
        $reportTitle = "Gesamte Historie"; break;
}

if (!empty($search)) {
    $prefixSearch = $isInventoryMode ? "" : "i.";
    $whereClauses[] = "({$prefixSearch}barcode LIKE ? OR {$prefixSearch}item_name LIKE ? OR {$prefixSearch}location LIKE ?)";
    $term = "%$search%"; $params[] = $term; $params[] = $term; $params[] = $term;
}
$whereSQL = $whereClauses ? "WHERE " . implode(" AND ", $whereClauses) : "";

try {
    if ($isInventoryMode) {
        $countSql = "SELECT COUNT(*) FROM `{$table_prefix}inventory` $whereSQL";
        $stmt = $pdo->prepare($countSql); $stmt->execute($params);
        $totalItems = $stmt->fetchColumn();
        $dataSql = "SELECT *, id as inventory_id, 'snapshot' as action, created_at as created_at, created_by as username FROM `{$table_prefix}inventory` $whereSQL ORDER BY $dbSort $order LIMIT $itemsPerPage OFFSET $offset";
    } else {
        $countSql = "SELECT COUNT(*) FROM `{$table_prefix}history` h LEFT JOIN `{$table_prefix}inventory` i ON h.inventory_id = i.id $whereSQL";
        $stmt = $pdo->prepare($countSql); $stmt->execute($params);
        $totalItems = $stmt->fetchColumn();
        $sortPrefix = (in_array($sort, ['barcode', 'bezeichnung', 'status', 'last_inventur', 'standort'])) ? 'i.' : 'h.';
        $realSortCol = ($sort == 'bezeichnung') ? 'item_name' : (($sort == 'standort') ? 'location' : $dbSort);
        $dataSql = "SELECT h.*, i.barcode, i.item_name, i.status, i.last_stocktake FROM `{$table_prefix}history` h LEFT JOIN `{$table_prefix}inventory` i ON h.inventory_id = i.id $whereSQL ORDER BY {$sortPrefix}{$realSortCol} $order LIMIT $itemsPerPage OFFSET $offset";
    }
    $totalPages = ceil($totalItems / $itemsPerPage);
    $stmt = $pdo->prepare($dataSql);
    $stmt->execute($params);
    $results = $stmt->fetchAll();
} catch (PDOException $e) {
    error_log('Report DB error: ' . $e->getMessage());
    die("Ein Fehler ist aufgetreten. Bitte die Seite neu laden.");
}

function getDetailText($row, $isInvMode) {
    if ($isInvMode) {
        if ($row['status'] == 'disposed') return '<span class="text-danger">Grund: '.h(str_replace('Ausgeschieden. Grund: ', '', $row['notes'])).'</span>';
        return empty($row['location']) ? '<span class="text-danger">Kein Standort!</span>' : h($row['location']);
    } else {
        if ($row['action'] == 'dispose') {
            $reason = str_replace('Ausgeschieden. Grund: ', '', $row['new_value']);
            return '<span class="text-danger">'.h($reason).'</span>';
        }
        if ($row['action'] == 'move') return "Von <s style='opacity:0.5'>".h($row['old_value'])."</s> nach <b>".h($row['new_value'])."</b>";
        if ($row['action'] == 'create') {
            $d = json_decode($row['new_value'], true);
            return "Initial: " . h(is_array($d) ? ($d['location'] ?? '-') : $row['new_value']);
        }
        return h($row['new_value']);
    }
}

$pageTitle = $isDisposalReport ? "Ausscheidungsprotokoll" : "Berichte";
require_once __DIR__ . '/header.php';
?>

<!-- Report-spezifische Styles -->
<style>
    /* Einheitliche Höhe für alle Formularfelder */
    #reportForm input,
    #reportForm select,
    #reportForm button {
        height: 44px !important;  /* Etwas höher damit Text nicht abgeschnitten */
        font-size: 0.9rem !important;
    }
    
    /* Berichtstyp-Dropdown noch kleiner */
    #reportForm select[name="type"] {
        font-size: 0.85rem !important;
    }
    
    /* Optgroup-Labels */
    #reportForm optgroup {
        font-weight: 600;
        font-size: 0.85rem;
    }
    
    /* Options */
    #reportForm option {
        font-size: 0.85rem;
    }
    
    /* Export-Buttons gleiche Größe */
    .export-btn {
        height: 44px !important;
        width: 44px;
        padding: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.2rem;
    }
</style>

<div class="container">
    <div class="print-header-only">
        <div class="print-title"><?= $isDisposalReport ? 'Ausscheidungsprotokoll' : h($reportTitle) ?></div>
        <div class="print-meta" style="margin-bottom: 0px;">
            <div style="margin-top: 20px;">Erstellt am: <strong><?= date('d.m.Y') ?></strong></div>
            <div style="min-width: 300px; margin-top: 20px;">Erstellt durch: _______________________________&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;_______________________________(Unterschrift)</div>
        </div>
    </div>

    <?php if (!$isPrintMode): ?>
    <div class="glass-panel no-print">
        <form method="GET" id="reportForm" style="display: flex; flex-wrap: wrap; align-items: flex-end; gap: 10px;">
            <div class="form-group" style="margin:0; width: 280px;">
                <label>Suche</label>
                <input type="text" id="reportSearch" name="q" value="<?= h($search) ?>" placeholder="Barcode, Name...">
            </div>
            <div class="form-group" style="margin:0; width: 280px;">
                <label>Berichtstyp</label>
                <select name="type" onchange="this.form.submit()">
                    <optgroup label="Historie">
                        <option value="all" <?= $reportType=='all'?'selected':'' ?>>Gesamtes Protokoll</option>
                        <option value="move" <?= $reportType=='move'?'selected':'' ?>>Nur Bewegungen</option>
                    </optgroup>
                    <optgroup label="Bestand">
                        <option value="active" <?= $reportType=='active'?'selected':'' ?>>Aktiver Bestand</option>
                        <option value="disposed" <?= $reportType=='disposed'?'selected':'' ?>>Ausgeschieden (Zeitraum)</option>
                        <option value="inventur_old" <?= $reportType=='inventur_old'?'selected':'' ?>>⚠️ Inv. überfällig</option>
                        <option value="no_location" <?= $reportType=='no_location'?'selected':'' ?>>⚠️ Ohne Standort</option>
                        <option value="unknown_loc" <?= $reportType=='unknown_loc'?'selected':'' ?>>⚠️ Unbekannter Standort</option>
                    </optgroup>
                </select>
            </div>
            <div class="form-group" style="margin:0; width: 150px;">
                <label>Von</label>
                <input type="date" name="start" value="<?= h($startDate) ?>" <?= ($isInventoryMode && !$isDisposalReport) ? 'disabled style="opacity:0.5;"' : '' ?>>
            </div>
            <div class="form-group" style="margin:0; width: 150px;">
                <label>Bis</label>
                <input type="date" name="end" value="<?= h($endDate) ?>" <?= ($isInventoryMode && !$isDisposalReport) ? 'disabled style="opacity:0.5;"' : '' ?>>
            </div>
            <!-- Spacer für rechtsbündige Buttons -->
            <div style="flex-grow: 1;"></div>
            <!-- Buttons rechtsbündig -->
            <div style="display: flex; gap: 5px;">
                <a href="report_export.php?format=excel&<?= http_build_query($_GET) ?>" class="btn btn-secondary export-btn" title="Excel Export">📊</a>
                <a href="report_export.php?format=pdf&<?= http_build_query($_GET) ?>" class="btn btn-secondary export-btn" title="PDF Export">📄</a>
                <a href="<?= buildUrl(['print_mode' => 1]) ?>" target="_blank" class="btn btn-secondary export-btn" title="Drucken">🖨️</a>
            </div>
        </form>
    </div>
    <?php endif; ?>

    <div class="glass-panel">
        <div class="flex-between" style="border-bottom:1px solid var(--glass-border); padding-bottom:1rem; margin-bottom:1rem;">
            <div>
                <h2 class="text-primary" style="margin:0;"><?= h($reportTitle) ?></h2>
                <?php if($search): ?><small class="text-muted">Suche: "<?= h($search) ?>"</small><?php endif; ?>
            </div>
            <div class="text-muted"><strong><?= $totalItems ?></strong> Einträge</div>
        </div>

        <div class="table-wrapper">
            <table>
                <thead>
                    <tr>
                        <th><?= renderSortLink('timestamp', 'Datum', $sort, $order) ?></th>
                        <th><?= renderSortLink('barcode', 'Barcode', $sort, $order) ?></th>
                        <th><?= renderSortLink('bezeichnung', 'Gegenstand', $sort, $order) ?></th>
                        <th><?= renderSortLink('standort', ($isDisposalReport ? 'Grund / Details' : 'Detail / Ort'), $sort, $order) ?></th>
                        <th class="<?= ($isPrintMode && $isDisposalReport) ? '' : 'hide-mobile' ?>"><?= renderSortLink('username', ($isDisposalReport ? 'Ausgeschieden von' : 'User'), $sort, $order) ?></th>
                        <th class="<?= ($isPrintMode && $isDisposalReport) ? 'hide-in-print' : '' ?>"><?= renderSortLink('status', 'Status', $sort, $order) ?></th>
                        <th class="text-right <?= ($isPrintMode && $isDisposalReport) ? 'hide-in-print' : '' ?>"><?= renderSortLink('last_inventur', 'Inv.', $sort, $order) ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($results) == 0): ?><tr><td colspan="7" class="text-center text-muted" style="padding: 3rem;">Keine Datensätze gefunden.</td></tr><?php endif; ?>
                    <?php foreach($results as $row): ?>
                    <tr>
                        <td><?= $row['created_at'] ? date('d.m.Y', strtotime($row['created_at'])) : '-' ?><?php if(!$isInventoryMode): ?><div class="text-muted <?= $isPrintMode ? 'hide-in-print' : '' ?>" style="font-size:0.75em"><?= date('H:i', strtotime($row['created_at'])) ?></div><?php endif; ?></td>
                        <td class="font-mono text-primary"><?= $isPrintMode ? h($row['barcode']) : '<a href="edit.php?id='.$row['inventory_id'].'">'.h($row['barcode']).'</a>' ?></td>
                        <td><div class="font-bold"><?= h($row['item_name']) ?></div><?php if(!$isInventoryMode && !$isDisposalReport): ?><small class="uppercase"><?= h($row['action']) ?></small><?php endif; ?></td>
                        <td style="font-size:0.9em;"><?= getDetailText($row, $isInventoryMode) ?></td>
                        <td class="<?= ($isPrintMode && $isDisposalReport) ? '' : 'hide-mobile' ?>"><?= h(!empty($row['username']) ? $row['username'] : '-') ?></td>
                        <td class="<?= ($isPrintMode && $isDisposalReport) ? 'hide-in-print' : '' ?>"><?php if(($row['status']??'') === 'active'): ?><span class="text-success font-bold uppercase" style="font-size:0.75rem">Aktiv</span><?php else: ?><span class="text-danger font-bold uppercase" style="font-size:0.75rem">Weg</span><?php endif; ?></td>
                        <td class="text-right font-mono <?= ($isPrintMode && $isDisposalReport) ? 'hide-in-print' : '' ?>"><?php if ($row['last_stocktake']) { $isOld = strtotime($row['last_stocktake']) < strtotime('-1 year'); echo '<span class="'.($isOld ? 'text-danger' : 'text-success').'">' . date('d.m.Y', strtotime($row['last_stocktake'])) . '</span>'; } else { echo '<span class="text-muted">-</span>'; } ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php if ($totalPages > 1 && !$isPrintMode): ?>
        <div class="flex-center gap-2 no-print" style="margin-top:2rem;">
            <?php if ($page > 1): ?><a href="<?= buildUrl(['page' => $page - 1]) ?>" class="btn btn-secondary">&laquo;</a><?php endif; ?>
            <span class="btn btn-secondary" style="cursor:default; border-color:var(--primary);"><?= $page ?> / <?= $totalPages ?></span>
            <?php if ($page < $totalPages): ?><a href="<?= buildUrl(['page' => $page + 1]) ?>" class="btn btn-secondary">&raquo;</a><?php endif; ?>
        </div>
        <?php endif; ?>
    </div>
</div>
<?php if ($isPrintMode): ?><script>window.onload = function() { window.print(); }</script><?php endif; ?>
<script>
    const searchInput = document.getElementById('reportSearch');
    const form = document.getElementById('reportForm');
    let searchTimeout = null;
    
    function submitReportSearch() {
        let val = cleanBarcode(searchInput.value);
        searchInput.value = val;
        form.submit();
    }
    
    if(searchInput) {
        // Auto-Submit mit 500ms Delay
        searchInput.addEventListener('input', (e) => {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                submitReportSearch();
            }, 500);
        });
        
        // Enter-Taste: Sofort submitten (ohne Delay)
        searchInput.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                clearTimeout(searchTimeout);
                submitReportSearch();
            }
        });
    }
</script>
<?php require_once __DIR__ . '/footer.php'; ?>