<?php
// core/list.php
ob_start();
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

$allowedPerPage = [25, 50, 100, 200];
$perPage = isset($_GET['per_page']) ? (int)$_GET['per_page'] : 25;
$itemsPerPage = in_array($perPage, $allowedPerPage) ? $perPage : 25;
$allowedSorts = ['barcode', 'item_name', 'category', 'location', 'created_at'];

// ERWEITERTE FILTER
$search = trim($_GET['q'] ?? '');
$filterCategory = trim($_GET['category'] ?? '');
$filterLocation = trim($_GET['location'] ?? '');
$filterStatus = $_GET['status'] ?? 'active'; // Standard: nur aktive
$filterDateFrom = $_GET['date_from'] ?? '';
$filterDateTo = $_GET['date_to'] ?? '';

$page   = max(1, intval($_GET['page'] ?? 1));
$sort   = $_GET['sort'] ?? 'created_at';
$order  = strtoupper($_GET['order'] ?? 'DESC');
if (!in_array($sort, $allowedSorts)) $sort = 'created_at';
if (!in_array($order, ['ASC', 'DESC'])) $order = 'DESC';
$offset = ($page - 1) * $itemsPerPage;

// --- BULK ACTION ---
$msgSuccess = [];
$msgError = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'move') {
    // CSRF-Schutz
    if (!validateCSRF($_POST['csrf_token'] ?? '')) {
        $msgError[] = "Ungültige Anfrage. Bitte laden Sie die Seite neu.";
    } else {
    $targetLocation = trim($_POST['target_location']);
    $selectedIds = $_POST['ids'] ?? [];

    if (empty($targetLocation)) { $msgError[] = "Bitte einen Ziel-Standort angeben."; } 
    elseif (empty($selectedIds)) { $msgError[] = "Keine Gegenstände ausgewählt."; } 
    else {
        $pdo->beginTransaction();
        try {
            $getOldStmt = $pdo->prepare("SELECT location FROM `{$table_prefix}inventory` WHERE id = ?");
            $updateStmt = $pdo->prepare("UPDATE `{$table_prefix}inventory` SET location = ? WHERE id = ?");
            $historyStmt = $pdo->prepare("INSERT INTO `{$table_prefix}history` (inventory_id, user_id, username, action, old_value, new_value) VALUES (?, ?, ?, 'move', ?, ?)");
            $count = 0;
            foreach ($selectedIds as $id) {
                $getOldStmt->execute([$id]);
                $item = $getOldStmt->fetch();
                if ($item && $item['location'] !== $targetLocation) {
                    $updateStmt->execute([$targetLocation, $id]);
                    $historyStmt->execute([$id, $currentUserId, $currentUser, $item['location'], $targetLocation]);
                    $count++;
                }
            }
            $pdo->commit();
            $msgSuccess[] = "$count Gegenstände nach '<b>" . h($targetLocation) . "</b>' verschoben.";
        } catch (Exception $e) { $pdo->rollBack(); $msgError[] = "Fehler: " . $e->getMessage(); }
    }
    } // Ende CSRF-Schutz
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'set_category') {
    if (!validateCSRF($_POST['csrf_token'] ?? '')) {
        $msgError[] = "Ungültige Anfrage.";
    } else {
        $targetCategory = trim($_POST['target_category'] ?? '');
        $selectedIds = $_POST['ids'] ?? [];

        $stmtCatCheck = $pdo->prepare("SELECT id FROM `{$table_prefix}categories` WHERE name = ?");
        $stmtCatCheck->execute([$targetCategory]);
        if (empty($targetCategory) || !$stmtCatCheck->fetch()) {
            $msgError[] = "Ungültige Kategorie.";
        } elseif (empty($selectedIds)) {
            $msgError[] = "Keine Artikel ausgewählt.";
        } else {
            $pdo->beginTransaction();
            try {
                $updStmt  = $pdo->prepare("UPDATE `{$table_prefix}inventory` SET category = ? WHERE id = ?");
                $histStmt = $pdo->prepare("INSERT INTO `{$table_prefix}history` (inventory_id, user_id, username, action, old_value, new_value) VALUES (?, ?, ?, 'edit', ?, ?)");
                $getStmt  = $pdo->prepare("SELECT category FROM `{$table_prefix}inventory` WHERE id = ?");
                $count = 0;
                foreach ($selectedIds as $id) {
                    $getStmt->execute([$id]);
                    $row = $getStmt->fetch();
                    if ($row && $row['category'] !== $targetCategory) {
                        $updStmt->execute([$targetCategory, $id]);
                        $histStmt->execute([$id, $currentUserId, $currentUser, $row['category'], $targetCategory]);
                        $count++;
                    }
                }
                $pdo->commit();
                $msgSuccess[] = "$count Artikel auf Kategorie '<b>" . h($targetCategory) . "</b>' gesetzt.";
            } catch (Exception $e) {
                $pdo->rollBack();
                $msgError[] = "Fehler: " . $e->getMessage();
            }
        }
    }
}

// --- ERWEITERTE SUCHE MIT FILTERN ---
$whereSQL = "WHERE 1=1"; 
$params = [];

// Status-Filter
if ($filterStatus === 'active') {
    $whereSQL .= " AND status = 'active'";
} elseif ($filterStatus === 'disposed') {
    $whereSQL .= " AND status = 'disposed'";
}
// 'all' = keine Einschränkung

// Freitext-Suche
if (!empty($search)) {
    $whereSQL .= " AND (barcode LIKE ? OR item_name LIKE ? OR location LIKE ? OR category LIKE ?)";
    $term = "%$search%";
    $params = array_merge($params, [$term, $term, $term, $term]);
}

// Kategorie-Filter
if (!empty($filterCategory)) {
    $whereSQL .= " AND category = ?";
    $params[] = $filterCategory;
}

// Standort-Filter
if (!empty($filterLocation)) {
    $whereSQL .= " AND location = ?";
    $params[] = $filterLocation;
}

// Datums-Filter (Von)
if (!empty($filterDateFrom)) {
    $whereSQL .= " AND DATE(created_at) >= ?";
    $params[] = $filterDateFrom;
}

// Datums-Filter (Bis)
if (!empty($filterDateTo)) {
    $whereSQL .= " AND DATE(created_at) <= ?";
    $params[] = $filterDateTo;
}

$countStmt = $pdo->prepare("SELECT COUNT(*) FROM `{$table_prefix}inventory` $whereSQL");
$countStmt->execute($params);
$totalItems = $countStmt->fetchColumn();
$totalPages = ceil($totalItems / $itemsPerPage);

$sql = "SELECT * FROM `{$table_prefix}inventory` $whereSQL ORDER BY $sort $order LIMIT $itemsPerPage OFFSET $offset";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$items = $stmt->fetchAll();

// Für Filter-Dropdowns: Alle verfügbaren Kategorien und Standorte laden
$stmtCats = $pdo->query("SELECT DISTINCT category FROM `{$table_prefix}inventory` WHERE category IS NOT NULL AND category != '' ORDER BY category ASC");
$categories = $stmtCats->fetchAll(PDO::FETCH_COLUMN);

$stmtLocs = $pdo->query("SELECT DISTINCT location FROM `{$table_prefix}inventory` WHERE location IS NOT NULL AND location != '' ORDER BY location ASC");
$locations = $stmtLocs->fetchAll(PDO::FETCH_COLUMN);

$pageTitle = "Verwaltung";
require_once __DIR__ . '/header.php';

function thLink($col, $label, $currentSort, $currentOrder) {
    $newOrder = ($currentSort === $col && $currentOrder === 'ASC') ? 'DESC' : 'ASC';
    $arrow = ($currentSort === $col) ? ($currentOrder === 'ASC' ? '▲' : '▼') : '';
    $cls = ($currentSort === $col) ? 'text-primary' : 'text-muted';
    $url = buildUrl(['sort' => $col, 'order' => $newOrder, 'page' => 1]);
    return "<a href='$url' class='$cls'>$label $arrow</a>";
}

// Aktive Filter zählen
$activeFilters = 0;
if (!empty($search)) $activeFilters++;
if (!empty($filterCategory)) $activeFilters++;
if (!empty($filterLocation)) $activeFilters++;
if ($filterStatus !== 'active') $activeFilters++; // Standard ist 'active'
if (!empty($filterDateFrom)) $activeFilters++;
if (!empty($filterDateTo)) $activeFilters++;
?>

<div class="container">
    <form method="POST" action="<?= buildUrl() ?>" id="bulkForm">
        <?= csrfField() ?>
        <input type="hidden" name="action" value="move">
        
        <!-- Suchleiste & Bulk-Aktionen -->
        <div class="glass-panel sticky-toolbar">
            <div class="flex-between flex-wrap gap-4" style="align-items: center;">
                <div class="input-group search-group" style="display: flex; gap: 0.5rem; flex: 1; min-width: 200px;">
                    <input type="text" id="searchInput" name="dummy_q" value="<?= h($search) ?>" placeholder="Suchen (Scan)..." style="flex: 1;">
                    <button type="button" onclick="triggerSearch()" class="btn btn-primary">🔍</button>
                    <button type="button" onclick="toggleFilters()" class="btn btn-secondary" id="filterToggle">
                        🔽 Filter <?php if ($activeFilters > 0): ?><span style="background: var(--danger); color: white; padding: 2px 6px; border-radius: 10px; font-size: 0.75rem; margin-left: 4px;"><?= $activeFilters ?></span><?php endif; ?>
                    </button>
                </div>

                <div class="flex-row gap-2 flex-wrap action-group" style="align-items: center;">
                    <span class="text-muted hide-mobile" style="font-size: 0.9rem; white-space: nowrap;">Markierte:</span>
                    <div class="input-group move-group" style="display: flex; gap: 0.5rem;">
                        <input type="text" name="target_location" list="dl_locs" placeholder="Ziel-Standort..." style="min-width: 150px;">
                        <button type="submit" name="action" value="move" class="btn btn-primary">Verschieben</button>
                    </div>
                    <datalist id="dl_locs"><?php foreach($locations as $l) echo "<option value=\"".h($l)."\">"; ?></datalist>
                    <div class="input-group" style="display: flex; gap: 0.5rem;">
                        <input type="text" name="target_category" list="dl_cats" placeholder="Ziel-Kategorie..." style="min-width: 150px;">
                        <button type="submit" name="action" value="set_category" class="btn btn-secondary">Kat. setzen</button>
                    </div>
                    <datalist id="dl_cats"><?php foreach($categories as $c) echo '<option value="'.h($c).'">'; ?></datalist>
                    <button type="submit" formaction="dispose.php" formmethod="POST" class="btn btn-danger" style="height: 42px; padding: 0 1rem;" title="Markierte ausscheiden">🗑️</button>
                </div>
            </div>
        </div>

        <!-- Erweiterte Filter (aufklappbar) -->
        <div class="glass-panel" id="filterPanel" style="display: <?= $activeFilters > 0 ? 'block' : 'none' ?>; margin-bottom: 1rem;">
            <h3 style="margin-top: 0; margin-bottom: 1rem; font-size: 1rem;">🔍 Erweiterte Filter</h3>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                
                <!-- Kategorie -->
                <div class="form-group" style="margin-bottom: 0;">
                    <label style="font-size: 0.85rem;">Kategorie</label>
                    <select id="filterCategory" onchange="applyFilters()" style="width: 100%;">
                        <option value="">Alle</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?= h($cat) ?>" <?= $filterCategory === $cat ? 'selected' : '' ?>><?= h($cat) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Standort -->
                <div class="form-group" style="margin-bottom: 0;">
                    <label style="font-size: 0.85rem;">Standort</label>
                    <select id="filterLocation" onchange="applyFilters()" style="width: 100%;">
                        <option value="">Alle</option>
                        <?php foreach ($locations as $loc): ?>
                            <option value="<?= h($loc) ?>" <?= $filterLocation === $loc ? 'selected' : '' ?>><?= h($loc) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Status -->
                <div class="form-group" style="margin-bottom: 0;">
                    <label style="font-size: 0.85rem;">Status</label>
                    <select id="filterStatus" onchange="applyFilters()" style="width: 100%;">
                        <option value="active" <?= $filterStatus === 'active' ? 'selected' : '' ?>>Nur Aktive</option>
                        <option value="disposed" <?= $filterStatus === 'disposed' ? 'selected' : '' ?>>Nur Ausgeschiedene</option>
                        <option value="all" <?= $filterStatus === 'all' ? 'selected' : '' ?>>Alle</option>
                    </select>
                </div>

                <!-- Datum Von -->
                <div class="form-group" style="margin-bottom: 0;">
                    <label style="font-size: 0.85rem;">Erfasst von</label>
                    <input type="date" id="filterDateFrom" onchange="applyFilters()" value="<?= h($filterDateFrom) ?>">
                </div>

                <!-- Datum Bis -->
                <div class="form-group" style="margin-bottom: 0;">
                    <label style="font-size: 0.85rem;">Erfasst bis</label>
                    <input type="date" id="filterDateTo" onchange="applyFilters()" value="<?= h($filterDateTo) ?>">
                </div>

            </div>

            <div style="text-align: center; margin-top: 1rem;">
                <button type="button" onclick="resetFilters()" class="btn btn-secondary btn-sm">✖ Alle Filter zurücksetzen</button>
            </div>
        </div>

        <?php if ($msgSuccess) echo '<div class="alert alert-success">'.implode('<br>', $msgSuccess).'</div>'; ?>
        <?php if ($msgError) echo '<div class="alert alert-error">'.implode('<br>', $msgError).'</div>'; ?>

        <!-- Ergebnis-Info -->
        <?php if ($totalItems > 0): ?>
            <div style="padding: 0.5rem 0; display: flex; align-items: center; gap: 1rem; flex-wrap: wrap;">
                <span style="color: var(--text-muted); font-size: 0.9rem;">
                    <?= number_format($totalItems) ?> Treffer gefunden
                    <?php if ($activeFilters > 0): ?>
                        <span style="color: var(--primary);">(<?= $activeFilters ?> Filter aktiv)</span>
                    <?php endif; ?>
                </span>
                <select onchange="window.location.href='<?= buildUrl(['page' => 1]) ?>&per_page='+this.value" style="font-size:0.85rem; padding:2px 6px; height:auto; width:auto;">
                    <?php foreach ($allowedPerPage as $n): ?>
                        <option value="<?= $n ?>" <?= $itemsPerPage === $n ? 'selected' : '' ?>><?= $n ?> pro Seite</option>
                    <?php endforeach; ?>
                </select>
            </div>
        <?php endif; ?>

        <?php if ($totalItems == 0): ?>
            <div class="glass-panel text-center text-muted" style="padding: 3rem;">
                Keine Treffer gefunden.
                <?php if ($activeFilters > 0): ?>
                    <br><br>
                    <button type="button" onclick="resetFilters()" class="btn btn-secondary">Filter zurücksetzen</button>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="glass-panel" style="padding: 0;">
                <div class="table-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th class="th-check"><input type="checkbox" onclick="toggleAll(this)"></th>
                                <th><?= thLink('barcode', 'Barcode', $sort, $order) ?></th>
                                <th><?= thLink('item_name', 'Bezeichnung', $sort, $order) ?></th>
                                <th class="hide-mobile"><?= thLink('category', 'Kategorie', $sort, $order) ?></th>
                                <th><?= thLink('location', 'Standort', $sort, $order) ?></th>
                                <th class="text-right hide-mobile"><?= thLink('created_at', 'Erfasst', $sort, $order) ?></th>
                                <th style="width: 50px;"></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($items as $item): ?>
                            <tr>
                                <td class="td-check"><input type="checkbox" name="ids[]" value="<?= $item['id'] ?>"></td>
                                <td class="font-mono text-primary font-bold" onclick="copyBarcode(this)" title="Klicken zum Kopieren" style="cursor:pointer"><?= h($item['barcode']) ?></td>
                                <td style="font-weight: 500;"><?= h($item['item_name']) ?></td>
                                <td class="text-muted hide-mobile"><?= h($item['category']) ?></td>
                                <td><span style="background: var(--table-hover); padding: 2px 6px; border-radius: 4px; font-size: 0.85rem;">📍 <?= h($item['location']) ?></span></td>
                                <td class="text-right text-muted hide-mobile" style="font-size: 0.85rem;"><?= date('d.m.Y', strtotime($item['created_at'])) ?></td>
                                <td><a href="edit.php?id=<?= $item['id'] ?>" class="btn btn-secondary btn-sm" title="Bearbeiten">✏️</a></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php if ($totalPages > 1): ?>
                <div class="flex-center gap-2" style="padding: 1.5rem;">
                    <?php if ($page > 1): ?><a href="<?= buildUrl(['page' => $page - 1]) ?>" class="btn btn-secondary">&laquo;</a><?php endif; ?>
                    <span class="btn btn-secondary" style="cursor: default; border-color: var(--primary); color: var(--text-main);"><?= $page ?> / <?= $totalPages ?></span>
                    <?php if ($page < $totalPages): ?><a href="<?= buildUrl(['page' => $page + 1]) ?>" class="btn btn-secondary">&raquo;</a><?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </form>
</div>

<script>
    // Auto-Submit für Suchfeld mit 500ms Delay
    const searchInput = document.getElementById('searchInput');
    let searchTimeout = null;
    
    if (searchInput) {
        // Auto-Submit mit 500ms Delay
        searchInput.addEventListener('input', function(e) {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(function() {
                triggerSearch();
            }, 500);
        });
        
        // Enter-Taste: Sofort submitten (ohne Delay)
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                clearTimeout(searchTimeout);
                triggerSearch();
            }
        });
    }
    
    // Toggle Filter-Panel
    function toggleFilters() {
        const panel = document.getElementById('filterPanel');
        const btn = document.getElementById('filterToggle');
        if (panel.style.display === 'none') {
            panel.style.display = 'block';
            btn.textContent = btn.textContent.replace('🔽', '🔼');
        } else {
            panel.style.display = 'none';
            btn.textContent = btn.textContent.replace('🔼', '🔽');
        }
    }

    // Filter anwenden (bei Änderung)
    function applyFilters() {
        const category = document.getElementById('filterCategory').value;
        const location = document.getElementById('filterLocation').value;
        const status = document.getElementById('filterStatus').value;
        const dateFrom = document.getElementById('filterDateFrom').value;
        const dateTo = document.getElementById('filterDateTo').value;
        
        const params = new URLSearchParams(window.location.search);
        
        // Filter setzen/entfernen
        if (category) params.set('category', category); else params.delete('category');
        if (location) params.set('location', location); else params.delete('location');
        if (status !== 'active') params.set('status', status); else params.delete('status'); // Standard ist 'active'
        if (dateFrom) params.set('date_from', dateFrom); else params.delete('date_from');
        if (dateTo) params.set('date_to', dateTo); else params.delete('date_to');
        
        // Zurück auf Seite 1
        params.set('page', '1');
        
        // Navigieren
        window.location.href = '?' + params.toString();
    }

    // Alle Filter zurücksetzen
    function resetFilters() {
        window.location.href = 'list.php';
    }

    // Suche auslösen
    function triggerSearch() {
        const searchValue = cleanBarcode(document.getElementById('searchInput').value);
        const params = new URLSearchParams(window.location.search);
        if (searchValue) { params.set('q', searchValue); } else { params.delete('q'); }
        params.set('page', '1');
        window.location.href = '?' + params.toString();
    }

    // Alle Checkboxen togglen
    function toggleAll(source) {
        const checkboxes = document.querySelectorAll('input[name="ids[]"]');
        checkboxes.forEach(cb => cb.checked = source.checked);
    }

    // Filter-Panel bei aktiven Filtern öffnen
    <?php if ($activeFilters > 0): ?>
    document.addEventListener('DOMContentLoaded', function() {
        document.getElementById('filterToggle').textContent = document.getElementById('filterToggle').textContent.replace('🔽', '🔼');
    });
    <?php endif; ?>
</script>

<?php require_once __DIR__ . '/footer.php'; ?>
