<?php
// core/dashboard.php
ob_start();
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

// ========================================
// STATISTIKEN LADEN
// ========================================

// 1. Gesamt-Statistiken
$statsQuery = "SELECT 
    COUNT(*) as total,
    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
    SUM(CASE WHEN status = 'disposed' THEN 1 ELSE 0 END) as disposed
FROM `{$table_prefix}inventory`";
$stats = $pdo->query($statsQuery)->fetch();

// 2. Top 5 Standorte
$topLocationsQuery = "SELECT location, COUNT(*) as count 
FROM `{$table_prefix}inventory` 
WHERE status = 'active' AND location IS NOT NULL AND location != ''
GROUP BY location 
ORDER BY count DESC 
LIMIT 5";
$topLocations = $pdo->query($topLocationsQuery)->fetchAll();

// 3. Top 5 Kategorien
$topCategoriesQuery = "SELECT category, COUNT(*) as count 
FROM `{$table_prefix}inventory` 
WHERE status = 'active' AND category IS NOT NULL AND category != ''
GROUP BY category 
ORDER BY count DESC 
LIMIT 5";
$topCategories = $pdo->query($topCategoriesQuery)->fetchAll();

// 4. Letzte Aktivitäten (5 neueste History-Einträge)
$recentActivityQuery = "SELECT h.*, i.barcode, i.item_name 
FROM `{$table_prefix}history` h
LEFT JOIN `{$table_prefix}inventory` i ON h.inventory_id = i.id
ORDER BY h.created_at DESC 
LIMIT 5";
$recentActivity = $pdo->query($recentActivityQuery)->fetchAll();

// 5. Neueste Artikel (Letzte 5)
$newestItemsQuery = "SELECT barcode, item_name, category, location, created_at 
FROM `{$table_prefix}inventory` 
WHERE status = 'active'
ORDER BY created_at DESC 
LIMIT 5";
$newestItems = $pdo->query($newestItemsQuery)->fetchAll();

// 6. Letzte Scans (Create-Aktionen aus History)
$lastScansQuery = "SELECT h.created_at, h.username, i.barcode, i.item_name
FROM `{$table_prefix}history` h
LEFT JOIN `{$table_prefix}inventory` i ON h.inventory_id = i.id
WHERE h.action = 'create'
ORDER BY h.created_at DESC
LIMIT 8";
$lastScans = $pdo->query($lastScansQuery)->fetchAll();

// 7. Artikel nach Monat (für Chart - letzte 6 Monate)
$monthlyStatsQuery = "SELECT 
    DATE_FORMAT(created_at, '%Y-%m') as month,
    COUNT(*) as count
FROM `{$table_prefix}inventory`
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
GROUP BY month
ORDER BY month ASC";
$monthlyStats = $pdo->query($monthlyStatsQuery)->fetchAll();

// Hilfsfunktion für Action-Icons
function getActionIcon($action) {
    $icons = [
        'create' => '➕',
        'move' => '🚚',
        'dispose' => '🗑️',
        'edit' => '✏️',
        'lost_inventory' => '❓',
    ];
    return $icons[$action] ?? '📝';
}

// Hilfsfunktion für Action-Text
function getActionText($action) {
    $texts = [
        'create' => 'Erfasst',
        'move' => 'Verschoben',
        'dispose' => 'Ausgeschieden',
        'edit' => 'Bearbeitet',
        'lost_inventory' => 'Als verloren markiert',
    ];
    return $texts[$action] ?? 'Aktion';
}

// Hilfsfunktion für relative Zeit
function timeAgo($datetime) {
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) return 'gerade eben';
    if ($diff < 3600) return floor($diff / 60) . ' Min';
    if ($diff < 86400) return floor($diff / 3600) . ' Std';
    if ($diff < 604800) return floor($diff / 86400) . ' Tag' . (floor($diff / 86400) > 1 ? 'e' : '');
    return date('d.m.Y', $timestamp);
}

$pageTitle = "Dashboard"; 
require_once __DIR__ . '/header.php';
?>

<!-- Dashboard-spezifische Styles -->
<style>
    /* iPad Hochformat: Statistik-Karten 2-2 Layout */
    @media (min-width: 768px) and (max-width: 1024px) {
        .stats-grid {
            grid-template-columns: repeat(2, 1fr) !important;
        }
    }
    
    /* Handy Hochformat: Schnellzugriff einspaltig */
    @media (max-width: 600px) {
        .dashboard-grid {
            grid-template-columns: 1fr !important;
        }
    }
</style>

<div class="container">
    
    <!-- Quick Actions - DIREKT ohne Header -->
    <div class="glass-panel" style="margin-bottom: 1rem;">
        <h3 style="margin-top: 0; margin-bottom: 0.75rem;">⚡ Schnellzugriff</h3>
        <div class="dashboard-grid">
            
            <a href="scan.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon text-success">📷</span>
                <div class="portal-title">Erfassen</div>
                <div class="text-muted">Neues Inventar</div>
            </a>

            <a href="move.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon text-warning">🚚</span>
                <div class="portal-title">Verschieben</div>
                <div class="text-muted">Standort ändern</div>
            </a>

            <a href="dispose.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon text-danger">🗑️</span>
                <div class="portal-title">Ausscheiden</div>
                <div class="text-muted">Defekt/Verkauft</div>
            </a>

            <a href="inventur.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon text-info">📋</span>
                <div class="portal-title">Inventur</div>
                <div class="text-muted">Bestand prüfen</div>
            </a>

            <a href="list.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon text-primary">🔍</span>
                <div class="portal-title">Suchen</div>
                <div class="text-muted">Liste & Filter</div>
            </a>

            <a href="report.php" class="glass-panel portal-card" style="text-decoration: none;">
                <span class="portal-icon" style="color: #a855f7;">📊</span>
                <div class="portal-title">Berichte</div>
                <div class="text-muted">Logbuch</div>
            </a>

        </div>
    </div>
    
    <!-- Statistik-Karten -->
    <div class="stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
        
        <!-- Gesamt -->
        <div class="glass-panel text-center" style="padding: 1.5rem;">
            <div style="font-size: 2.5rem; font-weight: bold; color: var(--primary);">
                <?= number_format($stats['total']) ?>
            </div>
            <div class="text-muted" style="font-size: 0.9rem;">Gesamt</div>
        </div>
        
        <!-- Aktiv -->
        <div class="glass-panel text-center" style="padding: 1.5rem;">
            <div style="font-size: 2.5rem; font-weight: bold; color: var(--success);">
                <?= number_format($stats['active']) ?>
            </div>
            <div class="text-muted" style="font-size: 0.9rem;">Aktiv</div>
        </div>
        
        <!-- Ausgeschieden -->
        <div class="glass-panel text-center" style="padding: 1.5rem;">
            <div style="font-size: 2.5rem; font-weight: bold; color: var(--danger);">
                <?= number_format($stats['disposed']) ?>
            </div>
            <div class="text-muted" style="font-size: 0.9rem;">Ausgeschieden</div>
        </div>
        
        <!-- Prozent Aktiv -->
        <div class="glass-panel text-center" style="padding: 1.5rem;">
            <?php $percentActive = $stats['total'] > 0 ? round(($stats['active'] / $stats['total']) * 100) : 0; ?>
            <div style="font-size: 2.5rem; font-weight: bold; color: var(--info);">
                <?= $percentActive ?>%
            </div>
            <div class="text-muted" style="font-size: 0.9rem;">Aktiv-Rate</div>
        </div>
        
    </div>
    
    <!-- Zwei-Spalten Layout -->
    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
        
        <!-- Top Standorte -->
        <div class="glass-panel">
            <h3 style="margin-top: 0; margin-bottom: 1rem; color: var(--primary);">📍 Top Standorte</h3>
            <?php if (count($topLocations) > 0): ?>
                <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                    <?php foreach ($topLocations as $loc): ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.5rem; background: var(--glass-bg-hover); border-radius: 6px;">
                            <span style="font-weight: 500;"><?= h($loc['location']) ?></span>
                            <span style="background: var(--primary); color: white; padding: 2px 8px; border-radius: 12px; font-size: 0.85rem; font-weight: 600;">
                                <?= $loc['count'] ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="text-muted" style="text-align: center;">Noch keine Daten</p>
            <?php endif; ?>
        </div>
        
        <!-- Top Kategorien -->
        <div class="glass-panel">
            <h3 style="margin-top: 0; margin-bottom: 1rem; color: var(--success);">🏷️ Top Kategorien</h3>
            <?php if (count($topCategories) > 0): ?>
                <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                    <?php foreach ($topCategories as $cat): ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.5rem; background: var(--glass-bg-hover); border-radius: 6px;">
                            <span style="font-weight: 500;"><?= h($cat['category']) ?></span>
                            <span style="background: var(--success); color: white; padding: 2px 8px; border-radius: 12px; font-size: 0.85rem; font-weight: 600;">
                                <?= $cat['count'] ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="text-muted" style="text-align: center;">Noch keine Daten</p>
            <?php endif; ?>
        </div>
        
    </div>
    
    <!-- Letzte Aktivitäten & Neueste Artikel -->
    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
        
        <!-- Letzte Aktivitäten -->
        <div class="glass-panel">
            <h3 style="margin-top: 0; margin-bottom: 1rem; color: var(--info);">🕐 Letzte Aktivitäten</h3>
            <?php if (count($recentActivity) > 0): ?>
                <div style="display: flex; flex-direction: column; gap: 0.75rem;">
                    <?php foreach ($recentActivity as $activity): ?>
                        <div style="display: flex; gap: 0.75rem; padding: 0.5rem; border-left: 3px solid var(--glass-border); padding-left: 0.75rem;">
                            <span style="font-size: 1.2rem;"><?= getActionIcon($activity['action']) ?></span>
                            <div style="flex: 1; min-width: 0;">
                                <div style="font-weight: 500; font-size: 0.9rem;">
                                    <?= getActionText($activity['action']) ?>
                                    <?php if ($activity['barcode']): ?>
                                        <span class="font-mono text-primary">#<?= h($activity['barcode']) ?></span>
                                    <?php endif; ?>
                                </div>
                                <?php if ($activity['item_name']): ?>
                                    <div class="text-muted" style="font-size: 0.85rem; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                                        <?= h($activity['item_name']) ?>
                                    </div>
                                <?php endif; ?>
                                <div class="text-muted" style="font-size: 0.75rem;">
                                    <?= h($activity['username']) ?> • <?= timeAgo($activity['created_at']) ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div style="text-align: center; margin-top: 1rem;">
                    <a href="report.php" class="btn btn-secondary btn-sm">Alle anzeigen →</a>
                </div>
            <?php else: ?>
                <p class="text-muted" style="text-align: center;">Noch keine Aktivitäten</p>
            <?php endif; ?>
        </div>
        
        <!-- Neueste Artikel -->
        <div class="glass-panel">
            <h3 style="margin-top: 0; margin-bottom: 1rem; color: var(--warning);">✨ Zuletzt erfasst</h3>
            <?php if (count($newestItems) > 0): ?>
                <div style="display: flex; flex-direction: column; gap: 0.75rem;">
                    <?php foreach ($newestItems as $item): ?>
                        <div style="padding: 0.5rem; border-left: 3px solid var(--warning); padding-left: 0.75rem;">
                            <div style="font-weight: 500; font-size: 0.9rem;">
                                <span class="font-mono text-primary">#<?= h($item['barcode']) ?></span>
                            </div>
                            <div style="font-size: 0.85rem; margin: 0.25rem 0;">
                                <?= h($item['item_name']) ?>
                            </div>
                            <div class="text-muted" style="font-size: 0.75rem;">
                                <?= h($item['category']) ?> • <?= h($item['location']) ?> • <?= timeAgo($item['created_at']) ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div style="text-align: center; margin-top: 1rem;">
                    <a href="list.php" class="btn btn-secondary btn-sm">Alle anzeigen →</a>
                </div>
            <?php else: ?>
                <p class="text-muted" style="text-align: center;">Noch keine Artikel</p>
            <?php endif; ?>
        </div>
        
    </div>

    <!-- Letzte Scans -->
    <div class="glass-panel" style="margin-bottom: 1rem;">
        <h3 style="margin-top: 0; margin-bottom: 1rem; color: var(--success);">📷 Letzte Scans</h3>
        <?php if (count($lastScans) > 0): ?>
            <div style="display: flex; flex-direction: column; gap: 0.4rem;">
                <?php foreach ($lastScans as $scan): ?>
                    <div style="display: flex; align-items: center; gap: 0.75rem; padding: 0.4rem 0.75rem; background: var(--glass-bg-hover); border-radius: 6px; font-size: 0.85rem;">
                        <span class="font-mono text-primary font-bold" onclick="copyBarcode(this)" title="Klicken zum Kopieren" style="cursor:pointer; flex-shrink:0;"><?= h($scan['barcode'] ?? '') ?></span>
                        <span style="flex: 1; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;"><?= h($scan['item_name'] ?? '') ?></span>
                        <span class="text-muted" style="flex-shrink:0;"><?= h($scan['username']) ?> • <?= timeAgo($scan['created_at']) ?></span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <p class="text-muted text-center">Noch keine Scans</p>
        <?php endif; ?>
    </div>

</div>

<?php require_once __DIR__ . '/footer.php'; ?>
