<?php
/**
 * update_apply.php – Update installieren (BIB)
 *
 * POST-Endpoint. Erwartet:
 *   csrf_token     – CSRF-Token
 *   target_version – Zielversion (z.B. "2.2")
 *
 * Gibt JSON zurück:
 *   success  – bool
 *   steps    – array mit Fortschrittsschritten
 *   error    – Fehlermeldung (nur bei Misserfolg)
 */

header('Content-Type: application/json; charset=utf-8');
set_time_limit(300);
ini_set('memory_limit', '256M');

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

// Auth-Prüfung
if (!is_admin()) {
    http_response_code(403);
    echo json_encode(['error' => 'Nicht autorisiert']);
    exit;
}

// Nur POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Nur POST erlaubt']);
    exit;
}

// CSRF prüfen
if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
    http_response_code(400);
    echo json_encode(['error' => 'Sicherheitsfehler (CSRF). Bitte Seite neu laden.']);
    exit;
}

// Zielversion validieren
$target_version = trim($_POST['target_version'] ?? '');
if (!preg_match('/^\d+\.\d+(\.\d+)?$/', $target_version)) {
    http_response_code(400);
    echo json_encode(['error' => 'Ungültige Version angegeben']);
    exit;
}

require_once __DIR__ . '/Updater.php';

try {
    global $pdo;

    $manifest = Updater::fetchManifest();
    if (!$manifest) {
        echo json_encode(['success' => false, 'steps' => [], 'error' => 'Manifest nicht abrufbar']);
        exit;
    }

    $entry = null;
    foreach ($manifest['versions'] as $v) {
        if (($v['version'] ?? '') === $target_version) {
            $entry = $v;
            break;
        }
    }

    if (!$entry) {
        echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version nicht im Manifest gefunden"]);
        exit;
    }

    $installed = Updater::getInstalledVersion($pdo);
    if (version_compare($target_version, $installed, '<=')) {
        echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version ist nicht neuer als die installierte Version $installed"]);
        exit;
    }

    $result = Updater::applyUpdate($entry, $pdo);
    echo json_encode($result);

} catch (Throwable $e) {
    Updater::log('FEHLER in update_apply.php: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'steps' => [], 'error' => 'Interner Fehler: ' . $e->getMessage()]);
}
