<?php
/**
 * Updater.php – Auto-Update Engine für BIB (Bibliotheksverwaltung)
 *
 * Prüft auf neue Versionen unter update.it-wil.at/bib,
 * lädt das Update-ZIP herunter und spielt es ein.
 *
 * Bewahrte Dateien (werden NICHT überschrieben):
 *   - includes/config.php
 *   - uploads/
 *   - backups/
 *   - vendor/
 */

define('BIB_APP_ID',       'bib');
define('BIB_UPDATE_SERVER', 'https://update.it-wil.at/bib');
define('BIB_APP_ROOT',      dirname(__DIR__));         // /bib
define('BIB_UPDATE_DIR',    __DIR__);                  // /bib/update
define('BIB_UPDATE_LOG',    __DIR__ . '/update.log');
define('BIB_UPDATE_LOCK',   __DIR__ . '/update.lock');
define('BIB_CACHE_TTL',     3600);                     // Manifest 1 Std. cachen

class Updater
{
    // ── Öffentliche API ───────────────────────────────────────────────────────

    /**
     * Manifest (versions.json) vom Update-Server holen.
     * Ergebnis wird bis BIB_CACHE_TTL Sekunden gecacht.
     */
    public static function fetchManifest(): ?array
    {
        $cache = BIB_UPDATE_DIR . '/manifest_cache.json';
        if (file_exists($cache) && (time() - filemtime($cache)) < BIB_CACHE_TTL) {
            $data = json_decode(file_get_contents($cache), true);
            if (is_array($data)) return $data;
        }

        $url  = BIB_UPDATE_SERVER . '/versions.json';
        $json = @file_get_contents($url, false, self::httpContext());
        if ($json === false) return null;

        $data = json_decode($json, true);
        if (!is_array($data)) return null;

        file_put_contents($cache, $json, LOCK_EX);
        return $data;
    }

    /**
     * Neueste verfügbare Version ermitteln, die neuer als $current ist.
     */
    public static function getAvailableUpdate(string $current): ?array
    {
        $manifest = self::fetchManifest();
        if (!$manifest || empty($manifest['versions'])) return null;

        $best = null;
        foreach ($manifest['versions'] as $entry) {
            if (!isset($entry['version'])) continue;
            if (!preg_match('/^\d+\.\d+(\.\d+)?$/', $entry['version'])) continue;

            if (version_compare($entry['version'], $current, '>')) {
                if ($best === null || version_compare($entry['version'], $best['version'], '>')) {
                    if (!empty($entry['min_php']) && version_compare(PHP_VERSION, $entry['min_php'], '<')) {
                        continue;
                    }
                    $best = $entry;
                }
            }
        }
        return $best;
    }

    /**
     * Update-Prozess ausführen.
     *
     * Schritte:
     *  1. Lock-Datei belegen
     *  2. Automatisches DB-Backup (backup_pdo)
     *  3. ZIP herunterladen + SHA-256 prüfen
     *  4. In Temp-Dir entpacken
     *  5. Dateien kopieren (bewahrte Pfade überspringen)
     *  6. Migration-SQL ausführen (optional)
     *  7. Neue Version in settings-Tabelle speichern
     *  8. Lock freigeben + Cache löschen
     *
     * @param array $version_entry Versions-Eintrag aus dem Manifest
     * @param PDO   $pdo           Datenbankverbindung
     * @return array ['success'=>bool, 'steps'=>string[], 'error'=>?string]
     */
    public static function applyUpdate(array $version_entry, PDO $pdo): array
    {
        $steps = [];
        $lock  = null;

        $lock = fopen(BIB_UPDATE_LOCK, 'w');
        if (!$lock || !flock($lock, LOCK_EX | LOCK_NB)) {
            return ['success' => false, 'steps' => $steps, 'error' => 'Update läuft bereits (Lock aktiv)'];
        }

        try {
            $ver = $version_entry['version'];
            self::log("=== Update auf v{$ver} gestartet ===");

            // 2. Automatisches DB-Backup vor dem Update
            $backup_dir = BIB_APP_ROOT . '/backups/';
            if (!is_dir($backup_dir)) mkdir($backup_dir, 0750, true);
            $backup_file = $backup_dir . 'pre_update_' . $ver . '_' . date('Y-m-d_H-i-s') . '.sql.gz';

            self::log("Erstelle Pre-Update-Backup…");
            $backup_result = self::backupPdo($backup_file, true, $pdo);
            if ($backup_result['success']) {
                $steps[] = "DB-Backup erstellt: " . basename($backup_file);
                self::log("Backup OK: " . basename($backup_file));
            } else {
                self::log("Backup-Warnung: " . ($backup_result['error'] ?? 'unbekannt'));
                $steps[] = "Backup-Warnung (Update wird trotzdem fortgesetzt)";
            }

            // 3. ZIP herunterladen
            $steps[] = "Lade ZIP herunter…";
            self::log("Lade ZIP: " . $version_entry['zip_url']);
            $zip_data = @file_get_contents($version_entry['zip_url'], false, self::httpContext());
            if ($zip_data === false) {
                throw new RuntimeException('ZIP konnte nicht heruntergeladen werden');
            }

            // SHA-256 prüfen
            $actual_hash = hash('sha256', $zip_data);
            if (!hash_equals(strtolower($version_entry['zip_sha256']), $actual_hash)) {
                throw new RuntimeException("ZIP-Prüfsumme ungültig");
            }
            $steps[] = "ZIP-Integrität OK (" . strlen($zip_data) . " Bytes)";
            self::log("ZIP-Prüfsumme OK");

            $tmp_zip = sys_get_temp_dir() . '/bib_update_' . time() . '.zip';
            file_put_contents($tmp_zip, $zip_data);
            unset($zip_data);

            // 4. Entpacken
            $tmp_dir = sys_get_temp_dir() . '/bib_update_extract_' . time();
            mkdir($tmp_dir, 0755, true);

            $zip = new ZipArchive();
            if ($zip->open($tmp_zip) !== true) {
                throw new RuntimeException('ZIP konnte nicht geöffnet werden');
            }
            $zip->extractTo($tmp_dir);
            $zip->close();
            @unlink($tmp_zip);
            $steps[] = "ZIP entpackt";
            self::log("ZIP entpackt nach: $tmp_dir");

            // 5. Dateien kopieren
            $preserved = self::getPreservedPaths();
            $copied    = self::copyFiles($tmp_dir, BIB_APP_ROOT, $preserved);
            $steps[] = "Dateien kopiert ($copied Dateien)";
            self::log("$copied Dateien kopiert");

            self::rmdir_recursive($tmp_dir);

            // 6. Migration ausführen (falls nötig)
            if (!empty($version_entry['requires_db_migration']) && !empty($version_entry['migration_url'])) {
                $steps[] = "Lade Migration-SQL…";
                self::log("Lade Migration: " . $version_entry['migration_url']);
                $sql_data = @file_get_contents($version_entry['migration_url'], false, self::httpContext());
                if ($sql_data === false) {
                    throw new RuntimeException('Migration-SQL konnte nicht heruntergeladen werden');
                }

                if (!empty($version_entry['migration_sha256'])) {
                    $sql_hash = hash('sha256', $sql_data);
                    if (!hash_equals(strtolower($version_entry['migration_sha256']), $sql_hash)) {
                        throw new RuntimeException('Migration-SQL Prüfsumme ungültig');
                    }
                }

                $mig_result = self::runMigration($sql_data, $pdo);
                self::log("Migration: {$mig_result['executed']} Statements");
                if (!empty($mig_result['errors'])) {
                    self::log("Migrationsfehler: " . implode('; ', $mig_result['errors']));
                }
                $steps[] = "Migration ausgeführt ({$mig_result['executed']} Statements)";
            }

            // 7. Version speichern
            self::saveInstalledVersion($ver, $pdo);
            $steps[] = "Version v{$ver} gespeichert";
            self::log("Version gespeichert: v{$ver}");

            @unlink(BIB_UPDATE_DIR . '/manifest_cache.json');

            self::log("=== Update auf v{$ver} erfolgreich ===");
            return ['success' => true, 'steps' => $steps, 'error' => null];

        } catch (Throwable $e) {
            $msg = $e->getMessage();
            self::log("FEHLER: $msg");
            return ['success' => false, 'steps' => $steps, 'error' => $msg];
        } finally {
            if ($lock) {
                flock($lock, LOCK_UN);
                fclose($lock);
                @unlink(BIB_UPDATE_LOCK);
            }
        }
    }

    /**
     * Bewahrte Pfade – werden beim Update NICHT überschrieben.
     */
    public static function getPreservedPaths(): array
    {
        return [
            BIB_APP_ROOT . '/includes/config.php',
            BIB_APP_ROOT . '/uploads',
            BIB_APP_ROOT . '/backups',
            BIB_APP_ROOT . '/vendor',
        ];
    }

    /**
     * Installierte Version aus der settings-Tabelle lesen.
     */
    public static function getInstalledVersion(PDO $pdo): string
    {
        try {
            $val = $pdo->query("SELECT setting_value FROM settings WHERE setting_key = 'app_version' LIMIT 1")->fetchColumn();
            return $val ?: '0.0';
        } catch (Throwable $e) {
            return '0.0';
        }
    }

    /**
     * Neue Version in der settings-Tabelle speichern.
     */
    public static function saveInstalledVersion(string $version, PDO $pdo): void
    {
        $pdo->prepare(
            "INSERT INTO settings (setting_key, setting_value) VALUES ('app_version', ?)
             ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)"
        )->execute([$version]);

        $pdo->prepare(
            "INSERT INTO settings (setting_key, setting_value) VALUES ('last_update_at', ?)
             ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)"
        )->execute([date('Y-m-d H:i:s')]);
    }

    /**
     * Zeile ans Update-Log anhängen (mit Auto-Rotation).
     */
    public static function log(string $msg): void
    {
        $line = '[' . date('Y-m-d H:i:s') . '] ' . $msg . PHP_EOL;

        if (file_exists(BIB_UPDATE_LOG) && filesize(BIB_UPDATE_LOG) > 262144) {
            $lines = file(BIB_UPDATE_LOG, FILE_IGNORE_NEW_LINES);
            if ($lines !== false) {
                file_put_contents(BIB_UPDATE_LOG, implode(PHP_EOL, array_slice($lines, -300)) . PHP_EOL);
            }
        }

        file_put_contents(BIB_UPDATE_LOG, $line, FILE_APPEND | LOCK_EX);
    }

    /**
     * Letzte $n Zeilen des Update-Logs lesen.
     */
    public static function readLog(int $n = 50): array
    {
        if (!file_exists(BIB_UPDATE_LOG)) return [];
        $lines = file(BIB_UPDATE_LOG, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!$lines) return [];
        return array_slice($lines, -$n);
    }

    // ── Private Hilfsmethoden ─────────────────────────────────────────────────

    /**
     * Dateien rekursiv kopieren, bewahrte Pfade überspringen.
     */
    private static function copyFiles(string $src, string $dst, array $preserved): int
    {
        $count   = 0;
        $root_re = realpath($dst);

        $iter = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($src, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iter as $item) {
            $rel  = ltrim(substr($item->getPathname(), strlen($src)), DIRECTORY_SEPARATOR);
            $dest = $dst . DIRECTORY_SEPARATOR . $rel;

            $real_dest_dir = realpath(dirname($dest)) ?: dirname($dest);
            if ($root_re && strpos($real_dest_dir, $root_re) !== 0) {
                continue;
            }

            if ($item->isDir()) {
                if (!is_dir($dest)) mkdir($dest, 0755, true);
                continue;
            }

            $real_dest = realpath($dest) ?: $dest;
            $skip = false;
            foreach ($preserved as $p) {
                $rp = realpath($p) ?: $p;
                if ($real_dest === $rp || strpos($real_dest . '/', rtrim($rp, '/\\') . '/') === 0) {
                    $skip = true;
                    break;
                }
            }
            if ($skip) continue;

            if (!is_dir(dirname($dest))) mkdir(dirname($dest), 0755, true);
            copy($item->getPathname(), $dest);
            $count++;
        }

        return $count;
    }

    /**
     * SQL-Migration ausführen.
     */
    private static function runMigration(string $sql, PDO $pdo): array
    {
        $stmts = []; $current = ''; $in_str = false; $str_chr = '';
        $len = strlen($sql);
        for ($i = 0; $i < $len; $i++) {
            $c = $sql[$i]; $prev = $i > 0 ? $sql[$i-1] : '';
            if (!$in_str && ($c === "'" || $c === '"' || $c === '`')) { $in_str = true; $str_chr = $c; }
            elseif ($in_str && $c === $str_chr && $prev !== '\\')      { $in_str = false; }
            if (!$in_str && $c === ';') {
                $s = trim($current);
                if ($s !== '' && !str_starts_with($s, '--') && !str_starts_with($s, '/*')) {
                    $stmts[] = $s;
                }
                $current = '';
            } else { $current .= $c; }
        }

        $executed = 0; $errors = [];
        $pdo->exec('SET FOREIGN_KEY_CHECKS = 0');
        foreach ($stmts as $stmt) {
            if (preg_match('/^\s*(--|\/\*)/s', $stmt)) continue;
            try { $pdo->exec($stmt); $executed++; }
            catch (PDOException $e) {
                $errors[] = substr($stmt, 0, 60) . '… → ' . $e->getMessage();
                if (count($errors) >= 10) { $errors[] = '(weitere Fehler abgeschnitten)'; break; }
            }
        }
        $pdo->exec('SET FOREIGN_KEY_CHECKS = 1');

        return ['executed' => $executed, 'errors' => $errors];
    }

    /**
     * DB-Backup über PDO erstellen (komprimiert oder unkomprimiert).
     * Vereinfachte Version basierend auf settings-general.php backup_pdo().
     */
    private static function backupPdo(string $filepath, bool $compress, PDO $pdo): array
    {
        try {
            $fh = $compress ? gzopen($filepath, 'wb9') : fopen($filepath, 'wb');
            if (!$fh) return ['success' => false, 'error' => 'Datei konnte nicht erstellt werden'];
            $w = fn(string $s) => $compress ? gzwrite($fh, $s) : fwrite($fh, $s);

            $w("-- BIB Pre-Update Backup\n-- Erstellt: " . date('Y-m-d H:i:s') . "\n\nSET NAMES 'utf8mb4';\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

            $tables = $pdo->query("SELECT table_name FROM information_schema.tables WHERE table_schema = DATABASE() AND table_type = 'BASE TABLE' ORDER BY table_name")->fetchAll(PDO::FETCH_COLUMN);
            foreach ($tables as $table) {
                $create = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(PDO::FETCH_NUM);
                $w("DROP TABLE IF EXISTS `$table`;\n" . $create[1] . ";\n\n");
                $total = (int)$pdo->query("SELECT COUNT(*) FROM `$table`")->fetchColumn();
                if ($total > 0) {
                    $chunk = 500; $offset = 0;
                    $cols = array_map(fn($c) => "`{$c['Field']}`", $pdo->query("SHOW COLUMNS FROM `$table`")->fetchAll(PDO::FETCH_ASSOC));
                    while ($offset < $total) {
                        $rows = $pdo->query("SELECT * FROM `$table` LIMIT $chunk OFFSET $offset")->fetchAll(PDO::FETCH_NUM);
                        if (empty($rows)) break;
                        $vals = [];
                        foreach ($rows as $row) {
                            $vals[] = '(' . implode(', ', array_map(fn($v) => $v === null ? 'NULL' : $pdo->quote($v), $row)) . ')';
                        }
                        $w("INSERT INTO `$table` (" . implode(', ', $cols) . ") VALUES\n" . implode(",\n", $vals) . ";\n");
                        $offset += $chunk;
                    }
                    $w("\n");
                }
            }
            $w("SET FOREIGN_KEY_CHECKS = 1;\n");
            $compress ? gzclose($fh) : fclose($fh);

            if (!file_exists($filepath) || filesize($filepath) < 50) {
                @unlink($filepath);
                return ['success' => false, 'error' => 'Backup zu klein'];
            }
            return ['success' => true, 'size' => filesize($filepath)];
        } catch (Throwable $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * HTTP-Stream-Context für Downloads.
     * @return resource
     */
    private static function httpContext()
    {
        return stream_context_create([
            'http' => [
                'timeout'       => 60,
                'user_agent'    => 'it-wil-updater/1.0 (bib)',
                'ignore_errors' => false,
            ],
            'ssl' => [
                'verify_peer'      => true,
                'verify_peer_name' => true,
            ],
        ]);
    }

    /**
     * Verzeichnis rekursiv löschen.
     */
    private static function rmdir_recursive(string $dir): void
    {
        if (!is_dir($dir)) return;
        $items = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ($items as $item) {
            $item->isDir() ? rmdir($item->getPathname()) : unlink($item->getPathname());
        }
        rmdir($dir);
    }
}
