<?php
/**
 * Buchsuche — search.php
 * LIKE-Suche auf Titel, Untertitel, Autor, ISBN, Schlagworte.
 * Filter: Sprache, Unterkategorie, nur verfügbare Exemplare.
 * Pagination: 20 Ergebnisse pro Seite.
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/session.php';

require_login();

$page_title   = 'Buchsuche';
$current_user = get_logged_in_user();
$lang         = load_language();

// ── Parameter ────────────────────────────────────────────────────────────────
$q        = trim($_GET['q']    ?? '');
$lang_f   = trim($_GET['lang'] ?? '');
$cat_f    = (int)($_GET['cat'] ?? 0);
$avail    = isset($_GET['avail']) && $_GET['avail'] === '1';
$page     = max(1, (int)($_GET['page'] ?? 1));
$per_page = 20;

// ── WHERE-Klausel aufbauen ────────────────────────────────────────────────────
$where  = [];
$params = [];

if ($q !== '') {
    $like    = '%' . $q . '%';
    $where[] = '(b.title LIKE ? OR b.subtitle LIKE ? OR b.author LIKE ? OR b.isbn = ? OR b.keywords LIKE ?)';
    array_push($params, $like, $like, $like, $q, $like);
}
if ($lang_f !== '') {
    $where[] = 'b.language = ?';
    $params[] = $lang_f;
}
if ($cat_f > 0) {
    $where[] = 'b.subcategory_id = ?';
    $params[] = $cat_f;
}

$where_sql  = $where ? 'WHERE ' . implode(' AND ', $where) : '';
$having_sql = $avail ? "HAVING available_copies > 0" : '';

// ── Gesamtanzahl (Subquery-Wrapper für HAVING) ────────────────────────────────
$total = (int)db_get_var(
    "SELECT COUNT(*) FROM (
        SELECT b.id,
               COALESCE(SUM(bc.status = 'verfügbar'), 0) AS available_copies
        FROM books b
        LEFT JOIN book_copies bc ON b.id = bc.book_id
        $where_sql
        GROUP BY b.id
        $having_sql
    ) _c",
    $params
);

$total_pages = max(1, (int)ceil($total / $per_page));
$page        = min($page, $total_pages);
$offset      = ($page - 1) * $per_page;

// ── Ergebnisse ────────────────────────────────────────────────────────────────
$books = db_get_results(
    "SELECT b.id, b.title, b.subtitle, b.author, b.isbn,
            b.image_url, b.year, b.language,
            bs.code AS subcategory_code, bs.name AS subcategory_name,
            COALESCE(SUM(bc.status = 'verfügbar'), 0)
                AS available_copies,
            COUNT(CASE WHEN bc.status NOT IN ('aussortiert','verloren') THEN 1 END)
                AS active_copies
     FROM books b
     LEFT JOIN book_copies bc ON b.id = bc.book_id
     LEFT JOIN book_subcategories bs ON b.subcategory_id = bs.id
     $where_sql
     GROUP BY b.id, b.title, b.subtitle, b.author, b.isbn,
              b.image_url, b.year, b.language, bs.code, bs.name
     $having_sql
     ORDER BY b.title_sort ASC, b.title ASC
     LIMIT $per_page OFFSET $offset",
    $params
);

// ── Filter-Optionen ───────────────────────────────────────────────────────────
$languages = db_get_results(
    "SELECT DISTINCT language FROM books
     WHERE language IS NOT NULL AND language != ''
     ORDER BY language"
);

$subcategories = db_get_results(
    "SELECT bs.id, bs.code, bs.name, bc.name AS category_name
     FROM book_subcategories bs
     JOIN book_categories bc ON bs.category_id = bc.id
     WHERE bs.active = 1
     ORDER BY bc.sort_order, bs.sort_order, bs.code"
);

// ── Hilfsfunktion: Paginierungs-URL ──────────────────────────────────────────
$get_base = array_filter([
    'q'    => $q,
    'lang' => $lang_f,
    'cat'  => $cat_f > 0 ? $cat_f : '',
    'avail'=> $avail ? '1' : '',
]);

$pag_url = function(int $p) use ($get_base): string {
    return BASE_URL . '/search.php?' . http_build_query(array_merge($get_base, ['page' => $p]));
};

$has_filter = ($q !== '' || $lang_f !== '' || $cat_f > 0 || $avail);

include __DIR__ . '/includes/header.php';
?>

<!-- Breadcrumb -->
<nav class="text-sm text-gray-500 mb-6" aria-label="Breadcrumb">
    <a href="<?= BASE_URL ?>/dashboard.php" class="hover:text-blue-600">Dashboard</a>
    <span class="mx-2">/</span>
    <span class="text-gray-900">Buchsuche</span>
</nav>

<!-- Search & Filter Card -->
<div class="bg-white rounded-lg shadow p-6 mb-6">
    <form method="GET" action="<?= BASE_URL ?>/search.php" id="searchForm">

        <!-- Search bar -->
        <div class="flex gap-3 mb-4">
            <input type="text"
                   name="q"
                   id="searchInput"
                   value="<?= esc_attr($q) ?>"
                   placeholder="Titel, Autor, ISBN, Schlagwort ..."
                   autofocus
                   class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-base">
            <button type="submit"
                    class="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium whitespace-nowrap">
                <svg class="inline h-5 w-5 mr-1 -mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                </svg>
                Suchen
            </button>
        </div>

        <!-- Filter row -->
        <div class="flex flex-wrap gap-x-6 gap-y-3 items-center">

            <!-- Language -->
            <div class="flex items-center gap-2">
                <label for="filterLang" class="text-sm text-gray-600 whitespace-nowrap">Sprache:</label>
                <select name="lang" id="filterLang"
                        onchange="document.getElementById('searchForm').submit()"
                        class="text-sm border border-gray-300 rounded px-2 py-1.5 focus:ring-1 focus:ring-blue-500">
                    <option value="">Alle</option>
                    <?php foreach ($languages as $l): ?>
                        <option value="<?= esc_attr($l['language']) ?>"
                            <?= $lang_f === $l['language'] ? 'selected' : '' ?>>
                            <?= esc_html($l['language']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- Subcategory (grouped) -->
            <div class="flex items-center gap-2">
                <label for="filterCat" class="text-sm text-gray-600 whitespace-nowrap">Kategorie:</label>
                <select name="cat" id="filterCat"
                        onchange="document.getElementById('searchForm').submit()"
                        class="text-sm border border-gray-300 rounded px-2 py-1.5 focus:ring-1 focus:ring-blue-500 max-w-xs">
                    <option value="0">Alle</option>
                    <?php
                    $last_cat = '';
                    foreach ($subcategories as $sc):
                        if ($sc['category_name'] !== $last_cat):
                            if ($last_cat !== '') echo '</optgroup>';
                            echo '<optgroup label="' . esc_attr($sc['category_name']) . '">';
                            $last_cat = $sc['category_name'];
                        endif;
                    ?>
                        <option value="<?= (int)$sc['id'] ?>"
                            <?= $cat_f === (int)$sc['id'] ? 'selected' : '' ?>>
                            <?= esc_html($sc['code'] . ' \u2013 ' . $sc['name']) ?>
                        </option>
                    <?php endforeach; if ($last_cat !== '') echo '</optgroup>'; ?>
                </select>
            </div>

            <!-- Available only -->
            <label class="flex items-center gap-2 text-sm text-gray-600 cursor-pointer">
                <input type="checkbox" name="avail" value="1"
                       <?= $avail ? 'checked' : '' ?>
                       onchange="document.getElementById('searchForm').submit()"
                       class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                Nur verfügbare Bücher
            </label>

            <!-- Clear filters -->
            <?php if ($has_filter): ?>
                <a href="<?= BASE_URL ?>/search.php"
                   class="text-sm text-red-600 hover:text-red-800 ml-auto">
                    Filter zurücksetzen ×
                </a>
            <?php endif; ?>
        </div>
    </form>
</div>

<!-- Result count + page info -->
<div class="flex items-center justify-between mb-4">
    <p class="text-sm text-gray-600">
        <?php if ($has_filter): ?>
            <strong><?= number_format($total, 0, ',', '.') ?></strong> Buch<?= $total !== 1 ? 'er' : '' ?> gefunden
            <?php if ($q !== ''): ?>
                für &bdquo;<strong><?= esc_html($q) ?></strong>&ldquo;
            <?php endif; ?>
        <?php else: ?>
            <strong><?= number_format($total, 0, ',', '.') ?></strong> Bücher im Bestand
        <?php endif; ?>
    </p>
    <?php if ($total_pages > 1): ?>
        <p class="text-sm text-gray-500">Seite <?= $page ?> von <?= $total_pages ?></p>
    <?php endif; ?>
</div>

<!-- Results grid -->
<?php if (empty($books)): ?>
    <div class="bg-white rounded-lg shadow p-12 text-center text-gray-500">
        <svg class="mx-auto h-16 w-16 text-gray-300 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                  d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
        </svg>
        <p class="text-xl font-medium mb-2">Keine Bücher gefunden</p>
        <p class="text-sm">
            <?php if ($has_filter): ?>
                Versuchen Sie einen anderen Suchbegriff oder entfernen Sie einige Filter.
            <?php else: ?>
                Der Bestand ist noch leer.
            <?php endif; ?>
        </p>
    </div>

<?php else: ?>

    <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-4 mb-8">
        <?php foreach ($books as $book): ?>
            <a href="<?= BASE_URL ?>/book.php?id=<?= (int)$book['id'] ?>"
               class="group bg-white rounded-lg shadow hover:shadow-md overflow-hidden flex flex-col">

                <!-- Cover -->
                <div class="aspect-[2/3] bg-gray-100 overflow-hidden flex-shrink-0">
                    <?php if ($book['image_url']): ?>
                        <img src="<?= esc_url($book['image_url']) ?>"
                             alt="<?= esc_attr($book['title']) ?>"
                             class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200"
                             loading="lazy">
                    <?php else: ?>
                        <div class="w-full h-full flex flex-col items-center justify-center p-3 text-center bg-gray-50">
                            <svg class="h-10 w-10 text-gray-300 mb-2 flex-shrink-0"
                                 fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                      d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                            </svg>
                            <?php if ($book['subcategory_code']): ?>
                                <span class="text-xs font-bold text-gray-400">
                                    <?= esc_html($book['subcategory_code']) ?>
                                </span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Metadata -->
                <div class="p-3 flex flex-col flex-1">
                    <p class="text-sm font-medium text-gray-900 line-clamp-2 group-hover:text-blue-600 leading-snug mb-1">
                        <?= esc_html($book['title']) ?>
                    </p>
                    <?php if ($book['author']): ?>
                        <p class="text-xs text-gray-500 truncate"><?= esc_html($book['author']) ?></p>
                    <?php endif; ?>
                    <?php if ($book['year']): ?>
                        <p class="text-xs text-gray-400"><?= (int)$book['year'] ?></p>
                    <?php endif; ?>

                    <!-- Availability badge -->
                    <div class="mt-auto pt-2">
                        <?php if ((int)$book['available_copies'] > 0): ?>
                            <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-green-100 text-green-800">
                                <?= (int)$book['available_copies'] ?> verfügbar
                            </span>
                        <?php elseif ((int)$book['active_copies'] > 0): ?>
                            <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-red-100 text-red-800">
                                ausgeliehen
                            </span>
                        <?php else: ?>
                            <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-gray-100 text-gray-600">
                                kein Exemplar
                            </span>
                        <?php endif; ?>
                    </div>
                </div>
            </a>
        <?php endforeach; ?>
    </div>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
        <nav class="flex justify-center mb-4" aria-label="Pagination">
            <div class="flex items-center gap-1">

                <?php if ($page > 1): ?>
                    <a href="<?= $pag_url($page - 1) ?>"
                       class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-l-md hover:bg-gray-50">
                        &laquo;
                    </a>
                <?php endif; ?>

                <?php
                for ($p = 1; $p <= $total_pages; $p++):
                    if ($p > 1 && $p < $total_pages && abs($p - $page) > 2):
                        if ($p === 2 || $p === $total_pages - 1):
                            echo '<span class="px-3 py-2 text-sm text-gray-400">...</span>';
                        endif;
                        continue;
                    endif;
                ?>
                    <a href="<?= $pag_url($p) ?>"
                       class="px-3 py-2 text-sm font-medium border <?= $p === $page ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50' ?>">
                        <?= $p ?>
                    </a>
                <?php endfor; ?>

                <?php if ($page < $total_pages): ?>
                    <a href="<?= $pag_url($page + 1) ?>"
                       class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-r-md hover:bg-gray-50">
                        &raquo;
                    </a>
                <?php endif; ?>

            </div>
        </nav>
    <?php endif; ?>

<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>
