<?php
/**
 * Installation Wizard
 * WordPress-style installation process
 */

// Calculate BASE_URL
require_once __DIR__ . '/../includes/base_url.php';

// Check if already installed
if (file_exists(__DIR__ . '/../includes/config.php')) {
    require_once __DIR__ . '/../includes/config.php';
    if (defined('INSTALLED') && INSTALLED) {
        header('Location: ' . BASE_URL . '/index.php');
        exit;
    }
}

session_start();

$step = $_GET['step'] ?? 1;
$error = null;
$success = null;

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        case 2:
            // Test database connection
            $result = test_database_connection($_POST);
            if ($result['success']) {
                $_SESSION['install_db'] = $_POST;
                header('Location: wizard.php?step=3');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
            
        case 3:
            // Create admin user
            $_SESSION['install_admin'] = $_POST;
            header('Location: wizard.php?step=4');
            exit;
            break;
            
        case 4:
            // School settings
            $_SESSION['install_school'] = $_POST;
            header('Location: wizard.php?step=5');
            exit;
            break;
            
        case 5:
            // Run installation
            $result = run_installation();
            if ($result['success']) {
                $_SESSION['installation_complete'] = true;
                header('Location: wizard.php?step=6');
                exit;
            } else {
                $error = $result['error'];
            }
            break;
    }
}

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installation - Bibliotheksverwaltung</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50">
    <div class="min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
        <div class="max-w-2xl w-full">
            <!-- Progress Bar -->
            <div class="mb-8">
                <div class="flex items-center justify-between mb-2">
                    <span class="text-sm font-medium text-gray-700">Schritt <?php echo $step; ?> von 6</span>
                    <span class="text-sm font-medium text-gray-700"><?php echo round(($step / 6) * 100); ?>%</span>
                </div>
                <div class="w-full bg-gray-200 rounded-full h-2">
                    <div class="bg-blue-600 h-2 rounded-full transition-all duration-300" style="width: <?php echo ($step / 6) * 100; ?>%"></div>
                </div>
            </div>

            <!-- Main Card -->
            <div class="bg-white rounded-xl shadow-lg p-8">
                <?php if ($error): ?>
                    <div class="mb-6 bg-red-50 border-l-4 border-red-500 p-4">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm text-red-700"><?php echo htmlspecialchars($error); ?></p>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <?php
                // Include step content
                switch ($step) {
                    case 1:
                        include 'steps/step1-welcome.php';
                        break;
                    case 2:
                        include 'steps/step2-database.php';
                        break;
                    case 3:
                        include 'steps/step3-admin.php';
                        break;
                    case 4:
                        include 'steps/step4-school.php';
                        break;
                    case 5:
                        include 'steps/step5-install.php';
                        break;
                    case 6:
                        include 'steps/step6-complete.php';
                        break;
                    default:
                        echo '<p>Ungültiger Schritt</p>';
                }
                ?>
            </div>

            <!-- Footer -->
            <div class="mt-6 text-center text-sm text-gray-600">
                Bibliotheksverwaltung v1.0 | © 2026
            </div>
        </div>
    </div>
</body>
</html>

<?php
/**
 * Helper Functions
 */

function test_database_connection($data) {
    try {
        $dsn = sprintf(
            "mysql:host=%s;dbname=%s;charset=utf8mb4",
            $data['db_host'],
            $data['db_name']
        );
        
        $pdo = new PDO($dsn, $data['db_user'], $data['db_pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        return ['success' => true];
    } catch (PDOException $e) {
        return [
            'success' => false,
            'error' => 'Datenbankverbindung fehlgeschlagen: ' . $e->getMessage()
        ];
    }
}

function run_installation() {
    try {
        $db = $_SESSION['install_db'];
        $admin = $_SESSION['install_admin'];
        $school = $_SESSION['install_school'];
        
        // Connect to database
        $dsn = sprintf("mysql:host=%s;dbname=%s;charset=utf8mb4", $db['db_host'], $db['db_name']);
        $pdo = new PDO($dsn, $db['db_user'], $db['db_pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        // Run SQL schema
        $sql = file_get_contents(__DIR__ . '/database.sql');
        $pdo->exec($sql);
        
        // Create triggers separately (PDO->exec can't handle DELIMITER)
        create_triggers($pdo);
        
        // Create admin user
        $passwordHash = password_hash($admin['password'], PASSWORD_ARGON2ID);
        $stmt = $pdo->prepare("
            INSERT INTO users (user_group_id, username, email, password_hash, first_name, last_name, auth_method, status)
            SELECT id, ?, ?, ?, ?, ?, 'local', 'aktiv'
            FROM user_groups WHERE name = 'Admin' LIMIT 1
        ");
        $stmt->execute([
            $admin['username'],
            $admin['email'],
            $passwordHash,
            $admin['first_name'],
            $admin['last_name']
        ]);
        
        // Update settings
        $settings = [
            'library_name' => $school['library_name'],
            'library_email' => $school['library_email'],
            'footer_text' => $school['footer_text'],
            'installation_complete' => 'true'
        ];
        
        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("UPDATE settings SET setting_value = ? WHERE setting_key = ?");
            $stmt->execute([$value, $key]);
        }
        
        // Create config.php
        create_config_file($db, $school);
        
        // Clear session
        unset($_SESSION['install_db'], $_SESSION['install_admin'], $_SESSION['install_school']);
        
        return ['success' => true];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function create_config_file($db, $school) {
    $date = date('Y-m-d H:i:s');
    $auth_key = bin2hex(random_bytes(32));
    $secure_auth_key = bin2hex(random_bytes(32));
    $logged_in_key = bin2hex(random_bytes(32));
    $nonce_key = bin2hex(random_bytes(32));
    
    $config = <<<PHP
<?php
/**
 * Database Configuration
 * Generated by Installation Wizard
 * Date: {$date}
 */

// Database Configuration
define('DB_HOST', '{$db['db_host']}');
define('DB_NAME', '{$db['db_name']}');
define('DB_USER', '{$db['db_user']}');
define('DB_PASS', '{$db['db_pass']}');
define('DB_CHARSET', 'utf8mb4');

// Installation Status
define('INSTALLED', true);

// Base URL (auto-detected)
if (!defined('BASE_URL')) {
    require_once __DIR__ . '/base_url.php';
}

// Security Keys
define('AUTH_KEY', '{$auth_key}');
define('SECURE_AUTH_KEY', '{$secure_auth_key}');
define('LOGGED_IN_KEY', '{$logged_in_key}');
define('NONCE_KEY', '{$nonce_key}');

// Error Reporting
define('DEBUG_MODE', false);

if (DEBUG_MODE) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}

// Timezone
date_default_timezone_set('Europe/Vienna');
PHP;
    
    file_put_contents(__DIR__ . '/../includes/config.php', $config);
}

function create_triggers($pdo) {
    // Trigger 1: Update overdue count when loan status changes
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS update_overdue_count_after_loan_update
        AFTER UPDATE ON loans
        FOR EACH ROW
        BEGIN
            IF NEW.status = 'überfällig' AND OLD.status != 'überfällig' THEN
                UPDATE users SET overdue_loans = overdue_loans + 1 WHERE id = NEW.user_id;
            ELSEIF NEW.status != 'überfällig' AND OLD.status = 'überfällig' THEN
                UPDATE users SET overdue_loans = overdue_loans - 1 WHERE id = NEW.user_id;
            END IF;
        END
    ");
    
    // Trigger 2: Update current loans count when loan created
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS update_current_loans_after_insert
        AFTER INSERT ON loans
        FOR EACH ROW
        BEGIN
            IF NEW.status IN ('aktiv', 'überfällig') THEN
                UPDATE users SET current_loans = current_loans + 1, total_loans = total_loans + 1 WHERE id = NEW.user_id;
            END IF;
        END
    ");
    
    // Trigger 3: Update current loans count when loan returned
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS update_current_loans_after_return
        AFTER UPDATE ON loans
        FOR EACH ROW
        BEGIN
            IF NEW.status = 'zurückgegeben' AND OLD.status IN ('aktiv', 'überfällig') THEN
                UPDATE users SET current_loans = current_loans - 1 WHERE id = NEW.user_id;
            END IF;
        END
    ");
}
?>

