-- ============================================================================
-- BIBLIOTHEKS-VERWALTUNGSSYSTEM - DATABASE SCHEMA
-- Version: 2.1
-- Date: 2026-02-22
-- Description: Complete database structure for library management system
--              Includes book_categories, book_subcategories (from Littera CSV)
--              Extended books and book_copies tables
-- ============================================================================

-- DROP DATABASE IF EXISTS bibliothek;
-- CREATE DATABASE IF NOT EXISTS bibliothek
--   CHARACTER SET utf8mb4
--   COLLATE utf8mb4_unicode_ci;
-- USE bibliothek;

-- ============================================================================
-- TABLE 1: BOOK_CATEGORIES (Obergruppen - replaces old categories table)
-- ============================================================================
CREATE TABLE IF NOT EXISTS book_categories (
    id         INT AUTO_INCREMENT PRIMARY KEY,
    code       VARCHAR(10)  NOT NULL UNIQUE COMMENT 'z.B. B, CO, KÜ',
    name       VARCHAR(100) NOT NULL         COMMENT 'z.B. Biologie, Comics',
    sort_order INT DEFAULT 0,
    active     TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_code (code)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 2: BOOK_SUBCATEGORIES (Untergruppen / Systematik = Sig1 in Littera)
-- ============================================================================
CREATE TABLE IF NOT EXISTS book_subcategories (
    id          INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    code        VARCHAR(10)  NOT NULL UNIQUE COMMENT 'z.B. Ba, De, GN — entspricht Sig1',
    name        VARCHAR(100) NOT NULL,
    sort_order  INT DEFAULT 0,
    active      TINYINT(1) DEFAULT 1,
    created_at  TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (category_id) REFERENCES book_categories(id) ON DELETE CASCADE,
    INDEX idx_code (code),
    INDEX idx_category (category_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 3: USER_GROUPS
-- ============================================================================
CREATE TABLE IF NOT EXISTS user_groups (
    id                INT AUTO_INCREMENT PRIMARY KEY,
    name              VARCHAR(50) UNIQUE NOT NULL,
    description       TEXT,
    max_loans         INT DEFAULT 5,
    loan_duration_days INT DEFAULT 28,
    can_extend        BOOLEAN DEFAULT TRUE,
    max_extensions    INT DEFAULT NULL COMMENT 'NULL = unlimited',
    can_reserve       BOOLEAN DEFAULT TRUE,
    can_manage_books  BOOLEAN DEFAULT FALSE,
    can_manage_users  BOOLEAN DEFAULT FALSE,
    can_manage_loans  BOOLEAN DEFAULT FALSE,
    created_at        TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 4: USERS
-- ============================================================================
CREATE TABLE IF NOT EXISTS users (
    id            INT AUTO_INCREMENT PRIMARY KEY,
    user_group_id INT NOT NULL,
    username      VARCHAR(100) UNIQUE,
    email         VARCHAR(255) UNIQUE NULL    COMMENT 'NULL erlaubt für Sokrates-Schüler ohne E-Mail',
    password_hash VARCHAR(255)   COMMENT 'NULL for LDAP users',
    first_name    VARCHAR(100) NOT NULL,
    last_name     VARCHAR(100) NOT NULL,
    class_name    VARCHAR(50),
    student_id    VARCHAR(50),
    birthday      DATE NULL              COMMENT 'Geburtsdatum (aus Sokrates-Import)',
    phone         VARCHAR(50),
    address       TEXT,
    auth_method   ENUM('local','ldap') DEFAULT 'local',
    ldap_dn       VARCHAR(255)   COMMENT 'LDAP Distinguished Name',
    status        ENUM('aktiv','gesperrt','inaktiv') DEFAULT 'aktiv',
    blocked_reason TEXT,
    blocked_until  DATE,
    total_loans    INT DEFAULT 0,
    current_loans  INT DEFAULT 0,
    overdue_loans  INT DEFAULT 0,
    last_login     DATETIME,
    created_at     TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at     TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (user_group_id) REFERENCES user_groups(id),

    INDEX idx_email      (email),
    INDEX idx_username   (username),
    INDEX idx_student_id (student_id),
    INDEX idx_status     (status),
    INDEX idx_class      (class_name),
    FULLTEXT idx_name    (first_name, last_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 5: BOOKS (Stammdaten — 1x pro Titel)
-- ============================================================================
CREATE TABLE IF NOT EXISTS books (
    id          INT AUTO_INCREMENT PRIMARY KEY,

    -- Identifikation
    isbn        VARCHAR(20) NULL COMMENT 'NULL erlaubt für Bücher ohne ISBN',

    -- Titeldaten
    title       VARCHAR(500) NOT NULL,
    subtitle    VARCHAR(500)          COMMENT 'Untertitel',
    title_sort  VARCHAR(500)          COMMENT 'Sortierform ohne Artikel (Der/Die/Das)',

    -- Hauptautor
    author       VARCHAR(255)         COMMENT 'Nachname Vorname',
    author_short VARCHAR(20)          COMMENT 'Kürzel für Etikett, z.B. Böl',
    
    -- Weitere beteiligte Personen
    author_additional VARCHAR(500)    COMMENT 'Weitere Autoren, kommagetrennt',
    editor            VARCHAR(255)    COMMENT 'Herausgeber',
    translator        VARCHAR(255)    COMMENT 'Übersetzer',
    illustrator       VARCHAR(255)    COMMENT 'Illustratoren',
    photographer      VARCHAR(255)    COMMENT 'Fotograf',
    graphic_artist    VARCHAR(255)    COMMENT 'Grafiker',
    other_persons     VARCHAR(255)    COMMENT 'Sonstige',

    -- Publikation
    publisher            VARCHAR(255),
    place_of_publication VARCHAR(100),
    year                 SMALLINT,
    edition              VARCHAR(50)   COMMENT 'z.B. 2. Aufl.',
    series               VARCHAR(255)  COMMENT 'Reihe / Gesamttitel',
    series_volume        VARCHAR(50)   COMMENT 'Band in Reihe',
    pages                SMALLINT UNSIGNED,
    language             VARCHAR(50) DEFAULT 'Deutsch',

    -- Inhalt & Erschließung
    description TEXT                  COMMENT 'Inhaltsangabe / Annotation',
    keywords    TEXT                  COMMENT 'Schlagworte — wichtig für Suche',

    -- Kategorisierung (Obergruppe über JOIN auf book_subcategories)
    subcategory_id INT                COMMENT '→ book_subcategories.id (= Sig1)',

    -- Wirtschaftlich
    price DECIMAL(10,2)               COMMENT 'Anschaffungspreis',

    -- Cover & Links
    image_url  VARCHAR(512)           COMMENT 'URL zum Coverbild',
    cover_link VARCHAR(512)           COMMENT 'Link zu ISBN.de o.ä.',

    -- Meta
    created_by INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (subcategory_id) REFERENCES book_subcategories(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by)     REFERENCES users(id) ON DELETE SET NULL,

    INDEX idx_isbn        (isbn),
    INDEX idx_title       (title(255)),
    INDEX idx_author      (author(100)),
    INDEX idx_subcategory (subcategory_id),
    INDEX idx_year        (year),
    INDEX idx_language    (language),
    FULLTEXT idx_search   (title, subtitle, author, keywords, description)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 6: BOOK_COPIES (Physische Exemplare)
-- ============================================================================
CREATE TABLE IF NOT EXISTS book_copies (
    id          INT AUTO_INCREMENT PRIMARY KEY,
    book_id     INT NOT NULL,

    -- Inventarnummer (steht auf dem Etikett)
    book_number VARCHAR(20)              COMMENT 'Inventarnummer, z.B. 1458 oder 1458_2',
    copy_index  TINYINT UNSIGNED DEFAULT 1 COMMENT '1 = erstes Exemplar, 2 = zweites, ...',

    -- Status & Zustand
    status ENUM(
        'verfügbar',
        'ausgeliehen',
        'reserviert',
        'in_bearbeitung',
        'verloren',
        'aussortiert'
    ) DEFAULT 'verfügbar',
    condition_note TEXT           COMMENT 'Zustandsnotizen',
    location       VARCHAR(100)   COMMENT 'Standort / Kategorie (z.B. De, Dd, MEd)',

    -- Littera-Import: Inventar & Signatur
    accession_number VARCHAR(20)  COMMENT 'Inventarnummer aus Littera (Exemplarnummer)',
    signature        VARCHAR(100) COMMENT 'Zusammengesetzte Signatur (z.B. De Bro 13751)',

    -- Erwerbung
    acquired_at      DATE         COMMENT 'Zugangsdatum (Legacy)',
    acquisition_date DATE         COMMENT 'Zugangsdatum',

    -- Meta
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (book_id) REFERENCES books(id) ON DELETE CASCADE,

    INDEX idx_book_id          (book_id),
    INDEX idx_book_number      (book_number),
    INDEX idx_status           (status),
    INDEX idx_copy_index       (copy_index),
    INDEX idx_accession_number (accession_number),
    INDEX idx_signature        (signature)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 7: LOANS
-- ============================================================================
CREATE TABLE IF NOT EXISTS loans (
    id                 INT AUTO_INCREMENT PRIMARY KEY,
    copy_id            INT NOT NULL,
    user_id            INT NOT NULL,
    loan_date          DATETIME NOT NULL,
    due_date           DATE NOT NULL,
    return_date        DATETIME,
    extensions         INT DEFAULT 0,
    last_extension_date DATETIME,
    status             ENUM('aktiv','überfällig','zurückgegeben','verloren','beschädigt') DEFAULT 'aktiv',
    loan_note          TEXT,
    return_note        TEXT,
    loaned_by          INT COMMENT 'Admin der die Ausleihe bearbeitet hat',
    returned_by        INT COMMENT 'Admin der die Rückgabe bearbeitet hat',
    created_at         TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at         TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (copy_id)      REFERENCES book_copies(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id)      REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (loaned_by)    REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (returned_by)  REFERENCES users(id) ON DELETE SET NULL,

    INDEX idx_copy        (copy_id),
    INDEX idx_user        (user_id),
    INDEX idx_status      (status),
    INDEX idx_due_date    (due_date),
    INDEX idx_loan_date   (loan_date),
    INDEX idx_return_date (return_date),
    INDEX idx_active_loans(status, due_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 8: RESERVATIONS
-- ============================================================================
CREATE TABLE IF NOT EXISTS reservations (
    id            INT AUTO_INCREMENT PRIMARY KEY,
    book_id       INT NOT NULL,
    user_id       INT NOT NULL,
    reserved_date DATETIME NOT NULL,
    valid_until   DATE,
    notified_date DATETIME,
    status        ENUM('aktiv','verfügbar','abgeholt','abgelaufen','storniert') DEFAULT 'aktiv',
    copy_id       INT COMMENT 'Konkretes Exemplar (wenn verfügbar)',
    created_at    TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (book_id) REFERENCES books(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (copy_id) REFERENCES book_copies(id) ON DELETE SET NULL,

    INDEX idx_book   (book_id),
    INDEX idx_user   (user_id),
    INDEX idx_status (status),
    INDEX idx_reserved_date (reserved_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 9: REMINDERS
-- ============================================================================
CREATE TABLE IF NOT EXISTS reminders (
    id            INT AUTO_INCREMENT PRIMARY KEY,
    loan_id       INT NOT NULL,
    user_id       INT NOT NULL,
    sent_date     DATETIME NOT NULL,
    email_to      VARCHAR(255) NOT NULL,
    email_subject VARCHAR(255),
    email_body    TEXT,
    status        ENUM('versendet','fehlgeschlagen','gelesen') DEFAULT 'versendet',
    error_message TEXT,
    read_date     DATETIME,
    created_at    TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (loan_id) REFERENCES loans(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,

    INDEX idx_loan      (loan_id),
    INDEX idx_user      (user_id),
    INDEX idx_sent_date (sent_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 10: LOAN_HISTORY (Abgeschlossene Ausleihen)
-- ============================================================================
CREATE TABLE IF NOT EXISTS loan_history (
    id          INT AUTO_INCREMENT PRIMARY KEY,
    user_id     INT     COMMENT 'NULL wenn Benutzer gelöscht',
    book_id     INT     COMMENT 'NULL wenn Buch gelöscht',
    copy_id     INT,
    loan_date   DATETIME NOT NULL,
    return_date DATETIME,
    was_overdue BOOLEAN DEFAULT FALSE,
    days_overdue INT DEFAULT 0,
    user_name   VARCHAR(200) COMMENT 'Denormalisiert für Historie',
    book_title  VARCHAR(255) COMMENT 'Denormalisiert für Historie',
    created_at  TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_user_history (user_id),
    INDEX idx_book_history (book_id),
    INDEX idx_dates        (loan_date, return_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 11: LOAN_HISTORY_ARCHIVE (Alte Ausleihen > 5 Jahre)
-- ============================================================================
CREATE TABLE IF NOT EXISTS loan_history_archive (
    id          INT AUTO_INCREMENT PRIMARY KEY,
    user_id     INT,
    book_id     INT,
    copy_id     INT,
    loan_date   DATETIME NOT NULL,
    return_date DATETIME,
    was_overdue BOOLEAN DEFAULT FALSE,
    days_overdue INT DEFAULT 0,
    user_name   VARCHAR(200),
    book_title  VARCHAR(255),
    created_at  TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_user_history (user_id),
    INDEX idx_book_history (book_id),
    INDEX idx_dates        (loan_date, return_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 12: ACTIVITY_LOG
-- ============================================================================
CREATE TABLE IF NOT EXISTS activity_log (
    id          INT AUTO_INCREMENT PRIMARY KEY,
    user_id     INT,
    action      VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50),
    entity_id   INT,
    details     TEXT,
    ip_address  VARCHAR(50),
    created_at  TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,

    INDEX idx_user_activity (user_id),
    INDEX idx_action        (action),
    INDEX idx_created       (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- TABLE 13: SETTINGS
-- ============================================================================
CREATE TABLE IF NOT EXISTS settings (
    setting_key   VARCHAR(100) PRIMARY KEY,
    setting_value TEXT,
    updated_at    TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- VIEWS
-- ============================================================================

CREATE OR REPLACE VIEW view_active_loans AS
SELECT
    l.id as loan_id,
    l.loan_date,
    l.due_date,
    l.extensions,
    CASE
        WHEN l.due_date < CURDATE() THEN 'überfällig'
        WHEN l.due_date = CURDATE() THEN 'heute_fällig'
        ELSE 'aktiv'
    END as loan_status,
    DATEDIFF(CURDATE(), l.due_date) as days_overdue,
    u.id as user_id,
    u.first_name,
    u.last_name,
    u.email,
    u.class_name,
    ug.name as user_group,
    b.id as book_id,
    b.title,
    b.author,
    b.isbn,
    bc.id as copy_id,
    bc.book_number,
    bc.copy_index,
    bsc.code as subcategory_code
FROM loans l
JOIN users u       ON l.user_id  = u.id
JOIN user_groups ug ON u.user_group_id = ug.id
JOIN book_copies bc ON l.copy_id  = bc.id
JOIN books b        ON bc.book_id = b.id
LEFT JOIN book_subcategories bsc ON b.subcategory_id = bsc.id
WHERE l.status IN ('aktiv','überfällig');

CREATE OR REPLACE VIEW view_overdue_loans AS
SELECT * FROM view_active_loans
WHERE due_date < CURDATE();

CREATE OR REPLACE VIEW view_available_copies AS
SELECT
    bc.id as copy_id,
    bc.book_number,
    bc.copy_index,
    bc.location,
    b.id as book_id,
    b.title,
    b.author,
    b.isbn,
    b.image_url,
    bsc.code  as subcategory_code,
    bsc.name  as subcategory_name,
    bcat.code as category_code,
    bcat.name as category_name
FROM book_copies bc
JOIN books b ON bc.book_id = b.id
LEFT JOIN book_subcategories bsc  ON b.subcategory_id = bsc.id
LEFT JOIN book_categories    bcat ON bsc.category_id  = bcat.id
WHERE bc.status = 'verfügbar';

-- ============================================================================
-- DEFAULT DATA: USER_GROUPS
-- ============================================================================
INSERT INTO user_groups (name, description, max_loans, loan_duration_days, can_extend, max_extensions, can_manage_books, can_manage_users, can_manage_loans) VALUES
('Admin', 'Administrators with full access', 99, 365, TRUE, NULL, TRUE, TRUE, TRUE),
('User',  'Regular users (students, teachers)', 5, 28, TRUE, 6, FALSE, FALSE, FALSE)
ON DUPLICATE KEY UPDATE name = name;

-- ============================================================================
-- DEFAULT DATA: BOOK_CATEGORIES (Obergruppen aus Littera-Systematikliste)
-- ============================================================================
INSERT INTO book_categories (code, name, sort_order) VALUES
('B',   'Biologie',              1),
('BB',  'Bibliothek',            2),
('C',   'Computer',              3),
('CO',  'Comics',                4),
('D',   'Deutsche Literatur',    5),
('E',   'Englisch',              6),
('ER',  'Ernährungslehre',       7),
('F',   'Französisch',           8),
('G',   'Geographie',            9),
('H',   'Geschichte',           10),
('HA',  'Hauswirtschaft',       11),
('J',   'Recht und Politik',    12),
('K',   'Kunst',                13),
('KO',  'Kommunikation',        14),
('KÜ',  'Küchenführung',        15),
('L',   'Leben',                16),
('LI',  'Literatur',            17),
('M',   'Musik',                18),
('ME',  'Medien',               19),
('N',   'Naturwissenschaft',    20),
('P',   'PPP',                  21),
('R',   'Religion',             22),
('S',   'Sport',                23),
('SP',  'Spanisch',             24),
('T',   'Textverarbeitung',     25),
('V',   'Volkswirtschaft',      26),
('W',   'Wirtschaft',           27),
('WE',  'Werken',               28),
('X',   'Lexika',               29),
('Z',   'Zeitschriften',        30)
ON DUPLICATE KEY UPDATE name = VALUES(name);

-- ============================================================================
-- DEFAULT DATA: BOOK_SUBCATEGORIES (75 Untergruppen aus Littera-Systematikliste)
-- ============================================================================
INSERT INTO book_subcategories (category_id, code, name, sort_order) VALUES
-- B - Biologie
((SELECT id FROM book_categories WHERE code='B'),  'Ba',   'Biologie Allgemein',                1),
((SELECT id FROM book_categories WHERE code='B'),  'Bm',   'Biologie Mensch',                   2),
-- BB - Bibliothek
((SELECT id FROM book_categories WHERE code='BB'), 'BBa',  'Bibliothek allgemein',               1),
-- C - Computer
((SELECT id FROM book_categories WHERE code='C'),  'Ci',   'Internet',                           1),
((SELECT id FROM book_categories WHERE code='C'),  'Ck',   'Kommunikations- u. Mediendesign',    2),
((SELECT id FROM book_categories WHERE code='C'),  'Cp',   'Programme',                          3),
-- CO - Comics
((SELECT id FROM book_categories WHERE code='CO'), 'GN',   'Graphic Novel',                      1),
-- D - Deutsche Literatur
((SELECT id FROM book_categories WHERE code='D'),  'Da',   'Deutsch Allgemein',                  1),
((SELECT id FROM book_categories WHERE code='D'),  'Dd',   'Drama',                              2),
((SELECT id FROM book_categories WHERE code='D'),  'De',   'Epik',                               3),
((SELECT id FROM book_categories WHERE code='D'),  'DJ',   'Deutsch Jugendliteratur',            4),
((SELECT id FROM book_categories WHERE code='D'),  'Dl',   'Lyrik',                              5),
((SELECT id FROM book_categories WHERE code='D'),  'Dlg',  'Deutsch Literatur Geschichte',       6),
((SELECT id FROM book_categories WHERE code='D'),  'Dn',   'Deutsch-Anthologien',                7),
-- E - Englisch
((SELECT id FROM book_categories WHERE code='E'),  'Ea',   'Englisch allgemein',                 1),
((SELECT id FROM book_categories WHERE code='E'),  'El',   'Literatur',                          2),
((SELECT id FROM book_categories WHERE code='E'),  'EUnt', 'Englisch Unterrichtssprache',        3),
-- ER - Ernährungslehre
((SELECT id FROM book_categories WHERE code='ER'), 'ERa',  'Ernährung Allgemein',                1),
((SELECT id FROM book_categories WHERE code='ER'), 'ERst', 'Störungen',                          2),
-- F - Französisch
((SELECT id FROM book_categories WHERE code='F'),  'Fa',   'Französisch Allgemein',              1),
((SELECT id FROM book_categories WHERE code='F'),  'Fl',   'Literatur',                          2),
-- G - Geographie
((SELECT id FROM book_categories WHERE code='G'),  'Ga',   'Geographie Allgemein',               1),
((SELECT id FROM book_categories WHERE code='G'),  'Gl',   'Länder',                             2),
-- H - Geschichte
((SELECT id FROM book_categories WHERE code='H'),  'Ha',   'Geschichte Allgemein',               1),
((SELECT id FROM book_categories WHERE code='H'),  'He',   'Epochen',                            2),
-- HA - Hauswirtschaft
((SELECT id FROM book_categories WHERE code='HA'), 'Ca',   'Cateringmanagement',                 1),
((SELECT id FROM book_categories WHERE code='HA'), 'HAb',  'Betriebsorganisation',               2),
-- J - Recht und Politik
((SELECT id FROM book_categories WHERE code='J'),  'Jg',   'Recht und Politik',                  1),
-- K - Kunst
((SELECT id FROM book_categories WHERE code='K'),  'Ka',   'Kunst allgemein',                    1),
((SELECT id FROM book_categories WHERE code='K'),  'Kd',   'Darstellende Kunst',                 2),
((SELECT id FROM book_categories WHERE code='K'),  'Kf',   'Fotographie',                        3),
((SELECT id FROM book_categories WHERE code='K'),  'Km',   'Musik',                              4),
((SELECT id FROM book_categories WHERE code='K'),  'Kt',   'Kunst Theater, Film',                5),
-- KO - Kommunikation
((SELECT id FROM book_categories WHERE code='KO'), 'KOa',  'Kommunikation allgemein',            1),
((SELECT id FROM book_categories WHERE code='KO'), 'KOp',  'Präsentation',                       2),
-- KÜ - Küchenführung
((SELECT id FROM book_categories WHERE code='KÜ'), 'KÜr',  'Rezepte',                            1),
((SELECT id FROM book_categories WHERE code='KÜ'), 'KÜs',  'Servierkunde',                       2),
-- L - Leben
((SELECT id FROM book_categories WHERE code='L'),  'Lg',   'Leben Gesundheit',                   1),
-- LI - Literatur
((SELECT id FROM book_categories WHERE code='LI'), 'Llg',  'Literaturgeschichte',                1),
-- M - Musik
((SELECT id FROM book_categories WHERE code='M'),  'Ma',   'Musik allgemein',                    1),
-- ME - Medien
((SELECT id FROM book_categories WHERE code='ME'), 'MEc',  'CDs',                                1),
((SELECT id FROM book_categories WHERE code='ME'), 'MEd',  'Medien DVD',                         2),
((SELECT id FROM book_categories WHERE code='ME'), 'MEh',  'Hörkassetten',                       3),
((SELECT id FROM book_categories WHERE code='ME'), 'MEsp', 'Gesellschaftsspiele / Brettspiele',  4),
((SELECT id FROM book_categories WHERE code='ME'), 'MEv',  'Videos',                             5),
-- N - Naturwissenschaft
((SELECT id FROM book_categories WHERE code='N'),  'Nc',   'Chemie',                             1),
((SELECT id FROM book_categories WHERE code='N'),  'Nm',   'Mathematik',                         2),
((SELECT id FROM book_categories WHERE code='N'),  'Np',   'Physik',                             3),
-- P - PPP
((SELECT id FROM book_categories WHERE code='P'),  'Bor',  'Berufsorientierung',                 1),
((SELECT id FROM book_categories WHERE code='P'),  'Pä',   'Pädagogik',                          2),
((SELECT id FROM book_categories WHERE code='P'),  'Pph',  'Philosophie',                        3),
((SELECT id FROM book_categories WHERE code='P'),  'Ps',   'Psychologie',                        4),
-- R - Religion
((SELECT id FROM book_categories WHERE code='R'),  'Ra',   'Religion Allgemein',                 1),
-- S - Sport
((SELECT id FROM book_categories WHERE code='S'),  'Sa',   'Sport allgemein',                    1),
((SELECT id FROM book_categories WHERE code='S'),  'Ssp',  'Sportarten',                         2),
-- SP - Spanisch
((SELECT id FROM book_categories WHERE code='SP'), 'SPa',  'Spanisch Allgemein',                 1),
((SELECT id FROM book_categories WHERE code='SP'), 'SPl',  'Spanisch Literatur',                 2),
-- T - Textverarbeitung
((SELECT id FROM book_categories WHERE code='T'),  'Tc',   'Computerunterstützte Datenverarbeitung', 1),
-- V - Volkswirtschaft
((SELECT id FROM book_categories WHERE code='V'),  'Vw',   'Volkswirtschaft',                    1),
-- W - Wirtschaft
((SELECT id FROM book_categories WHERE code='W'),  'Wa',   'Wirtschaft Allgemein',               1),
((SELECT id FROM book_categories WHERE code='W'),  'Wbw',  'Betriebswirtschaft',                 2),
((SELECT id FROM book_categories WHERE code='W'),  'Wrw',  'Rechnungswesen',                     3),
-- WE - Werken
((SELECT id FROM book_categories WHERE code='WE'), 'WEa',  'Werken allgemein',                   1),
-- X - Lexika
((SELECT id FROM book_categories WHERE code='X'),  'Xa',   'Lexika Allgemein',                   1),
((SELECT id FROM book_categories WHERE code='X'),  'Xd',   'Deutsch',                            2),
((SELECT id FROM book_categories WHERE code='X'),  'Xe',   'Englisch',                           3),
((SELECT id FROM book_categories WHERE code='X'),  'Xf',   'Französisch',                        4),
((SELECT id FROM book_categories WHERE code='X'),  'Xg',   'Geographie',                         5),
((SELECT id FROM book_categories WHERE code='X'),  'Xh',   'Geschichte',                         6),
((SELECT id FROM book_categories WHERE code='X'),  'Xk',   'Kunst',                              7),
((SELECT id FROM book_categories WHERE code='X'),  'Xmu',  'Musik',                              8),
((SELECT id FROM book_categories WHERE code='X'),  'Xn',   'Naturwissenschaft',                  9),
((SELECT id FROM book_categories WHERE code='X'),  'Xsp',  'Spanisch',                          10),
-- Z - Zeitschriften
((SELECT id FROM book_categories WHERE code='Z'),  'Ze',   'Englisch',                           1),
((SELECT id FROM book_categories WHERE code='Z'),  'Zf',   'Französisch',                        2)
ON DUPLICATE KEY UPDATE name = VALUES(name);

-- ============================================================================
-- DEFAULT DATA: SETTINGS
-- ============================================================================
INSERT INTO settings (setting_key, setting_value) VALUES
-- General
('installation_complete',         'false'),
('library_name',                  'Schulbibliothek'),
('library_email',                 'bibliothek@schule.at'),
('footer_text',                   'Schulbibliothek | © 2026'),
('language',                      'de'),
('session_timeout_minutes',       '60'),
-- Book numbering (next free inventory number for new books)
('next_book_number',              '1'),
-- Loan Settings
('loan_duration_default',         '28'),
('max_extensions',                '6'),
('max_loans_per_user',            '5'),
-- Reminder Settings
('reminder_enabled',              'true'),
('reminder_days_after_due',       '7'),
('reminder_frequency',            'weekly'),
-- LDAP Settings
('ldap_enabled',                  'false'),
('ldap_host',                     ''),
('ldap_port',                     '389'),
('ldap_base_dn',                  ''),
('ldap_bind_dn',                  ''),
('ldap_bind_password',            ''),
('ldap_username_attribute',       'sAMAccountName'),
('ldap_email_attribute',          'mail'),
('ldap_firstname_attribute',      'givenName'),
('ldap_lastname_attribute',       'sn'),
-- SMTP Settings
('smtp_enabled',                  'false'),
('smtp_host',                     'smtp.office365.com'),
('smtp_port',                     '587'),
('smtp_username',                 ''),
('smtp_password',                 ''),
('smtp_password',                 ''),
('smtp_from_email',               ''),
('smtp_from_name',                ''),
-- ISBN API Settings
('isbn_api_primary',              'dnb'),
('isbn_api_second',               'google'),
('isbn_api_use_both',             'true'),
('isbn_api_timeout',              '8'),
-- Barcode / Label Settings
('barcode_format',                'code128'),
('barcode_label_format',          'avery5160'),
-- Cleanup Settings
('activity_log_retention_years',  '5'),
('loan_history_archive_years',    '5')
ON DUPLICATE KEY UPDATE setting_key = setting_key;

-- ============================================================================
-- END OF SCHEMA v2.1
-- ============================================================================
