<?php
/**
 * Core Functions
 * Helper functions used throughout the application
 */

/**
 * Get or set a setting value
 */
function getSetting($key, $default = null) {
    $value = db_get_var("SELECT setting_value FROM settings WHERE setting_key = ?", [$key]);
    return $value !== null ? $value : $default;
}

function setSetting($key, $value) {
    $existing = db_get_var("SELECT setting_key FROM settings WHERE setting_key = ?", [$key]);
    
    if ($existing) {
        return db_update('settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
    } else {
        return db_insert('settings', ['setting_key' => $key, 'setting_value' => $value]);
    }
}

/**
 * Sanitize for JavaScript
 */
function esc_js($text) {
    return addslashes($text);
}

/**
 * Sanitize output (XSS protection)
 */
function esc_html($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}

function esc_attr($text) {
    return htmlspecialchars($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

function esc_url($url) {
    return filter_var($url, FILTER_SANITIZE_URL);
}

/**
 * Redirect helper
 */
function redirect($url) {
    header('Location: ' . $url);
    exit;
}

/**
 * Current URL helper
 */
function current_url() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    return $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
}

/**
 * Generate random string
 */
function generate_random_string($length = 32) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Password validation
 */
function validate_password($password) {
    $errors = [];
    
    if (strlen($password) < 8) {
        $errors[] = 'Passwort muss mindestens 8 Zeichen lang sein';
    }
    if (!preg_match('/[A-Z]/', $password)) {
        $errors[] = 'Passwort muss mindestens einen Großbuchstaben enthalten';
    }
    if (!preg_match('/[a-z]/', $password)) {
        $errors[] = 'Passwort muss mindestens einen Kleinbuchstaben enthalten';
    }
    if (!preg_match('/[0-9]/', $password)) {
        $errors[] = 'Passwort muss mindestens eine Zahl enthalten';
    }
    if (!preg_match('/[^A-Za-z0-9]/', $password)) {
        $errors[] = 'Passwort muss mindestens ein Sonderzeichen enthalten';
    }
    
    return $errors;
}

/**
 * Hash password (Argon2id)
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_ARGON2ID);
}

/**
 * Verify password
 */
function verify_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Format date
 */
function format_date($date, $format = 'd.m.Y') {
    if (!$date) return '-';
    return date($format, strtotime($date));
}

/**
 * Format datetime
 */
function format_datetime($datetime, $format = 'd.m.Y H:i') {
    if (!$datetime) return '-';
    return date($format, strtotime($datetime));
}

/**
 * Calculate days difference
 */
function days_diff($date1, $date2 = null) {
    if (!$date2) $date2 = date('Y-m-d');
    $d1 = new DateTime($date1);
    $d2 = new DateTime($date2);
    return $d1->diff($d2)->days;
}

/**
 * Log activity
 * Accepts user_id as parameter or uses session (can be NULL for LDAP-only users)
 */
function log_activity($user_id_or_action, $entity_type = null, $entity_id = null, $details = null, $action_param = null) {
    // Support both old and new signatures
    if ($action_param !== null) {
        // New signature: log_activity($user_id, $action, $entity_type, $entity_id, $details)
        $user_id = $user_id_or_action;
        $action = $entity_type;
        $entity_type = $entity_id;
        $entity_id = $details;
        $details = $action_param;
    } else {
        // Old signature: log_activity($action, $entity_type, $entity_id, $details)
        $action = $user_id_or_action;
        $user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;
    }
    
    $ip = $_SERVER['REMOTE_ADDR'] ?? null;
    
    // Only log if user has DB ID (skip LDAP-only users)
    if ($user_id !== null) {
        db_insert('activity_log', [
            'user_id' => $user_id,
            'action' => $action,
            'entity_type' => $entity_type,
            'entity_id' => $entity_id,
            'details' => $details,
            'ip_address' => $ip
        ]);
    }
}

/**
 * Check if user is logged in
 * Works for both DB users and LDAP-only users
 */
function is_logged_in() {
    // user_id can be NULL for LDAP-only users, but username must exist
    return isset($_SESSION['username']) && 
           (isset($_SESSION['user_id']) || ($_SESSION['ldap_only'] ?? false));
}

/**
 * Check if user is admin
 */
function is_admin() {
    return is_logged_in() && ($_SESSION['user_group'] ?? '') === 'Admin';
}

/**
 * Require login
 */
function require_login() {
    if (!is_logged_in()) {
        redirect(BASE_URL . '/login.php');
    }
}

/**
 * Require admin
 */
function require_admin() {
    require_login();
    if (!is_admin()) {
        die('Zugriff verweigert. Nur Administratoren haben Zugriff auf diesen Bereich.');
    }
}

/**
 * Get current logged in user
 * Returns DB data for DB users, Session data for LDAP-only users
 */
function get_logged_in_user() {
    if (!is_logged_in()) return null;
    
    // LDAP-only user (not yet in DB)
    if ($_SESSION['ldap_only'] ?? false) {
        return [
            'id' => null,
            'username' => $_SESSION['username'],
            'email' => $_SESSION['email'],
            'first_name' => $_SESSION['first_name'] ?? '',
            'last_name' => $_SESSION['last_name'] ?? '',
            'auth_method' => 'ldap',
            'status' => 'aktiv',
            'group_name' => $_SESSION['user_group'] ?? 'User',
            'user_group_id' => $_SESSION['user_group_id'] ?? 2,
            'ldap_only' => true
        ];
    }
    
    // DB user (local or already saved LDAP user)
    return db_get_row(
        "SELECT u.*, ug.name as group_name 
         FROM users u 
         JOIN user_groups ug ON u.user_group_id = ug.id 
         WHERE u.id = ?",
        [$_SESSION['user_id']]
    );
}

/**
 * CSRF Token functions
 */
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = generate_random_string(32);
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

function csrf_field() {
    $token = generate_csrf_token();
    return '<input type="hidden" name="csrf_token" value="' . esc_attr($token) . '">';
}

/**
 * Flash messages
 */
function set_flash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function get_flash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

/**
 * Get available languages
 */
function get_available_languages() {
    return [
        'de' => 'Deutsch',
        'en' => 'English'
    ];
}

/**
 * Get current language
 */
function get_current_language() {
    // Check session first (fast)
    if (isset($_SESSION['language'])) {
        return $_SESSION['language'];
    }
    
    // Try to get from settings (requires DB)
    // Only if function exists (defensive programming)
    if (function_exists('getSetting')) {
        try {
            return getSetting('language', 'de');
        } catch (Exception $e) {
            // DB not ready yet, use default
            return 'de';
        }
    }
    
    // Default fallback
    return 'de';
}

/**
 * Load language file
 */
function load_language($lang = null) {
    if (!$lang) {
        $lang = get_current_language();
    }
    
    $file = __DIR__ . '/lang/' . $lang . '.php';
    if (file_exists($file)) {
        return require $file;
    }
    
    // Fallback to German
    return require __DIR__ . '/lang/de.php';
}
