<?php
/**
 * Database Connection Handler
 * PDO-based database connection with error handling
 */

if (!defined('DB_HOST')) {
    die('Configuration file not loaded');
}

try {
    $dsn = sprintf(
        "mysql:host=%s;dbname=%s;charset=%s",
        DB_HOST,
        DB_NAME,
        DB_CHARSET
    );
    
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
    ];
    
    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
    
} catch (PDOException $e) {
    if (DEBUG_MODE) {
        die('Database Connection Failed: ' . $e->getMessage());
    } else {
        die('Database connection error. Please contact the administrator.');
    }
}

/**
 * Helper function: Execute query and return results
 */
function db_query($sql, $params = []) {
    global $pdo;
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        error_log('Database Query Error: ' . $e->getMessage());
        if (DEBUG_MODE) {
            throw $e;
        }
        return false;
    }
}

/**
 * Helper function: Get single row
 */
function db_get_row($sql, $params = []) {
    $stmt = db_query($sql, $params);
    return $stmt ? $stmt->fetch() : null;
}

/**
 * Helper function: Get all rows
 */
function db_get_results($sql, $params = []) {
    $stmt = db_query($sql, $params);
    return $stmt ? $stmt->fetchAll() : [];
}

/**
 * Helper function: Get single value
 */
function db_get_var($sql, $params = []) {
    $stmt = db_query($sql, $params);
    return $stmt ? $stmt->fetchColumn() : null;
}

/**
 * Helper function: Insert and return last ID
 */
function db_insert($table, $data) {
    global $pdo;
    
    $fields = array_keys($data);
    $placeholders = array_fill(0, count($fields), '?');
    
    $sql = sprintf(
        "INSERT INTO %s (%s) VALUES (%s)",
        $table,
        implode(', ', $fields),
        implode(', ', $placeholders)
    );
    
    $stmt = db_query($sql, array_values($data));
    return $stmt ? $pdo->lastInsertId() : false;
}

/**
 * Helper function: Update rows
 */
function db_update($table, $data, $where, $whereParams = []) {
    $setParts = [];
    foreach (array_keys($data) as $field) {
        $setParts[] = "$field = ?";
    }
    
    $sql = sprintf(
        "UPDATE %s SET %s WHERE %s",
        $table,
        implode(', ', $setParts),
        $where
    );
    
    $params = array_merge(array_values($data), $whereParams);
    return db_query($sql, $params);
}

/**
 * Helper function: Delete rows
 */
function db_delete($table, $where, $params = []) {
    $sql = sprintf("DELETE FROM %s WHERE %s", $table, $where);
    return db_query($sql, $params);
}

/**
 * Helper function: Get last insert ID
 */
function db_insert_id() {
    global $pdo;
    return $pdo->lastInsertId();
}
