<?php
/**
 * Authentication System
 * Handles both local and LDAP authentication
 */

/**
 * Authenticate user with username and password
 * Tries LDAP first (if enabled), then local
 */
function authenticate_user($username, $password) {
    global $pdo;
    
    // Try LDAP first if enabled
    if (getSetting('ldap_enabled') === '1') {
        $ldap_result = authenticate_ldap($username, $password);
        if ($ldap_result['success']) {
            // Check if user already exists in DB
            $existing_user = db_get_row(
                "SELECT u.*, ug.name as group_name 
                 FROM users u 
                 JOIN user_groups ug ON u.user_group_id = ug.id 
                 WHERE u.username = ? AND u.auth_method = 'ldap'",
                [$ldap_result['user_data']['username']]
            );
            
            if ($existing_user) {
                // User exists in DB - update and return
                db_query("UPDATE users SET last_login = NOW() WHERE id = ?", [$existing_user['id']]);
                return ['success' => true, 'user' => $existing_user];
            } else {
                // User NOT in DB - return LDAP-only user (will be saved on first loan)
                return [
                    'success' => true, 
                    'user' => [
                        'id' => null,  // NULL = not in DB yet
                        'username' => $ldap_result['user_data']['username'],
                        'email' => $ldap_result['user_data']['email'],
                        'first_name' => $ldap_result['user_data']['first_name'],
                        'last_name' => $ldap_result['user_data']['last_name'],
                        'auth_method' => 'ldap',
                        'status' => 'aktiv',
                        'group_name' => 'User',
                        'user_group_id' => 2,  // Default User group
                        'ldap_only' => true     // Flag: not yet in DB
                    ]
                ];
            }
        }
    }
    
    // Try local authentication
    $user = db_get_row(
        "SELECT u.*, ug.name as group_name 
         FROM users u 
         JOIN user_groups ug ON u.user_group_id = ug.id 
         WHERE (u.username = ? OR u.email = ?) 
         AND u.auth_method = 'local' 
         AND u.status = 'aktiv'",
        [$username, $username]
    );
    
    if (!$user || !$user['password_hash']) {
        return ['success' => false, 'error' => 'Benutzername oder Passwort falsch'];
    }
    
    // Verify password
    if (!password_verify($password, $user['password_hash'])) {
        return ['success' => false, 'error' => 'Benutzername oder Passwort falsch'];
    }
    
    // Check if password needs rehashing (security improvement)
    if (password_needs_rehash($user['password_hash'], PASSWORD_ARGON2ID)) {
        $new_hash = password_hash($password, PASSWORD_ARGON2ID);
        db_update('users', ['password_hash' => $new_hash], 'id = ?', [$user['id']]);
    }
    
    return ['success' => true, 'user' => $user];
}

/**
 * Authenticate via LDAP/Active Directory
 */
function authenticate_ldap($username, $password) {
    if (!extension_loaded('ldap')) {
        return ['success' => false, 'error' => 'LDAP extension not installed'];
    }
    
    $server = getSetting('ldap_host');
    $port = (int)getSetting('ldap_port', '389');
    $base_dn = getSetting('ldap_base_dn');
    $bind_dn = getSetting('ldap_bind_dn');
    $bind_password = getSetting('ldap_bind_password');
    $username_attr = getSetting('ldap_username_attribute', 'sAMAccountName');
    $email_attr = getSetting('ldap_email_attribute', 'mail');
    $firstname_attr = getSetting('ldap_firstname_attribute', 'givenName');
    $lastname_attr = getSetting('ldap_lastname_attribute', 'sn');
    
    // Build filter - simple user search
    $user_filter = '(&(objectClass=user)(' . $username_attr . '=' . ldap_escape($username, '', LDAP_ESCAPE_FILTER) . '))';
    
    if (!$server || !$base_dn) {
        return ['success' => false, 'error' => 'LDAP not properly configured'];
    }
    
    try {
        // Connect to LDAP server
        $ldap = ldap_connect($server, $port);
        if (!$ldap) {
            return ['success' => false, 'error' => 'Could not connect to LDAP server'];
        }
        
        // Set LDAP options
        ldap_set_option($ldap, LDAP_OPT_PROTOCOL_VERSION, 3);
        ldap_set_option($ldap, LDAP_OPT_REFERRALS, 0);
        ldap_set_option($ldap, LDAP_OPT_NETWORK_TIMEOUT, 10);
        
        // Bind with service account
        if (!@ldap_bind($ldap, $bind_dn, $bind_password)) {
            ldap_close($ldap);
            return ['success' => false, 'error' => 'LDAP bind failed'];
        }
        
        // Search for user
        $search = @ldap_search($ldap, $base_dn, $user_filter, [
            'dn', $username_attr, $email_attr, $firstname_attr, $lastname_attr
        ]);
        
        if (!$search) {
            ldap_close($ldap);
            return ['success' => false, 'error' => 'User not found in LDAP'];
        }
        
        $entries = ldap_get_entries($ldap, $search);
        if ($entries['count'] === 0) {
            ldap_close($ldap);
            return ['success' => false, 'error' => 'User not found in LDAP'];
        }
        
        $user_dn = $entries[0]['dn'];
        
        // Try to bind as user (authenticate)
        if (!@ldap_bind($ldap, $user_dn, $password)) {
            ldap_close($ldap);
            return ['success' => false, 'error' => 'Invalid LDAP credentials'];
        }
        
        // Extract user data
        $user_data = [
            'username' => $entries[0][strtolower($username_attr)][0] ?? $username,
            'email' => $entries[0][strtolower($email_attr)][0] ?? '',
            'first_name' => $entries[0][strtolower($firstname_attr)][0] ?? '',
            'last_name' => $entries[0][strtolower($lastname_attr)][0] ?? '',
            'ldap_dn' => $user_dn
        ];
        
        ldap_close($ldap);
        
        return ['success' => true, 'user_data' => $user_data];
        
    } catch (Exception $e) {
        if (isset($ldap)) {
            ldap_close($ldap);
        }
        error_log('LDAP Authentication Error: ' . $e->getMessage());
        return ['success' => false, 'error' => 'LDAP authentication failed'];
    }
}

/**
 * Save LDAP-only user to database
 * Called when LDAP user performs first loan
 * Returns user ID or false on failure
 */
function save_ldap_user_to_db() {
    // Check if current user is LDAP-only
    if (!isset($_SESSION['ldap_only']) || $_SESSION['ldap_only'] !== true) {
        return $_SESSION['user_id'] ?? false;
    }
    
    // Create user in database
    $user_group_id = db_get_var("SELECT id FROM user_groups WHERE name = 'User' LIMIT 1") ?? 2;
    
    $new_user_id = db_insert('users', [
        'user_group_id' => $user_group_id,
        'username' => $_SESSION['username'],
        'email' => $_SESSION['email'],
        'first_name' => $_SESSION['first_name'],
        'last_name' => $_SESSION['last_name'],
        'password_hash' => password_hash(bin2hex(random_bytes(16)), PASSWORD_ARGON2ID), // Random password (unused)
        'auth_method' => 'ldap',
        'status' => 'aktiv',
        'created_at' => date('Y-m-d H:i:s'),
        'last_login' => date('Y-m-d H:i:s')
    ]);
    
    if ($new_user_id) {
        // Update session - user is now in DB
        $_SESSION['user_id'] = $new_user_id;
        $_SESSION['ldap_only'] = false;
        
        // Log activity
        log_activity($new_user_id, 'ldap_user_created', 'users', $new_user_id, 'LDAP user auto-created: ' . $_SESSION['username']);
        
        return $new_user_id;
    }
    
    return false;
}

/**
 * Sync LDAP user to local database
 * Creates user if not exists, updates if exists
 * @deprecated - Use save_ldap_user_to_db() instead for on-demand creation
 */
function sync_ldap_user($ldap_data) {
    global $pdo;
    
    // Check if user exists
    $user = db_get_row(
        "SELECT u.*, ug.name as group_name 
         FROM users u 
         JOIN user_groups ug ON u.user_group_id = ug.id 
         WHERE u.username = ? OR u.email = ?",
        [$ldap_data['username'], $ldap_data['email']]
    );
    
    if ($user) {
        // Update existing user
        db_update('users', [
            'email' => $ldap_data['email'],
            'first_name' => $ldap_data['first_name'],
            'last_name' => $ldap_data['last_name'],
            'ldap_dn' => $ldap_data['ldap_dn'],
            'auth_method' => 'ldap',
            'last_login' => date('Y-m-d H:i:s')
        ], 'id = ?', [$user['id']]);
        
        // Reload user data
        return db_get_row(
            "SELECT u.*, ug.name as group_name 
             FROM users u 
             JOIN user_groups ug ON u.user_group_id = ug.id 
             WHERE u.id = ?",
            [$user['id']]
        );
    } else {
        // Create new user (as regular user, not admin)
        $user_group_id = db_get_var("SELECT id FROM user_groups WHERE name = 'User' LIMIT 1");
        
        $new_user_id = db_insert('users', [
            'user_group_id' => $user_group_id,
            'username' => $ldap_data['username'],
            'email' => $ldap_data['email'],
            'first_name' => $ldap_data['first_name'],
            'last_name' => $ldap_data['last_name'],
            'ldap_dn' => $ldap_data['ldap_dn'],
            'auth_method' => 'ldap',
            'status' => 'aktiv',
            'last_login' => date('Y-m-d H:i:s')
        ]);
        
        if ($new_user_id) {
            return db_get_row(
                "SELECT u.*, ug.name as group_name 
                 FROM users u 
                 JOIN user_groups ug ON u.user_group_id = ug.id 
                 WHERE u.id = ?",
                [$new_user_id]
            );
        }
    }
    
    return null;
}

/**
 * Start user session
 */
function login_user($user) {
    // Regenerate session ID (prevent session fixation)
    session_regenerate_id(true);
    
    // Set session variables
    $_SESSION['user_id'] = $user['id'];  // Can be NULL for LDAP-only users
    $_SESSION['username'] = $user['username'];
    $_SESSION['user_group'] = $user['group_name'];
    $_SESSION['user_group_id'] = $user['user_group_id'];
    $_SESSION['first_name'] = $user['first_name'];
    $_SESSION['last_name'] = $user['last_name'];
    $_SESSION['email'] = $user['email'];
    $_SESSION['auth_method'] = $user['auth_method'] ?? 'local';
    $_SESSION['ldap_only'] = $user['ldap_only'] ?? false;  // Flag for LDAP-only users
    $_SESSION['logged_in_at'] = time();
    
    // Update last login (only if user exists in DB)
    if ($user['id'] !== null) {
        db_query("UPDATE users SET last_login = NOW() WHERE id = ?", [$user['id']]);
        
        // Log activity (only if user exists in DB)
        log_activity($user['id'], 'user_login', 'users', $user['id'], 'User logged in: ' . $user['username']);
    }
}

/**
 * Logout user
 */
function logout_user() {
    if (is_logged_in()) {
        log_activity('user_logout', 'users', $_SESSION['user_id'], 'User logged out: ' . $_SESSION['username']);
    }
    
    // Destroy session
    $_SESSION = [];
    
    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time() - 3600, '/');
    }
    
    session_destroy();
}

/**
 * Check rate limiting (brute force protection)
 * Max 5 attempts per 15 minutes per IP
 */
function check_rate_limit($identifier = null) {
    if (!$identifier) {
        $identifier = $_SERVER['REMOTE_ADDR'];
    }
    
    $cache_file = sys_get_temp_dir() . '/bib_login_' . md5($identifier) . '.tmp';
    
    if (!file_exists($cache_file)) {
        return ['allowed' => true, 'attempts' => 0];
    }
    
    $data = json_decode(file_get_contents($cache_file), true);
    $attempts = $data['attempts'] ?? 0;
    $first_attempt = $data['first_attempt'] ?? time();
    
    // Reset if 15 minutes passed
    if (time() - $first_attempt > 900) {
        unlink($cache_file);
        return ['allowed' => true, 'attempts' => 0];
    }
    
    // Check if limit exceeded
    if ($attempts >= 5) {
        $remaining = 900 - (time() - $first_attempt);
        return [
            'allowed' => false, 
            'attempts' => $attempts,
            'wait_time' => ceil($remaining / 60)
        ];
    }
    
    return ['allowed' => true, 'attempts' => $attempts];
}

/**
 * Record failed login attempt
 */
function record_failed_attempt($identifier = null) {
    if (!$identifier) {
        $identifier = $_SERVER['REMOTE_ADDR'];
    }
    
    $cache_file = sys_get_temp_dir() . '/bib_login_' . md5($identifier) . '.tmp';
    
    if (file_exists($cache_file)) {
        $data = json_decode(file_get_contents($cache_file), true);
        $attempts = $data['attempts'] + 1;
        $first_attempt = $data['first_attempt'];
    } else {
        $attempts = 1;
        $first_attempt = time();
    }
    
    file_put_contents($cache_file, json_encode([
        'attempts' => $attempts,
        'first_attempt' => $first_attempt
    ]));
}

/**
 * Clear rate limit (after successful login)
 */
function clear_rate_limit($identifier = null) {
    if (!$identifier) {
        $identifier = $_SERVER['REMOTE_ADDR'];
    }
    
    $cache_file = sys_get_temp_dir() . '/bib_login_' . md5($identifier) . '.tmp';
    
    if (file_exists($cache_file)) {
        unlink($cache_file);
    }
}
