<?php
/**
 * User Dashboard
 * Main dashboard for logged-in users
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/session.php';

// Require login
require_login();

$page_title = 'Dashboard';
$current_user = get_logged_in_user();
$lang = load_language();

// Get user statistics
$stats = [
    'current_loans' => db_get_var("SELECT COUNT(*) FROM loans WHERE user_id = ? AND status IN ('aktiv', 'überfällig')", [$current_user['id']]),
    'overdue_loans' => db_get_var("SELECT COUNT(*) FROM loans WHERE user_id = ? AND status = 'überfällig'", [$current_user['id']]),
    'reservations' => db_get_var("SELECT COUNT(*) FROM reservations WHERE user_id = ? AND status IN ('aktiv', 'verfügbar')", [$current_user['id']]),
    'total_loans' => $current_user['total_loans'] ?? 0
];

// Get active loans with book details
$active_loans = db_get_results("
    SELECT l.*, bc.book_number, bc.signature,
           b.title, b.author, b.isbn, b.image_url,
           DATEDIFF(l.due_date, CURDATE()) as days_remaining
    FROM loans l
    JOIN book_copies bc ON l.copy_id = bc.id
    JOIN books b ON bc.book_id = b.id
    WHERE l.user_id = ? AND l.status IN ('aktiv', 'überfällig')
    ORDER BY l.due_date ASC
    LIMIT 5
", [$current_user['id']]);

// Get active reservations
$active_reservations = db_get_results("
    SELECT r.*, b.title, b.author, b.isbn, b.image_url,
           (SELECT COUNT(*) FROM reservations r2 
            WHERE r2.book_id = r.book_id 
            AND r2.status = 'aktiv' 
            AND r2.reserved_date < r.reserved_date) as queue_position
    FROM reservations r
    JOIN books b ON r.book_id = b.id
    WHERE r.user_id = ? AND r.status IN ('aktiv', 'verfügbar')
    ORDER BY r.status DESC, r.reserved_date ASC
    LIMIT 5
", [$current_user['id']]);

// Get new books (last 30 days)
$new_books = db_get_results("
    SELECT b.*, COUNT(bc.id) as available_copies
    FROM books b
    LEFT JOIN book_copies bc ON b.id = bc.book_id AND bc.status = 'verfügbar'
    WHERE b.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
    GROUP BY b.id
    ORDER BY b.created_at DESC
    LIMIT 6
");

// Get popular books (most borrowed in last 90 days)
$popular_books = db_get_results("
    SELECT b.*, COUNT(l.id) as loan_count,
           (SELECT COUNT(*) FROM book_copies bc2 WHERE bc2.book_id = b.id AND bc2.status = 'verfügbar') as available_copies
    FROM books b
    LEFT JOIN book_copies bc ON b.id = bc.book_id
    LEFT JOIN loans l ON bc.id = l.copy_id
        AND l.loan_date >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)
    GROUP BY b.id
    HAVING loan_count > 0
    ORDER BY loan_count DESC
    LIMIT 6
");

include __DIR__ . '/includes/header.php';
?>

<!-- Welcome Section -->
<div class="mb-8">
    <h1 class="text-3xl font-bold text-gray-900">
        <?php echo $lang['welcome'] ?? 'Willkommen'; ?>, <?php echo esc_html($current_user['first_name']); ?>!
    </h1>
    <p class="mt-2 text-gray-600">
        <?php echo $lang['dashboard_subtitle'] ?? 'Hier ist eine Übersicht über Ihre Bibliotheksaktivitäten'; ?>
    </p>
</div>

<!-- Statistics Cards -->
<div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
    <!-- Current Loans -->
    <div class="bg-white rounded-lg shadow p-6 hover-lift">
        <div class="flex items-center">
            <div class="flex-shrink-0 bg-blue-500 rounded-md p-3">
                <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                </svg>
            </div>
            <div class="ml-4">
                <p class="text-sm font-medium text-gray-500"><?php echo $lang['current_loans'] ?? 'Aktuelle Ausleihen'; ?></p>
                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['current_loans']; ?></p>
            </div>
        </div>
    </div>

    <!-- Overdue Loans -->
    <div class="bg-white rounded-lg shadow p-6 hover-lift">
        <div class="flex items-center">
            <div class="flex-shrink-0 <?php echo $stats['overdue_loans'] > 0 ? 'bg-red-500' : 'bg-green-500'; ?> rounded-md p-3">
                <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="ml-4">
                <p class="text-sm font-medium text-gray-500"><?php echo $lang['overdue_loans'] ?? 'Überfällig'; ?></p>
                <p class="text-2xl font-bold <?php echo $stats['overdue_loans'] > 0 ? 'text-red-600' : 'text-green-600'; ?>">
                    <?php echo $stats['overdue_loans']; ?>
                </p>
            </div>
        </div>
    </div>

    <!-- Active Reservations -->
    <div class="bg-white rounded-lg shadow p-6 hover-lift">
        <div class="flex items-center">
            <div class="flex-shrink-0 bg-yellow-500 rounded-md p-3">
                <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                </svg>
            </div>
            <div class="ml-4">
                <p class="text-sm font-medium text-gray-500"><?php echo $lang['reservations'] ?? 'Vormerkungen'; ?></p>
                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['reservations']; ?></p>
            </div>
        </div>
    </div>

    <!-- Total Loans -->
    <div class="bg-white rounded-lg shadow p-6 hover-lift">
        <div class="flex items-center">
            <div class="flex-shrink-0 bg-purple-500 rounded-md p-3">
                <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                </svg>
            </div>
            <div class="ml-4">
                <p class="text-sm font-medium text-gray-500"><?php echo $lang['total_loans'] ?? 'Gesamt ausgeliehen'; ?></p>
                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total_loans']; ?></p>
            </div>
        </div>
    </div>
</div>

<!-- Quick Search -->
<div class="bg-white rounded-lg shadow mb-8 p-6">
    <h2 class="text-xl font-bold text-gray-900 mb-4">
        <svg class="inline h-6 w-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
        </svg>
        <?php echo $lang['quick_search'] ?? 'Schnellsuche'; ?>
    </h2>
    <form action="<?php echo BASE_URL; ?>/search.php" method="GET" class="flex gap-4">
        <input type="text" 
               name="q" 
               placeholder="<?php echo $lang['search_placeholder'] ?? 'Titel, Autor, ISBN...'; ?>"
               class="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
        <button type="submit" 
                class="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium">
            <?php echo $lang['search'] ?? 'Suchen'; ?>
        </button>
    </form>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
    <!-- Active Loans -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                </svg>
                <?php echo $lang['my_loans'] ?? 'Meine Ausleihen'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($active_loans)): ?>
                <div class="text-center py-8 text-gray-500">
                    <svg class="mx-auto h-12 w-12 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                    </svg>
                    <p><?php echo $lang['no_active_loans'] ?? 'Keine aktiven Ausleihen'; ?></p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($active_loans as $loan): ?>
                        <div class="flex gap-4 p-4 bg-gray-50 rounded-lg hover:bg-gray-100">
                            <div class="flex-shrink-0">
                                <?php if ($loan['image_url']): ?>
                                    <img src="<?php echo esc_url($loan['image_url']); ?>" 
                                         alt="<?php echo esc_attr($loan['title']); ?>"
                                         class="w-16 h-20 object-cover rounded">
                                <?php else: ?>
                                    <div class="w-16 h-20 bg-gray-300 rounded flex items-center justify-center">
                                        <svg class="h-8 w-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                        </svg>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="flex-1 min-w-0">
                                <p class="font-medium text-gray-900 truncate"><?php echo esc_html($loan['title']); ?></p>
                                <p class="text-sm text-gray-500"><?php echo esc_html($loan['author']); ?></p>
                                <p class="text-xs text-gray-400 mt-1">
                                    <?php echo $lang['book_number'] ?? 'Buchnr.'; ?>: <?php echo esc_html($loan['book_number']); ?>
                                </p>
                                <div class="mt-2">
                                    <?php if ($loan['status'] === 'überfällig'): ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                            <?php echo abs($loan['days_remaining']); ?> <?php echo $lang['days_overdue'] ?? 'Tage überfällig'; ?>
                                        </span>
                                    <?php elseif ($loan['days_remaining'] <= 3): ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                            <?php echo $loan['days_remaining']; ?> <?php echo $lang['days_remaining'] ?? 'Tage verbleibend'; ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                            <?php echo $loan['days_remaining']; ?> <?php echo $lang['days_remaining'] ?? 'Tage verbleibend'; ?>
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <?php if ($stats['current_loans'] > 5): ?>
                    <div class="mt-4 text-center">
                        <a href="<?php echo BASE_URL; ?>/loans.php" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                            <?php echo $lang['view_all_loans'] ?? 'Alle Ausleihen anzeigen'; ?> →
                        </a>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>

    <!-- Active Reservations -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                </svg>
                <?php echo $lang['my_reservations'] ?? 'Meine Vormerkungen'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($active_reservations)): ?>
                <div class="text-center py-8 text-gray-500">
                    <svg class="mx-auto h-12 w-12 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                    </svg>
                    <p><?php echo $lang['no_active_reservations'] ?? 'Keine aktiven Vormerkungen'; ?></p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($active_reservations as $reservation): ?>
                        <div class="flex gap-4 p-4 bg-gray-50 rounded-lg hover:bg-gray-100">
                            <div class="flex-shrink-0">
                                <?php if ($reservation['image_url']): ?>
                                    <img src="<?php echo esc_url($reservation['image_url']); ?>" 
                                         alt="<?php echo esc_attr($reservation['title']); ?>"
                                         class="w-16 h-20 object-cover rounded">
                                <?php else: ?>
                                    <div class="w-16 h-20 bg-gray-300 rounded flex items-center justify-center">
                                        <svg class="h-8 w-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                        </svg>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="flex-1 min-w-0">
                                <p class="font-medium text-gray-900 truncate"><?php echo esc_html($reservation['title']); ?></p>
                                <p class="text-sm text-gray-500"><?php echo esc_html($reservation['author']); ?></p>
                                <div class="mt-2">
                                    <?php if ($reservation['status'] === 'verfügbar'): ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                            Abholbereit
                                        </span>
                                    <?php else: ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                            <?php echo $lang['position'] ?? 'Position'; ?>: <?php echo $reservation['queue_position'] + 1; ?>
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <?php if ($stats['reservations'] > 5): ?>
                    <div class="mt-4 text-center">
                        <a href="<?php echo BASE_URL; ?>/reservations.php" class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                            <?php echo $lang['view_all_reservations'] ?? 'Alle Vormerkungen anzeigen'; ?> →
                        </a>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- New Books & Popular Books -->
<div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
    <!-- New Books -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                </svg>
                <?php echo $lang['new_books'] ?? 'Neue Bücher'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($new_books)): ?>
                <p class="text-center py-8 text-gray-500"><?php echo $lang['no_new_books'] ?? 'Keine neuen Bücher'; ?></p>
            <?php else: ?>
                <div class="grid grid-cols-2 sm:grid-cols-3 gap-4">
                    <?php foreach ($new_books as $book): ?>
                        <a href="<?php echo BASE_URL; ?>/book.php?id=<?php echo $book['id']; ?>" 
                           class="group">
                            <div class="aspect-[2/3] bg-gray-200 rounded-lg overflow-hidden mb-2">
                                <?php if ($book['image_url']): ?>
                                    <img src="<?php echo esc_url($book['image_url']); ?>" 
                                         alt="<?php echo esc_attr($book['title']); ?>"
                                         class="w-full h-full object-cover group-hover:scale-105 transition">
                                <?php else: ?>
                                    <div class="w-full h-full flex items-center justify-center">
                                        <svg class="h-16 w-16 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                        </svg>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <p class="text-sm font-medium text-gray-900 truncate group-hover:text-blue-600">
                                <?php echo esc_html($book['title']); ?>
                            </p>
                            <p class="text-xs text-gray-500 truncate"><?php echo esc_html($book['author']); ?></p>
                            <?php if ($book['available_copies'] > 0): ?>
                                <span class="inline-block mt-1 px-2 py-0.5 text-xs font-medium bg-green-100 text-green-800 rounded">
                                    <?php echo $lang['available'] ?? 'Verfügbar'; ?>
                                </span>
                            <?php endif; ?>
                        </a>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Popular Books -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-orange-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"/>
                </svg>
                <?php echo $lang['popular_books'] ?? 'Beliebte Bücher'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($popular_books)): ?>
                <p class="text-center py-8 text-gray-500"><?php echo $lang['no_popular_books'] ?? 'Keine Daten verfügbar'; ?></p>
            <?php else: ?>
                <div class="grid grid-cols-2 sm:grid-cols-3 gap-4">
                    <?php foreach ($popular_books as $book): ?>
                        <a href="<?php echo BASE_URL; ?>/book.php?id=<?php echo $book['id']; ?>" 
                           class="group">
                            <div class="aspect-[2/3] bg-gray-200 rounded-lg overflow-hidden mb-2 relative">
                                <?php if ($book['image_url']): ?>
                                    <img src="<?php echo esc_url($book['image_url']); ?>" 
                                         alt="<?php echo esc_attr($book['title']); ?>"
                                         class="w-full h-full object-cover group-hover:scale-105 transition">
                                <?php else: ?>
                                    <div class="w-full h-full flex items-center justify-center">
                                        <svg class="h-16 w-16 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                        </svg>
                                    </div>
                                <?php endif; ?>
                                <div class="absolute top-2 right-2 bg-orange-500 text-white text-xs font-bold px-2 py-1 rounded">
                                    🔥 <?php echo $book['loan_count']; ?>
                                </div>
                            </div>
                            <p class="text-sm font-medium text-gray-900 truncate group-hover:text-blue-600">
                                <?php echo esc_html($book['title']); ?>
                            </p>
                            <p class="text-xs text-gray-500 truncate"><?php echo esc_html($book['author']); ?></p>
                            <?php if ($book['available_copies'] > 0): ?>
                                <span class="inline-block mt-1 px-2 py-0.5 text-xs font-medium bg-green-100 text-green-800 rounded">
                                    <?php echo $lang['available'] ?? 'Verfügbar'; ?>
                                </span>
                            <?php endif; ?>
                        </a>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
