<?php
/**
 * Buchdetail — book.php
 * Zeigt alle Metadaten, Exemplare und den persönlichen Ausleih-/Vormerkungsstatus.
 * Benutzer können sich selbst vormerken (POST action=reserve) oder
 * ihre eigene Vormerkung stornieren (POST action=cancel_reservation).
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/session.php';

require_login();

$current_user = get_logged_in_user();
$user_id      = (int)$current_user['id'];
$lang         = load_language();

// ── Buch-ID validieren ────────────────────────────────────────────────────────
$book_id = (int)($_GET['id'] ?? 0);
if ($book_id <= 0) {
    set_flash('error', 'Ungültige Buch-ID.');
    header('Location: ' . BASE_URL . '/search.php');
    exit;
}

// ── POST-Handler ──────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf_token($_POST['csrf_token'] ?? '');
    $action = $_POST['action'] ?? '';

    if ($action === 'reserve') {
        // Bereits vorgemerkt?
        $existing = db_get_var(
            "SELECT id FROM reservations
             WHERE book_id = ? AND user_id = ? AND status IN ('aktiv','verfügbar')",
            [$book_id, $user_id]
        );
        if ($existing) {
            set_flash('warning', 'Sie haben bereits eine aktive Vormerkung für dieses Buch.');
        } else {
            // Freies Exemplar vorhanden?
            $avail_copy = db_get_row(
                "SELECT id FROM book_copies WHERE book_id = ? AND status = 'verfügbar' LIMIT 1",
                [$book_id]
            );
            if ($avail_copy) {
                // Direkt reservieren
                $res_id = db_insert('reservations', [
                    'book_id'       => $book_id,
                    'user_id'       => $user_id,
                    'copy_id'       => $avail_copy['id'],
                    'status'        => 'verfügbar',
                    'reserved_date' => date('Y-m-d H:i:s'),
                ]);
                db_update('book_copies', ['status' => 'reserviert'], ['id' => $avail_copy['id']]);
                set_flash('success', 'Vormerkung erstellt – ein Exemplar liegt bereit. Bitte beim Bibliothekar abholen.');
            } else {
                // Warteschlange
                $res_id = db_insert('reservations', [
                    'book_id'       => $book_id,
                    'user_id'       => $user_id,
                    'copy_id'       => null,
                    'status'        => 'aktiv',
                    'reserved_date' => date('Y-m-d H:i:s'),
                ]);
                set_flash('success', 'Vormerkung erstellt. Sie werden informiert, sobald ein Exemplar zurückgegeben wird.');
            }
            log_activity($user_id, 'reservation_created', 'reservations', $res_id ?? 0,
                json_encode(['book_id' => $book_id, 'source' => 'book.php']));
        }

    } elseif ($action === 'cancel_reservation') {
        $res_id = (int)($_POST['reservation_id'] ?? 0);
        // Nur eigene, noch offene Vormerkungen stornieren
        $res = db_get_row(
            "SELECT * FROM reservations
             WHERE id = ? AND user_id = ? AND status IN ('aktiv','verfügbar')",
            [$res_id, $user_id]
        );
        if ($res) {
            db_update('reservations', ['status' => 'storniert'], ['id' => $res_id]);
            // Reserviertes Exemplar wieder freigeben
            if ($res['copy_id']) {
                db_update('book_copies', ['status' => 'verfügbar'], ['id' => $res['copy_id']]);
            }
            log_activity($user_id, 'reservation_cancelled', 'reservations', $res_id,
                json_encode(['book_id' => $book_id, 'by_user' => true]));
            set_flash('success', 'Vormerkung wurde storniert.');
        } else {
            set_flash('error', 'Vormerkung nicht gefunden oder bereits abgeschlossen.');
        }
    }

    header('Location: ' . BASE_URL . '/book.php?id=' . $book_id);
    exit;
}

// ── Buch + Kategorie laden ────────────────────────────────────────────────────
$book = db_get_row(
    "SELECT b.*,
            bs.code AS subcategory_code, bs.name AS subcategory_name,
            bc.code AS category_code,    bc.name AS category_name
     FROM books b
     LEFT JOIN book_subcategories bs ON b.subcategory_id = bs.id
     LEFT JOIN book_categories    bc ON bs.category_id   = bc.id
     WHERE b.id = ?",
    [$book_id]
);

if (!$book) {
    set_flash('error', 'Buch nicht gefunden.');
    header('Location: ' . BASE_URL . '/search.php');
    exit;
}

$page_title = mb_substr($book['title'], 0, 60);

// ── Exemplare (ohne aussortiert/verloren) ─────────────────────────────────────
$copies = db_get_results(
    "SELECT id, book_number, copy_index, signature, status, location, condition_note
     FROM book_copies
     WHERE book_id = ? AND status NOT IN ('aussortiert','verloren')
     ORDER BY copy_index ASC, id ASC",
    [$book_id]
);

// ── Persönlicher Status ───────────────────────────────────────────────────────
// Aktive Ausleihe
$my_loan = db_get_row(
    "SELECT l.id, l.loan_date, l.due_date, l.status,
            bc.book_number, bc.signature
     FROM loans l
     JOIN book_copies bc ON l.copy_id = bc.id
     WHERE bc.book_id = ? AND l.user_id = ? AND l.return_date IS NULL
     LIMIT 1",
    [$book_id, $user_id]
);

// Aktive Vormerkung
$my_reservation = db_get_row(
    "SELECT r.*,
            (SELECT COUNT(*) FROM reservations r2
             WHERE r2.book_id = r.book_id
               AND r2.status = 'aktiv'
               AND r2.reserved_date < r.reserved_date) + 1 AS queue_position
     FROM reservations r
     WHERE r.book_id = ? AND r.user_id = ? AND r.status IN ('aktiv','verfügbar')
     LIMIT 1",
    [$book_id, $user_id]
);

// Verfügbare Exemplare
$available_copies = (int)db_get_var(
    "SELECT COUNT(*) FROM book_copies WHERE book_id = ? AND status = 'verfügbar'",
    [$book_id]
);
$total_active_copies = count($copies);

$csrf_token = generate_csrf_token();

// ── Rückkehr-URL für Breadcrumb ───────────────────────────────────────────────
$back_url = isset($_SERVER['HTTP_REFERER']) &&
            str_contains($_SERVER['HTTP_REFERER'], '/search.php')
            ? $_SERVER['HTTP_REFERER']
            : BASE_URL . '/search.php';

// ── Hilfsfunktion: Exemplar-Status-Badge ─────────────────────────────────────
function copy_status_badge(string $status): string {
    $map = [
        'verfügbar'      => 'bg-green-100 text-green-800',
        'ausgeliehen'    => 'bg-red-100 text-red-800',
        'reserviert'     => 'bg-yellow-100 text-yellow-800',
        'in_bearbeitung' => 'bg-blue-100 text-blue-800',
    ];
    $class = $map[$status] ?? 'bg-gray-100 text-gray-600';
    return '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' . $class . '">'
         . esc_html($status) . '</span>';
}

include __DIR__ . '/includes/header.php';
?>

<!-- Breadcrumb -->
<nav class="text-sm text-gray-500 mb-6" aria-label="Breadcrumb">
    <a href="<?= BASE_URL ?>/dashboard.php" class="hover:text-blue-600">Dashboard</a>
    <span class="mx-2">/</span>
    <a href="<?= esc_url($back_url) ?>" class="hover:text-blue-600">Buchsuche</a>
    <span class="mx-2">/</span>
    <span class="text-gray-900 truncate"><?= esc_html(mb_substr($book['title'], 0, 40)) ?></span>
</nav>

<!-- ── Hauptbereich ─────────────────────────────────────────────────────────── -->
<div class="grid grid-cols-1 md:grid-cols-3 gap-8 mb-8">

    <!-- Cover -->
    <div class="md:col-span-1">
        <div class="bg-white rounded-lg shadow overflow-hidden">
            <?php if ($book['image_url']): ?>
                <img src="<?= esc_url($book['image_url']) ?>"
                     alt="<?= esc_attr($book['title']) ?>"
                     class="w-full object-contain max-h-96 bg-gray-50">
            <?php else: ?>
                <div class="w-full h-64 bg-gray-100 flex flex-col items-center justify-center text-gray-300">
                    <svg class="h-24 w-24 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                              d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                    </svg>
                    <span class="text-sm text-gray-400">Kein Cover vorhanden</span>
                </div>
            <?php endif; ?>
        </div>

        <!-- Availability summary under cover -->
        <div class="mt-4 bg-white rounded-lg shadow p-4 text-center">
            <?php if ($available_copies > 0): ?>
                <div class="text-green-600">
                    <svg class="mx-auto h-8 w-8 mb-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                              d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <p class="font-semibold"><?= $available_copies ?> Exemplar<?= $available_copies !== 1 ? 'e' : '' ?> verfügbar</p>
                    <p class="text-xs text-gray-500 mt-1">Beim Bibliothekar ausleihen</p>
                </div>
            <?php elseif ($total_active_copies > 0): ?>
                <div class="text-red-600">
                    <svg class="mx-auto h-8 w-8 mb-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                              d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <p class="font-semibold">Alle Exemplare ausgeliehen</p>
                    <p class="text-xs text-gray-500 mt-1">Vormerken möglich</p>
                </div>
            <?php else: ?>
                <div class="text-gray-500">
                    <p class="font-semibold">Keine Exemplare vorhanden</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Metadata + Status + Actions -->
    <div class="md:col-span-2 space-y-6">

        <!-- Title & Author -->
        <div class="bg-white rounded-lg shadow p-6">
            <h1 class="text-2xl font-bold text-gray-900 leading-tight mb-1">
                <?= esc_html($book['title']) ?>
            </h1>
            <?php if ($book['subtitle']): ?>
                <p class="text-lg text-gray-600 mb-2"><?= esc_html($book['subtitle']) ?></p>
            <?php endif; ?>
            <?php if ($book['author']): ?>
                <p class="text-base text-blue-700 font-medium mb-3"><?= esc_html($book['author']) ?></p>
            <?php endif; ?>
            <?php if ($book['author_additional']): ?>
                <p class="text-sm text-gray-500"><?= esc_html($book['author_additional']) ?></p>
            <?php endif; ?>

            <!-- Meta grid -->
            <dl class="mt-4 grid grid-cols-2 gap-x-6 gap-y-2 text-sm">
                <?php if ($book['isbn']): ?>
                    <div><dt class="text-gray-500">ISBN</dt><dd class="font-medium font-mono"><?= esc_html($book['isbn']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['year']): ?>
                    <div><dt class="text-gray-500">Erscheinungsjahr</dt><dd class="font-medium"><?= (int)$book['year'] ?></dd></div>
                <?php endif; ?>
                <?php if ($book['publisher']): ?>
                    <div><dt class="text-gray-500">Verlag</dt><dd class="font-medium"><?= esc_html($book['publisher']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['place_of_publication']): ?>
                    <div><dt class="text-gray-500">Erscheinungsort</dt><dd class="font-medium"><?= esc_html($book['place_of_publication']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['pages']): ?>
                    <div><dt class="text-gray-500">Seiten</dt><dd class="font-medium"><?= (int)$book['pages'] ?></dd></div>
                <?php endif; ?>
                <?php if ($book['language']): ?>
                    <div><dt class="text-gray-500">Sprache</dt><dd class="font-medium"><?= esc_html($book['language']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['edition']): ?>
                    <div><dt class="text-gray-500">Auflage</dt><dd class="font-medium"><?= esc_html($book['edition']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['subcategory_name']): ?>
                    <div><dt class="text-gray-500">Kategorie</dt>
                         <dd class="font-medium"><?= esc_html($book['subcategory_code'] . ' – ' . $book['subcategory_name']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['series']): ?>
                    <div class="col-span-2"><dt class="text-gray-500">Reihe</dt>
                         <dd class="font-medium"><?= esc_html($book['series']) ?><?= $book['series_volume'] ? ', Bd. ' . esc_html($book['series_volume']) : '' ?></dd></div>
                <?php endif; ?>
                <?php if ($book['editor']): ?>
                    <div><dt class="text-gray-500">Herausgeber</dt><dd class="font-medium"><?= esc_html($book['editor']) ?></dd></div>
                <?php endif; ?>
                <?php if ($book['translator']): ?>
                    <div><dt class="text-gray-500">Übersetzung</dt><dd class="font-medium"><?= esc_html($book['translator']) ?></dd></div>
                <?php endif; ?>
            </dl>

            <!-- Description -->
            <?php if ($book['description']): ?>
                <div class="mt-4 pt-4 border-t">
                    <h3 class="text-sm font-semibold text-gray-700 mb-2">Inhaltsbeschreibung</h3>
                    <div class="text-sm text-gray-600 leading-relaxed" id="descText">
                        <?php
                        $desc = esc_html($book['description']);
                        if (mb_strlen($book['description']) > 500):
                            $short = esc_html(mb_substr($book['description'], 0, 500));
                        ?>
                            <span id="descShort"><?= $short ?> <button onclick="document.getElementById('descShort').style.display='none';document.getElementById('descFull').style.display='inline'" class="text-blue-600 hover:underline text-xs">... mehr anzeigen</button></span>
                            <span id="descFull" style="display:none"><?= $desc ?> <button onclick="document.getElementById('descShort').style.display='inline';document.getElementById('descFull').style.display='none'" class="text-blue-600 hover:underline text-xs">weniger</button></span>
                        <?php else: ?>
                            <?= $desc ?>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Keywords -->
            <?php if ($book['keywords']): ?>
                <div class="mt-3">
                    <h3 class="text-xs font-semibold text-gray-500 uppercase tracking-wide mb-2">Schlagworte</h3>
                    <div class="flex flex-wrap gap-1">
                        <?php foreach (preg_split('/[,;]+/', $book['keywords']) as $kw): ?>
                            <?php $kw = trim($kw); if ($kw === '') continue; ?>
                            <a href="<?= BASE_URL ?>/search.php?q=<?= urlencode($kw) ?>"
                               class="px-2 py-0.5 bg-gray-100 hover:bg-blue-100 hover:text-blue-700 rounded text-xs text-gray-600">
                                <?= esc_html($kw) ?>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <!-- ── Persönlicher Status & Aktionen ─────────────────────────────── -->
        <?php if ($my_loan): ?>
            <!-- User has this book on loan -->
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-5">
                <div class="flex items-start gap-3">
                    <svg class="h-6 w-6 text-blue-500 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                              d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                    </svg>
                    <div>
                        <p class="font-semibold text-blue-900">Sie haben dieses Buch ausgeliehen</p>
                        <p class="text-sm text-blue-700 mt-1">
                            Exemplar: <strong><?= esc_html($my_loan['book_number']) ?></strong>
                            &middot;
                            Rückgabe bis:
                            <strong class="<?= strtotime($my_loan['due_date']) < time() ? 'text-red-700' : '' ?>">
                                <?= date('d.m.Y', strtotime($my_loan['due_date'])) ?>
                            </strong>
                        </p>
                        <?php if (strtotime($my_loan['due_date']) < time()): ?>
                            <p class="text-sm text-red-700 font-medium mt-1">
                                Dieses Buch ist überfällig – bitte umgehend zurückgeben.
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

        <?php elseif ($my_reservation): ?>
            <!-- User has a reservation -->
            <?php if ($my_reservation['status'] === 'verfügbar'): ?>
                <div class="bg-green-50 border border-green-200 rounded-lg p-5">
                    <div class="flex items-start gap-3">
                        <svg class="h-6 w-6 text-green-500 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                  d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <div>
                            <p class="font-semibold text-green-900">Ein Exemplar liegt für Sie bereit!</p>
                            <p class="text-sm text-green-700 mt-1">
                                Bitte beim Bibliothekar abholen.
                            </p>
                        </div>
                    </div>
                    <form method="POST" class="mt-3"
                          onsubmit="return confirm('Vormerkung wirklich stornieren?')">
                        <?= csrf_field() ?>
                        <input type="hidden" name="action" value="cancel_reservation">
                        <input type="hidden" name="reservation_id" value="<?= (int)$my_reservation['id'] ?>">
                        <button type="submit"
                                class="text-sm text-red-600 hover:text-red-800 hover:underline">
                            Vormerkung stornieren
                        </button>
                    </form>
                </div>
            <?php else: ?>
                <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-5">
                    <div class="flex items-start gap-3">
                        <svg class="h-6 w-6 text-yellow-500 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                  d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                        </svg>
                        <div>
                            <p class="font-semibold text-yellow-900">Sie haben dieses Buch vorgemerkt</p>
                            <p class="text-sm text-yellow-700 mt-1">
                                Warteschlangen-Position: <strong><?= (int)$my_reservation['queue_position'] ?></strong>
                                &middot;
                                Vorgemerkt seit: <?= date('d.m.Y', strtotime($my_reservation['reserved_date'])) ?>
                            </p>
                        </div>
                    </div>
                    <form method="POST" class="mt-3"
                          onsubmit="return confirm('Vormerkung wirklich stornieren?')">
                        <?= csrf_field() ?>
                        <input type="hidden" name="action" value="cancel_reservation">
                        <input type="hidden" name="reservation_id" value="<?= (int)$my_reservation['id'] ?>">
                        <button type="submit"
                                class="text-sm text-red-600 hover:text-red-800 hover:underline">
                            Vormerkung stornieren
                        </button>
                    </form>
                </div>
            <?php endif; ?>

        <?php elseif ($current_user['status'] !== 'inaktiv'): ?>
            <!-- Reservation button -->
            <div class="bg-white rounded-lg shadow p-5">
                <h3 class="font-semibold text-gray-800 mb-2">Vormerken</h3>
                <?php if ($available_copies > 0): ?>
                    <p class="text-sm text-gray-600 mb-3">
                        Aktuell <?= $available_copies ?> Exemplar<?= $available_copies !== 1 ? 'e' : '' ?> verfügbar.
                        Sie können sich trotzdem vormerken lassen – das Exemplar wird dann für Sie reserviert.
                    </p>
                <?php else: ?>
                    <p class="text-sm text-gray-600 mb-3">
                        Alle Exemplare sind derzeit ausgeliehen. Tragen Sie sich in die Warteliste ein –
                        Sie werden benachrichtigt, sobald ein Exemplar zurückgegeben wird.
                    </p>
                <?php endif; ?>
                <form method="POST">
                    <?= csrf_field() ?>
                    <input type="hidden" name="action" value="reserve">
                    <button type="submit"
                            class="px-5 py-2.5 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium text-sm">
                        <svg class="inline h-4 w-4 mr-1 -mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                  d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                        </svg>
                        Vormerken
                    </button>
                </form>
            </div>
        <?php endif; ?>

    </div><!-- /col-2 -->
</div><!-- /grid -->

<!-- ── Exemplare ────────────────────────────────────────────────────────────── -->
<?php if (!empty($copies)): ?>
<div class="bg-white rounded-lg shadow mb-8">
    <div class="p-5 border-b">
        <h2 class="text-lg font-semibold text-gray-900">
            Exemplare <span class="text-gray-400 font-normal">(<?= count($copies) ?>)</span>
        </h2>
    </div>
    <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
            <thead class="bg-gray-50 text-gray-500 text-xs uppercase">
                <tr>
                    <th class="px-5 py-3 text-left font-medium">Exemplar</th>
                    <th class="px-5 py-3 text-left font-medium">Signatur</th>
                    <th class="px-5 py-3 text-left font-medium">Standort</th>
                    <th class="px-5 py-3 text-left font-medium">Status</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
                <?php foreach ($copies as $copy): ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-5 py-3 font-mono text-gray-700">
                            <?= esc_html($copy['book_number'] ?? ('Ex. ' . $copy['copy_index'])) ?>
                        </td>
                        <td class="px-5 py-3 text-gray-600">
                            <?= esc_html($copy['signature'] ?? '–') ?>
                        </td>
                        <td class="px-5 py-3 text-gray-600">
                            <?= $copy['location'] ? esc_html($copy['location']) : '–' ?>
                        </td>
                        <td class="px-5 py-3">
                            <?= copy_status_badge($copy['status'] ?? '') ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>
<?php endif; ?>

<!-- ── Weitere Bücher von diesem Autor ───────────────────────────────────────── -->
<?php if ($book['author']):
    $same_author = db_get_results(
        "SELECT b.id, b.title, b.image_url, b.year,
                COALESCE(SUM(bc.status = 'verfügbar'), 0) AS available_copies
         FROM books b
         LEFT JOIN book_copies bc ON b.id = bc.book_id
         WHERE b.author LIKE ? AND b.id != ?
         GROUP BY b.id, b.title, b.image_url, b.year
         ORDER BY b.title_sort ASC
         LIMIT 5",
        ['%' . $book['author'] . '%', $book_id]
    );
    if (!empty($same_author)):
?>
<div class="bg-white rounded-lg shadow mb-8">
    <div class="p-5 border-b">
        <h2 class="text-lg font-semibold text-gray-900">
            Weitere Bücher von
            <a href="<?= BASE_URL ?>/search.php?q=<?= urlencode($book['author']) ?>"
               class="text-blue-600 hover:underline"><?= esc_html($book['author']) ?></a>
        </h2>
    </div>
    <div class="p-5">
        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 gap-4">
            <?php foreach ($same_author as $related): ?>
                <a href="<?= BASE_URL ?>/book.php?id=<?= (int)$related['id'] ?>"
                   class="group text-center">
                    <div class="aspect-[2/3] bg-gray-100 rounded overflow-hidden mb-2">
                        <?php if ($related['image_url']): ?>
                            <img src="<?= esc_url($related['image_url']) ?>"
                                 alt="<?= esc_attr($related['title']) ?>"
                                 class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200"
                                 loading="lazy">
                        <?php else: ?>
                            <div class="w-full h-full flex items-center justify-center">
                                <svg class="h-10 w-10 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                          d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                </svg>
                            </div>
                        <?php endif; ?>
                    </div>
                    <p class="text-xs font-medium text-gray-800 group-hover:text-blue-600 line-clamp-2 leading-snug">
                        <?= esc_html($related['title']) ?>
                    </p>
                    <?php if ((int)$related['available_copies'] > 0): ?>
                        <span class="inline-block mt-1 px-1.5 py-0.5 bg-green-100 text-green-700 rounded text-xs">verfügbar</span>
                    <?php endif; ?>
                </a>
            <?php endforeach; ?>
        </div>
    </div>
</div>
<?php endif; endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>
