<?php
/**
 * Sokrates-Import
 * Importiert Schuelerdaten aus dem Schulverwaltungssystem Sokrates (CSV-Export)
 *
 * CSV-Format: Semikolon-getrennt, UTF-8 oder Windows-1252
 * Spalten: Schuelerkennzahl, Nachname, Vorname, Lesergruppe, Geburtsdatum,
 *          Mobiltelefon, Strasse, PLZ, Ort, Email (ignoriert)
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang = load_language();
$page_title = $lang['sokrates_import_title'] ?? 'Sokrates-Import';
$current_user = get_logged_in_user();

define('SOKRATES_DIR', __DIR__ . '/../uploads/sokrates/');

// Initialize wizard session
if (!isset($_SESSION['sokrates_import'])) {
    $_SESSION['sokrates_import'] = ['step' => 1, 'filename' => null, 'deactivate_missing' => false, 'stats' => []];
}
$import = &$_SESSION['sokrates_import'];

// ─── FILE UPLOAD ──────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'upload') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', $lang['security_error'] ?? 'Sicherheitsfehler.');
        header('Location: users-sokrates-import.php');
        exit;
    }

    if (!is_dir(SOKRATES_DIR)) {
        mkdir(SOKRATES_DIR, 0755, true);
    }

    $file = $_FILES['csv_file'] ?? null;
    if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
        set_flash('error', 'Upload-Fehler (Code ' . ($file['error'] ?? -1) . ').');
        header('Location: users-sokrates-import.php');
        exit;
    }

    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if ($ext !== 'csv') {
        set_flash('error', 'Nur CSV-Dateien erlaubt.');
        header('Location: users-sokrates-import.php');
        exit;
    }

    $filename = 'sokrates_' . date('YmdHis') . '.csv';
    if (!move_uploaded_file($file['tmp_name'], SOKRATES_DIR . $filename)) {
        set_flash('error', 'Konnte Datei nicht speichern.');
        header('Location: users-sokrates-import.php');
        exit;
    }

    $import['filename'] = $filename;
    $import['deactivate_missing'] = !empty($_POST['deactivate_missing']);
    $import['step'] = 2;
    $import['stats'] = [];

    header('Location: users-sokrates-import.php');
    exit;
}

// ─── AJAX HANDLERS ────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] !== 'upload') {
    header('Content-Type: application/json; charset=utf-8');
    try {
        switch ($_POST['action']) {
            case 'preview': handlePreview(); break;
            case 'import':  handleImport();  break;
            case 'reset':   handleReset();   break;
            default: echo json_encode(['success' => false, 'error' => 'Unbekannte Aktion']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// ─── CSV LESEN ────────────────────────────────────────────────────────────────

/**
 * Liest die Sokrates-CSV und gibt assoziative Zeilen zurueck.
 * Erkennt automatisch UTF-8 vs. Windows-1252.
 * Trenner: Semikolon.
 */
function readSokratesCsv($limit = null) {
    global $import;
    $filepath = SOKRATES_DIR . $import['filename'];
    if (!file_exists($filepath)) {
        throw new Exception('CSV-Datei nicht gefunden.');
    }

    $handle = fopen($filepath, 'r');
    if (!$handle) throw new Exception('Kann CSV nicht oeffnen.');

    // Encoding-Erkennung
    $sample = fread($handle, 4096);
    $isUtf8 = mb_check_encoding($sample, 'UTF-8');
    rewind($handle);

    // Header lesen (Semikolon-getrennt)
    $headerRaw = fgetcsv($handle, 0, ';');
    if (!$headerRaw) { fclose($handle); throw new Exception('CSV leer oder ungueltig.'); }

    $headers = array_map(function($v) use ($isUtf8) {
        $v = $isUtf8 ? $v : iconv('Windows-1252', 'UTF-8//IGNORE', $v);
        return trim($v);
    }, $headerRaw);

    $data = [];
    $count = 0;
    while (($row = fgetcsv($handle, 0, ';')) !== false) {
        if (!$isUtf8) {
            $row = array_map(fn($v) => iconv('Windows-1252', 'UTF-8//IGNORE', $v), $row);
        }
        if (count($row) !== count($headers)) continue;
        $assoc = array_combine($headers, array_map('trim', $row));
        if (empty(array_filter($assoc))) continue; // Leerzeile
        $data[] = $assoc;
        $count++;
        if ($limit !== null && $count >= $limit) break;
    }
    fclose($handle);
    return $data;
}

/**
 * Parst Datum im Format d.m.Y nach Y-m-d
 */
function parseSokratesDate($str) {
    $str = trim($str);
    if (empty($str)) return null;
    $d = DateTime::createFromFormat('d.m.Y', $str);
    if ($d) return $d->format('Y-m-d');
    $d = DateTime::createFromFormat('Y-m-d', $str);
    if ($d) return $d->format('Y-m-d');
    return null;
}

/**
 * Baut Adresse aus Teilfeldern zusammen
 */
function buildAddress($row) {
    $strasse = trim($row['Strasse'] ?? $row['Straße'] ?? '');
    $plz     = trim($row['PLZ'] ?? '');
    $ort     = trim($row['Ort'] ?? '');
    $parts = [];
    if ($strasse) $parts[] = $strasse;
    if ($plz || $ort) $parts[] = trim("$plz $ort");
    return implode(', ', $parts) ?: null;
}

/**
 * Ermittelt Student-ID aus der Zeile (verschiedene Spaltennamen moeglich)
 */
function getStudentId($row) {
    foreach (['Schuelerkennzahl', 'Schülerkennzahl', 'SchülerKZ', 'StudentID', 'student_id'] as $k) {
        if (!empty($row[$k])) return trim($row[$k]);
    }
    return null;
}

// ─── AJAX HANDLER: VORSCHAU ───────────────────────────────────────────────────

function handlePreview() {
    global $import;
    if (!$import['filename']) {
        echo json_encode(['success' => false, 'error' => 'Keine Datei hochgeladen.']);
        return;
    }

    $rows = readSokratesCsv(20);
    $total = countSokratesRows();

    // Schüler-Gruppe ermitteln
    $schuelerGroupId = db_get_var("SELECT id FROM user_groups WHERE name IN ('Schueler', 'Schüler', 'User', 'Schüler*innen') ORDER BY id LIMIT 1") ?? 2;

    // Email-Spalte vorhanden?
    $hasEmail = false;
    if (!empty($rows)) {
        $hasEmail = array_key_exists('Email', $rows[0]) || array_key_exists('E-Mail', $rows[0]);
    }

    $preview = [];
    foreach ($rows as $row) {
        $studentId = getStudentId($row);
        $lastName  = trim($row['Nachname'] ?? '');
        $firstName = trim($row['Vorname'] ?? '');
        $className = trim($row['Lesergruppe'] ?? '');
        $birthday  = parseSokratesDate($row['Geburtsdatum'] ?? '');

        // Status bestimmen: Neu / Update / Unveraendert
        $existing = null;
        if ($studentId) {
            $existing = db_get_row("SELECT id, last_name, first_name, class_name, birthday, status FROM users WHERE student_id = ?", [$studentId]);
        }

        if (!$existing) {
            $status = 'neu';
        } elseif (
            $existing['last_name'] !== $lastName ||
            $existing['first_name'] !== $firstName ||
            $existing['class_name'] !== $className ||
            ($birthday && $existing['birthday'] !== $birthday)
        ) {
            $status = 'update';
        } else {
            $status = 'unveraendert';
        }

        $preview[] = [
            'student_id' => $studentId,
            'last_name'  => $lastName,
            'first_name' => $firstName,
            'class_name' => $className,
            'birthday'   => $birthday,
            'status'     => $status,
        ];
    }

    echo json_encode([
        'success'   => true,
        'total'     => $total,
        'preview'   => $preview,
        'has_email' => $hasEmail,
        'deactivate_missing' => $import['deactivate_missing'],
    ]);
}

function countSokratesRows() {
    global $import;
    $filepath = SOKRATES_DIR . $import['filename'];
    $handle = fopen($filepath, 'r');
    fgetcsv($handle, 0, ';'); // Header
    $count = 0;
    while (fgetcsv($handle, 0, ';') !== false) $count++;
    fclose($handle);
    return $count;
}

// ─── AJAX HANDLER: IMPORT ────────────────────────────────────────────────────

function handleImport() {
    global $import, $pdo;

    if (!$import['filename']) {
        echo json_encode(['success' => false, 'error' => 'Keine Datei hochgeladen.']);
        return;
    }

    set_time_limit(0);

    // Schueler-Gruppe
    $schuelerGroupId = db_get_var("SELECT id FROM user_groups WHERE name IN ('Schueler', 'Schüler', 'User', 'Schüler*innen') ORDER BY id LIMIT 1") ?? 2;

    $stats = [
        'total'       => 0,
        'inserted'    => 0,
        'updated'     => 0,
        'unchanged'   => 0,
        'deactivated' => 0,
        'errors'      => [],
    ];

    $rows = readSokratesCsv();
    $stats['total'] = count($rows);

    // Alle Student-IDs aus CSV sammeln (fuer optionale Deaktivierung)
    $csvStudentIds = [];

    $pdo->beginTransaction();
    try {
        foreach ($rows as $row) {
            $studentId = getStudentId($row);
            if (!$studentId) {
                $stats['errors'][] = 'Zeile ohne Schuelerkennzahl uebersprungen.';
                continue;
            }
            $csvStudentIds[] = $studentId;

            $lastName  = trim($row['Nachname'] ?? '');
            $firstName = trim($row['Vorname'] ?? '');
            $className = trim($row['Lesergruppe'] ?? '');
            $birthday  = parseSokratesDate($row['Geburtsdatum'] ?? '');
            $phone     = trim($row['Mobiltelefon'] ?? '') ?: null;
            $address   = buildAddress($row);

            // Vorhandenen Benutzer suchen
            $existing = db_get_row("SELECT id, email, username, ldap_dn, password_hash FROM users WHERE student_id = ?", [$studentId]);

            if (!$existing) {
                // INSERT
                db_insert('users', [
                    'user_group_id' => $schuelerGroupId,
                    'username'      => null,
                    'email'         => null,
                    'last_name'     => $lastName,
                    'first_name'    => $firstName,
                    'birthday'      => $birthday,
                    'class_name'    => $className,
                    'student_id'    => $studentId,
                    'phone'         => $phone,
                    'address'       => $address,
                    'auth_method'   => 'ldap',
                    'status'        => 'aktiv',
                    'created_at'    => date('Y-m-d H:i:s'),
                ]);
                $stats['inserted']++;
            } else {
                // UPDATE: Stammdaten immer aktualisieren
                // email/username/ldap_dn/password_hash nur wenn noch NULL
                $updateData = [
                    'last_name'  => $lastName,
                    'first_name' => $firstName,
                    'birthday'   => $birthday,
                    'class_name' => $className,
                    'student_id' => $studentId,
                    'status'     => 'aktiv',
                ];
                if ($phone !== null) $updateData['phone'] = $phone;
                if ($address !== null) $updateData['address'] = $address;

                // Sensible Felder nur setzen wenn noch leer
                // (werden beim LDAP-Login ausgefuellt)

                db_update('users', $updateData, 'id = ?', [$existing['id']]);
                $stats['updated']++;
            }
        }

        // Optional: Nicht-CSV-Schueler deaktivieren
        if ($import['deactivate_missing'] && !empty($csvStudentIds)) {
            $placeholders = implode(',', array_fill(0, count($csvStudentIds), '?'));
            $deactivated = db_query(
                "UPDATE users SET status = 'inaktiv' WHERE student_id IS NOT NULL AND student_id NOT IN ($placeholders) AND status = 'aktiv'",
                $csvStudentIds
            );
            // Zaehlen
            $stats['deactivated'] = db_get_var(
                "SELECT ROW_COUNT()"
            ) ?? 0;
        }

        $pdo->commit();

        // Aktivitaet loggen
        $adminId = $_SESSION['user_id'] ?? null;
        if ($adminId) {
            log_activity($adminId, 'sokrates_import', 'users', 0, json_encode([
                'inserted' => $stats['inserted'],
                'updated'  => $stats['updated'],
                'total'    => $stats['total'],
                'file'     => $import['filename'],
            ]));
        }

        $import['step']  = 3;
        $import['stats'] = $stats;

        echo json_encode(['success' => true, 'stats' => $stats]);

    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'error' => $e->getMessage(), 'stats' => $stats]);
    }
}

// ─── AJAX HANDLER: RESET ─────────────────────────────────────────────────────

function handleReset() {
    global $import;
    // Datei loeschen
    if ($import['filename'] && file_exists(SOKRATES_DIR . $import['filename'])) {
        unlink(SOKRATES_DIR . $import['filename']);
    }
    $_SESSION['sokrates_import'] = ['step' => 1, 'filename' => null, 'deactivate_missing' => false, 'stats' => []];
    echo json_encode(['success' => true]);
}

// ─── PAGE OUTPUT ──────────────────────────────────────────────────────────────

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Header -->
    <div class="mb-8">
        <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['sokrates_import_title'] ?? 'Sokrates-Import'; ?></h1>
        <p class="mt-2 text-sm text-gray-600"><?php echo $lang['sokrates_import_desc'] ?? 'Schuelerdaten aus dem Schulverwaltungssystem Sokrates importieren'; ?></p>
    </div>

    <!-- Flash -->
    <?php if ($flash): ?>
        <?php $c = $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'yellow'); ?>
        <div class="mb-6 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo esc_html($flash['message']); ?></p>
        </div>
    <?php endif; ?>

    <!-- Progress Steps -->
    <div class="mb-8">
        <div class="flex items-center">
            <?php
            $steps = [
                1 => $lang['sokrates_step_upload'] ?? 'Upload',
                2 => $lang['sokrates_step_preview'] ?? 'Vorschau',
                3 => $lang['sokrates_step_result'] ?? 'Ergebnis',
            ];
            $i = 0;
            foreach ($steps as $num => $label):
                $i++;
                $active   = $import['step'] >= $num;
                $complete = $import['step'] > $num;
            ?>
                <div class="flex-1 text-center">
                    <div class="w-10 h-10 mx-auto rounded-full <?php echo $active ? 'bg-blue-600' : 'bg-gray-300'; ?> text-white flex items-center justify-center font-bold text-sm">
                        <?php echo $complete ? '&#10003;' : $num; ?>
                    </div>
                    <p class="mt-2 text-xs font-medium <?php echo $active ? 'text-blue-600' : 'text-gray-500'; ?>">
                        <?php echo esc_html($label); ?>
                    </p>
                </div>
                <?php if ($num < count($steps)): ?>
                    <div class="flex-1 border-t-2 <?php echo $import['step'] > $num ? 'border-blue-600' : 'border-gray-300'; ?> mx-2 mt-[-10px]"></div>
                <?php endif; ?>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- ═══════════════════════════════════════ SCHRITT 1: UPLOAD ════ -->
    <?php if ($import['step'] === 1): ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['sokrates_step1_title'] ?? 'Schritt 1: CSV-Datei hochladen'; ?></h2>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['sokrates_step1_desc'] ?? 'Exportieren Sie die Schuelerliste aus Sokrates als CSV-Datei (Semikolon-getrennt).'; ?></p>
        </div>
        <div class="p-6">
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6 text-sm text-blue-800">
                <p class="font-semibold mb-1"><?php echo $lang['csv_expected_columns'] ?? 'Erwartete CSV-Spalten (Sokrates-Export):'; ?></p>
                <p class="font-mono text-xs">Schuelerkennzahl; Nachname; Vorname; Lesergruppe; Geburtsdatum; Mobiltelefon; Strasse; PLZ; Ort; Email</p>
                <p class="mt-2 text-xs text-blue-600"><?php echo $lang['csv_encoding_info'] ?? 'Encoding: UTF-8 oder Windows-1252 wird automatisch erkannt. Trennzeichen: Semikolon.'; ?></p>
            </div>

            <form method="POST" enctype="multipart/form-data" class="space-y-6">
                <input type="hidden" name="action" value="upload">
                <?php echo csrf_field(); ?>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['csv_file'] ?? 'CSV-Datei'; ?> <span class="text-red-500">*</span></label>
                    <input type="file" name="csv_file" accept=".csv" required
                           class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                </div>

                <div class="flex items-center gap-3 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <input type="checkbox" name="deactivate_missing" id="deactivate_missing" value="1"
                           class="h-4 w-4 text-blue-600 rounded">
                    <label for="deactivate_missing" class="text-sm text-gray-700">
                        <span class="font-medium"><?php echo $lang['deactivate_missing_label'] ?? 'Schulabgaenger inaktiv setzen:'; ?></span>
                        <?php echo $lang['deactivate_missing_desc'] ?? 'Schueler die in der Datenbank vorhanden sind, aber nicht in der CSV, werden auf Status "inaktiv" gesetzt.'; ?>
                    </label>
                </div>

                <div>
                    <button type="submit" class="px-6 py-2 bg-blue-600 text-white font-medium rounded-md hover:bg-blue-700">
                        <?php echo $lang['upload_and_continue'] ?? 'Datei hochladen und weiter'; ?>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- ════════════════════════════════════════ SCHRITT 2: VORSCHAU ═ -->
    <?php elseif ($import['step'] === 2): ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['sokrates_step2_title'] ?? 'Schritt 2: Datenvorschau'; ?></h2>
            <p class="mt-1 text-sm text-gray-500">
                Datei: <code class="bg-gray-100 px-1 rounded"><?php echo esc_html($import['filename']); ?></code>
                <?php if ($import['deactivate_missing']): ?>
                    <span class="ml-2 bg-yellow-100 text-yellow-800 text-xs px-2 py-0.5 rounded"><?php echo $lang['deactivate_missing_badge'] ?? 'Schulabgaenger werden inaktiv gesetzt'; ?></span>
                <?php endif; ?>
            </p>
        </div>
        <div class="p-6">
            <!-- Vorschau-Container -->
            <div id="previewContainer">
                <p class="text-gray-500 text-sm"><?php echo $lang['click_load_preview_sokrates'] ?? 'Klicken Sie "Vorschau laden" um die ersten 20 Eintraege zu pruefen.'; ?></p>
            </div>

            <div class="flex gap-3 mt-6" id="previewButtons">
                <button onclick="loadPreview()" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    <?php echo $lang['load_preview_sokrates'] ?? 'Vorschau laden'; ?>
                </button>
                <button onclick="startImport()" id="importBtn" class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700">
                    <?php echo $lang['start_import_sokrates'] ?? 'Import starten'; ?>
                </button>
                <button onclick="resetWizard()" class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                    <?php echo $lang['cancel_new_file'] ?? 'Abbrechen / Neue Datei'; ?>
                </button>
            </div>

            <!-- Import-Fortschritt -->
            <div id="importProgress" class="hidden mt-6 text-center py-8">
                <div class="inline-block animate-spin rounded-full h-16 w-16 border-b-4 border-blue-600"></div>
                <p class="mt-4 text-gray-700 font-medium text-lg"><?php echo $lang['sokrates_import_running'] ?? 'Import laeuft...'; ?></p>
                <p class="text-sm text-gray-500 mt-2"><?php echo $lang['sokrates_import_running_desc'] ?? 'Bitte Fenster nicht schliessen.'; ?></p>
            </div>
        </div>
    </div>

    <!-- ════════════════════════════════════════ SCHRITT 3: ERGEBNIS ═ -->
    <?php elseif ($import['step'] === 3): ?>
    <?php $st = $import['stats']; ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['sokrates_step3_title'] ?? 'Schritt 3: Import abgeschlossen'; ?></h2>
        </div>
        <div class="p-6">
            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-blue-50 rounded-lg p-4 text-center">
                    <p class="text-3xl font-bold text-blue-700"><?php echo number_format($st['total']); ?></p>
                    <p class="text-sm text-blue-600 mt-1"><?php echo $lang['total_in_csv'] ?? 'Gesamt in CSV'; ?></p>
                </div>
                <div class="bg-green-50 rounded-lg p-4 text-center">
                    <p class="text-3xl font-bold text-green-700"><?php echo number_format($st['inserted']); ?></p>
                    <p class="text-sm text-green-600 mt-1"><?php echo $lang['newly_created'] ?? 'Neu angelegt'; ?></p>
                </div>
                <div class="bg-yellow-50 rounded-lg p-4 text-center">
                    <p class="text-3xl font-bold text-yellow-700"><?php echo number_format($st['updated']); ?></p>
                    <p class="text-sm text-yellow-600 mt-1"><?php echo $lang['updated_label'] ?? 'Aktualisiert'; ?></p>
                </div>
                <?php if ($import['deactivate_missing']): ?>
                <div class="bg-gray-50 rounded-lg p-4 text-center">
                    <p class="text-3xl font-bold text-gray-700"><?php echo number_format($st['deactivated']); ?></p>
                    <p class="text-sm text-gray-600 mt-1"><?php echo $lang['deactivated_label'] ?? 'Inaktiv gesetzt'; ?></p>
                </div>
                <?php else: ?>
                <div class="bg-gray-50 rounded-lg p-4 text-center">
                    <p class="text-3xl font-bold text-gray-700"><?php echo number_format($st['total'] - $st['inserted'] - $st['updated']); ?></p>
                    <p class="text-sm text-gray-600 mt-1"><?php echo $lang['unchanged_label'] ?? 'Unveraendert'; ?></p>
                </div>
                <?php endif; ?>
            </div>

            <?php if (!empty($st['errors'])): ?>
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <p class="text-sm font-semibold text-red-800 mb-2"><?php echo $lang['warnings_label'] ?? 'Warnungen'; ?> (<?php echo count($st['errors']); ?>):</p>
                    <ul class="text-sm text-red-700 list-disc list-inside space-y-1">
                        <?php foreach (array_slice($st['errors'], 0, 10) as $err): ?>
                            <li><?php echo esc_html($err); ?></li>
                        <?php endforeach; ?>
                        <?php if (count($st['errors']) > 10): ?>
                            <li class="text-red-500">... <?php echo $lang['and'] ?? 'und'; ?> <?php echo count($st['errors']) - 10; ?> <?php echo $lang['more'] ?? 'weitere'; ?></li>
                        <?php endif; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <div class="flex gap-3">
                <a href="<?php echo BASE_URL; ?>/admin/users-list.php"
                   class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                    <?php echo $lang['to_user_list'] ?? 'Zur Benutzerliste'; ?>
                </a>
                <button onclick="resetWizard()" class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                    <?php echo $lang['new_import_sokrates'] ?? 'Neuer Import'; ?>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>

</div>
</div>

<script>
const SOKRATES_CONFIRM_IMPORT = '<?php echo esc_js($lang['sokrates_confirm_import'] ?? 'Import jetzt starten? Benutzer werden angelegt oder aktualisiert.'); ?>';
const SOKRATES_RESET_CONFIRM  = '<?php echo esc_js($lang['sokrates_reset_confirm'] ?? 'Wizard zuruecksetzen und Datei loeschen?'); ?>';

function esc(str) {
    if (!str) return '';
    const d = document.createElement('div');
    d.textContent = str;
    return d.innerHTML;
}

function statusBadge(s) {
    const map = {
        'neu':          'bg-green-100 text-green-800',
        'update':       'bg-yellow-100 text-yellow-800',
        'unveraendert': 'bg-gray-100 text-gray-600',
    };
    const label = {
        'neu':          '<?php echo esc_js($lang['newly_created'] ?? 'Neu'); ?>',
        'update':       '<?php echo esc_js($lang['updated_label'] ?? 'Update'); ?>',
        'unveraendert': '<?php echo esc_js($lang['unchanged_label'] ?? 'Unveraendert'); ?>',
    };
    return '<span class="px-2 py-0.5 rounded text-xs font-semibold ' + (map[s] || '') + '">' + (label[s] || s) + '</span>';
}

function loadPreview() {
    const container = document.getElementById('previewContainer');
    container.innerHTML = '<div class="text-center py-6"><div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div><p class="mt-2 text-sm text-gray-500"><?php echo esc_js($lang['loading'] ?? 'Lade Vorschau...'); ?></p></div>';

    const fd = new FormData();
    fd.append('action', 'preview');

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (!data.success) { container.innerHTML = '<p class="text-red-600"><?php echo esc_js($lang['error'] ?? 'Fehler'); ?>: ' + esc(data.error) + '</p>'; return; }

            let html = '<div class="flex gap-4 mb-4">';
            html += '<div class="bg-blue-50 rounded-lg px-4 py-2"><span class="text-2xl font-bold text-blue-700">' + data.total.toLocaleString() + '</span><span class="text-sm text-blue-600 ml-1"><?php echo esc_js($lang['total_in_csv'] ?? 'Eintraege gesamt'); ?></span></div>';
            if (data.has_email) {
                html += '<div class="bg-yellow-50 border border-yellow-200 rounded-lg px-4 py-2 text-sm text-yellow-800"><strong><?php echo esc_js($lang['info'] ?? 'Hinweis'); ?>:</strong> CSV enthalt eine Email-Spalte &mdash; wird beim Import ignoriert.</div>';
            }
            if (data.deactivate_missing) {
                html += '<div class="bg-orange-50 border border-orange-200 rounded-lg px-4 py-2 text-sm text-orange-800"><strong><?php echo esc_js($lang['deactivate_missing_label'] ?? 'Schulabgaenger:'); ?></strong> <?php echo esc_js($lang['deactivate_missing_desc'] ?? 'Nicht in CSV enthaltene Schueler werden inaktiv gesetzt.'); ?></div>';
            }
            html += '</div>';

            // Statistik aus Vorschau
            const counts = {neu: 0, update: 0, unveraendert: 0};
            data.preview.forEach(r => { if (counts[r.status] !== undefined) counts[r.status]++; });
            html += '<p class="text-xs text-gray-500 mb-3"><?php echo esc_js($lang['preview_label'] ?? 'Vorschau'); ?> (<?php echo esc_js($lang['first_20'] ?? 'erste 20'); ?>): ';
            html += '<span class="text-green-700 font-medium">' + counts.neu + ' <?php echo esc_js($lang['newly_created'] ?? 'neu'); ?></span>, ';
            html += '<span class="text-yellow-700 font-medium">' + counts.update + ' <?php echo esc_js($lang['updated_label'] ?? 'update'); ?></span>, ';
            html += '<span class="text-gray-600 font-medium">' + counts.unveraendert + ' <?php echo esc_js($lang['unchanged_label'] ?? 'unveraendert'); ?></span></p>';

            html += '<div class="overflow-x-auto"><table class="min-w-full text-xs border-collapse">';
            html += '<thead class="bg-gray-100"><tr>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['student_id'] ?? 'Kennzahl'); ?></th>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['last_name'] ?? 'Nachname'); ?></th>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['first_name'] ?? 'Vorname'); ?></th>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['class_name'] ?? 'Klasse'); ?></th>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['birthday'] ?? 'Geburtsdatum'); ?></th>';
            html += '<th class="border px-2 py-1 text-left"><?php echo esc_js($lang['status'] ?? 'Status'); ?></th>';
            html += '</tr></thead><tbody>';

            data.preview.forEach(r => {
                const rowClass = r.status === 'neu' ? 'bg-green-50' : (r.status === 'update' ? 'bg-yellow-50' : '');
                html += '<tr class="' + rowClass + ' hover:bg-gray-50">';
                html += '<td class="border px-2 py-1 font-mono">' + esc(r.student_id) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(r.last_name) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(r.first_name) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(r.class_name) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(r.birthday || '-') + '</td>';
                html += '<td class="border px-2 py-1">' + statusBadge(r.status) + '</td>';
                html += '</tr>';
            });

            html += '</tbody></table></div>';
            container.innerHTML = html;
        })
        .catch(err => {
            container.innerHTML = '<p class="text-red-600">Netzwerkfehler: ' + esc(err.message) + '</p>';
        });
}

function startImport() {
    if (!confirm(SOKRATES_CONFIRM_IMPORT)) return;

    document.getElementById('previewButtons').classList.add('hidden');
    document.getElementById('importProgress').classList.remove('hidden');

    const fd = new FormData();
    fd.append('action', 'import');

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                window.location.reload();
            } else {
                alert('Import fehlgeschlagen: ' + data.error);
                document.getElementById('importProgress').classList.add('hidden');
                document.getElementById('previewButtons').classList.remove('hidden');
            }
        })
        .catch(err => {
            alert('Netzwerkfehler: ' + err.message);
            document.getElementById('importProgress').classList.add('hidden');
            document.getElementById('previewButtons').classList.remove('hidden');
        });
}

function resetWizard() {
    if (!confirm(SOKRATES_RESET_CONFIRM)) return;
    const fd = new FormData();
    fd.append('action', 'reset');
    fetch('', {method: 'POST', body: fd})
        .then(() => window.location.reload())
        .catch(() => window.location.reload());
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
