<?php
/**
 * User Management - List + New User (Tab Layout)
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$current_user = get_logged_in_user();
$lang = load_language();
$page_title = $lang['user_management'] ?? 'Benutzerverwaltung';

// Active tab
$active_tab = $_GET['tab'] ?? 'list';

// ─── HANDLE DELETE ────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    if ($_POST['action'] === 'delete') {
        if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
            set_flash($lang['security_error'] ?? 'Sicherheitsfehler', 'error');
        } else {
            $user_id = (int)($_POST['user_id'] ?? 0);
            $active_loans = db_get_var(
                "SELECT COUNT(*) FROM loans WHERE user_id = ? AND status IN ('aktiv', 'überfällig')",
                [$user_id]
            );
            if ($active_loans > 0) {
                set_flash(sprintf($lang['error_user_has_loans'] ?? 'Benutzer hat noch %d aktive Ausleihen und kann nicht gelöscht werden.', $active_loans), 'error');
            } else {
                $user = db_get_row("SELECT username FROM users WHERE id = ?", [$user_id]);
                db_query("DELETE FROM users WHERE id = ?", [$user_id]);
                log_activity('user_deleted', 'users', $user_id, "Deleted user: {$user['username']}");
                set_flash(sprintf($lang['success_user_deleted'] ?? 'Benutzer "%s" wurde erfolgreich gelöscht', $user['username']), 'success');
            }
        }
        header('Location: users-list.php?tab=list');
        exit;
    }

    // ─── HANDLE NEW USER ──────────────────────────────────────────────────────
    if ($_POST['action'] === 'create_user') {
        if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
            set_flash($lang['security_error'] ?? 'Sicherheitsfehler', 'error');
            header('Location: users-list.php?tab=new');
            exit;
        }

        $username       = trim($_POST['username'] ?? '');
        $email          = trim($_POST['email'] ?? '');
        $password       = $_POST['password'] ?? '';
        $password_confirm = $_POST['password_confirm'] ?? '';
        $first_name     = trim($_POST['first_name'] ?? '');
        $last_name      = trim($_POST['last_name'] ?? '');
        $user_group_id  = (int)($_POST['user_group_id'] ?? 2);
        $class_name     = trim($_POST['class_name'] ?? '');
        $student_id     = trim($_POST['student_id'] ?? '');

        $errors = [];
        if (empty($username))                          $errors[] = $lang['error_username_required'] ?? 'Benutzername erforderlich';
        if (empty($email))                             $errors[] = $lang['error_email_required'] ?? 'E-Mail erforderlich';
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = $lang['error_email_invalid'] ?? 'Ungültige E-Mail';
        if (empty($password))                          $errors[] = $lang['error_password_required'] ?? 'Passwort erforderlich';
        if ($password !== $password_confirm)           $errors[] = $lang['error_password_mismatch'] ?? 'Passwörter stimmen nicht überein';
        if (strlen($password) < 8)                     $errors[] = $lang['error_password_too_short'] ?? 'Passwort muss mindestens 8 Zeichen lang sein';
        if (empty($first_name))                        $errors[] = $lang['error_first_name_required'] ?? 'Vorname erforderlich';
        if (empty($last_name))                         $errors[] = $lang['error_last_name_required'] ?? 'Nachname erforderlich';
        if (db_get_row("SELECT id FROM users WHERE username = ?", [$username]))
            $errors[] = $lang['error_username_exists'] ?? 'Benutzername existiert bereits';
        if (db_get_row("SELECT id FROM users WHERE email = ?", [$email]))
            $errors[] = $lang['error_email_exists'] ?? 'E-Mail existiert bereits';

        if (empty($errors)) {
            $password_hash = password_hash($password, PASSWORD_ARGON2ID);
            $result = db_query("
                INSERT INTO users (user_group_id, username, email, password_hash,
                    first_name, last_name, class_name, student_id,
                    auth_method, status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'local', 'aktiv', NOW())
            ", [$user_group_id, $username, $email, $password_hash,
                $first_name, $last_name, $class_name, $student_id]);

            if ($result) {
                log_activity('user_created', 'users', 0, "Created user: $username");
                set_flash(sprintf($lang['success_user_created'] ?? 'Benutzer "%s" wurde erfolgreich angelegt', $username), 'success');
                header('Location: users-list.php?tab=list');
                exit;
            } else {
                $errors[] = $lang['error_user_create_failed'] ?? 'Fehler beim Anlegen des Benutzers';
            }
        }

        // On error: stay on new-tab with errors
        set_flash(implode('<br>', $errors), 'error');
        header('Location: users-list.php?tab=new');
        exit;
    }
}

// ─── DATA FOR LIST TAB ────────────────────────────────────────────────────────
$user_groups = db_get_results("SELECT * FROM user_groups ORDER BY name");

$search        = $_GET['search'] ?? '';
$filter_status = $_GET['status'] ?? '';
$filter_group  = $_GET['group'] ?? '';
$page          = max(1, (int)($_GET['page'] ?? 1));
$per_page      = 50;
$offset        = ($page - 1) * $per_page;

$where  = ["1=1"];
$params = [];

if (!empty($search)) {
    $where[] = "(u.first_name LIKE ? OR u.last_name LIKE ? OR u.username LIKE ? OR u.email LIKE ? OR u.student_id LIKE ?)";
    $s = "%$search%";
    for ($i = 0; $i < 5; $i++) $params[] = $s;
}
if (!empty($filter_status)) { $where[] = "u.status = ?";        $params[] = $filter_status; }
if (!empty($filter_group))  { $where[] = "u.user_group_id = ?"; $params[] = $filter_group; }

$where_clause = implode(' AND ', $where);
$total_users  = db_get_var("SELECT COUNT(*) FROM users u WHERE $where_clause", $params);
$users        = db_get_results(
    "SELECT u.*, ug.name as group_name
     FROM users u
     LEFT JOIN user_groups ug ON u.user_group_id = ug.id
     WHERE $where_clause
     ORDER BY u.last_name, u.first_name
     LIMIT $per_page OFFSET $offset",
    $params
);
$total_pages = ceil($total_users / $per_page);

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Page Header -->
        <div class="mb-6">
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['user_management'] ?? 'Benutzerverwaltung'; ?></h1>
        </div>

        <!-- Flash Message -->
        <?php if ($flash): ?>
            <?php
            $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue'];
            $c  = $fc[$flash['type']] ?? 'blue';
            ?>
            <div class="mb-6 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
                <p class="text-sm text-<?php echo $c; ?>-700"><?php echo $flash['message']; ?></p>
            </div>
        <?php endif; ?>

        <!-- Tab Navigation -->
        <div class="mb-6 border-b border-gray-200">
            <nav class="-mb-px flex space-x-8">
                <a href="?tab=list"
                   class="tab-link border-b-2 py-4 px-1 text-sm font-medium
                          <?php echo $active_tab === 'list'
                              ? 'border-blue-500 text-blue-600'
                              : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?>">
                    <?php echo $lang['user_overview'] ?? 'Benutzerübersicht'; ?>
                </a>
                <a href="?tab=new"
                   class="tab-link border-b-2 py-4 px-1 text-sm font-medium
                          <?php echo $active_tab === 'new'
                              ? 'border-blue-500 text-blue-600'
                              : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?>">
                    <?php echo $lang['new_user'] ?? 'Neuer Benutzer'; ?>
                </a>
            </nav>
        </div>

        <!-- ══════════════════════════════════════════════════════════ -->
        <!-- TAB: LIST                                                  -->
        <!-- ══════════════════════════════════════════════════════════ -->
        <?php if ($active_tab === 'list'): ?>

            <div class="bg-white rounded-lg shadow">

                <!-- Filter bar -->
                <div class="p-6 border-b">
                    <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                        <input type="hidden" name="tab" value="list">
                        <input type="text" name="search" placeholder="<?php echo esc_attr($lang['search_name_email'] ?? 'Name, Benutzername, E-Mail...'); ?>"
                               value="<?php echo esc_attr($search); ?>"
                               class="px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">

                        <select name="group" class="px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            <option value=""><?php echo $lang['all_groups'] ?? 'Alle Gruppen'; ?></option>
                            <?php foreach ($user_groups as $g): ?>
                                <option value="<?php echo $g['id']; ?>" <?php echo $filter_group == $g['id'] ? 'selected' : ''; ?>>
                                    <?php echo esc_html($g['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>

                        <select name="status" class="px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            <option value=""><?php echo $lang['all_statuses'] ?? 'Alle Status'; ?></option>
                            <option value="aktiv"    <?php echo $filter_status === 'aktiv'    ? 'selected' : ''; ?>><?php echo $lang['status_active'] ?? 'Aktiv'; ?></option>
                            <option value="gesperrt" <?php echo $filter_status === 'gesperrt' ? 'selected' : ''; ?>><?php echo $lang['status_blocked'] ?? 'Gesperrt'; ?></option>
                            <option value="inaktiv"  <?php echo $filter_status === 'inaktiv'  ? 'selected' : ''; ?>><?php echo $lang['status_inactive'] ?? 'Inaktiv'; ?></option>
                        </select>

                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            <?php echo $lang['filter'] ?? 'Filtern'; ?>
                        </button>
                    </form>
                </div>

                <!-- Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['name'] ?? 'Name'; ?></th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['username'] ?? 'Benutzername'; ?></th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['email'] ?? 'E-Mail'; ?></th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['group'] ?? 'Gruppe'; ?></th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['status'] ?? 'Status'; ?></th>
                                <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase"><?php echo $lang['actions'] ?? 'Aktionen'; ?></th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php if (empty($users)): ?>
                                <tr>
                                    <td colspan="6" class="px-4 py-6 text-center text-gray-500">
                                        <?php echo $lang['no_users_found'] ?? 'Keine Benutzer gefunden'; ?>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($users as $user): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-4 py-3">
                                            <div class="flex items-center">
                                                <div class="h-8 w-8 rounded-full bg-blue-500 flex items-center justify-center flex-shrink-0">
                                                    <span class="text-white text-xs font-medium">
                                                        <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                                                    </span>
                                                </div>
                                                <div class="ml-3">
                                                    <div class="text-sm font-medium text-gray-900">
                                                        <?php echo esc_html($user['last_name'] . ' ' . $user['first_name']); ?>
                                                    </div>
                                                    <?php if (!empty($user['class_name'])): ?>
                                                        <div class="text-xs text-gray-500"><?php echo esc_html($user['class_name']); ?></div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-4 py-3 text-sm text-gray-900"><?php echo esc_html($user['username']); ?></td>
                                        <td class="px-4 py-3 text-sm text-gray-500"><?php echo esc_html($user['email']); ?></td>
                                        <td class="px-4 py-3 text-sm text-gray-500"><?php echo esc_html($user['group_name'] ?? '-'); ?></td>
                                        <td class="px-4 py-3">
                                            <?php
                                            $sc = ['aktiv'=>'bg-green-100 text-green-800','gesperrt'=>'bg-red-100 text-red-800','inaktiv'=>'bg-gray-100 text-gray-800'];
                                            $cls = $sc[$user['status']] ?? 'bg-gray-100 text-gray-800';
                                            $status_labels = ['aktiv' => $lang['status_active'] ?? 'Aktiv', 'gesperrt' => $lang['status_blocked'] ?? 'Gesperrt', 'inaktiv' => $lang['status_inactive'] ?? 'Inaktiv'];
                                            ?>
                                            <span class="px-2 py-1 text-xs font-semibold rounded-full <?php echo $cls; ?>">
                                                <?php echo esc_html($status_labels[$user['status']] ?? $user['status']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-3 text-center whitespace-nowrap">
                                            <a href="users-edit.php?id=<?php echo $user['id']; ?>"
                                               class="inline-flex items-center justify-center w-8 h-8 bg-blue-100 text-blue-700 rounded hover:bg-blue-200"
                                               title="<?php echo esc_attr($lang['edit'] ?? 'Bearbeiten'); ?>">
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                                </svg>
                                            </a>
                                            <?php if ($user['id'] != $current_user['id']): ?>
                                                <button onclick="deleteUser(<?php echo $user['id']; ?>, '<?php echo esc_js($user['username']); ?>')"
                                                        class="inline-flex items-center justify-center w-8 h-8 bg-red-100 text-red-700 rounded hover:bg-red-200 ml-1"
                                                        title="<?php echo esc_attr($lang['delete'] ?? 'Löschen'); ?>">
                                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                                                    </svg>
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="bg-gray-50 px-6 py-4 border-t flex justify-between items-center">
                        <div class="text-sm text-gray-700">
                            <?php echo $lang['showing'] ?? 'Zeige'; ?> <?php echo $offset + 1; ?> <?php echo $lang['to'] ?? 'bis'; ?> <?php echo min($offset + $per_page, $total_users); ?> <?php echo $lang['of'] ?? 'von'; ?> <?php echo $total_users; ?> <?php echo $lang['users'] ?? 'Benutzern'; ?>
                        </div>
                        <div class="flex gap-2">
                            <?php for ($i = 1; $i <= min($total_pages, 10); $i++): ?>
                                <a href="?tab=list&page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($filter_status); ?>&group=<?php echo urlencode($filter_group); ?>"
                                   class="px-3 py-1 border rounded text-sm <?php echo $i === $page ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-700 hover:bg-gray-50'; ?>">
                                    <?php echo $i; ?>
                                </a>
                            <?php endfor; ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Delete Form (hidden) -->
            <form id="delete-form" method="POST" style="display:none;">
                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="user_id" id="delete-user-id">
            </form>

        <?php endif; ?>

        <!-- ══════════════════════════════════════════════════════════ -->
        <!-- TAB: NEW USER                                              -->
        <!-- ══════════════════════════════════════════════════════════ -->
        <?php if ($active_tab === 'new'): ?>

            <div class="max-w-4xl">
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6 border-b">
                        <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['create_new_user'] ?? 'Neuen Benutzer anlegen'; ?></h2>
                    </div>

                    <form method="POST" class="p-6 space-y-6">
                        <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                        <input type="hidden" name="action" value="create_user">

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['first_name'] ?? 'Vorname'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="first_name" required
                                       value="<?php echo isset($_POST['first_name']) ? esc_attr($_POST['first_name']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['last_name'] ?? 'Nachname'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="last_name" required
                                       value="<?php echo isset($_POST['last_name']) ? esc_attr($_POST['last_name']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['username'] ?? 'Benutzername'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="username" required
                                       value="<?php echo isset($_POST['username']) ? esc_attr($_POST['username']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['email'] ?? 'E-Mail'; ?> <span class="text-red-500">*</span></label>
                                <input type="email" name="email" required
                                       value="<?php echo isset($_POST['email']) ? esc_attr($_POST['email']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['user_group'] ?? 'Benutzergruppe'; ?> <span class="text-red-500">*</span></label>
                                <select name="user_group_id" required
                                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                    <?php foreach ($user_groups as $g): ?>
                                        <option value="<?php echo $g['id']; ?>"
                                            <?php echo (isset($_POST['user_group_id']) && $_POST['user_group_id'] == $g['id']) ? 'selected' : ''; ?>>
                                            <?php echo esc_html($g['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['class_name'] ?? 'Klasse'; ?></label>
                                <input type="text" name="class_name"
                                       value="<?php echo isset($_POST['class_name']) ? esc_attr($_POST['class_name']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['student_id'] ?? 'Schüler-ID'; ?></label>
                                <input type="text" name="student_id"
                                       value="<?php echo isset($_POST['student_id']) ? esc_attr($_POST['student_id']) : ''; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>

                        <!-- Password -->
                        <div class="border-t pt-6">
                            <h3 class="text-lg font-medium text-gray-900 mb-4"><?php echo $lang['password'] ?? 'Passwort'; ?></h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['password'] ?? 'Passwort'; ?> <span class="text-red-500">*</span></label>
                                    <input type="password" name="password" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['password_confirm'] ?? 'Passwort bestätigen'; ?> <span class="text-red-500">*</span></label>
                                    <input type="password" name="password_confirm" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                </div>
                            </div>
                            <p class="mt-2 text-xs text-gray-500"><?php echo $lang['password_min_hint'] ?? 'Mindestens 8 Zeichen, Groß-/Kleinbuchstaben, Zahl und Sonderzeichen'; ?></p>
                        </div>

                        <div class="flex justify-end gap-3 border-t pt-6">
                            <a href="?tab=list" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                                <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                            </a>
                            <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                                <?php echo $lang['create_user'] ?? 'Benutzer anlegen'; ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

        <?php endif; ?>

    </div>
</div>

<script>
function deleteUser(userId, username) {
    if (confirm('<?php echo esc_js($lang['confirm_delete_user'] ?? 'Möchten Sie diesen Benutzer wirklich löschen?'); ?> (' + username + ')')) {
        document.getElementById('delete-user-id').value = userId;
        document.getElementById('delete-form').submit();
    }
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
