<?php
/**
 * User Edit - Edit existing user
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$current_user = get_logged_in_user();
$lang = load_language();
$page_title = $lang['edit_user'] ?? 'Benutzer bearbeiten';

// Get user ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    set_flash($lang['invalid_user_id'] ?? 'Ungültige Benutzer-ID', 'error');
    header('Location: users-list.php?tab=list');
    exit;
}

$userId = intval($_GET['id']);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash($lang['security_error'] ?? 'Sicherheitsfehler', 'error');
        header('Location: users-edit.php?id=' . $userId);
        exit;
    }

    // ─── UPDATE BASE DATA ─────────────────────────────────────────────────────
    if ($_POST['action'] === 'update_user') {

        $errors = [];
        $first_name    = trim($_POST['first_name'] ?? '');
        $last_name     = trim($_POST['last_name'] ?? '');
        $email         = trim($_POST['email'] ?? '');
        $username      = trim($_POST['username'] ?? '');
        $user_group_id = intval($_POST['user_group_id'] ?? 2);
        $class_name    = trim($_POST['class_name'] ?? '');
        $student_id    = trim($_POST['student_id'] ?? '');
        $phone         = trim($_POST['phone'] ?? '');
        $status        = in_array($_POST['status'], ['aktiv', 'gesperrt', 'inaktiv']) ? $_POST['status'] : 'aktiv';
        $blocked_reason = trim($_POST['blocked_reason'] ?? '');
        $blocked_until  = !empty($_POST['blocked_until']) ? $_POST['blocked_until'] : null;

        if (empty($first_name)) $errors[] = $lang['error_first_name_required'] ?? 'Vorname erforderlich';
        if (empty($last_name))  $errors[] = $lang['error_last_name_required'] ?? 'Nachname erforderlich';
        if (empty($email))      $errors[] = $lang['error_email_required'] ?? 'E-Mail erforderlich';
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = $lang['error_email_invalid'] ?? 'Ungültige E-Mail';
        if (empty($username))   $errors[] = $lang['error_username_required'] ?? 'Benutzername erforderlich';

        // Check duplicates (exclude current user)
        if (db_get_var("SELECT id FROM users WHERE username = ? AND id != ?", [$username, $userId]))
            $errors[] = $lang['error_username_exists'] ?? 'Benutzername existiert bereits';
        if (db_get_var("SELECT id FROM users WHERE email = ? AND id != ?", [$email, $userId]))
            $errors[] = $lang['error_email_exists'] ?? 'E-Mail existiert bereits';

        if (empty($errors)) {
            db_query("
                UPDATE users SET
                    first_name = ?, last_name = ?, email = ?, username = ?,
                    user_group_id = ?, class_name = ?, student_id = ?, phone = ?,
                    status = ?, blocked_reason = ?, blocked_until = ?,
                    updated_at = NOW()
                WHERE id = ?
            ", [
                $first_name, $last_name, $email, $username,
                $user_group_id, $class_name, $student_id, $phone,
                $status, $blocked_reason ?: null, $blocked_until,
                $userId
            ]);

            log_activity('user_updated', 'users', $userId, "Updated user: $username");
            set_flash($lang['user_saved'] ?? 'Benutzer erfolgreich gespeichert', 'success');
        } else {
            set_flash(implode('<br>', $errors), 'error');
        }

        header('Location: users-edit.php?id=' . $userId);
        exit;
    }

    // ─── CHANGE PASSWORD ──────────────────────────────────────────────────────
    if ($_POST['action'] === 'change_password') {

        $password         = $_POST['password'] ?? '';
        $password_confirm = $_POST['password_confirm'] ?? '';
        $errors = [];

        if (empty($password))                $errors[] = $lang['error_password_required'] ?? 'Passwort erforderlich';
        if (strlen($password) < 8)           $errors[] = $lang['error_password_too_short'] ?? 'Passwort muss mindestens 8 Zeichen lang sein';
        if ($password !== $password_confirm) $errors[] = $lang['error_password_mismatch'] ?? 'Passwörter stimmen nicht überein';

        if (empty($errors)) {
            $hash = password_hash($password, PASSWORD_ARGON2ID);
            db_query("UPDATE users SET password_hash = ?, updated_at = NOW() WHERE id = ?", [$hash, $userId]);
            log_activity('password_changed', 'users', $userId, 'Password changed by admin');
            set_flash($lang['password_changed_success'] ?? 'Passwort erfolgreich geändert', 'success');
        } else {
            set_flash(implode('<br>', $errors), 'error');
        }

        header('Location: users-edit.php?id=' . $userId);
        exit;
    }
}

// Load user
$user = db_get_row("SELECT * FROM users WHERE id = ?", [$userId]);

if (!$user) {
    set_flash($lang['user_not_found'] ?? 'Benutzer nicht gefunden', 'error');
    header('Location: users-list.php?tab=list');
    exit;
}

// Load groups
$user_groups = db_get_results("SELECT * FROM user_groups ORDER BY name");

// Load loan stats
$active_loans = db_get_var(
    "SELECT COUNT(*) FROM loans WHERE user_id = ? AND return_date IS NULL", [$userId]
);
$total_loans = db_get_var(
    "SELECT COUNT(*) FROM loans WHERE user_id = ?", [$userId]
);

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Back -->
        <div class="mb-6">
            <a href="users-list.php?tab=list" class="inline-flex items-center text-sm text-gray-600 hover:text-gray-900">
                &larr; <?php echo $lang['back_to_overview'] ?? 'Zurück zur Übersicht'; ?>
            </a>
        </div>

        <!-- Flash -->
        <?php if ($flash): ?>
            <?php
            $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue'];
            $c  = $fc[$flash['type']] ?? 'blue';
            ?>
            <div class="mb-6 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
                <p class="text-sm text-<?php echo $c; ?>-700"><?php echo $flash['message']; ?></p>
            </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

            <!-- ── MAIN FORM (2 cols) ───────────────────────────────────────── -->
            <div class="lg:col-span-2 space-y-6">

                <!-- Stammdaten -->
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-bold text-gray-900"><?php echo $lang['master_data'] ?? 'Stammdaten'; ?></h2>
                    </div>
                    <form method="POST" class="p-6 space-y-5">
                        <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                        <input type="hidden" name="action" value="update_user">

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-5">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['first_name'] ?? 'Vorname'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="first_name" required
                                       value="<?php echo esc_attr($user['first_name']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['last_name'] ?? 'Nachname'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="last_name" required
                                       value="<?php echo esc_attr($user['last_name']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['username'] ?? 'Benutzername'; ?> <span class="text-red-500">*</span></label>
                                <input type="text" name="username" required
                                       value="<?php echo esc_attr($user['username']); ?>"
                                       <?php echo $user['auth_method'] === 'ldap' ? 'readonly class="w-full px-3 py-2 border border-gray-200 rounded-md bg-gray-50 text-gray-500"' : 'class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"'; ?>>
                                <?php if ($user['auth_method'] === 'ldap'): ?>
                                    <p class="mt-1 text-xs text-gray-500"><?php echo $lang['ldap_cannot_change_username'] ?? 'LDAP-Benutzer – Benutzername nicht änderbar'; ?></p>
                                <?php endif; ?>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['email'] ?? 'E-Mail'; ?> <span class="text-red-500">*</span></label>
                                <input type="email" name="email" required
                                       value="<?php echo esc_attr($user['email']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['user_group'] ?? 'Benutzergruppe'; ?></label>
                                <select name="user_group_id"
                                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                    <?php foreach ($user_groups as $g): ?>
                                        <option value="<?php echo $g['id']; ?>"
                                            <?php echo $user['user_group_id'] == $g['id'] ? 'selected' : ''; ?>>
                                            <?php echo esc_html($g['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['phone'] ?? 'Telefon'; ?></label>
                                <input type="text" name="phone"
                                       value="<?php echo esc_attr($user['phone'] ?? ''); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['class_name'] ?? 'Klasse'; ?></label>
                                <input type="text" name="class_name"
                                       value="<?php echo esc_attr($user['class_name'] ?? ''); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['student_id'] ?? 'Schüler-ID'; ?></label>
                                <input type="text" name="student_id"
                                       value="<?php echo esc_attr($user['student_id'] ?? ''); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>

                        <!-- Status -->
                        <div class="border-t pt-5">
                            <h3 class="text-sm font-semibold text-gray-700 mb-3"><?php echo $lang['status_and_block'] ?? 'Status & Sperren'; ?></h3>
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-5">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['status'] ?? 'Status'; ?></label>
                                    <select name="status" id="status-select"
                                            class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"
                                            onchange="toggleBlockFields(this.value)">
                                        <option value="aktiv"    <?php echo $user['status'] === 'aktiv'    ? 'selected' : ''; ?>>&#10003; <?php echo $lang['status_active'] ?? 'Aktiv'; ?></option>
                                        <option value="inaktiv"  <?php echo $user['status'] === 'inaktiv'  ? 'selected' : ''; ?>>&#9208; <?php echo $lang['status_inactive'] ?? 'Inaktiv'; ?></option>
                                        <option value="gesperrt" <?php echo $user['status'] === 'gesperrt' ? 'selected' : ''; ?>>&#128683; <?php echo $lang['status_blocked'] ?? 'Gesperrt'; ?></option>
                                    </select>
                                </div>
                                <div id="block-reason-field" class="<?php echo $user['status'] !== 'gesperrt' ? 'hidden' : ''; ?>">
                                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['block_reason'] ?? 'Sperrgrund'; ?></label>
                                    <input type="text" name="blocked_reason"
                                           value="<?php echo esc_attr($user['blocked_reason'] ?? ''); ?>"
                                           placeholder="z.B. Bücher nicht zurückgegeben"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                </div>
                                <div id="block-until-field" class="<?php echo $user['status'] !== 'gesperrt' ? 'hidden' : ''; ?>">
                                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['blocked_until'] ?? 'Gesperrt bis'; ?></label>
                                    <input type="date" name="blocked_until"
                                           value="<?php echo esc_attr($user['blocked_until'] ? date('Y-m-d', strtotime($user['blocked_until'])) : ''); ?>"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                                    <p class="mt-1 text-xs text-gray-500"><?php echo $lang['blocked_until_empty'] ?? 'Leer = dauerhaft gesperrt'; ?></p>
                                </div>
                            </div>
                        </div>

                        <div class="flex justify-end border-t pt-4">
                            <button type="submit"
                                    class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                                &#128190; <?php echo $lang['save_changes'] ?? 'Änderungen speichern'; ?>
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Passwort ändern -->
                <?php if ($user['auth_method'] !== 'ldap'): ?>
                <div class="bg-white rounded-lg shadow">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-bold text-gray-900"><?php echo $lang['change_password'] ?? 'Passwort ändern'; ?></h2>
                    </div>
                    <form method="POST" class="p-6 space-y-4">
                        <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                        <input type="hidden" name="action" value="change_password">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-5">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['new_password'] ?? 'Neues Passwort'; ?> <span class="text-red-500">*</span></label>
                                <input type="password" name="password" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['password_confirm'] ?? 'Passwort bestätigen'; ?> <span class="text-red-500">*</span></label>
                                <input type="password" name="password_confirm" required
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                        <p class="text-xs text-gray-500"><?php echo $lang['password_requirements'] ?? 'Mindestens 8 Zeichen'; ?></p>
                        <div class="flex justify-end">
                            <button type="submit"
                                    class="px-6 py-2 bg-yellow-500 text-white rounded-md hover:bg-yellow-600">
                                &#128273; <?php echo $lang['set_password'] ?? 'Passwort setzen'; ?>
                            </button>
                        </div>
                    </form>
                </div>
                <?php endif; ?>

            </div>

            <!-- ── SIDEBAR (1 col) ──────────────────────────────────────────── -->
            <div class="space-y-6">

                <!-- Metadaten -->
                <div class="bg-white rounded-lg shadow p-5">
                    <h3 class="text-sm font-bold text-gray-700 uppercase mb-4"><?php echo $lang['user_info'] ?? 'Info'; ?></h3>

                    <!-- Avatar -->
                    <div class="flex items-center mb-4">
                        <div class="h-14 w-14 rounded-full bg-blue-500 flex items-center justify-center">
                            <span class="text-white text-xl font-bold">
                                <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                            </span>
                        </div>
                        <div class="ml-3">
                            <div class="font-semibold text-gray-900">
                                <?php echo esc_html($user['first_name'] . ' ' . $user['last_name']); ?>
                            </div>
                            <div class="text-sm text-gray-500"><?php echo esc_html($user['username']); ?></div>
                        </div>
                    </div>

                    <dl class="space-y-3 text-sm">
                        <div class="flex justify-between">
                            <dt class="text-gray-500">ID</dt>
                            <dd class="font-mono text-gray-900">#<?php echo $user['id']; ?></dd>
                        </div>
                        <div class="flex justify-between">
                            <dt class="text-gray-500"><?php echo $lang['auth_method'] ?? 'Auth'; ?></dt>
                            <dd>
                                <span class="px-2 py-0.5 text-xs rounded-full <?php echo $user['auth_method'] === 'ldap' ? 'bg-purple-100 text-purple-800' : 'bg-gray-100 text-gray-800'; ?>">
                                    <?php echo strtoupper($user['auth_method']); ?>
                                </span>
                            </dd>
                        </div>
                        <div class="flex justify-between">
                            <dt class="text-gray-500"><?php echo $lang['status'] ?? 'Status'; ?></dt>
                            <dd>
                                <?php
                                $sc = ['aktiv'=>'bg-green-100 text-green-800','gesperrt'=>'bg-red-100 text-red-800','inaktiv'=>'bg-gray-100 text-gray-800'];
                                $cls = $sc[$user['status']] ?? 'bg-gray-100 text-gray-800';
                                $status_labels = ['aktiv' => $lang['status_active'] ?? 'Aktiv', 'gesperrt' => $lang['status_blocked'] ?? 'Gesperrt', 'inaktiv' => $lang['status_inactive'] ?? 'Inaktiv'];
                                ?>
                                <span class="px-2 py-0.5 text-xs font-semibold rounded-full <?php echo $cls; ?>">
                                    <?php echo esc_html($status_labels[$user['status']] ?? $user['status']); ?>
                                </span>
                            </dd>
                        </div>
                        <div class="flex justify-between">
                            <dt class="text-gray-500"><?php echo $lang['last_login'] ?? 'Letzter Login'; ?></dt>
                            <dd class="text-gray-900 text-right">
                                <?php echo $user['last_login'] ? date('d.m.Y H:i', strtotime($user['last_login'])) : '–'; ?>
                            </dd>
                        </div>
                        <div class="flex justify-between">
                            <dt class="text-gray-500"><?php echo $lang['created_at_label'] ?? 'Erstellt'; ?></dt>
                            <dd class="text-gray-900"><?php echo date('d.m.Y', strtotime($user['created_at'])); ?></dd>
                        </div>
                    </dl>
                </div>

                <!-- Ausleih-Stats -->
                <div class="bg-white rounded-lg shadow p-5">
                    <h3 class="text-sm font-bold text-gray-700 uppercase mb-4"><?php echo $lang['loans_sidebar'] ?? 'Ausleihen'; ?></h3>
                    <div class="grid grid-cols-2 gap-3">
                        <div class="bg-blue-50 rounded-lg p-3 text-center">
                            <div class="text-2xl font-bold text-blue-700"><?php echo $active_loans; ?></div>
                            <div class="text-xs text-blue-600 mt-1"><?php echo $lang['loans_active'] ?? 'Aktiv'; ?></div>
                        </div>
                        <div class="bg-gray-50 rounded-lg p-3 text-center">
                            <div class="text-2xl font-bold text-gray-700"><?php echo $total_loans; ?></div>
                            <div class="text-xs text-gray-600 mt-1"><?php echo $lang['loans_total'] ?? 'Gesamt'; ?></div>
                        </div>
                    </div>
                </div>

                <!-- Danger Zone -->
                <?php if ($userId !== $current_user['id']): ?>
                <div class="bg-white rounded-lg shadow p-5 border border-red-200">
                    <h3 class="text-sm font-bold text-red-700 uppercase mb-3"><?php echo $lang['danger_zone'] ?? 'Gefahrenzone'; ?></h3>
                    <?php if ($active_loans > 0): ?>
                        <p class="text-xs text-gray-500 mb-3">
                            <?php echo sprintf($lang['user_has_active_loans_msg'] ?? 'Benutzer hat %d aktive Ausleihe(n) – Löschen nicht möglich.', $active_loans); ?>
                        </p>
                        <button disabled
                                class="w-full px-4 py-2 bg-gray-100 text-gray-400 rounded-md text-sm cursor-not-allowed">
                            &#128465; <?php echo $lang['delete_user'] ?? 'Benutzer löschen'; ?>
                        </button>
                    <?php else: ?>
                        <form method="POST" action="users-list.php">
                            <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="user_id" value="<?php echo $userId; ?>">
                            <button type="submit"
                                    onclick="return confirm('<?php echo esc_js($lang['confirm_delete_user'] ?? 'Möchten Sie diesen Benutzer wirklich löschen?'); ?> (<?php echo esc_js($user['username']); ?>)')"
                                    class="w-full px-4 py-2 bg-red-100 text-red-700 rounded-md hover:bg-red-200 text-sm">
                                &#128465; <?php echo $lang['delete_user'] ?? 'Benutzer löschen'; ?>
                            </button>
                        </form>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

            </div>
        </div>
    </div>
</div>

<script>
function toggleBlockFields(status) {
    const reason = document.getElementById('block-reason-field');
    const until  = document.getElementById('block-until-field');
    if (status === 'gesperrt') {
        reason.classList.remove('hidden');
        until.classList.remove('hidden');
    } else {
        reason.classList.add('hidden');
        until.classList.add('hidden');
    }
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
