<?php
/**
 * Neuen Benutzer anlegen
 * Tab 1: Manuell (lokaler Account, Argon2id-Passwort)
 * Tab 2: Via LDAP (AJAX-Suche + optionales Merge mit Sokrates-User)
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$lang         = load_language();
$page_title   = $lang['create_new_user'] ?? 'Neuen Benutzer anlegen';
$current_user = get_logged_in_user();
$ldap_enabled = getSetting('ldap_enabled') === '1';
$user_groups  = db_get_results("SELECT id, name FROM user_groups ORDER BY name");

$active_tab = $_GET['tab'] ?? 'manual';
$errors     = [];
$form       = [];

// ─── AJAX: LDAP-Benutzer anlegen ─────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create_ldap_user') {
    header('Content-Type: application/json; charset=utf-8');

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        echo json_encode(['success' => false, 'error' => $lang['security_error'] ?? 'Sicherheitsfehler']);
        exit;
    }
    if (!is_admin()) {
        echo json_encode(['success' => false, 'error' => 'Zugriff verweigert']);
        exit;
    }

    $username    = trim($_POST['username'] ?? '');
    $email       = trim($_POST['email'] ?? '');
    $first_name  = trim($_POST['first_name'] ?? '');
    $last_name   = trim($_POST['last_name'] ?? '');
    $ldap_dn     = trim($_POST['ldap_dn'] ?? '');
    $group_id    = (int)($_POST['user_group_id'] ?? 2);

    if (empty($username) || empty($first_name) || empty($last_name)) {
        echo json_encode(['success' => false, 'error' => $lang['error_first_name_required'] ?? 'Vorname, Nachname und Benutzername sind Pflichtfelder']);
        exit;
    }

    // Pruefen ob Benutzername schon vergeben
    if (db_get_row("SELECT id FROM users WHERE username = ?", [$username])) {
        echo json_encode(['success' => false, 'error' => $lang['error_username_exists'] ?? 'Benutzername bereits vergeben']);
        exit;
    }

    // Merge-Versuch: Sokrates-User mit gleichen Namen und email=NULL suchen
    $merge_candidate = db_get_row(
        "SELECT id FROM users WHERE first_name = ? AND last_name = ? AND email IS NULL AND username IS NULL AND auth_method = 'ldap'",
        [$first_name, $last_name]
    );

    if ($merge_candidate) {
        // MERGE: bestehenden Sokrates-User mit LDAP-Daten anreichern
        db_update('users', [
            'user_group_id' => $group_id,
            'username'      => $username,
            'email'         => $email ?: null,
            'ldap_dn'       => $ldap_dn ?: null,
            'status'        => 'aktiv',
        ], 'id = ?', [$merge_candidate['id']]);

        $new_id = $merge_candidate['id'];
        $action_label = 'ldap_user_merged';
        $msg = 'LDAP-Benutzer mit bestehendem Sokrates-Eintrag verbunden.';
    } else {
        // INSERT: komplett neuer Benutzer
        $new_id = db_insert('users', [
            'user_group_id' => $group_id,
            'username'      => $username,
            'email'         => $email ?: null,
            'first_name'    => $first_name,
            'last_name'     => $last_name,
            'ldap_dn'       => $ldap_dn ?: null,
            'auth_method'   => 'ldap',
            'status'        => 'aktiv',
            'created_at'    => date('Y-m-d H:i:s'),
        ]);
        $action_label = 'ldap_user_created';
        $msg = 'LDAP-Benutzer angelegt.';
    }

    if ($new_id) {
        log_activity($_SESSION['user_id'], $action_label, 'users', $new_id, "LDAP user: $username");
        echo json_encode(['success' => true, 'user_id' => $new_id, 'message' => $msg]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Datenbankfehler']);
    }
    exit;
}

// ─── FORMULAR: Manuellen Benutzer anlegen ────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create_local') {

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', $lang['security_error'] ?? 'Sicherheitsfehler.');
        header('Location: users-add.php?tab=manual');
        exit;
    }

    $form = [
        'first_name'       => trim($_POST['first_name'] ?? ''),
        'last_name'        => trim($_POST['last_name'] ?? ''),
        'email'            => trim($_POST['email'] ?? ''),
        'username'         => trim($_POST['username'] ?? ''),
        'phone'            => trim($_POST['phone'] ?? ''),
        'address'          => trim($_POST['address'] ?? ''),
        'class_name'       => trim($_POST['class_name'] ?? ''),
        'birthday'         => trim($_POST['birthday'] ?? ''),
        'user_group_id'    => (int)($_POST['user_group_id'] ?? 2),
        'password'         => $_POST['password'] ?? '',
        'password_confirm' => $_POST['password_confirm'] ?? '',
    ];

    // Validierung
    if (empty($form['first_name']))   $errors[] = $lang['error_first_name_required'] ?? 'Vorname ist erforderlich';
    if (empty($form['last_name']))    $errors[] = $lang['error_last_name_required'] ?? 'Nachname ist erforderlich';
    if (empty($form['email']))        $errors[] = $lang['error_email_required'] ?? 'E-Mail ist erforderlich';
    if (!filter_var($form['email'], FILTER_VALIDATE_EMAIL)) $errors[] = $lang['error_email_invalid'] ?? 'Ungueltige E-Mail-Adresse';
    if (empty($form['password']))     $errors[] = $lang['error_password_required'] ?? 'Passwort ist erforderlich';
    if ($form['password'] !== $form['password_confirm']) $errors[] = $lang['error_password_mismatch'] ?? 'Passwoerter stimmen nicht ueberein';

    $pw_errors = validate_password($form['password']);
    $errors = array_merge($errors, $pw_errors);

    // Eindeutigkeit pruefen
    if (!empty($form['email']) && db_get_row("SELECT id FROM users WHERE email = ?", [$form['email']])) {
        $errors[] = $lang['error_email_exists'] ?? 'E-Mail-Adresse bereits vergeben';
    }
    if (!empty($form['username']) && db_get_row("SELECT id FROM users WHERE username = ?", [$form['username']])) {
        $errors[] = $lang['error_username_exists'] ?? 'Benutzername bereits vergeben';
    }

    if (empty($errors)) {
        $password_hash = hash_password($form['password']);
        $new_id = db_insert('users', [
            'user_group_id' => $form['user_group_id'],
            'username'      => $form['username'] ?: null,
            'email'         => $form['email'],
            'password_hash' => $password_hash,
            'first_name'    => $form['first_name'],
            'last_name'     => $form['last_name'],
            'birthday'      => $form['birthday'] ?: null,
            'phone'         => $form['phone'] ?: null,
            'address'       => $form['address'] ?: null,
            'class_name'    => $form['class_name'] ?: null,
            'auth_method'   => 'local',
            'status'        => 'aktiv',
            'created_at'    => date('Y-m-d H:i:s'),
        ]);

        if ($new_id) {
            log_activity($_SESSION['user_id'], 'user_created', 'users', $new_id,
                'Lokal angelegt: ' . $form['first_name'] . ' ' . $form['last_name']);
            set_flash('success', sprintf($lang['success_user_created'] ?? 'Benutzer "%s" wurde erfolgreich angelegt', $form['first_name'] . ' ' . $form['last_name']));
            header('Location: users-list.php');
            exit;
        }
        $errors[] = $lang['error_user_create_failed'] ?? 'Datenbankfehler beim Anlegen des Benutzers';
    }

    // Fehler: Formular wieder anzeigen
    set_flash('error', implode('<br>', array_map('esc_html', $errors)));
    // $form bleibt gesetzt fuer Vorausfuellung
}

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Page Header -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['create_new_user'] ?? 'Neuen Benutzer anlegen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['create_user_subtitle'] ?? 'Benutzer manuell oder via LDAP anlegen'; ?></p>
        </div>
        <a href="users-list.php" class="text-sm text-gray-500 hover:text-gray-700">
            &larr; <?php echo $lang['to_user_list'] ?? 'Zur Benutzerliste'; ?>
        </a>
    </div>

    <!-- Flash -->
    <?php if ($flash): ?>
        <?php $c = $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'yellow'); ?>
        <div class="mb-6 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo $flash['message']; ?></p>
        </div>
    <?php endif; ?>

    <!-- Tab Navigation -->
    <div class="mb-6 border-b border-gray-200">
        <nav class="-mb-px flex space-x-8">
            <a href="?tab=manual"
               class="border-b-2 py-4 px-1 text-sm font-medium <?php echo $active_tab === 'manual' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?>">
                <?php echo $lang['create_manually'] ?? 'Manuell anlegen'; ?>
            </a>
            <?php if ($ldap_enabled): ?>
            <a href="?tab=ldap"
               class="border-b-2 py-4 px-1 text-sm font-medium <?php echo $active_tab === 'ldap' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?>">
                <?php echo $lang['via_ldap_ad'] ?? 'Via LDAP/AD'; ?>
            </a>
            <?php endif; ?>
        </nav>
    </div>

    <!-- ══════════════════════════════════════════ TAB: MANUELL ══════ -->
    <?php if ($active_tab === 'manual'): ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['create_user_local'] ?? 'Lokalen Benutzer anlegen'; ?></h2>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['local_user_desc'] ?? 'Fuer Lehrer, Externe und Gaeste (ohne Active Directory).'; ?></p>
        </div>
        <form method="POST" class="p-6 space-y-6">
            <input type="hidden" name="action" value="create_local">
            <?php echo csrf_field(); ?>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['first_name'] ?? 'Vorname'; ?> <span class="text-red-500">*</span></label>
                    <input type="text" name="first_name" required
                           value="<?php echo esc_attr($form['first_name'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['last_name'] ?? 'Nachname'; ?> <span class="text-red-500">*</span></label>
                    <input type="text" name="last_name" required
                           value="<?php echo esc_attr($form['last_name'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['email'] ?? 'E-Mail'; ?> <span class="text-red-500">*</span></label>
                    <input type="email" name="email" required
                           value="<?php echo esc_attr($form['email'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['username'] ?? 'Benutzername'; ?></label>
                    <input type="text" name="username"
                           value="<?php echo esc_attr($form['username'] ?? ''); ?>"
                           placeholder="<?php echo esc_attr($lang['username_optional_hint'] ?? 'Optional (fuer Login)'); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['user_group'] ?? 'Benutzergruppe'; ?> <span class="text-red-500">*</span></label>
                    <select name="user_group_id" required
                            class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                        <?php foreach ($user_groups as $g): ?>
                            <option value="<?php echo $g['id']; ?>"
                                <?php echo (($form['user_group_id'] ?? 2) == $g['id']) ? 'selected' : ''; ?>>
                                <?php echo esc_html($g['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['birthday'] ?? 'Geburtsdatum'; ?></label>
                    <input type="date" name="birthday"
                           value="<?php echo esc_attr($form['birthday'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['phone'] ?? 'Telefon'; ?></label>
                    <input type="tel" name="phone"
                           value="<?php echo esc_attr($form['phone'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['class_name'] ?? 'Klasse'; ?></label>
                    <input type="text" name="class_name"
                           value="<?php echo esc_attr($form['class_name'] ?? ''); ?>"
                           placeholder="z.B. 3HLW"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                </div>

                <div class="md:col-span-2">
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['address'] ?? 'Adresse'; ?></label>
                    <textarea name="address" rows="2"
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500"><?php echo esc_html($form['address'] ?? ''); ?></textarea>
                </div>

            </div>

            <!-- Passwort -->
            <div class="border-t pt-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4"><?php echo $lang['password'] ?? 'Passwort'; ?> <span class="text-red-500">*</span></h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['password'] ?? 'Passwort'; ?></label>
                        <input type="password" name="password" required autocomplete="new-password"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['password_confirm'] ?? 'Passwort bestaetigen'; ?></label>
                        <input type="password" name="password_confirm" required autocomplete="new-password"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                    </div>
                </div>
                <p class="mt-2 text-xs text-gray-500"><?php echo $lang['password_min_hint'] ?? 'Mind. 8 Zeichen, Gross-/Kleinbuchstaben, Zahl und Sonderzeichen.'; ?></p>
            </div>

            <div class="flex justify-end gap-3 border-t pt-6">
                <a href="users-list.php" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                    <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                </a>
                <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium">
                    <?php echo $lang['create_user'] ?? 'Benutzer anlegen'; ?>
                </button>
            </div>
        </form>
    </div>

    <!-- ══════════════════════════════════════════════ TAB: LDAP ═════ -->
    <?php elseif ($active_tab === 'ldap' && $ldap_enabled): ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['create_via_ldap'] ?? 'Benutzer via LDAP/Active Directory anlegen'; ?></h2>
            <p class="mt-1 text-sm text-gray-500">
                Suche im Active Directory und lege den Benutzer direkt an.
                Wenn ein passender Sokrates-Eintrag existiert, werden die Daten zusammengefuehrt.
            </p>
        </div>
        <div class="p-6">

            <!-- LDAP-Suche -->
            <div class="mb-6">
                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_search_label'] ?? 'Name suchen'; ?></label>
                <div class="flex gap-3">
                    <input type="text" id="ldapSearchInput" placeholder="<?php echo esc_attr($lang['ldap_search_placeholder'] ?? 'Nachname oder Vorname (mind. 2 Zeichen)...'); ?>"
                           class="flex-1 px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                    <button onclick="ldapSearch()" id="ldapSearchBtn"
                            class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                        <?php echo $lang['ldap_search_btn'] ?? 'Suchen'; ?>
                    </button>
                </div>
            </div>

            <!-- Suchergebnisse -->
            <div id="ldapResults" class="mb-6"></div>

            <!-- Anlegen-Dialog (versteckt) -->
            <div id="ldapCreatePanel" class="hidden border-t pt-6">
                <h3 class="text-lg font-semibold text-gray-900 mb-4"><?php echo $lang['ldap_create_user_panel'] ?? 'Benutzer anlegen'; ?></h3>
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4 text-sm text-blue-800" id="ldapMergeHint"></div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['first_name'] ?? 'Vorname'; ?></label>
                        <input type="text" id="ldapFirstName" readonly
                               class="w-full px-3 py-2 border border-gray-200 bg-gray-50 rounded-md text-gray-700">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['last_name'] ?? 'Nachname'; ?></label>
                        <input type="text" id="ldapLastName" readonly
                               class="w-full px-3 py-2 border border-gray-200 bg-gray-50 rounded-md text-gray-700">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['ldap_username_ad'] ?? 'Benutzername (AD)'; ?></label>
                        <input type="text" id="ldapUsername"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['email'] ?? 'E-Mail'; ?></label>
                        <input type="email" id="ldapEmail"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['user_group'] ?? 'Benutzergruppe'; ?></label>
                        <select id="ldapGroupId"
                                class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500">
                            <?php foreach ($user_groups as $g): ?>
                                <option value="<?php echo $g['id']; ?>"><?php echo esc_html($g['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <input type="hidden" id="ldapDn">

                <div id="ldapCreateError" class="hidden mb-4 bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-700"></div>

                <div class="flex gap-3">
                    <button onclick="createLdapUser()"
                            class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 font-medium">
                        <?php echo $lang['create_user'] ?? 'Benutzer anlegen'; ?>
                    </button>
                    <button onclick="closeLdapPanel()"
                            class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50">
                        <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                    </button>
                </div>
            </div>

        </div>
    </div>

    <?php elseif ($active_tab === 'ldap' && !$ldap_enabled): ?>
    <div class="bg-white rounded-lg shadow p-6">
        <div class="text-center py-8">
            <p class="text-gray-600 mb-4"><?php echo $lang['ldap_not_enabled_msg'] ?? 'LDAP/Active Directory ist nicht aktiviert.'; ?></p>
            <a href="settings-system.php" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                <?php echo $lang['ldap_system_settings_link'] ?? 'Zu den Systemeinstellungen'; ?>
            </a>
        </div>
    </div>
    <?php endif; ?>

</div>
</div>

<?php if ($active_tab === 'ldap' && $ldap_enabled): ?>
<script>
const CSRF_TOKEN = '<?php echo esc_js(generate_csrf_token()); ?>';

function esc(str) {
    if (!str) return '';
    const d = document.createElement('div');
    d.textContent = str;
    return d.innerHTML;
}

function ldapSearch() {
    const q = document.getElementById('ldapSearchInput').value.trim();
    if (q.length < 2) {
        alert('<?php echo esc_js($lang['ldap_search_placeholder'] ?? 'Bitte mindestens 2 Zeichen eingeben.'); ?>');
        return;
    }

    const results = document.getElementById('ldapResults');
    results.innerHTML = '<div class="text-center py-4"><div class="inline-block animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div><p class="mt-2 text-sm text-gray-500"><?php echo esc_js($lang['loading'] ?? 'Laedt...'); ?></p></div>';
    closeLdapPanel();

    fetch('api/ldap-search.php?q=' + encodeURIComponent(q))
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                results.innerHTML = '<div class="bg-red-50 border border-red-200 rounded-lg p-4 text-sm text-red-700">' + esc(data.error) + '</div>';
                return;
            }
            if (!Array.isArray(data) || data.length === 0) {
                results.innerHTML = '<div class="bg-gray-50 border border-gray-200 rounded-lg p-4 text-sm text-gray-600"><?php echo esc_js($lang['ldap_no_results'] ?? 'Keine Treffer im Active Directory.'); ?></div>';
                return;
            }

            let html = '<div class="overflow-x-auto"><table class="min-w-full text-sm border-collapse">';
            html += '<thead class="bg-gray-100"><tr>';
            html += '<th class="border px-3 py-2 text-left font-medium text-gray-700"><?php echo esc_js($lang['name'] ?? 'Name'); ?></th>';
            html += '<th class="border px-3 py-2 text-left font-medium text-gray-700"><?php echo esc_js($lang['username'] ?? 'Benutzername'); ?></th>';
            html += '<th class="border px-3 py-2 text-left font-medium text-gray-700"><?php echo esc_js($lang['email'] ?? 'E-Mail'); ?></th>';
            html += '<th class="border px-3 py-2 text-center font-medium text-gray-700"><?php echo esc_js($lang['actions'] ?? 'Aktion'); ?></th>';
            html += '</tr></thead><tbody>';

            data.forEach((user, idx) => {
                html += '<tr class="hover:bg-gray-50">';
                html += '<td class="border px-3 py-2">' + esc(user.last_name + ', ' + user.first_name) + '</td>';
                html += '<td class="border px-3 py-2 font-mono text-xs">' + esc(user.username) + '</td>';
                html += '<td class="border px-3 py-2">' + esc(user.email || '-') + '</td>';
                html += '<td class="border px-3 py-2 text-center">';
                html += '<button onclick=\'openLdapPanel(' + JSON.stringify(user) + ')\' ';
                html += 'class="px-3 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700"><?php echo esc_js($lang['create_user'] ?? 'Anlegen'); ?></button>';
                html += '</td>';
                html += '</tr>';
            });

            html += '</tbody></table></div>';
            html += '<p class="mt-2 text-xs text-gray-500">' + data.length + ' <?php echo esc_js($lang['results'] ?? 'Ergebnis(se) gefunden'); ?></p>';

            results.innerHTML = html;
        })
        .catch(err => {
            results.innerHTML = '<div class="bg-red-50 border border-red-200 rounded-lg p-4 text-sm text-red-700">Netzwerkfehler: ' + esc(err.message) + '</div>';
        });
}

function openLdapPanel(user) {
    document.getElementById('ldapFirstName').value = user.first_name || '';
    document.getElementById('ldapLastName').value  = user.last_name  || '';
    document.getElementById('ldapUsername').value  = user.username   || '';
    document.getElementById('ldapEmail').value     = user.email      || '';
    document.getElementById('ldapDn').value        = user.ldap_dn    || '';
    document.getElementById('ldapCreateError').classList.add('hidden');

    // Merge-Hinweis anzeigen
    const hint = document.getElementById('ldapMergeHint');
    hint.innerHTML = 'Beim Anlegen wird geprueft, ob ein passender Sokrates-Eintrag (gleicher Name, noch kein Benutzername) vorhanden ist und damit verbunden werden kann.';

    document.getElementById('ldapCreatePanel').classList.remove('hidden');
    document.getElementById('ldapCreatePanel').scrollIntoView({behavior: 'smooth', block: 'start'});
}

function closeLdapPanel() {
    document.getElementById('ldapCreatePanel').classList.add('hidden');
}

function createLdapUser() {
    const fd = new FormData();
    fd.append('action',        'create_ldap_user');
    fd.append('csrf_token',    CSRF_TOKEN);
    fd.append('first_name',    document.getElementById('ldapFirstName').value);
    fd.append('last_name',     document.getElementById('ldapLastName').value);
    fd.append('username',      document.getElementById('ldapUsername').value);
    fd.append('email',         document.getElementById('ldapEmail').value);
    fd.append('ldap_dn',       document.getElementById('ldapDn').value);
    fd.append('user_group_id', document.getElementById('ldapGroupId').value);

    const errEl = document.getElementById('ldapCreateError');
    errEl.classList.add('hidden');

    fetch('users-add.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                // Weiterleitung zur Benutzerliste mit Erfolgs-Meldung
                window.location.href = 'users-list.php?created=1&msg=' + encodeURIComponent(data.message || 'Benutzer angelegt');
            } else {
                errEl.textContent = data.error || 'Unbekannter Fehler';
                errEl.classList.remove('hidden');
            }
        })
        .catch(err => {
            errEl.textContent = 'Netzwerkfehler: ' + err.message;
            errEl.classList.remove('hidden');
        });
}

// Suche auch per Enter-Taste
document.getElementById('ldapSearchInput').addEventListener('keydown', function(e) {
    if (e.key === 'Enter') ldapSearch();
});
</script>
<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>
