<?php
/**
 * Einstellungen: Benutzerverwaltung
 * Tabs: Benutzergruppen · LDAP / Active Directory
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$page_title   = 'Einstellungen: Benutzerverwaltung';
$current_user = get_logged_in_user();

// Load all settings
$settings = [];
$result = db_query("SELECT setting_key, setting_value FROM settings");
while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// ─── POST Handlers ──────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', $lang['security_error_retry'] ?? 'Sicherheitsfehler. Bitte versuchen Sie es erneut.');
        header('Location: settings-users.php'); exit;
    }

    // ── Benutzergruppe speichern (inline edit) ───────────────────────────────
    if ($action === 'save_group') {
        $id   = intval($_POST['id'] ?? 0);
        $name = trim($_POST['name'] ?? '');
        $desc = trim($_POST['description'] ?? '');
        if (!$id || !$name) {
            set_flash('error', $lang['invalid_input'] ?? 'Ungültige Eingabe.');
        } elseif (db_get_var("SELECT COUNT(*) FROM user_groups WHERE name = ? AND id != ?", [$name, $id])) {
            set_flash('error', sprintf($lang['name_already_taken'] ?? 'Name ist bereits vergeben.'));
        } else {
            db_update('user_groups', [
                'name'               => $name,
                'description'        => $desc ?: null,
                'max_loans'          => max(1, (int)($_POST['max_loans'] ?? 5)),
                'loan_duration_days' => max(1, (int)($_POST['loan_duration_days'] ?? 28)),
                'can_extend'         => isset($_POST['can_extend']) ? 1 : 0,
                'max_extensions'     => $_POST['max_extensions'] !== '' ? max(0, (int)$_POST['max_extensions']) : null,
                'can_reserve'        => isset($_POST['can_reserve']) ? 1 : 0,
            ], 'id = ?', [$id]);
            log_activity($current_user['id'], 'group_updated', 'user_groups', $id, "Group '{$name}' updated");
            set_flash('success', ($lang['group_saved'] ?? 'Gruppe gespeichert.'));
        }
        header('Location: settings-users.php?tab=benutzergruppen'); exit;
    }

    // ── Neue Benutzergruppe anlegen ──────────────────────────────────────────
    if ($action === 'add_group') {
        $name = trim($_POST['name'] ?? '');
        $desc = trim($_POST['description'] ?? '');
        if (!$name) {
            set_flash('error', $lang['name_required'] ?? 'Name ist ein Pflichtfeld.');
        } elseif (db_get_var("SELECT COUNT(*) FROM user_groups WHERE name = ?", [$name])) {
            set_flash('error', sprintf($lang['name_already_taken'] ?? 'Name ist bereits vergeben.'));
        } else {
            $newId = db_insert('user_groups', [
                'name'               => $name,
                'description'        => $desc ?: null,
                'max_loans'          => max(1, (int)($_POST['max_loans'] ?? 5)),
                'loan_duration_days' => max(1, (int)($_POST['loan_duration_days'] ?? 28)),
                'can_extend'         => isset($_POST['can_extend']) ? 1 : 0,
                'max_extensions'     => $_POST['max_extensions'] !== '' ? max(0, (int)$_POST['max_extensions']) : null,
                'can_reserve'        => isset($_POST['can_reserve']) ? 1 : 0,
            ]);
            log_activity($current_user['id'], 'group_created', 'user_groups', $newId, "Group '{$name}' created");
            set_flash('success', ($lang['group_created'] ?? 'Gruppe angelegt.'));
        }
        header('Location: settings-users.php?tab=benutzergruppen'); exit;
    }

    // ── Benutzergruppe löschen ───────────────────────────────────────────────
    if ($action === 'delete_group') {
        $id   = intval($_POST['id'] ?? 0);
        $used = (int)db_get_var("SELECT COUNT(*) FROM users WHERE user_group_id = ?", [$id]);
        if ($used > 0) {
            set_flash('error', sprintf($lang['group_delete_has_users'] ?? 'Gruppe kann nicht gelöscht werden: %d Benutzer zugewiesen.', $used));
        } else {
            $name = db_get_var("SELECT name FROM user_groups WHERE id = ?", [$id]);
            db_delete('user_groups', 'id = ?', [$id]);
            log_activity($current_user['id'], 'group_deleted', 'user_groups', $id, "Group '{$name}' deleted");
            set_flash('success', ($lang['group_deleted'] ?? 'Gruppe gelöscht.'));
        }
        header('Location: settings-users.php?tab=benutzergruppen'); exit;
    }

    // ── LDAP-Einstellungen speichern ─────────────────────────────────────────
    if ($action === 'save_ldap') {
        $updates = [
            'ldap_enabled'             => isset($_POST['ldap_enabled']) ? '1' : '0',
            'ldap_host'                => trim($_POST['ldap_host'] ?? ''),
            'ldap_port'                => trim($_POST['ldap_port'] ?? '389'),
            'ldap_base_dn'             => trim($_POST['ldap_base_dn'] ?? ''),
            'ldap_bind_dn'             => trim($_POST['ldap_bind_dn'] ?? ''),
            'ldap_username_attribute'  => trim($_POST['ldap_username_attribute'] ?? 'sAMAccountName'),
            'ldap_email_attribute'     => trim($_POST['ldap_email_attribute'] ?? 'mail'),
            'ldap_firstname_attribute' => trim($_POST['ldap_firstname_attribute'] ?? 'givenName'),
            'ldap_lastname_attribute'  => trim($_POST['ldap_lastname_attribute'] ?? 'sn'),
        ];
        if (!empty($_POST['ldap_bind_password'])) {
            $updates['ldap_bind_password'] = $_POST['ldap_bind_password'];
        }
        foreach ($updates as $key => $value) {
            setSetting($key, $value);
        }
        log_activity($current_user['id'], 'settings_updated', 'settings', 0, 'LDAP settings updated');
        set_flash('success', $lang['ldap_settings_saved'] ?? 'LDAP-Einstellungen gespeichert.');
        header('Location: settings-users.php?tab=ldap'); exit;
    }

    // ── LDAP-Verbindung testen ───────────────────────────────────────────────
    if ($action === 'test_ldap') {
        if (!function_exists('ldap_connect')) {
            set_flash('error', $lang['error_ldap_not_available'] ?? 'LDAP-Extension ist nicht verfügbar.');
        } else {
            $test_host     = trim($_POST['ldap_host'] ?? '');
            $test_port     = (int)trim($_POST['ldap_port'] ?? '389');
            $test_bind_dn  = trim($_POST['ldap_bind_dn'] ?? '');
            $test_bind_pw  = $_POST['ldap_bind_password'] ?? getSetting('ldap_bind_password');
            $ldap_conn = @ldap_connect($test_host, $test_port);
            if ($ldap_conn) {
                ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3);
                ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0);
                if (@ldap_bind($ldap_conn, $test_bind_dn, $test_bind_pw)) {
                    set_flash('success', $lang['success_ldap_connection'] ?? 'LDAP-Verbindung erfolgreich!');
                    ldap_close($ldap_conn);
                } else {
                    set_flash('error', $lang['error_ldap_bind_failed'] ?? 'LDAP-Bind fehlgeschlagen. Bind-DN und Passwort prüfen.');
                }
            } else {
                set_flash('error', $lang['error_ldap_connection_failed'] ?? 'LDAP-Verbindung fehlgeschlagen. Host und Port prüfen.');
            }
        }
        header('Location: settings-users.php?tab=ldap'); exit;
    }
}

// ─── Daten laden ─────────────────────────────────────────────────────────────
$groups = db_get_results("
    SELECT g.id, g.name, g.description, g.max_loans, g.loan_duration_days,
           g.can_extend, g.max_extensions, g.can_reserve,
           COUNT(u.id) AS user_count
    FROM user_groups g
    LEFT JOIN users u ON u.user_group_id = g.id
    GROUP BY g.id
    ORDER BY g.id
");

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-8">
        <h1 class="text-2xl font-bold text-gray-900"><?php echo $lang['settings_users_title'] ?? 'Einstellungen: Benutzerverwaltung'; ?></h1>
        <p class="mt-1 text-sm text-gray-600"><?php echo $lang['settings_users_desc'] ?? 'Benutzergruppen und LDAP/Active Directory'; ?></p>
    </div>

    <!-- Flash-Meldung -->
    <?php if ($flash): ?>
        <?php $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue']; $c = $fc[$flash['type']] ?? 'blue'; ?>
        <div class="mb-6 flash-message bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo esc_html($flash['message']); ?></p>
        </div>
    <?php endif; ?>

    <!-- Tab-Navigation -->
    <div class="mb-6 border-b border-gray-200">
        <nav class="-mb-px flex space-x-8">
            <a href="#benutzergruppen" data-tab="benutzergruppen" class="usr-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['tab_user_groups'] ?? 'Benutzergruppen'; ?></a>
            <a href="#ldap"            data-tab="ldap"            class="usr-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['tab_ldap'] ?? 'LDAP / Active Directory'; ?></a>
        </nav>
    </div>

    <!-- ══════════════════════════════════════════════════════════════
         TAB: BENUTZERGRUPPEN
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-benutzergruppen" class="usr-section hidden">

        <!-- Inline-Edit-Tabelle -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden mb-8">
            <div class="px-5 py-4 border-b border-gray-100 flex items-center justify-between">
                <h2 class="text-base font-semibold text-gray-900"><?php echo $lang['groups_heading'] ?? 'Gruppen'; ?></h2>
                <p class="text-xs text-gray-400"><?php echo $lang['groups_inline_edit_hint'] ?? 'Änderungen direkt in der Tabelle vornehmen und mit'; ?>
                    <svg class="inline h-3.5 w-3.5 mb-0.5 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                    <?php echo $lang['confirm'] ?? 'bestätigen'; ?></p>
            </div>
            <div class="overflow-x-auto">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase"><?php echo $lang['name'] ?? 'Name'; ?></th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase w-24"><?php echo $lang['max_loans_short'] ?? 'Max. Ausleihen'; ?></th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase w-28"><?php echo $lang['loan_duration_days'] ?? 'Ausleihdauer (Tage)'; ?></th>
                            <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase w-24"><?php echo $lang['extend'] ?? 'Verlängern'; ?></th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase w-28"><?php echo $lang['max_extensions'] ?? 'Max. Verlängerungen'; ?></th>
                            <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase w-20"><?php echo $lang['reserve'] ?? 'Vormerken'; ?></th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase w-16"><?php echo $lang['users_count'] ?? 'Nutzer'; ?></th>
                            <th class="px-4 py-3 w-16"></th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        <?php foreach ($groups as $g): ?>
                            <tr class="hover:bg-gray-50 align-top">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                    <input type="hidden" name="action" value="save_group">
                                    <input type="hidden" name="id"     value="<?php echo $g['id']; ?>">
                                    <!-- Name -->
                                    <td class="px-4 py-3">
                                        <input type="text" name="name" required maxlength="50"
                                               value="<?php echo esc_attr($g['name']); ?>"
                                               class="w-full px-2 py-1.5 border border-gray-300 rounded text-sm focus:ring-2 focus:ring-blue-500">
                                        <input type="text" name="description" maxlength="255"
                                               value="<?php echo esc_attr($g['description'] ?? ''); ?>"
                                               placeholder="<?php echo esc_attr($lang['description_optional'] ?? 'Beschreibung (optional)'); ?>"
                                               class="mt-1 w-full px-2 py-1 border border-gray-200 rounded text-xs text-gray-500 focus:ring-1 focus:ring-blue-400">
                                    </td>
                                    <!-- Max. Ausleihen -->
                                    <td class="px-4 py-3">
                                        <input type="number" name="max_loans" min="1" max="999" required
                                               value="<?php echo (int)$g['max_loans']; ?>"
                                               class="w-16 px-2 py-1.5 border border-gray-300 rounded text-sm text-center focus:ring-2 focus:ring-blue-500">
                                    </td>
                                    <!-- Ausleihdauer -->
                                    <td class="px-4 py-3">
                                        <input type="number" name="loan_duration_days" min="1" max="999" required
                                               value="<?php echo (int)$g['loan_duration_days']; ?>"
                                               class="w-16 px-2 py-1.5 border border-gray-300 rounded text-sm text-center focus:ring-2 focus:ring-blue-500">
                                    </td>
                                    <!-- Verlängern -->
                                    <td class="px-4 py-3 text-center">
                                        <input type="checkbox" name="can_extend" value="1"
                                               <?php echo $g['can_extend'] ? 'checked' : ''; ?>
                                               class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                                    </td>
                                    <!-- Max. Verlängerungen -->
                                    <td class="px-4 py-3">
                                        <input type="number" name="max_extensions" min="0" max="99"
                                               value="<?php echo $g['max_extensions'] !== null ? (int)$g['max_extensions'] : ''; ?>"
                                               placeholder="∞"
                                               class="w-16 px-2 py-1.5 border border-gray-300 rounded text-sm text-center focus:ring-2 focus:ring-blue-500">
                                        <p class="text-xs text-gray-400 mt-0.5"><?php echo $lang['unlimited_hint'] ?? 'leer = unbegrenzt'; ?></p>
                                    </td>
                                    <!-- Vormerken -->
                                    <td class="px-4 py-3 text-center">
                                        <input type="checkbox" name="can_reserve" value="1"
                                               <?php echo $g['can_reserve'] ? 'checked' : ''; ?>
                                               class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                                    </td>
                                    <!-- Nutzer-Zähler -->
                                    <td class="px-4 py-3 text-center">
                                        <span class="text-sm text-gray-500"><?php echo (int)$g['user_count']; ?></span>
                                    </td>
                                    <!-- Aktionen -->
                                    <td class="px-4 py-3 text-right whitespace-nowrap">
                                        <button type="submit" title="<?php echo esc_attr($lang['save'] ?? 'Speichern'); ?>"
                                                class="p-1.5 bg-blue-600 text-white rounded hover:bg-blue-700 transition-colors">
                                            <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                                        </button>
                                </form>
                                <form method="POST" class="inline ml-1"
                                      onsubmit="return confirm('<?php echo addslashes($lang['confirm_delete_group'] ?? 'Gruppe wirklich loschen? Diese Aktion kann nicht ruckgangig gemacht werden.'); ?>')">
                                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                    <input type="hidden" name="action" value="delete_group">
                                    <input type="hidden" name="id"     value="<?php echo $g['id']; ?>">
                                    <button type="submit" title="<?php echo $g['user_count'] > 0 ? esc_attr($lang['group_has_users'] ?? 'Gruppe hat zugewiesene Benutzer') : esc_attr($lang['delete'] ?? 'Loschen'); ?>"
                                            class="p-1.5 border border-red-300 text-red-500 rounded hover:bg-red-50 transition-colors disabled:opacity-40 disabled:cursor-not-allowed"
                                            <?php echo $g['user_count'] > 0 ? 'disabled' : ''; ?>>
                                        <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/></svg>
                                    </button>
                                </form>
                                    </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if (empty($groups)): ?>
                            <tr>
                                <td colspan="8" class="px-4 py-6 text-center text-gray-400 text-sm"><?php echo $lang['no_groups'] ?? 'Keine Gruppen vorhanden'; ?></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Neue Gruppe anlegen -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="px-5 py-4 border-b border-gray-100">
                <h2 class="text-base font-semibold text-gray-900"><?php echo $lang['create_new_group'] ?? 'Neue Gruppe anlegen'; ?></h2>
            </div>
            <div class="p-5">
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="add_group">

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                <?php echo $lang['name'] ?? 'Name'; ?> <span class="text-red-500">*</span>
                            </label>
                            <input type="text" name="name" required maxlength="50"
                                   placeholder="z.B. Schüler, Lehrer, Extern..."
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['description'] ?? 'Beschreibung'; ?></label>
                            <input type="text" name="description" maxlength="255"
                                   placeholder="<?php echo esc_attr($lang['description_optional'] ?? 'Beschreibung (optional)'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['max_loans_short'] ?? 'Max. Ausleihen'; ?></label>
                            <input type="number" name="max_loans" min="1" max="999" value="5"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['loan_duration_days'] ?? 'Ausleihdauer (Tage)'; ?></label>
                            <input type="number" name="loan_duration_days" min="1" max="999" value="28"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['max_extensions'] ?? 'Max. Verlängerungen'; ?></label>
                            <input type="number" name="max_extensions" min="0" max="99" placeholder="∞ <?php echo esc_attr($lang['unlimited_hint'] ?? 'leer = unbegrenzt'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div class="flex flex-col justify-end gap-3 pb-0.5">
                            <label class="flex items-center gap-2 cursor-pointer">
                                <input type="checkbox" name="can_extend" value="1" checked
                                       class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                                <span class="text-sm text-gray-700"><?php echo $lang['extend_allowed'] ?? 'Verlängern erlaubt'; ?></span>
                            </label>
                            <label class="flex items-center gap-2 cursor-pointer">
                                <input type="checkbox" name="can_reserve" value="1" checked
                                       class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                                <span class="text-sm text-gray-700"><?php echo $lang['reserve_allowed'] ?? 'Vormerken erlaubt'; ?></span>
                            </label>
                        </div>
                    </div>

                    <div class="flex justify-end border-t pt-4">
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm font-medium transition-colors">
                            <?php echo $lang['create_group'] ?? 'Gruppe anlegen'; ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div><!-- /tab-benutzergruppen -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: LDAP / ACTIVE DIRECTORY
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-ldap" class="usr-section hidden">

        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="p-6 border-b border-gray-100">
                <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['ldap_ad_title'] ?? 'LDAP / Active Directory'; ?></h2>
                <p class="mt-1 text-sm text-gray-500"><?php echo $lang['ldap_ad_desc'] ?? 'Authentifizierung gegen Active Directory'; ?></p>
            </div>
            <div class="p-6">

                <!-- Info-Box -->
                <div class="mb-6 bg-blue-50 border-l-4 border-blue-400 p-4 rounded">
                    <div class="flex gap-3">
                        <svg class="h-5 w-5 text-blue-400 shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                        </svg>
                        <div>
                            <p class="text-sm font-medium text-blue-800"><?php echo $lang['ldap_on_demand_title'] ?? 'On-Demand User Creation'; ?></p>
                            <p class="mt-1 text-sm text-blue-700">
                                <?php echo $lang['ldap_on_demand_desc'] ?? 'LDAP-Benutzer werden nicht massenweise synchronisiert. Sie werden automatisch beim ersten Ausleihvorgang in der Datenbank angelegt bzw. mit einem vorhandenen Sokrates-Import-Datensatz zusammengefuhrt.'; ?>
                            </p>
                        </div>
                    </div>
                </div>

                <form method="POST" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="save_ldap">

                    <div class="flex items-center">
                        <input type="checkbox" name="ldap_enabled" id="ldap_enabled" value="1"
                               <?php echo ($settings['ldap_enabled'] ?? '0') === '1' ? 'checked' : ''; ?>
                               class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                        <label for="ldap_enabled" class="ml-2 text-sm font-medium text-gray-700">
                            <?php echo $lang['enable_ldap'] ?? 'LDAP-Authentifizierung aktivieren'; ?>
                        </label>
                    </div>

                    <div class="border-t pt-6">
                        <h3 class="text-base font-semibold text-gray-800 mb-4"><?php echo $lang['connection_settings'] ?? 'Verbindungseinstellungen'; ?></h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['ldap_host'] ?? 'LDAP-Server'; ?> <span class="text-red-500">*</span>
                                </label>
                                <input type="text" name="ldap_host" required
                                       value="<?php echo esc_attr($settings['ldap_host'] ?? ''); ?>"
                                       placeholder="ldap://server.example.com"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['ldap_port'] ?? 'Port'; ?> <span class="text-red-500">*</span>
                                </label>
                                <input type="number" name="ldap_port" required
                                       value="<?php echo esc_attr($settings['ldap_port'] ?? 389); ?>"
                                       placeholder="389"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div class="md:col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['ldap_base_dn'] ?? 'Base DN'; ?> <span class="text-red-500">*</span>
                                </label>
                                <input type="text" name="ldap_base_dn" required
                                       value="<?php echo esc_attr($settings['ldap_base_dn'] ?? ''); ?>"
                                       placeholder="OU=VirtualSchool,DC=schule,DC=intern"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div class="md:col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['ldap_bind_dn'] ?? 'Bind DN'; ?> <span class="text-red-500">*</span>
                                </label>
                                <input type="text" name="ldap_bind_dn" required
                                       value="<?php echo esc_attr($settings['ldap_bind_dn'] ?? ''); ?>"
                                       placeholder="CN=Service-Account,CN=Users,DC=schule,DC=intern"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div class="md:col-span-2">
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_bind_password'] ?? 'Bind Passwort'; ?></label>
                                <input type="password" name="ldap_bind_password"
                                       placeholder="<?php echo esc_attr($lang['leave_empty_to_keep'] ?? 'Leer lassen um beizubehalten'); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                    </div>

                    <div class="border-t pt-6">
                        <h3 class="text-base font-semibold text-gray-800 mb-4"><?php echo $lang['user_attributes'] ?? 'Benutzer-Attribute'; ?></h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_username_attr'] ?? 'Benutzername-Attribut'; ?></label>
                                <input type="text" name="ldap_username_attribute"
                                       value="<?php echo esc_attr($settings['ldap_username_attribute'] ?? 'sAMAccountName'); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_email_attr'] ?? 'E-Mail-Attribut'; ?></label>
                                <input type="text" name="ldap_email_attribute"
                                       value="<?php echo esc_attr($settings['ldap_email_attribute'] ?? 'mail'); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_firstname_attr'] ?? 'Vorname-Attribut'; ?></label>
                                <input type="text" name="ldap_firstname_attribute"
                                       value="<?php echo esc_attr($settings['ldap_firstname_attribute'] ?? 'givenName'); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['ldap_lastname_attr'] ?? 'Nachname-Attribut'; ?></label>
                                <input type="text" name="ldap_lastname_attribute"
                                       value="<?php echo esc_attr($settings['ldap_lastname_attribute'] ?? 'sn'); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                    </div>

                    <div class="flex justify-between border-t pt-6">
                        <!-- Test-Button: separates Formular, speichert nicht -->
                        <button type="button" onclick="testLdap(this)"
                                class="px-5 py-2 border border-blue-600 text-blue-600 rounded-md hover:bg-blue-50 text-sm font-medium transition-colors">
                            <?php echo $lang['test_connection'] ?? 'Verbindung testen'; ?>
                        </button>
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                            <?php echo $lang['save'] ?? 'Speichern'; ?>
                        </button>
                    </div>
                </form>

            </div>
        </div>

    </div><!-- /tab-ldap -->

</div>
</div>

<script>
(function () {
    function getActiveTab() {
        if (window.location.hash) return window.location.hash.substring(1);
        return new URLSearchParams(window.location.search).get('tab') || 'benutzergruppen';
    }

    function showTab(id) {
        document.querySelectorAll('.usr-section').forEach(function (s) {
            s.classList.add('hidden');
        });
        document.querySelectorAll('.usr-tab').forEach(function (t) {
            if (t.dataset.tab === id) {
                t.classList.add('border-blue-500', 'text-blue-600');
                t.classList.remove('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            } else {
                t.classList.remove('border-blue-500', 'text-blue-600');
                t.classList.add('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            }
        });
        var sec = document.getElementById('tab-' + id);
        if (sec) sec.classList.remove('hidden');
    }

    document.querySelectorAll('.usr-tab').forEach(function (t) {
        t.addEventListener('click', function (e) {
            e.preventDefault();
            var id = this.dataset.tab;
            history.pushState(null, null, location.pathname + '#' + id);
            showTab(id);
        });
    });

    showTab(getActiveTab());

    window.addEventListener('hashchange', function () {
        showTab(window.location.hash.substring(1) || 'benutzergruppen');
    });
})();

// LDAP-Test via POST (separate form submit)
function testLdap(btn) {
    var form = btn.closest('form');
    var orig = btn.textContent;
    btn.disabled = true;
    btn.textContent = '<?php echo addslashes($lang['testing'] ?? 'Teste...'); ?>';

    var data = new FormData(form);
    data.set('action', 'test_ldap');

    fetch('settings-users.php', { method: 'POST', body: data })
        .then(function (r) { return r.text(); })
        .then(function () {
            // Server redirect after test → reload to show flash message
            location.href = 'settings-users.php?tab=ldap';
        })
        .catch(function () {
            btn.disabled = false;
            btn.textContent = orig;
            alert('<?php echo addslashes($lang['ldap_test_error'] ?? 'Fehler beim Verbindungstest.'); ?>');
        });
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
