<?php
/**
 * Einstellungen: Verleihverwaltung
 * Tabs: Ausleihen · E-Mail / SMTP
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$page_title   = 'Einstellungen: Verleihverwaltung';
$current_user = get_logged_in_user();

// Load all settings
$settings = [];
$result = db_query("SELECT setting_key, setting_value FROM settings");
while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// ─── POST Handlers ──────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        // JSON-Handler abfangen
        if ($action === 'send_test_email') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => $lang['security_error_retry'] ?? 'Sicherheitsfehler.']);
            exit;
        }
        set_flash('error', $lang['security_error_retry'] ?? 'Sicherheitsfehler. Bitte versuchen Sie es erneut.');
        header('Location: settings-lending.php'); exit;
    }

    // ── Ausleih-Einstellungen speichern ──────────────────────────────────────
    if ($action === 'save_loans') {
        $updates = [
            'loan_duration_default'  => max(1, (int)($_POST['loan_duration_default'] ?? 28)),
            'max_extensions'         => max(0, (int)($_POST['max_extensions'] ?? 6)),
            'max_loans_per_user'     => max(1, (int)($_POST['max_loans_per_user'] ?? 5)),
            'reminder_enabled'       => isset($_POST['reminder_enabled']) ? 'true' : 'false',
            'reminder_days_after_due'=> max(1, (int)($_POST['reminder_days_after_due'] ?? 7)),
            'reminder_frequency'     => in_array($_POST['reminder_frequency'] ?? '', ['daily','weekly','monthly'])
                                            ? $_POST['reminder_frequency'] : 'weekly',
        ];
        foreach ($updates as $key => $value) {
            setSetting($key, $value);
        }
        log_activity($current_user['id'], 'settings_updated', 'settings', 0, 'Loan settings updated');
        set_flash('success', $lang['loan_settings_saved'] ?? 'Ausleih-Einstellungen gespeichert.');
        header('Location: settings-lending.php?tab=ausleihen'); exit;
    }

    // ── SMTP-Einstellungen speichern ─────────────────────────────────────────
    if ($action === 'save_smtp') {
        $updates = [
            'smtp_enabled'     => isset($_POST['smtp_enabled']) ? 'true' : 'false',
            'smtp_host'        => trim($_POST['smtp_host'] ?? ''),
            'smtp_port'        => max(1, (int)($_POST['smtp_port'] ?? 587)),
            'smtp_username'    => trim($_POST['smtp_username'] ?? ''),
            'smtp_from_email'  => trim($_POST['smtp_from_email'] ?? ''),
            'smtp_from_name'   => trim($_POST['smtp_from_name'] ?? ''),
        ];
        if (!empty($_POST['smtp_password'])) {
            $updates['smtp_password'] = $_POST['smtp_password'];
        }
        foreach ($updates as $key => $value) {
            setSetting($key, $value);
        }
        log_activity($current_user['id'], 'settings_updated', 'settings', 0, 'SMTP settings updated');
        set_flash('success', $lang['smtp_settings_saved'] ?? 'E-Mail-Einstellungen gespeichert.');
        header('Location: settings-lending.php?tab=email'); exit;
    }

    // ── Test-E-Mail senden (AJAX, JSON-Response) ─────────────────────────────
    if ($action === 'send_test_email') {
        header('Content-Type: application/json');

        $to = getSetting('library_email');
        if (empty($to)) {
            echo json_encode(['success' => false, 'message' => 'Keine Bibliotheks-E-Mail-Adresse konfiguriert (Einstellungen: Allgemein).']);
            exit;
        }

        $from_name  = getSetting('smtp_from_name',  getSetting('library_name', 'Bibliothek'));
        $from_email = getSetting('smtp_from_email', $to);
        $subject    = '[Test] Bibliotheksverwaltung E-Mail-Test';
        $body       = 'Dies ist eine Test-E-Mail der Bibliotheksverwaltung HLW Leoben.' . "\r\n\r\n"
                    . 'Versandzeitpunkt: ' . date('d.m.Y H:i:s') . "\r\n"
                    . 'System: ' . (getSetting('library_name') ?: 'Bibliotheksverwaltung');

        // PHPMailer bevorzugen, falls verfügbar
        if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            try {
                $mail = new PHPMailer\PHPMailer\PHPMailer(true);
                $mail->isSMTP();
                $mail->Host       = getSetting('smtp_host');
                $mail->Port       = (int)getSetting('smtp_port', 587);
                $mail->SMTPAuth   = true;
                $mail->Username   = getSetting('smtp_username');
                $mail->Password   = getSetting('smtp_password');
                $mail->SMTPSecure = ($mail->Port === 465) ? 'ssl' : 'tls';
                $mail->setFrom($from_email, $from_name);
                $mail->addAddress($to);
                $mail->Subject = $subject;
                $mail->Body    = $body;
                $mail->send();
                log_activity($current_user['id'], 'test_email_sent', 'settings', 0, "Test email sent to {$to}");
                echo json_encode(['success' => true, 'message' => "Test-E-Mail erfolgreich an {$to} gesendet (SMTP)."]);
            } catch (\Exception $e) {
                echo json_encode(['success' => false, 'message' => 'SMTP-Fehler: ' . $e->getMessage()]);
            }
            exit;
        }

        // Fallback: PHP mail()
        $headers  = "From: {$from_name} <{$from_email}>\r\n";
        $headers .= "Reply-To: {$from_email}\r\n";
        $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $headers .= "X-Mailer: BibliothekVerwaltung\r\n";

        if (@mail($to, $subject, $body, $headers)) {
            log_activity($current_user['id'], 'test_email_sent', 'settings', 0, "Test email sent to {$to}");
            echo json_encode(['success' => true, 'message' => "Test-E-Mail an {$to} übergeben (PHP mail()). Prüfen Sie Ihren Posteingang."]);
        } else {
            echo json_encode(['success' => false, 'message' => 'E-Mail-Versand fehlgeschlagen. PHP mail() konnte keine E-Mail übergeben. SMTP nicht konfiguriert oder mail() deaktiviert.']);
        }
        exit;
    }
}

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-8">
        <h1 class="text-2xl font-bold text-gray-900"><?php echo $lang['settings_lending_title'] ?? 'Einstellungen: Verleihverwaltung'; ?></h1>
        <p class="mt-1 text-sm text-gray-600"><?php echo $lang['settings_lending_desc'] ?? 'Ausleih-Parameter und E-Mail / SMTP-Konfiguration'; ?></p>
    </div>

    <!-- Flash-Meldung -->
    <?php if ($flash): ?>
        <?php $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue']; $c = $fc[$flash['type']] ?? 'blue'; ?>
        <div class="mb-6 flash-message bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo esc_html($flash['message']); ?></p>
        </div>
    <?php endif; ?>

    <!-- Tab-Navigation -->
    <div class="mb-6 border-b border-gray-200">
        <nav class="-mb-px flex space-x-8">
            <a href="#ausleihen" data-tab="ausleihen" class="lnd-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['tab_loans'] ?? 'Ausleihen'; ?></a>
            <a href="#email"     data-tab="email"     class="lnd-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['tab_email_smtp'] ?? 'E-Mail / SMTP'; ?></a>
        </nav>
    </div>

    <!-- ══════════════════════════════════════════════════════════════
         TAB: AUSLEIHEN
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-ausleihen" class="lnd-section hidden">

        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="p-6 border-b border-gray-100">
                <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['loan_settings'] ?? 'Ausleih-Einstellungen'; ?></h2>
                <p class="mt-1 text-sm text-gray-500">
                    <?php echo $lang['loan_settings_desc'] ?? 'Systemweite Standardwerte. Pro Benutzergruppe können abweichende Werte festgelegt werden.'; ?>
                </p>
            </div>
            <div class="p-6">
                <form method="POST" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="save_loans">

                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 items-start">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <?php echo $lang['default_loan_duration'] ?? 'Standard-Ausleihdauer (Tage)'; ?>
                            </label>
                            <input type="number" name="loan_duration_default" min="1" max="365" required
                                   value="<?php echo esc_attr($settings['loan_duration_default'] ?? 28); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <p class="mt-1 text-xs text-gray-500"><?php echo $lang['default_loan_duration_help'] ?? 'Fallback, wenn Gruppe keine eigene Dauer hat'; ?></p>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <?php echo $lang['max_extensions_default'] ?? 'Max. Verlängerungen (Standard)'; ?>
                            </label>
                            <input type="number" name="max_extensions" min="0" max="20" required
                                   value="<?php echo esc_attr($settings['max_extensions'] ?? 6); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <?php echo $lang['max_loans_default'] ?? 'Max. Ausleihen pro Benutzer (Standard)'; ?>
                            </label>
                            <input type="number" name="max_loans_per_user" min="1" max="50" required
                                   value="<?php echo esc_attr($settings['max_loans_per_user'] ?? 5); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <!-- Mahnungen -->
                    <div class="border-t pt-6">
                        <h3 class="text-base font-semibold text-gray-800 mb-4"><?php echo $lang['reminders'] ?? 'Mahnungen'; ?></h3>

                        <div class="mb-4 flex items-center">
                            <input type="checkbox" name="reminder_enabled" id="reminder_enabled" value="1"
                                   <?php echo ($settings['reminder_enabled'] ?? 'true') === 'true' ? 'checked' : ''; ?>
                                   class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                            <label for="reminder_enabled" class="ml-2 text-sm text-gray-700">
                                <?php echo $lang['reminders_enabled'] ?? 'Mahnungen aktiviert'; ?>
                            </label>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['reminder_after_days'] ?? 'Mahnung nach X Tagen Überfälligkeit'; ?>
                                </label>
                                <input type="number" name="reminder_days_after_due" min="1" max="90" required
                                       value="<?php echo esc_attr($settings['reminder_days_after_due'] ?? 7); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <p class="mt-1 text-xs text-gray-500"><?php echo $lang['reminder_after_days_help'] ?? 'Tage nach Ablauf der Leihfrist'; ?></p>
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <?php echo $lang['reminder_frequency'] ?? 'Wiederholungsfrequenz'; ?>
                                </label>
                                <select name="reminder_frequency"
                                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="daily"   <?php echo ($settings['reminder_frequency'] ?? 'weekly') === 'daily'   ? 'selected' : ''; ?>><?php echo $lang['frequency_daily'] ?? 'Täglich'; ?></option>
                                    <option value="weekly"  <?php echo ($settings['reminder_frequency'] ?? 'weekly') === 'weekly'  ? 'selected' : ''; ?>><?php echo $lang['frequency_weekly'] ?? 'Wöchentlich'; ?></option>
                                    <option value="monthly" <?php echo ($settings['reminder_frequency'] ?? 'weekly') === 'monthly' ? 'selected' : ''; ?>><?php echo $lang['frequency_monthly'] ?? 'Monatlich'; ?></option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div class="flex justify-end border-t pt-4">
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                            <?php echo $lang['save'] ?? 'Speichern'; ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div><!-- /tab-ausleihen -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: E-MAIL / SMTP
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-email" class="lnd-section hidden">

        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="p-6 border-b border-gray-100">
                <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['email_smtp'] ?? 'E-Mail / SMTP'; ?></h2>
                <p class="mt-1 text-sm text-gray-500">
                    <?php echo $lang['email_smtp_desc'] ?? 'Konfiguration für E-Mail-Versand (Mahnungen, Benachrichtigungen)'; ?>
                </p>
            </div>
            <div class="p-6">
                <form method="POST" id="smtpForm" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="save_smtp">

                    <div class="flex items-center">
                        <input type="checkbox" name="smtp_enabled" id="smtp_enabled" value="1"
                               <?php echo ($settings['smtp_enabled'] ?? 'false') === 'true' ? 'checked' : ''; ?>
                               class="h-4 w-4 text-blue-600 border-gray-300 rounded">
                        <label for="smtp_enabled" class="ml-2 text-sm font-medium text-gray-700">
                            <?php echo $lang['smtp_enable'] ?? 'SMTP aktivieren'; ?>
                        </label>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['smtp_host'] ?? 'SMTP-Server'; ?></label>
                            <input type="text" name="smtp_host"
                                   value="<?php echo esc_attr($settings['smtp_host'] ?? ''); ?>"
                                   placeholder="smtp.office365.com"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['smtp_port_label'] ?? 'Port'; ?></label>
                            <input type="number" name="smtp_port"
                                   value="<?php echo esc_attr($settings['smtp_port'] ?? 587); ?>"
                                   placeholder="587"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <p class="mt-1 text-xs text-gray-500"><?php echo $lang['smtp_port_help'] ?? '587 (STARTTLS) oder 465 (SSL)'; ?></p>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['smtp_username'] ?? 'Benutzername'; ?></label>
                            <input type="text" name="smtp_username"
                                   value="<?php echo esc_attr($settings['smtp_username'] ?? ''); ?>"
                                   placeholder="bibliothek@schule.at"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['smtp_password_label'] ?? 'Passwort'; ?></label>
                            <input type="password" name="smtp_password"
                                   placeholder="<?php echo esc_attr($lang['leave_empty_to_keep'] ?? 'Leer lassen um beizubehalten'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['sender_email'] ?? 'Absender-E-Mail'; ?></label>
                            <input type="email" name="smtp_from_email"
                                   value="<?php echo esc_attr($settings['smtp_from_email'] ?? ''); ?>"
                                   placeholder="bibliothek@schule.at"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['sender_name'] ?? 'Absender-Name'; ?></label>
                            <input type="text" name="smtp_from_name"
                                   value="<?php echo esc_attr($settings['smtp_from_name'] ?? ''); ?>"
                                   placeholder="Schulbibliothek"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <!-- Test-E-Mail Bereich -->
                    <div class="border-t pt-6">
                        <h3 class="text-base font-semibold text-gray-800 mb-2"><?php echo $lang['test_email_section'] ?? 'Test-E-Mail'; ?></h3>
                        <p class="text-sm text-gray-500 mb-3">
                            <?php echo $lang['test_email_desc'] ?? 'Sendet eine Test-E-Mail an die unter Einstellungen → Allgemein hinterlegte Bibliotheks-E-Mail-Adresse'; ?>
                            <?php $libMail = getSetting('library_email'); if ($libMail): ?>
                                (<strong><?php echo esc_html($libMail); ?></strong>)
                            <?php else: ?>
                                <span class="text-yellow-600"><?php echo $lang['not_configured_yet'] ?? '(noch nicht konfiguriert)'; ?></span>
                            <?php endif; ?>.
                        </p>
                        <div class="flex items-center gap-3">
                            <button type="button" id="testEmailBtn"
                                    class="px-5 py-2 border border-blue-600 text-blue-600 rounded-md hover:bg-blue-50 text-sm font-medium transition-colors">
                                <?php echo $lang['send_test_email'] ?? 'Test-E-Mail senden'; ?>
                            </button>
                            <span id="testEmailResult" class="hidden text-sm font-medium"></span>
                        </div>
                    </div>

                    <div class="flex justify-end border-t pt-4">
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                            <?php echo $lang['save'] ?? 'Speichern'; ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div><!-- /tab-email -->

</div>
</div>

<script>
(function () {
    function getActiveTab() {
        if (window.location.hash) return window.location.hash.substring(1);
        return new URLSearchParams(window.location.search).get('tab') || 'ausleihen';
    }

    function showTab(id) {
        document.querySelectorAll('.lnd-section').forEach(function (s) {
            s.classList.add('hidden');
        });
        document.querySelectorAll('.lnd-tab').forEach(function (t) {
            if (t.dataset.tab === id) {
                t.classList.add('border-blue-500', 'text-blue-600');
                t.classList.remove('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            } else {
                t.classList.remove('border-blue-500', 'text-blue-600');
                t.classList.add('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            }
        });
        var sec = document.getElementById('tab-' + id);
        if (sec) sec.classList.remove('hidden');
    }

    document.querySelectorAll('.lnd-tab').forEach(function (t) {
        t.addEventListener('click', function (e) {
            e.preventDefault();
            var id = this.dataset.tab;
            history.pushState(null, null, location.pathname + '#' + id);
            showTab(id);
        });
    });

    showTab(getActiveTab());

    window.addEventListener('hashchange', function () {
        showTab(window.location.hash.substring(1) || 'ausleihen');
    });
})();

// Test-E-Mail AJAX
document.getElementById('testEmailBtn').addEventListener('click', function () {
    var btn    = this;
    var result = document.getElementById('testEmailResult');
    var form   = document.getElementById('smtpForm');
    var origText = btn.textContent;

    btn.disabled    = true;
    btn.textContent = '<?php echo addslashes($lang['sending'] ?? 'Sende...'); ?>';
    result.className = 'hidden text-sm font-medium';

    var data = new FormData(form);
    data.set('action', 'send_test_email');

    fetch('settings-lending.php', { method: 'POST', body: data })
        .then(function (r) { return r.json(); })
        .then(function (json) {
            btn.disabled    = false;
            btn.textContent = origText;
            result.textContent = json.message;
            result.className   = 'text-sm font-medium ' + (json.success ? 'text-green-600' : 'text-red-600');
        })
        .catch(function () {
            btn.disabled    = false;
            btn.textContent = origText;
            result.textContent = '<?php echo addslashes($lang['network_error_test'] ?? 'Netzwerkfehler beim Test.'); ?>';
            result.className   = 'text-sm font-medium text-red-600';
        });
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
