<?php
/**
 * Einstellungen: Bestandsverwaltung
 * Tabs: Barcodes · ISBN-API · Kategorien · Littera-Import · Unterkategorien
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang         = load_language();
$page_title   = $lang['settings_inventory_title'] ?? 'Einstellungen: Bestandsverwaltung';
$current_user = get_logged_in_user();

// Load all settings
$settings = [];
$result = db_query("SELECT setting_key, setting_value FROM settings");
while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// ─── POST Handlers ──────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', 'Sicherheitsfehler. Bitte versuchen Sie es erneut.');
        header('Location: settings-inventory.php'); exit;
    }

    // ── Barcode-Einstellungen speichern ──────────────────────────────────────
    if ($action === 'save_barcode') {
        setSetting('barcode_format',       $_POST['barcode_format'] ?? 'code128');
        setSetting('barcode_label_format', $_POST['barcode_label_format'] ?? 'avery5160');
        log_activity($current_user['id'], 'settings_updated', 'settings', 0, 'Barcode settings updated');
        set_flash('success', 'Barcode-Einstellungen gespeichert.');
        header('Location: settings-inventory.php?tab=barcodes'); exit;
    }

    // ── ISBN-API-Einstellungen speichern ─────────────────────────────────────
    if ($action === 'save_isbn') {
        $validApis = ['dnb', 'google', 'isbnde'];
        $primary = in_array($_POST['isbn_api_primary'] ?? '', $validApis)
            ? $_POST['isbn_api_primary'] : 'dnb';
        $second  = in_array($_POST['isbn_api_second']  ?? '', $validApis)
            ? $_POST['isbn_api_second']  : 'google';
        if ($second === $primary) {
            $second = ($primary === 'dnb') ? 'google' : 'dnb';
        }
        setSetting('isbn_api_primary',  $primary);
        setSetting('isbn_api_second',   $second);
        setSetting('isbn_api_use_both', isset($_POST['isbn_api_use_both']) ? 'true' : 'false');
        setSetting('isbn_api_timeout',  (int)($_POST['isbn_api_timeout'] ?? 8));
        log_activity($current_user['id'], 'settings_updated', 'settings', 0, 'ISBN API settings updated');
        set_flash('success', 'ISBN-API-Einstellungen gespeichert.');
        header('Location: settings-inventory.php?tab=isbn'); exit;
    }

    // ── Kategorie hinzufügen ─────────────────────────────────────────────────
    if ($action === 'add_category') {
        $code = strtoupper(trim($_POST['code'] ?? ''));
        $name = trim($_POST['name'] ?? '');
        $sort = intval($_POST['sort_order'] ?? 0);
        if (!$code || !$name) {
            set_flash('error', 'Code und Name sind Pflichtfelder.');
        } elseif (db_get_var("SELECT COUNT(*) FROM book_categories WHERE code = ?", [$code])) {
            set_flash('error', "Code '{$code}' ist bereits vergeben.");
        } else {
            db_insert('book_categories', ['code' => $code, 'name' => $name, 'sort_order' => $sort, 'active' => 1]);
            set_flash('success', "Kategorie '{$name}' angelegt.");
        }
        header('Location: settings-inventory.php?tab=kategorien'); exit;
    }

    // ── Kategorie aktualisieren ──────────────────────────────────────────────
    if ($action === 'update_category') {
        $id     = intval($_POST['id'] ?? 0);
        $code   = strtoupper(trim($_POST['code'] ?? ''));
        $name   = trim($_POST['name'] ?? '');
        $sort   = intval($_POST['sort_order'] ?? 0);
        $active = isset($_POST['active']) ? 1 : 0;
        if (!$id || !$code || !$name) {
            set_flash('error', 'Ungültige Eingabe.');
        } elseif (db_get_var("SELECT COUNT(*) FROM book_categories WHERE code = ? AND id != ?", [$code, $id])) {
            set_flash('error', "Code '{$code}' ist bereits vergeben.");
        } else {
            db_update('book_categories', ['code' => $code, 'name' => $name, 'sort_order' => $sort, 'active' => $active], 'id = ?', [$id]);
            set_flash('success', 'Kategorie gespeichert.');
        }
        header('Location: settings-inventory.php?tab=kategorien'); exit;
    }

    // ── Kategorie löschen ────────────────────────────────────────────────────
    if ($action === 'delete_category') {
        $id  = intval($_POST['id'] ?? 0);
        $sub = (int)db_get_var("SELECT COUNT(*) FROM book_subcategories WHERE category_id = ?", [$id]);
        if ($sub > 0) {
            set_flash('error', "Kategorie kann nicht gelöscht werden: {$sub} Untergruppe(n) vorhanden.");
        } else {
            $name = db_get_var("SELECT name FROM book_categories WHERE id = ?", [$id]);
            db_delete('book_categories', 'id = ?', [$id]);
            set_flash('success', "Kategorie '{$name}' gelöscht.");
        }
        header('Location: settings-inventory.php?tab=kategorien'); exit;
    }

    // ── Untergruppe hinzufügen ───────────────────────────────────────────────
    if ($action === 'add_subcategory') {
        $cat_id = intval($_POST['category_id'] ?? 0);
        $code   = trim($_POST['code'] ?? '');
        $name   = trim($_POST['name'] ?? '');
        $sort   = intval($_POST['sort_order'] ?? 0);
        if (!$cat_id || !$code || !$name) {
            set_flash('error', 'Kategorie, Code und Name sind Pflichtfelder.');
        } elseif (db_get_var("SELECT COUNT(*) FROM book_subcategories WHERE code = ?", [$code])) {
            set_flash('error', "Code '{$code}' ist bereits vergeben.");
        } else {
            db_insert('book_subcategories', ['category_id' => $cat_id, 'code' => $code, 'name' => $name, 'sort_order' => $sort, 'active' => 1]);
            set_flash('success', "Untergruppe '{$name}' angelegt.");
        }
        header('Location: settings-inventory.php?tab=unterkategorien'); exit;
    }

    // ── Untergruppe aktualisieren ────────────────────────────────────────────
    if ($action === 'update_subcategory') {
        $id     = intval($_POST['id'] ?? 0);
        $cat_id = intval($_POST['category_id'] ?? 0);
        $code   = trim($_POST['code'] ?? '');
        $name   = trim($_POST['name'] ?? '');
        $sort   = intval($_POST['sort_order'] ?? 0);
        $active = isset($_POST['active']) ? 1 : 0;
        if (!$id || !$cat_id || !$code || !$name) {
            set_flash('error', 'Ungültige Eingabe.');
        } elseif (db_get_var("SELECT COUNT(*) FROM book_subcategories WHERE code = ? AND id != ?", [$code, $id])) {
            set_flash('error', "Code '{$code}' ist bereits vergeben.");
        } else {
            db_update('book_subcategories', ['category_id' => $cat_id, 'code' => $code, 'name' => $name, 'sort_order' => $sort, 'active' => $active], 'id = ?', [$id]);
            set_flash('success', 'Untergruppe gespeichert.');
        }
        header('Location: settings-inventory.php?tab=unterkategorien'); exit;
    }

    // ── Untergruppe löschen ──────────────────────────────────────────────────
    if ($action === 'delete_subcategory') {
        $id   = intval($_POST['id'] ?? 0);
        $used = (int)db_get_var("SELECT COUNT(*) FROM books WHERE subcategory_id = ?", [$id]);
        if ($used > 0) {
            set_flash('error', "Untergruppe kann nicht gelöscht werden: {$used} Buch/Bücher verwenden sie.");
        } else {
            $name = db_get_var("SELECT name FROM book_subcategories WHERE id = ?", [$id]);
            db_delete('book_subcategories', 'id = ?', [$id]);
            set_flash('success', "Untergruppe '{$name}' gelöscht.");
        }
        header('Location: settings-inventory.php?tab=unterkategorien'); exit;
    }
}

// ─── Daten laden ─────────────────────────────────────────────────────────────
$categories = db_get_results("
    SELECT id, code, name, sort_order, active
    FROM book_categories
    ORDER BY sort_order, name
");

$editCat = null;
if (isset($_GET['edit_cat'])) {
    $editCat = db_get_row("SELECT * FROM book_categories WHERE id = ?", [intval($_GET['edit_cat'])]);
}

$editSub = null;
if (isset($_GET['edit_sub'])) {
    $editSub = db_get_row("SELECT * FROM book_subcategories WHERE id = ?", [intval($_GET['edit_sub'])]);
}

$subcategories = db_get_results("
    SELECT s.id, s.category_id, s.code, s.name, s.sort_order, s.active, c.name AS cat_name
    FROM book_subcategories s
    JOIN book_categories c ON s.category_id = c.id
    ORDER BY c.sort_order, c.name, s.sort_order, s.name
");

// ISBN-Einstellungen
$apiPrimary = $settings['isbn_api_primary'] ?? 'dnb';
$apiSecond  = $settings['isbn_api_second']  ?? 'google';
$apiUseBoth = ($settings['isbn_api_use_both'] ?? 'true') === 'true';
$apiOptions = [
    'dnb'    => '🏛 DNB (Deutsche Nationalbibliothek)',
    'google' => '🔍 Google Books',
    'isbnde' => '📖 ISBN.de',
];

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-8">
        <h1 class="text-2xl font-bold text-gray-900"><?php echo $lang['settings_inventory_title'] ?? 'Einstellungen: Bestandsverwaltung'; ?></h1>
        <p class="mt-1 text-sm text-gray-600"><?php echo $lang['settings_inventory_desc'] ?? 'Barcodes, ISBN-API, Kategorien und Littera-Import'; ?></p>
    </div>

    <!-- Flash-Meldung -->
    <?php if ($flash): ?>
        <?php $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue']; $c = $fc[$flash['type']] ?? 'blue'; ?>
        <div class="mb-6 flash-message bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo esc_html($flash['message']); ?></p>
        </div>
    <?php endif; ?>

    <!-- Tab-Navigation -->
    <div class="mb-6 border-b border-gray-200">
        <nav class="-mb-px flex space-x-8 overflow-x-auto">
            <a href="#barcodes"        data-tab="barcodes"        class="inv-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['barcode_settings'] ?? 'Barcodes'; ?></a>
            <a href="#isbn"            data-tab="isbn"            class="inv-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium">ISBN-API</a>
            <a href="#kategorien"      data-tab="kategorien"      class="inv-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['admin_categories'] ?? 'Kategorien'; ?></a>
            <a href="#unterkategorien" data-tab="unterkategorien" class="inv-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['subcategories_tab'] ?? 'Unterkategorien'; ?></a>
            <a href="#littera"         data-tab="littera"         class="inv-tab whitespace-nowrap border-b-2 py-4 px-1 text-sm font-medium"><?php echo $lang['littera_settings_title'] ?? 'Littera-Import'; ?></a>
        </nav>
    </div>

    <!-- ══════════════════════════════════════════════════════════════
         TAB: BARCODES
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-barcodes" class="inv-section hidden">

        <!-- "in Vorbereitung" Hinweis -->
        <div class="mb-6 bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded">
            <div class="flex items-start gap-3">
                <svg class="h-5 w-5 text-yellow-500 mt-0.5 shrink-0" fill="currentColor" viewBox="0 0 20 20">
                    <path fill-rule="evenodd" d="M8.485 2.495c.673-1.167 2.357-1.167 3.03 0l6.28 10.875c.673 1.167-.17 2.625-1.516 2.625H3.72c-1.347 0-2.189-1.458-1.515-2.625L8.485 2.495zM10 5a.75.75 0 01.75.75v3.5a.75.75 0 01-1.5 0v-3.5A.75.75 0 0110 5zm0 9a1 1 0 100-2 1 1 0 000 2z" clip-rule="evenodd"/>
                </svg>
                <div>
                    <p class="text-sm font-semibold text-yellow-800"><?php echo $lang['barcode_print_in_prep'] ?? 'Barcode-Druck ist in Vorbereitung'; ?></p>
                    <p class="mt-1 text-sm text-yellow-700">
                        <?php echo $lang['barcode_print_not_impl'] ?? 'Das direkte Drucken von Barcode-Etiketten aus dem Browser ist noch nicht implementiert. Die folgenden Einstellungen werden gespeichert und beim Rollout des Barcode-Drucks verwendet.'; ?>
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="p-6 border-b border-gray-100">
                <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['barcode_settings'] ?? 'Barcode-Einstellungen'; ?></h2>
            </div>
            <div class="p-6">
                <form method="POST" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="save_barcode">

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            <?php echo $lang['barcode_format'] ?? 'Barcode-Format'; ?>
                        </label>
                        <select name="barcode_format"
                                class="w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="code128" <?php echo ($settings['barcode_format'] ?? 'code128') === 'code128' ? 'selected' : ''; ?>>Code 128</option>
                            <option value="ean13"   <?php echo ($settings['barcode_format'] ?? 'code128') === 'ean13'   ? 'selected' : ''; ?>>EAN-13</option>
                        </select>
                        <p class="mt-1 text-xs text-gray-500"><?php echo $lang['barcode_format_help'] ?? 'Code 128 empfohlen — unterstützt alle Buchungsnummern.'; ?></p>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            <?php echo $lang['label_format_label'] ?? 'Etiketten-Format'; ?>
                        </label>
                        <select name="barcode_label_format"
                                class="w-full max-w-xs px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="avery5160" <?php echo ($settings['barcode_label_format'] ?? 'avery5160') === 'avery5160' ? 'selected' : ''; ?>>Avery 5160 (2,625" × 1")</option>
                            <option value="avery5161" <?php echo ($settings['barcode_label_format'] ?? 'avery5160') === 'avery5161' ? 'selected' : ''; ?>>Avery 5161 (1" × 4")</option>
                        </select>
                    </div>

                    <div class="flex justify-end border-t pt-4">
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                            <?php echo $lang['save'] ?? 'Speichern'; ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div><!-- /tab-barcodes -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: ISBN-API
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-isbn" class="inv-section hidden">

        <div class="bg-white rounded-xl shadow-sm border border-gray-200">
            <div class="p-6 border-b border-gray-100">
                <h2 class="text-lg font-semibold text-gray-900">ISBN-API</h2>
                <p class="mt-1 text-sm text-gray-500"><?php echo $lang['isbn_api_desc'] ?? 'Automatischer Abruf von Buchdaten beim Erfassen neuer Bücher'; ?></p>
            </div>
            <div class="p-6 space-y-6">

                <!-- Info-Box Quellen -->
                <div class="bg-blue-50 border-l-4 border-blue-400 p-4 rounded">
                    <h3 class="text-sm font-semibold text-blue-800 mb-2"><?php echo $lang['isbn_api_sources'] ?? 'Verfügbare Quellen'; ?></h3>
                    <div class="text-sm text-blue-700 space-y-1.5">
                        <div><strong>🏛 DNB</strong> — Deutsche Nationalbibliothek, offizielle API, kostenlos &amp; ohne Registrierung.
                            Beste Abdeckung für deutsche &amp; österreichische Bücher. Kein Cover, kein Preis.</div>
                        <div><strong>🔍 Google Books</strong> — offizielle REST-API. Gut für internationale Bücher, liefert Cover-Bilder.</div>
                        <div><strong>📖 ISBN.de</strong> — deutschsprachiges Buchhandels-Portal. Liefert Cover-Bilder und Preise.</div>
                    </div>
                </div>

                <form method="POST" class="space-y-6">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <input type="hidden" name="action" value="save_isbn">

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <?php echo $lang['isbn_primary_api'] ?? 'Primäre API'; ?>
                                <span class="text-xs text-gray-400 font-normal ml-1"><?php echo $lang['isbn_primary_help'] ?? '(wird zuerst abgefragt)'; ?></span>
                            </label>
                            <select name="isbn_api_primary" id="isbn_primary" onchange="updateSecondOptions()"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <?php foreach ($apiOptions as $val => $label): ?>
                                    <option value="<?php echo $val; ?>" <?php echo $apiPrimary === $val ? 'selected' : ''; ?>>
                                        <?php echo $label; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <?php echo $lang['isbn_second_api'] ?? 'Zweite API'; ?>
                                <span class="text-xs text-gray-400 font-normal ml-1"><?php echo $lang['isbn_second_help'] ?? '(ergänzt fehlende Felder)'; ?></span>
                            </label>
                            <select name="isbn_api_second" id="isbn_second"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 <?php echo !$apiUseBoth ? 'opacity-50' : ''; ?>"
                                    <?php echo !$apiUseBoth ? 'disabled' : ''; ?>>
                                <?php foreach ($apiOptions as $val => $label): ?>
                                    <?php if ($val !== $apiPrimary): ?>
                                        <option value="<?php echo $val; ?>" <?php echo $apiSecond === $val ? 'selected' : ''; ?>>
                                            <?php echo $label; ?>
                                        </option>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <div class="flex items-start gap-3 p-4 bg-gray-50 rounded-lg border border-gray-200">
                        <input type="checkbox" name="isbn_api_use_both" id="isbn_api_use_both" value="1"
                               <?php echo $apiUseBoth ? 'checked' : ''; ?>
                               onchange="toggleSecond(this.checked)"
                               class="mt-0.5 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                        <div>
                            <label for="isbn_api_use_both" class="block text-sm font-medium text-gray-700 cursor-pointer">
                                <?php echo $lang['isbn_use_both'] ?? 'Beide APIs abfragen'; ?>
                            </label>
                            <p class="text-xs text-gray-500 mt-1">
                                <?php echo $lang['isbn_use_both_help'] ?? 'Empfohlen: Jede API liefert andere Felder. DNB = Metadaten & Schlagworte, Google/ISBN.de = Cover & Preis. Zusammen ergibt sich ein vollständigerer Datensatz.'; ?>
                            </p>
                        </div>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            <?php echo $lang['isbn_timeout_label'] ?? 'Timeout pro API (Sekunden)'; ?>
                        </label>
                        <input type="number" name="isbn_api_timeout" min="3" max="30"
                               value="<?php echo esc_attr($settings['isbn_api_timeout'] ?? 8); ?>"
                               class="w-32 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <p class="mt-1 text-xs text-gray-500"><?php echo $lang['isbn_timeout_help'] ?? 'Bei aktivierten beiden APIs gilt der Timeout jeweils pro Anfrage.'; ?></p>
                    </div>

                    <div class="flex justify-end border-t pt-4">
                        <button type="submit"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium">
                            <?php echo $lang['save'] ?? 'Speichern'; ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>

    </div><!-- /tab-isbn -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: KATEGORIEN
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-kategorien" class="inv-section hidden">

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 items-start">

            <!-- Formular (Neu / Bearbeiten) -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                <h3 class="font-semibold text-gray-800 mb-4">
                    <?php echo $editCat ? ($lang['category_edit'] ?? 'Kategorie bearbeiten') : ($lang['category_new'] ?? 'Neue Kategorie'); ?>
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <?php if ($editCat): ?>
                        <input type="hidden" name="action" value="update_category">
                        <input type="hidden" name="id"     value="<?php echo $editCat['id']; ?>">
                    <?php else: ?>
                        <input type="hidden" name="action" value="add_category">
                    <?php endif; ?>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Code <span class="text-red-500">*</span>
                            <span class="text-xs text-gray-400 font-normal">(z.B. DE)</span>
                        </label>
                        <input type="text" name="code" required maxlength="10"
                               value="<?php echo esc_attr($editCat['code'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm font-mono uppercase focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Name <span class="text-red-500">*</span>
                        </label>
                        <input type="text" name="name" required
                               value="<?php echo esc_attr($editCat['name'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['sort_order'] ?? 'Sortierung'; ?></label>
                        <input type="number" name="sort_order" min="0"
                               value="<?php echo esc_attr($editCat['sort_order'] ?? 0); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <?php if ($editCat): ?>
                        <div class="flex items-center gap-2">
                            <input type="checkbox" name="active" id="cat_active" value="1"
                                   <?php echo $editCat['active'] ? 'checked' : ''; ?>>
                            <label for="cat_active" class="text-sm text-gray-700"><?php echo $lang['active'] ?? 'Aktiv'; ?></label>
                        </div>
                    <?php endif; ?>

                    <div class="flex gap-2">
                        <button type="submit"
                                class="flex-1 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                            <?php echo $editCat ? ($lang['save'] ?? 'Speichern') : ($lang['category_create'] ?? 'Anlegen'); ?>
                        </button>
                        <?php if ($editCat): ?>
                            <a href="settings-inventory.php?tab=kategorien"
                               class="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-200 transition-colors">
                                <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>

            <!-- Tabelle -->
            <div class="lg:col-span-2 bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                <table class="w-full text-sm">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Sort</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aktiv</th>
                            <th class="px-4 py-3"></th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        <?php foreach ($categories as $cat): ?>
                            <tr class="<?php echo $editCat && (int)$editCat['id'] === (int)$cat['id'] ? 'bg-blue-50' : 'hover:bg-gray-50'; ?>">
                                <td class="px-4 py-3 font-mono font-bold text-gray-900"><?php echo esc_html($cat['code']); ?></td>
                                <td class="px-4 py-3 text-gray-700"><?php echo esc_html($cat['name']); ?></td>
                                <td class="px-4 py-3 text-gray-500"><?php echo (int)$cat['sort_order']; ?></td>
                                <td class="px-4 py-3">
                                    <span class="px-2 py-0.5 rounded text-xs font-medium <?php echo $cat['active'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-500'; ?>">
                                        <?php echo $cat['active'] ? 'ja' : 'nein'; ?>
                                    </span>
                                </td>
                                <td class="px-4 py-3 text-right whitespace-nowrap">
                                    <a href="settings-inventory.php?tab=kategorien&edit_cat=<?php echo $cat['id']; ?>"
                                       class="text-xs text-blue-600 hover:underline mr-3">Bearbeiten</a>
                                    <form method="POST" class="inline"
                                          onsubmit="return confirm('Kategorie wirklich löschen?')">
                                        <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                        <input type="hidden" name="action" value="delete_category">
                                        <input type="hidden" name="id"     value="<?php echo $cat['id']; ?>">
                                        <button type="submit" class="text-xs text-red-600 hover:underline">Löschen</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if (empty($categories)): ?>
                            <tr>
                                <td colspan="5" class="px-4 py-6 text-center text-gray-400 text-sm">
                                    Keine Kategorien vorhanden
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

        </div>

    </div><!-- /tab-kategorien -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: LITTERA-IMPORT
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-littera" class="inv-section hidden">

        <div class="max-w-2xl space-y-6">
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                <h2 class="text-lg font-semibold text-gray-900 mb-3">Littera-Import</h2>
                <p class="text-sm text-gray-600 mb-4">
                    Importiert Bücher und Exemplare aus dem Legacy-System <strong>Littera</strong>
                    (Microsoft Access / Jet-Datenbank). Der Import-Assistent führt Sie in mehreren
                    Schritten durch den Prozess — von der Prüfung der CSV-Quelldateien bis zur
                    Übernahme in die Datenbank.
                </p>

                <!-- Voraussetzungen -->
                <div class="mb-5 bg-blue-50 border border-blue-200 rounded-lg p-4 text-sm text-blue-800">
                    <p class="font-medium mb-1">Voraussetzungen</p>
                    <ul class="list-disc list-inside space-y-1 text-blue-700">
                        <li>CSV-Exporte der Littera-Datenbank (mdbtools empfohlen: <code class="font-mono text-xs bg-blue-100 px-1 rounded">brew install mdbtools</code>)</li>
                        <li>Dateien <code class="font-mono text-xs bg-blue-100 px-1 rounded">titel.csv</code>, <code class="font-mono text-xs bg-blue-100 px-1 rounded">exemplar.csv</code>, <code class="font-mono text-xs bg-blue-100 px-1 rounded">verlag.csv</code> u.a. in <code class="font-mono text-xs bg-blue-100 px-1 rounded">uploads/littera/csv_export/</code></li>
                        <li>Zeichenkodierung <strong>Windows-1252</strong> (nicht in UTF-8 konvertieren)</li>
                    </ul>
                </div>

                <!-- Verfügbarkeit der CSV-Dateien -->
                <p class="text-sm font-medium text-gray-700 mb-2">Benötigte Dateien:</p>
                <ul class="mb-5 space-y-1">
                    <?php foreach (['titel.csv', 'exemplar.csv', 'verlag.csv', 'sprache.csv', 'medienart.csv'] as $f):
                        $path   = dirname(__DIR__) . '/uploads/littera/csv_export/' . $f;
                        $exists = file_exists($path);
                    ?>
                        <li class="flex items-center gap-2 text-sm">
                            <span class="<?php echo $exists ? 'text-green-600' : 'text-gray-400'; ?> font-bold">
                                <?php echo $exists ? '✓' : '○'; ?>
                            </span>
                            <code class="font-mono text-xs bg-gray-100 px-1.5 py-0.5 rounded"><?php echo $f; ?></code>
                            <span class="text-xs <?php echo $exists ? 'text-green-600' : 'text-gray-400'; ?>">
                                <?php echo $exists ? 'vorhanden' : 'fehlt'; ?>
                            </span>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <a href="books-littera-import.php"
                   class="inline-block px-6 py-3 bg-blue-600 text-white font-semibold rounded-lg hover:bg-blue-700 transition-colors text-sm">
                    Zum Littera-Import-Assistenten
                </a>
            </div>
        </div>

    </div><!-- /tab-littera -->

    <!-- ══════════════════════════════════════════════════════════════
         TAB: UNTERKATEGORIEN
         ══════════════════════════════════════════════════════════════ -->
    <div id="tab-unterkategorien" class="inv-section hidden">

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 items-start">

            <!-- Formular (Neu / Bearbeiten) -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                <h3 class="font-semibold text-gray-800 mb-4">
                    <?php echo $editSub ? 'Untergruppe bearbeiten' : 'Neue Untergruppe'; ?>
                </h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                    <?php if ($editSub): ?>
                        <input type="hidden" name="action" value="update_subcategory">
                        <input type="hidden" name="id"     value="<?php echo $editSub['id']; ?>">
                    <?php else: ?>
                        <input type="hidden" name="action" value="add_subcategory">
                    <?php endif; ?>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Kategorie <span class="text-red-500">*</span>
                        </label>
                        <select name="category_id" required
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                            <option value="">— wählen —</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>"
                                        <?php echo $editSub && (int)$editSub['category_id'] === (int)$cat['id'] ? 'selected' : ''; ?>>
                                    <?php echo esc_html($cat['code'] . ' — ' . $cat['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Code (Sig1) <span class="text-red-500">*</span>
                        </label>
                        <input type="text" name="code" required maxlength="10"
                               value="<?php echo esc_attr($editSub['code'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm font-mono focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            Name <span class="text-red-500">*</span>
                        </label>
                        <input type="text" name="name" required
                               value="<?php echo esc_attr($editSub['name'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Sortierung</label>
                        <input type="number" name="sort_order" min="0"
                               value="<?php echo esc_attr($editSub['sort_order'] ?? 0); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <?php if ($editSub): ?>
                        <div class="flex items-center gap-2">
                            <input type="checkbox" name="active" id="sub_active" value="1"
                                   <?php echo $editSub['active'] ? 'checked' : ''; ?>>
                            <label for="sub_active" class="text-sm text-gray-700">Aktiv</label>
                        </div>
                    <?php endif; ?>

                    <div class="flex gap-2">
                        <button type="submit"
                                class="flex-1 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                            <?php echo $editSub ? 'Speichern' : 'Anlegen'; ?>
                        </button>
                        <?php if ($editSub): ?>
                            <a href="settings-inventory.php?tab=unterkategorien"
                               class="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-200 transition-colors">
                                Abbrechen
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>

            <!-- Tabelle + Filter -->
            <div class="lg:col-span-2">

                    <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                    <table class="w-full text-sm">
                        <thead class="bg-gray-50 border-b border-gray-200">
                            <tr>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Sig1</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Kategorie</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aktiv</th>
                                <th class="px-4 py-3"></th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            <?php foreach ($subcategories as $sub): ?>
                                <tr class="<?php echo $editSub && (int)$editSub['id'] === (int)$sub['id'] ? 'bg-blue-50' : 'hover:bg-gray-50'; ?>">
                                    <td class="px-4 py-3 font-mono font-bold text-gray-900"><?php echo esc_html($sub['code']); ?></td>
                                    <td class="px-4 py-3 text-gray-700"><?php echo esc_html($sub['name']); ?></td>
                                    <td class="px-4 py-3 text-gray-500 text-xs"><?php echo esc_html($sub['cat_name']); ?></td>
                                    <td class="px-4 py-3">
                                        <span class="px-2 py-0.5 rounded text-xs font-medium <?php echo $sub['active'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-500'; ?>">
                                            <?php echo $sub['active'] ? 'ja' : 'nein'; ?>
                                        </span>
                                    </td>
                                    <td class="px-4 py-3 text-right whitespace-nowrap">
                                        <a href="settings-inventory.php?tab=unterkategorien&edit_sub=<?php echo $sub['id']; ?>"
                                           class="text-xs text-blue-600 hover:underline mr-3">Bearbeiten</a>
                                        <form method="POST" class="inline"
                                              onsubmit="return confirm('Untergruppe wirklich löschen?')">
                                            <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                                            <input type="hidden" name="action" value="delete_subcategory">
                                            <input type="hidden" name="id"     value="<?php echo $sub['id']; ?>">
                                            <button type="submit" class="text-xs text-red-600 hover:underline">Löschen</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            <?php if (empty($subcategories)): ?>
                                <tr>
                                    <td colspan="5" class="px-4 py-6 text-center text-gray-400 text-sm">Keine Einträge</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>

            </div>

        </div>

    </div><!-- /tab-unterkategorien -->

</div>
</div>

<script>
(function () {
    function getActiveTab() {
        if (window.location.hash) return window.location.hash.substring(1);
        return new URLSearchParams(window.location.search).get('tab') || 'barcodes';
    }

    function showTab(id) {
        document.querySelectorAll('.inv-section').forEach(function (s) {
            s.classList.add('hidden');
        });
        document.querySelectorAll('.inv-tab').forEach(function (t) {
            if (t.dataset.tab === id) {
                t.classList.add('border-blue-500', 'text-blue-600');
                t.classList.remove('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            } else {
                t.classList.remove('border-blue-500', 'text-blue-600');
                t.classList.add('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
            }
        });
        var sec = document.getElementById('tab-' + id);
        if (sec) sec.classList.remove('hidden');
    }

    document.querySelectorAll('.inv-tab').forEach(function (t) {
        t.addEventListener('click', function (e) {
            e.preventDefault();
            var id = this.dataset.tab;
            history.pushState(null, null, location.pathname + '#' + id);
            showTab(id);
        });
    });

    showTab(getActiveTab());

    window.addEventListener('hashchange', function () {
        showTab(window.location.hash.substring(1) || 'barcodes');
    });
})();

// ISBN-API: zweite API-Dropdown dynamisch aktualisieren
function updateSecondOptions() {
    var primary = document.getElementById('isbn_primary').value;
    var second  = document.getElementById('isbn_second');
    var current = second.value;
    var options = {
        dnb:    '\uD83C\uDFDB\uFE0F DNB (Deutsche Nationalbibliothek)',
        google: '\uD83D\uDD0D Google Books',
        isbnde: '\uD83D\uDCD6 ISBN.de'
    };
    second.innerHTML = '';
    for (var val in options) {
        if (val === primary) continue;
        var opt = document.createElement('option');
        opt.value = val;
        opt.textContent = options[val];
        if (val === current) opt.selected = true;
        second.appendChild(opt);
    }
}

function toggleSecond(enabled) {
    var sel = document.getElementById('isbn_second');
    sel.disabled = !enabled;
    sel.classList.toggle('opacity-50', !enabled);
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
