<?php
/**
 * Einstellungen: Allgemein — settings-general.php
 * Tabs: Allgemein | Datenverwaltung | Datenbackup
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../update/Updater.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = $current_user['id'];

// ── Backup-Verzeichnis ────────────────────────────────────────────────────────
define('BACKUP_DIR', realpath(__DIR__ . '/..') . '/backups/');
if (!is_dir(BACKUP_DIR)) mkdir(BACKUP_DIR, 0750, true);
$htaccess = BACKUP_DIR . '.htaccess';
if (!file_exists($htaccess)) {
    file_put_contents($htaccess, "Require all denied\nDeny from all\n");
}

// ── Hilfsfunktionen Backup ────────────────────────────────────────────────────
function human_filesize(int $bytes): string {
    if ($bytes >= 1048576) return number_format($bytes / 1048576, 1, ',', '.') . ' MB';
    if ($bytes >= 1024)    return number_format($bytes / 1024,    1, ',', '.') . ' KB';
    return $bytes . ' B';
}
function safe_backup_filename(string $name): string { return basename($name); }

function backup_mysqldump(string $filepath, bool $compress): array {
    $candidates = ['/Applications/ServBay/bin/mysqldump', '/usr/local/bin/mysqldump', '/usr/bin/mysqldump', 'mysqldump'];
    $binary = null;
    foreach ($candidates as $c) { if (@is_executable($c)) { $binary = $c; break; } }
    if (!$binary) return ['success' => false, 'error' => 'mysqldump nicht gefunden'];

    $tmpconf = tempnam(sys_get_temp_dir(), 'bib_') . '.cnf';
    file_put_contents($tmpconf, "[client]\npassword=" . DB_PASS . "\n");
    chmod($tmpconf, 0600);
    $cmd = sprintf(
        '%s --defaults-extra-file=%s --no-tablespaces --triggers --single-transaction'
        . ' --add-drop-table --default-character-set=utf8mb4 -h %s -u %s %s',
        escapeshellarg($binary), escapeshellarg($tmpconf),
        escapeshellarg(DB_HOST), escapeshellarg(DB_USER), escapeshellarg(DB_NAME)
    );
    $desc = [0 => ['pipe','r'], 1 => ['pipe','w'], 2 => ['pipe','w']];
    $proc = proc_open($cmd, $desc, $pipes);
    fclose($pipes[0]);
    if ($compress) {
        $gz = gzopen($filepath, 'wb9');
        while (!feof($pipes[1])) gzwrite($gz, fread($pipes[1], 65536));
        gzclose($gz);
    } else {
        $out = fopen($filepath, 'wb');
        while (!feof($pipes[1])) fwrite($out, fread($pipes[1], 65536));
        fclose($out);
    }
    $stderr = stream_get_contents($pipes[2]);
    fclose($pipes[1]); fclose($pipes[2]);
    $exitcode = proc_close($proc);
    @unlink($tmpconf);
    if ($exitcode !== 0 || !file_exists($filepath) || filesize($filepath) < 100) {
        @unlink($filepath);
        return ['success' => false, 'error' => $stderr ?: 'Exit-Code: ' . $exitcode];
    }
    return ['success' => true, 'size' => filesize($filepath)];
}

function backup_pdo(string $filepath, bool $compress): array {
    global $pdo;
    $fh = $compress ? gzopen($filepath, 'wb9') : fopen($filepath, 'wb');
    if (!$fh) return ['success' => false, 'error' => 'Datei konnte nicht erstellt werden'];
    $w = fn(string $s) => $compress ? gzwrite($fh, $s) : fwrite($fh, $s);
    $w("-- Bibliothek SQL-Backup\n-- Datenbank: " . DB_NAME . "\n-- Erstellt: " . date('Y-m-d H:i:s') . "\n\n");
    $w("SET NAMES 'utf8mb4';\nSET FOREIGN_KEY_CHECKS = 0;\n\n");
    $tables = $pdo->query("SELECT table_name FROM information_schema.tables WHERE table_schema = DATABASE() AND table_type = 'BASE TABLE' ORDER BY table_name")->fetchAll(PDO::FETCH_COLUMN);
    foreach ($tables as $table) {
        $create = $pdo->query("SHOW CREATE TABLE `$table`")->fetch(PDO::FETCH_NUM);
        $w("DROP TABLE IF EXISTS `$table`;\n" . $create[1] . ";\n\n");
        $total = (int)$pdo->query("SELECT COUNT(*) FROM `$table`")->fetchColumn();
        if ($total > 0) {
            $chunk = 500; $offset = 0;
            $cols = array_map(fn($c) => "`{$c['Field']}`", $pdo->query("SHOW COLUMNS FROM `$table`")->fetchAll(PDO::FETCH_ASSOC));
            while ($offset < $total) {
                $rows = $pdo->query("SELECT * FROM `$table` LIMIT $chunk OFFSET $offset")->fetchAll(PDO::FETCH_NUM);
                if (empty($rows)) break;
                $vals = [];
                foreach ($rows as $row) {
                    $vals[] = '(' . implode(', ', array_map(fn($v) => $v === null ? 'NULL' : $pdo->quote($v), $row)) . ')';
                }
                $w("INSERT INTO `$table` (" . implode(', ', $cols) . ") VALUES\n" . implode(",\n", $vals) . ";\n");
                $offset += $chunk;
            }
            $w("\n");
        }
    }
    $views = $pdo->query("SELECT table_name FROM information_schema.tables WHERE table_schema = DATABASE() AND table_type = 'VIEW' ORDER BY table_name")->fetchAll(PDO::FETCH_COLUMN);
    foreach ($views as $view) {
        $create = $pdo->query("SHOW CREATE VIEW `$view`")->fetch(PDO::FETCH_NUM);
        $w("DROP VIEW IF EXISTS `$view`;\n" . $create[1] . ";\n\n");
    }
    $triggers = $pdo->query("SHOW TRIGGERS")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($triggers as $trigger) {
        $name = $trigger['Trigger'];
        $create = $pdo->query("SHOW CREATE TRIGGER `$name`")->fetch(PDO::FETCH_ASSOC);
        $w("DROP TRIGGER IF EXISTS `$name`;\nDELIMITER ;;\n" . $create['SQL Original Statement'] . ";;\nDELIMITER ;\n\n");
    }
    $w("SET FOREIGN_KEY_CHECKS = 1;\n");
    $compress ? gzclose($fh) : fclose($fh);
    if (!file_exists($filepath) || filesize($filepath) < 50) { @unlink($filepath); return ['success' => false, 'error' => 'Erzeugtes Backup zu klein']; }
    return ['success' => true, 'size' => filesize($filepath)];
}

function restore_sql_file(string $filepath): array {
    global $pdo;
    if (str_ends_with($filepath, '.gz')) {
        $fh = gzopen($filepath, 'rb'); $sql = '';
        while (!gzeof($fh)) $sql .= gzread($fh, 65536);
        gzclose($fh);
    } else { $sql = file_get_contents($filepath); }
    if ($sql === false) return ['success' => false, 'statements' => 0, 'errors' => ['Datei konnte nicht gelesen werden']];
    $sql = preg_replace('/DELIMITER\s+;;.*?DELIMITER\s+;/s', '', $sql);
    $stmts = []; $current = ''; $in_str = false; $str_chr = ''; $len = strlen($sql);
    for ($i = 0; $i < $len; $i++) {
        $c = $sql[$i]; $prev = $i > 0 ? $sql[$i-1] : '';
        if (!$in_str && ($c === "'" || $c === '"' || $c === '`')) { $in_str = true; $str_chr = $c; }
        elseif ($in_str && $c === $str_chr && $prev !== '\\') { $in_str = false; }
        if (!$in_str && $c === ';') {
            $s = trim($current);
            if ($s !== '' && !str_starts_with($s, '--') && !str_starts_with($s, '/*')) $stmts[] = $s;
            $current = '';
        } else { $current .= $c; }
    }
    $executed = 0; $errors = [];
    $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
    foreach ($stmts as $stmt) {
        if (preg_match('/^\s*(--|\/\*)/s', $stmt)) continue;
        try { $pdo->exec($stmt); $executed++; }
        catch (PDOException $e) {
            $errors[] = substr($stmt, 0, 60) . '… → ' . $e->getMessage();
            if (count($errors) >= 10) { $errors[] = '(weitere Fehler abgeschnitten)'; break; }
        }
    }
    $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
    return ['success' => count($errors) === 0, 'statements' => $executed, 'errors' => $errors];
}

// ── POST: Backup-Download (GET) ───────────────────────────────────────────────
if (($_GET['action'] ?? '') === 'download') {
    $file = safe_backup_filename($_GET['file'] ?? '');
    $path = BACKUP_DIR . $file;
    if ($file && is_file($path) && str_starts_with(realpath($path), realpath(BACKUP_DIR))) {
        $mime = str_ends_with($file, '.gz') ? 'application/gzip' : 'application/octet-stream';
        header('Content-Type: ' . $mime);
        header('Content-Disposition: attachment; filename="' . $file . '"');
        header('Content-Length: ' . filesize($path));
        readfile($path); exit;
    }
    set_flash('error', 'Backup-Datei nicht gefunden.');
    header('Location: settings-general.php?tab=datenbackup'); exit;
}

// ── POST-Handler ──────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', 'Sicherheitsfehler (CSRF). Bitte erneut versuchen.');
        header('Location: settings-general.php'); exit;
    }
    $action = $_POST['action'] ?? '';

    // ── Allgemein speichern ──
    if ($action === 'save_general') {
        $timeout_min = max(5, min(480, (int)($_POST['session_timeout_minutes'] ?? 60)));
        $updates = [
            'library_name'            => trim($_POST['library_name'] ?? ''),
            'library_email'           => trim($_POST['library_email'] ?? ''),
            'language'                => $_POST['language'] ?? 'de',
            'footer_text'             => trim($_POST['footer_text'] ?? ''),
            'session_timeout_minutes' => $timeout_min,
        ];
        foreach ($updates as $k => $v) setSetting($k, $v);
        // Sprachänderung sofort in Session übernehmen, damit sie ohne Neuanmeldung wirkt
        $_SESSION['language'] = $updates['language'];
        log_activity($admin_id, 'settings_updated', 'settings', 0, 'Allgemeine Einstellungen gespeichert');
        set_flash('success', 'Allgemeine Einstellungen gespeichert.');
        header('Location: settings-general.php?tab=allgemein'); exit;
    }

    // ── Datenverwaltung speichern ──
    if ($action === 'save_data') {
        $updates = [
            'activity_log_retention_years' => (int)($_POST['activity_log_retention_years'] ?? 5),
            'loan_history_archive_years'   => (int)($_POST['loan_history_archive_years'] ?? 5),
        ];
        foreach ($updates as $k => $v) setSetting($k, $v);
        log_activity($admin_id, 'settings_updated', 'settings', 0, 'Datenverwaltungs-Einstellungen gespeichert');
        set_flash('success', 'Datenverwaltungs-Einstellungen gespeichert.');
        header('Location: settings-general.php?tab=datenverwaltung'); exit;
    }

    // ── Backup erstellen ──
    if ($action === 'create_backup') {
        $compress = isset($_POST['compress']);
        $method   = ($_POST['method'] ?? 'auto') === 'pdo' ? 'pdo' : 'auto';
        $ext      = $compress ? '.sql.gz' : '.sql';
        $filename = 'backup_' . date('Y-m-d_H-i-s') . $ext;
        $filepath = BACKUP_DIR . $filename;
        if ($method === 'pdo') {
            $result = backup_pdo($filepath, $compress);
            $result['method'] = 'pdo';
        } else {
            $result = backup_mysqldump($filepath, $compress);
            if ($result['success']) { $result['method'] = 'mysqldump'; }
            else {
                $result = backup_pdo($filepath, $compress);
                $result['method'] = 'pdo';
            }
        }
        if ($result['success']) {
            log_activity($admin_id, 'backup_created', 'settings', 0,
                json_encode(['file' => $filename, 'size' => filesize($filepath), 'method' => $result['method'] ?? $method]));
            set_flash('success', 'Backup erstellt: <strong>' . esc_html($filename) . '</strong> (' . human_filesize(filesize($filepath)) . ')');
        } else {
            set_flash('error', 'Backup fehlgeschlagen: ' . esc_html($result['error']));
        }
        header('Location: settings-general.php?tab=datenbackup'); exit;
    }

    // ── Backup löschen (AJAX) ──
    if ($action === 'delete_backup') {
        header('Content-Type: application/json; charset=utf-8');
        $file = safe_backup_filename($_POST['file'] ?? '');
        $path = BACKUP_DIR . $file;
        if ($file && is_file($path) && str_starts_with(realpath($path), realpath(BACKUP_DIR))) {
            @unlink($path);
            log_activity($admin_id, 'backup_deleted', 'settings', 0, json_encode(['file' => $file]));
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['error' => 'Datei nicht gefunden']);
        }
        exit;
    }

    // ── Update per AJAX (gibt JSON zurück) ──
    if ($action === 'check_update_ajax') {
        header('Content-Type: application/json; charset=utf-8');
        global $pdo;
        try {
            $installed = Updater::getInstalledVersion($pdo);
            $available = Updater::getAvailableUpdate($installed);
            echo json_encode([
                'installed'  => $installed,
                'available'  => $available ? $available['version'] : null,
                'changelog'  => $available ? ($available['changelog'] ?? '') : null,
                'released'   => $available ? ($available['released'] ?? '') : null,
                'up_to_date' => $available === null,
            ]);
        } catch (Throwable $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
        exit;
    }

    // ── Update installieren (AJAX, gibt JSON zurück) ──
    if ($action === 'apply_update') {
        header('Content-Type: application/json; charset=utf-8');
        set_time_limit(300);
        ini_set('memory_limit', '256M');
        global $pdo;

        $target_version = trim($_POST['target_version'] ?? '');
        if (!preg_match('/^\d+\.\d+(\.\d+)?$/', $target_version)) {
            echo json_encode(['success' => false, 'steps' => [], 'error' => 'Ungültige Version']);
            exit;
        }

        try {
            $manifest = Updater::fetchManifest();
            if (!$manifest) {
                echo json_encode(['success' => false, 'steps' => [], 'error' => 'Manifest nicht abrufbar']);
                exit;
            }

            $entry = null;
            foreach ($manifest['versions'] as $v) {
                if (($v['version'] ?? '') === $target_version) { $entry = $v; break; }
            }
            if (!$entry) {
                echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version nicht gefunden"]);
                exit;
            }

            $installed = Updater::getInstalledVersion($pdo);
            if (version_compare($target_version, $installed, '<=')) {
                echo json_encode(['success' => false, 'steps' => [], 'error' => "Version $target_version ist nicht neuer als installierte Version $installed"]);
                exit;
            }

            $result = Updater::applyUpdate($entry, $pdo);
            echo json_encode($result);
        } catch (Throwable $e) {
            Updater::log('FEHLER: ' . $e->getMessage());
            echo json_encode(['success' => false, 'steps' => [], 'error' => $e->getMessage()]);
        }
        exit;
    }

    // ── Wiederherstellen ──
    if ($action === 'restore_backup') {
        if (!isset($_POST['confirm_restore']) || $_POST['confirm_restore'] !== 'JA') {
            set_flash('error', 'Bestätigung fehlt. Bitte "JA" eingeben.');
            header('Location: settings-general.php?tab=datenbackup'); exit;
        }
        if (empty($_FILES['backup_file']['tmp_name']) || $_FILES['backup_file']['error'] !== UPLOAD_ERR_OK) {
            set_flash('error', 'Upload fehlgeschlagen (Fehlercode: ' . ($_FILES['backup_file']['error'] ?? '?') . ')');
            header('Location: settings-general.php?tab=datenbackup'); exit;
        }
        $orig_name = $_FILES['backup_file']['name'];
        $tmp_path  = $_FILES['backup_file']['tmp_name'];
        if ($_FILES['backup_file']['size'] > 100 * 1024 * 1024) {
            set_flash('error', 'Datei zu groß (max. 100 MB).'); header('Location: settings-general.php?tab=datenbackup'); exit;
        }
        if (!preg_match('/\.(sql|sql\.gz)$/i', $orig_name)) {
            set_flash('error', 'Nur .sql und .sql.gz Dateien erlaubt.'); header('Location: settings-general.php?tab=datenbackup'); exit;
        }
        $result = restore_sql_file($tmp_path);
        log_activity($admin_id, 'backup_restored', 'settings', 0,
            json_encode(['file' => $orig_name, 'statements' => $result['statements'], 'errors' => count($result['errors'])]));
        if ($result['success']) {
            set_flash('success', "Wiederherstellung abgeschlossen: <strong>{$result['statements']}</strong> Statements.");
        } else {
            $err_list = implode('; ', array_slice($result['errors'], 0, 3));
            set_flash('error', "Wiederherstellung mit Fehlern ({$result['statements']} OK): " . esc_html($err_list));
        }
        header('Location: settings-general.php?tab=datenbackup'); exit;
    }
}

// ── Seiten-Daten ──────────────────────────────────────────────────────────────
$settings = [];
$result = db_query("SELECT setting_key, setting_value FROM settings");
while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

$backup_files = [];
if (is_dir(BACKUP_DIR)) {
    foreach (glob(BACKUP_DIR . 'backup_*.sql*') as $f) {
        if (!is_file($f)) continue;
        $backup_files[] = ['name' => basename($f), 'size' => filesize($f), 'mtime' => filemtime($f)];
    }
    usort($backup_files, fn($a, $b) => $b['mtime'] - $a['mtime']);
}
$last_backup = !empty($backup_files) ? date('d.m.Y H:i', $backup_files[0]['mtime']) : null;

$db_stats = db_get_row(
    "SELECT SUM(data_length + index_length) AS total_size, SUM(table_rows) AS total_rows, COUNT(*) AS table_count
     FROM information_schema.tables WHERE table_schema = DATABASE() AND table_type = 'BASE TABLE'"
);

$mysqldump_available = false;
foreach (['/Applications/ServBay/bin/mysqldump', '/usr/local/bin/mysqldump', '/usr/bin/mysqldump'] as $c) {
    if (@is_executable($c)) { $mysqldump_available = true; break; }
}

$lang = load_language();
$page_title = $lang['settings_general_title'] ?? 'Einstellungen: Allgemein';
$csrf_token = generate_csrf_token();
$flash      = get_flash();

include __DIR__ . '/includes/header.php';
?>

<div class="max-w-7xl mx-auto p-6">

  <div class="mb-6">
    <h1 class="text-2xl font-bold text-gray-900"><?php echo $lang['settings_general_title'] ?? 'Einstellungen: Allgemein'; ?></h1>
  </div>

  <?php if ($flash): ?>
    <?php $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue']; $c = $fc[$flash['type']] ?? 'blue'; ?>
    <div class="mb-6 bg-<?= $c ?>-50 border-l-4 border-<?= $c ?>-400 p-4 rounded">
      <p class="text-sm text-<?= $c ?>-700"><?= $flash['message'] ?></p>
    </div>
  <?php endif; ?>

  <!-- Tab-Navigation -->
  <div class="mb-6 border-b border-gray-200">
    <nav class="-mb-px flex space-x-6 overflow-x-auto">
      <a href="#allgemein"       class="tab-link py-4 px-1 text-sm font-medium whitespace-nowrap border-b-2"><?php echo $lang['tab_general'] ?? 'Allgemein'; ?></a>
      <a href="#datenverwaltung" class="tab-link py-4 px-1 text-sm font-medium whitespace-nowrap border-b-2"><?php echo $lang['tab_data_management'] ?? 'Datenverwaltung'; ?></a>
      <a href="#datenbackup"     class="tab-link py-4 px-1 text-sm font-medium whitespace-nowrap border-b-2"><?php echo $lang['tab_data_backup'] ?? 'Datenbackup'; ?></a>
      <a href="#updates"         class="tab-link py-4 px-1 text-sm font-medium whitespace-nowrap border-b-2">
        Updates
        <span id="bib-update-tab-badge" class="hidden ml-1 text-xs bg-red-100 text-red-700 px-1.5 py-0.5 rounded-full font-semibold">NEU</span>
      </a>
    </nav>
  </div>

  <!-- ════ TAB: ALLGEMEIN ════ -->
  <div id="allgemein" class="settings-section">
    <div class="bg-white rounded-lg shadow p-6">
      <h2 class="text-lg font-semibold text-gray-900 mb-5"><?php echo $lang['settings_general_subtitle'] ?? 'Allgemeine Einstellungen'; ?></h2>
      <form method="POST" class="space-y-5">
        <input type="hidden" name="csrf_token" value="<?= esc_attr($csrf_token) ?>">
        <input type="hidden" name="action" value="save_general">

        <div class="grid grid-cols-1 md:grid-cols-2 gap-5">
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['library_name'] ?? 'Bibliotheksname'; ?></label>
            <input type="text" name="library_name" required
                   value="<?= esc_attr($settings['library_name'] ?? '') ?>"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
          </div>
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['library_email'] ?? 'Bibliotheks-E-Mail'; ?></label>
            <input type="email" name="library_email" required
                   value="<?= esc_attr($settings['library_email'] ?? '') ?>"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
          </div>
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">
              <?php echo $lang['language'] ?? 'Sprache'; ?>
            </label>
            <select name="language" class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
              <option value="de" <?= ($settings['language'] ?? 'de') === 'de' ? 'selected' : '' ?>>Deutsch</option>
              <option value="en" <?= ($settings['language'] ?? 'de') === 'en' ? 'selected' : '' ?>>English</option>
            </select>
          </div>
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['footer_text'] ?? 'Footer-Text'; ?></label>
            <input type="text" name="footer_text"
                   value="<?= esc_attr($settings['footer_text'] ?? '') ?>"
                   placeholder="z.B. Schulbibliothek HLW Leoben | &copy; 2026"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
          </div>
        </div>

        <div class="border-t pt-5">
          <h3 class="text-sm font-semibold text-gray-700 mb-3"><?php echo $lang['session_section'] ?? 'Session'; ?></h3>
          <div class="flex items-start gap-6">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['session_timeout_minutes'] ?? 'Session-Timeout (Minuten)'; ?></label>
              <input type="number" name="session_timeout_minutes" min="5" max="480" required
                     value="<?= esc_attr($settings['session_timeout_minutes'] ?? 60) ?>"
                     class="w-28 px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
              <p class="mt-1 text-xs text-gray-500"><?php echo $lang['session_timeout_help'] ?? 'Gültige Werte: 5–480 Min.'; ?></p>
            </div>
            <div class="mt-6 text-xs text-gray-400 space-y-0.5">
              <div>30 Min. = kurz &nbsp;|&nbsp; <span class="font-medium text-gray-500">60 Min. = Standard</span> &nbsp;|&nbsp; 120 Min. = lang</div>
            </div>
          </div>
        </div>

        <div class="flex justify-end border-t pt-4">
          <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium"><?php echo $lang['save'] ?? 'Speichern'; ?></button>
        </div>
      </form>
    </div>
  </div>

  <!-- ════ TAB: DATENVERWALTUNG ════ -->
  <div id="datenverwaltung" class="settings-section hidden">
    <div class="bg-white rounded-lg shadow p-6">
      <h2 class="text-lg font-semibold text-gray-900 mb-1"><?php echo $lang['data_management'] ?? 'Datenverwaltung'; ?></h2>
      <p class="text-sm text-gray-500 mb-5"><?php echo $lang['data_management_desc'] ?? 'Aufbewahrungsfristen und Archivierung'; ?></p>
      <form method="POST" class="space-y-5">
        <input type="hidden" name="csrf_token" value="<?= esc_attr($csrf_token) ?>">
        <input type="hidden" name="action" value="save_data">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-5">
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['activity_log_retention'] ?? 'Activity-Log Aufbewahrung (Jahre)'; ?></label>
            <input type="number" name="activity_log_retention_years" min="1" max="20"
                   value="<?= esc_attr($settings['activity_log_retention_years'] ?? 5) ?>"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
            <p class="mt-1 text-xs text-gray-500"><?php echo $lang['activity_log_retention_help'] ?? 'Ältere Einträge werden automatisch gelöscht'; ?></p>
          </div>
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['loan_history_archive'] ?? 'Ausleihhistorie Archivierung (Jahre)'; ?></label>
            <input type="number" name="loan_history_archive_years" min="1" max="20"
                   value="<?= esc_attr($settings['loan_history_archive_years'] ?? 5) ?>"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
            <p class="mt-1 text-xs text-gray-500"><?php echo $lang['loan_history_archive_help'] ?? 'Ältere Einträge werden in Archiv-Tabelle verschoben'; ?></p>
          </div>
        </div>
        <div class="flex justify-end border-t pt-4">
          <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium"><?php echo $lang['save'] ?? 'Speichern'; ?></button>
        </div>
      </form>
    </div>
  </div>

  <!-- ════ TAB: DATENBACKUP ════ -->
  <div id="datenbackup" class="settings-section hidden">

    <!-- DB-Info-Kacheln -->
    <div class="grid grid-cols-3 gap-4 mb-6">
      <div class="bg-white rounded-lg shadow p-4 text-center">
        <p class="text-2xl font-bold text-blue-600"><?= human_filesize((int)($db_stats['total_size'] ?? 0)) ?></p>
        <p class="text-xs text-gray-500 mt-1"><?php echo $lang['db_size'] ?? 'Datenbankgröße'; ?></p>
      </div>
      <div class="bg-white rounded-lg shadow p-4 text-center">
        <p class="text-2xl font-bold text-gray-900"><?= number_format((int)($db_stats['table_count'] ?? 0)) ?></p>
        <p class="text-xs text-gray-500 mt-1"><?php echo $lang['tables'] ?? 'Tabellen'; ?></p>
      </div>
      <div class="bg-white rounded-lg shadow p-4 text-center">
        <p class="text-xl font-bold text-gray-900"><?= esc_html($last_backup ?? '—') ?></p>
        <p class="text-xs text-gray-500 mt-1"><?php echo $lang['last_backup'] ?? 'Letztes Backup'; ?></p>
      </div>
    </div>

    <!-- Backup erstellen -->
    <div class="bg-white rounded-lg shadow mb-6">
      <div class="p-5 border-b">
        <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['create_backup'] ?? 'Backup erstellen'; ?></h2>
        <p class="text-sm text-gray-500 mt-0.5"><?php echo $lang['create_backup_desc'] ?? 'SQL-Dump der kompletten Datenbank'; ?></p>
      </div>
      <form method="POST" class="p-5">
        <input type="hidden" name="action" value="create_backup">
        <?php echo csrf_field(); ?>
        <div class="grid grid-cols-1 sm:grid-cols-2 gap-5 mb-5">
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['method'] ?? 'Methode'; ?></label>
            <div class="space-y-2">
              <label class="flex items-start gap-3 cursor-pointer">
                <input type="radio" name="method" value="auto" checked class="mt-0.5 text-blue-600">
                <div>
                  <span class="text-sm font-medium text-gray-900"><?php echo $lang['method_auto'] ?? 'Automatisch'; ?>
                    <?php if ($mysqldump_available): ?>
                      <span class="ml-1 text-xs bg-green-100 text-green-700 px-1.5 py-0.5 rounded"><?php echo $lang['mysqldump_available'] ?? 'mysqldump verfügbar'; ?></span>
                    <?php else: ?>
                      <span class="ml-1 text-xs bg-yellow-100 text-yellow-700 px-1.5 py-0.5 rounded"><?php echo $lang['php_fallback'] ?? 'PHP-Fallback'; ?></span>
                    <?php endif; ?>
                  </span>
                  <p class="text-xs text-gray-500"><?php echo $lang['mysqldump_preferred'] ?? 'mysqldump bevorzugt, PHP-Export als Fallback'; ?></p>
                </div>
              </label>
              <label class="flex items-start gap-3 cursor-pointer">
                <input type="radio" name="method" value="pdo" class="mt-0.5 text-blue-600">
                <div>
                  <span class="text-sm font-medium text-gray-900"><?php echo $lang['php_export_pdo'] ?? 'PHP-Export (PDO)'; ?></span>
                  <p class="text-xs text-gray-500"><?php echo $lang['php_export_no_tool'] ?? 'Kein externes Tool benötigt'; ?></p>
                </div>
              </label>
            </div>
          </div>
          <div>
            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['options'] ?? 'Optionen'; ?></label>
            <label class="flex items-center gap-3 cursor-pointer">
              <input type="checkbox" name="compress" value="1" class="w-4 h-4 rounded border-gray-300 text-blue-600">
              <div>
                <span class="text-sm font-medium text-gray-900"><?php echo $lang['gzip_compression'] ?? 'gzip-Komprimierung'; ?></span>
                <p class="text-xs text-gray-500"><?php echo $lang['gzip_smaller_file'] ?? 'Kleinere Datei (.sql.gz)'; ?></p>
              </div>
            </label>
          </div>
        </div>
        <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium text-sm flex items-center gap-2">
          <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M9 19l3 3m0 0l3-3m-3 3V10"/>
          </svg>
          <?php echo $lang['create_and_save_backup'] ?? 'Backup erstellen & speichern'; ?>
        </button>
      </form>
    </div>

    <!-- Gespeicherte Backups -->
    <div class="bg-white rounded-lg shadow mb-6">
      <div class="p-5 border-b flex items-center justify-between">
        <h2 class="text-lg font-semibold text-gray-900"><?php echo $lang['saved_backups'] ?? 'Gespeicherte Backups'; ?> <span class="text-sm font-normal text-gray-500">(<?= count($backup_files) ?>)</span></h2>
        <p class="text-xs text-gray-400">Verzeichnis: <code class="bg-gray-100 px-1 rounded">backups/</code></p>
      </div>
      <?php if (empty($backup_files)): ?>
        <div class="p-8 text-center text-gray-500 text-sm"><?php echo $lang['no_backups_yet'] ?? 'Noch keine Backups vorhanden.'; ?></div>
      <?php else: ?>
      <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
          <thead class="bg-gray-50 border-b">
            <tr>
              <th class="px-4 py-3 text-left font-medium text-gray-700"><?php echo $lang['filename'] ?? 'Dateiname'; ?></th>
              <th class="px-4 py-3 text-center font-medium text-gray-700"><?php echo $lang['file_size'] ?? 'Größe'; ?></th>
              <th class="px-4 py-3 text-center font-medium text-gray-700"><?php echo $lang['created_col'] ?? 'Erstellt'; ?></th>
              <th class="px-4 py-3 text-center font-medium text-gray-700"><?php echo $lang['actions'] ?? 'Aktionen'; ?></th>
            </tr>
          </thead>
          <tbody class="divide-y divide-gray-100">
          <?php foreach ($backup_files as $i => $bf): ?>
            <tr class="hover:bg-gray-50 <?= $i === 0 ? 'bg-blue-50' : '' ?>" id="bf-<?= esc_attr($bf['name']) ?>">
              <td class="px-4 py-3">
                <span class="font-mono text-xs text-gray-700"><?= esc_html($bf['name']) ?></span>
                <?php if ($i === 0): ?><span class="ml-2 text-xs bg-blue-100 text-blue-700 px-1.5 py-0.5 rounded"><?php echo $lang['newest'] ?? 'neuestes'; ?></span><?php endif; ?>
                <?php if (str_ends_with($bf['name'], '.gz')): ?><span class="ml-1 text-xs bg-gray-100 text-gray-500 px-1.5 py-0.5 rounded">gzip</span><?php endif; ?>
              </td>
              <td class="px-4 py-3 text-center text-gray-600"><?= esc_html(human_filesize($bf['size'])) ?></td>
              <td class="px-4 py-3 text-center text-gray-600 whitespace-nowrap"><?= esc_html(date('d.m.Y H:i', $bf['mtime'])) ?></td>
              <td class="px-4 py-3 text-center">
                <div class="flex items-center justify-center gap-2">
                  <a href="?action=download&file=<?= urlencode($bf['name']) ?>"
                     class="px-3 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700"><?php echo $lang['download'] ?? 'Download'; ?></a>
                  <button onclick="deleteBackup('<?= esc_js($bf['name']) ?>')"
                          class="px-3 py-1 bg-red-100 text-red-700 text-xs rounded hover:bg-red-200"><?php echo $lang['delete'] ?? 'Löschen'; ?></button>
                </div>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <?php endif; ?>
    </div>

    <!-- Wiederherstellung -->
    <div class="bg-white rounded-lg shadow border border-red-200">
      <div class="p-5 border-b bg-red-50 flex items-center gap-3 cursor-pointer"
           onclick="document.getElementById('restorePanel').classList.toggle('hidden')">
        <svg class="w-5 h-5 text-red-600 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                d="M12 9v2m0 4h.01M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
        </svg>
        <h2 class="text-lg font-semibold text-red-800"><?php echo $lang['restore_database'] ?? 'Datenbank wiederherstellen'; ?></h2>
        <span class="ml-auto text-sm text-red-600">&#9660; <?php echo $lang['expand'] ?? 'aufklappen'; ?></span>
      </div>
      <div id="restorePanel" class="hidden p-5">
        <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-5 text-sm text-red-800">
          <?php echo $lang['restore_warning'] ?? '<strong>Achtung!</strong> Die Wiederherstellung überschreibt alle aktuellen Daten unwiderruflich. Erstellen Sie vorher unbedingt ein aktuelles Backup.'; ?>
        </div>
        <form method="POST" enctype="multipart/form-data" onsubmit="return confirmRestore(this)">
          <input type="hidden" name="action" value="restore_backup">
          <?php echo csrf_field(); ?>
          <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 mb-4">
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['backup_file_label'] ?? 'Backup-Datei'; ?> <span class="text-red-500">*</span></label>
              <input type="file" name="backup_file" required accept=".sql,.gz"
                     class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm">
            </div>
            <div>
              <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['confirm_restore_label'] ?? 'Bestätigung: Geben Sie JA ein'; ?> <span class="text-red-500">*</span></label>
              <input type="text" name="confirm_restore" required placeholder="JA"
                     class="w-full px-3 py-2 border border-red-300 rounded-md text-sm">
            </div>
          </div>
          <button type="submit" class="px-6 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 font-medium text-sm">
            <?php echo $lang['restore_database'] ?? 'Datenbank wiederherstellen'; ?>
          </button>
        </form>
      </div>
    </div>

  </div><!-- /datenbackup -->

  <!-- ════ TAB: UPDATES ════ -->
  <div id="updates" class="settings-section hidden">
    <div class="bg-white rounded-lg shadow p-6">
      <h2 class="text-lg font-semibold text-gray-900 mb-1">System-Updates</h2>
      <p class="text-sm text-gray-500 mb-5">
        Installierte Version: <strong id="bib-installed-ver"><?= esc_html(Updater::getInstalledVersion($pdo)) ?></strong>
      </p>

      <!-- Status-Panel -->
      <div id="bib-update-status" class="mb-6 p-4 rounded-lg border border-gray-200 bg-gray-50 text-sm text-gray-500">
        Prüfe auf Updates…
      </div>

      <!-- Update-Formular (wird bei verfügbarem Update sichtbar) -->
      <div id="bib-update-form-wrap" class="hidden">
        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-4 text-sm text-yellow-800">
          <strong>Hinweis:</strong> Das Update überschreibt alle Programmdateien.
          Folgende Daten bleiben unverändert: <code>includes/config.php</code>, <code>uploads/</code>, <code>backups/</code>, <code>vendor/</code>.
          Ein automatisches DB-Backup wird vor dem Update erstellt.
        </div>
        <input type="hidden" id="bib-target-version" value="">
        <button type="button" onclick="bibDoUpdate()"
                id="bib-update-btn"
                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium text-sm">
          Jetzt aktualisieren
        </button>
      </div>

      <!-- Fortschritts-Log -->
      <div id="bib-update-progress-wrap" class="hidden mt-4">
        <pre id="bib-progress-log" class="bg-gray-50 rounded p-4 text-xs overflow-y-auto max-h-48 whitespace-pre-wrap"></pre>
        <div id="bib-update-done" class="hidden mt-3 text-center">
          <button onclick="location.reload()" class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 font-medium text-sm">
            Seite neu laden
          </button>
        </div>
      </div>

      <!-- Update-Verlauf -->
      <div class="mt-8 border-t pt-4">
        <h3 class="text-sm font-semibold text-gray-700 mb-3">Update-Verlauf</h3>
        <?php $log_lines = Updater::readLog(20); ?>
        <?php if (empty($log_lines)): ?>
          <p class="text-xs text-gray-400">Noch keine Updates durchgeführt.</p>
        <?php else: ?>
          <pre class="bg-gray-50 rounded p-3 text-xs overflow-y-auto max-h-40 text-gray-600"><?= esc_html(implode("\n", $log_lines)) ?></pre>
        <?php endif; ?>
      </div>
    </div>
  </div><!-- /updates -->

</div>

<script>
const CSRF_TOKEN = '<?= esc_js($csrf_token) ?>';

// Tab-Switching
document.addEventListener('DOMContentLoaded', function() {
    const tabs     = document.querySelectorAll('.tab-link');
    const sections = document.querySelectorAll('.settings-section');
    function showSection(id) {
        sections.forEach(s => s.classList.add('hidden'));
        const target = document.getElementById(id);
        if (target) target.classList.remove('hidden');
        tabs.forEach(t => {
            const isActive = t.getAttribute('href') === '#' + id;
            t.classList.toggle('border-blue-500', isActive);
            t.classList.toggle('text-blue-600', isActive);
            t.classList.toggle('border-transparent', !isActive);
            t.classList.toggle('text-gray-500', !isActive);
        });
    }
    function getActiveId() {
        if (window.location.hash) return window.location.hash.substring(1);
        return new URLSearchParams(window.location.search).get('tab') || 'allgemein';
    }
    tabs.forEach(t => t.addEventListener('click', function(e) {
        e.preventDefault();
        const id = this.getAttribute('href').substring(1);
        history.pushState(null, null, '#' + id);
        showSection(id);
    }));
    showSection(getActiveId());
    window.addEventListener('hashchange', () => showSection(window.location.hash.substring(1)));
});

function deleteBackup(filename) {
    if (!confirm('Backup "' + filename + '" wirklich loschen?')) return;
    const fd = new FormData();
    fd.append('action', 'delete_backup');
    fd.append('csrf_token', CSRF_TOKEN);
    fd.append('file', filename);
    fetch('settings-general.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.error) { alert('Fehler: ' + data.error); return; }
            const row = document.getElementById('bf-' + filename);
            if (row) { row.style.opacity = '0'; row.style.transition = 'opacity 0.3s'; setTimeout(() => row.remove(), 350); }
        })
        .catch(err => alert('Netzwerkfehler: ' + err.message));
}

function confirmRestore(form) {
    if (form.querySelector('[name="confirm_restore"]').value !== 'JA') {
        alert('Bitte "JA" zur Bestatigung eingeben.'); return false;
    }
    return confirm('Letzter Schritt: Datenbank wirklich wiederherstellen? Alle aktuellen Daten werden uberschrieben!');
}

// ── Update-System ────────────────────────────────────────────────────────────
let bibUpdateData = null;
const BIB_CSRF = '<?= esc_js($csrf_token) ?>';

// Versionscheck beim Wechsel auf den Updates-Tab
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.tab-link').forEach(function(tab) {
        tab.addEventListener('click', function() {
            if (this.getAttribute('href') === '#updates' && !bibUpdateData) {
                bibCheckUpdate();
            }
        });
    });

    // Auch direkt laden wenn URL bereits #updates ist
    if (window.location.hash === '#updates') {
        bibCheckUpdate();
    }
});

function bibCheckUpdate() {
    const status = document.getElementById('bib-update-status');
    status.className = 'mb-6 p-4 rounded-lg border border-gray-200 bg-gray-50 text-sm text-gray-500';
    status.textContent = 'Prüfe auf Updates…';

    const fd = new FormData();
    fd.append('action', 'check_update_ajax');
    fd.append('csrf_token', BIB_CSRF);

    fetch('settings-general.php', { method: 'POST', body: fd })
        .then(r => r.json())
        .then(data => {
            bibUpdateData = data;
            const badge = document.getElementById('bib-update-tab-badge');

            if (data.error) {
                status.className = 'mb-6 p-4 rounded-lg border border-red-200 bg-red-50 text-sm text-red-700';
                status.textContent = 'Fehler: ' + data.error;
                return;
            }

            document.getElementById('bib-installed-ver').textContent = 'v' + data.installed;

            if (data.up_to_date) {
                status.className = 'mb-6 p-4 rounded-lg border border-green-200 bg-green-50 text-sm text-green-700';
                status.innerHTML = '✅ Das System ist aktuell <strong>(v' + data.installed + ')</strong>.';
            } else {
                status.className = 'mb-6 p-4 rounded-lg border border-blue-200 bg-blue-50 text-sm text-blue-800';
                status.innerHTML = '<strong>Update verfügbar: v' + data.available + '</strong>'
                    + (data.released ? ' <span class="text-blue-600">(vom ' + data.released + ')</span>' : '')
                    + (data.changelog ? '<br><br><em>' + bibEscape(data.changelog) + '</em>' : '');

                document.getElementById('bib-target-version').value = data.available;
                document.getElementById('bib-update-form-wrap').classList.remove('hidden');
                badge.classList.remove('hidden');
            }
        })
        .catch(err => {
            const status = document.getElementById('bib-update-status');
            status.className = 'mb-6 p-4 rounded-lg border border-red-200 bg-red-50 text-sm text-red-700';
            status.textContent = 'Update-Server nicht erreichbar: ' + err.message;
        });
}

function bibDoUpdate() {
    const btn       = document.getElementById('bib-update-btn');
    const formWrap  = document.getElementById('bib-update-form-wrap');
    const progress  = document.getElementById('bib-update-progress-wrap');
    const log       = document.getElementById('bib-progress-log');
    const doneDiv   = document.getElementById('bib-update-done');
    const version   = document.getElementById('bib-target-version').value;

    if (!confirm('System wirklich auf v' + version + ' aktualisieren?\n\nEin DB-Backup wird automatisch erstellt.')) return;

    btn.disabled    = true;
    btn.textContent = 'Aktualisiere…';
    formWrap.classList.add('hidden');
    progress.classList.remove('hidden');
    log.textContent = 'Starte Update auf v' + version + '…\n';

    const fd = new FormData();
    fd.append('action',         'apply_update');
    fd.append('csrf_token',     BIB_CSRF);
    fd.append('target_version', version);

    fetch('settings-general.php', { method: 'POST', body: fd })
        .then(r => r.json())
        .then(data => {
            if (data.steps && data.steps.length) {
                log.textContent += data.steps.join('\n') + '\n';
            }
            if (data.success) {
                log.textContent += '\n✅ Update erfolgreich!';
                doneDiv.classList.remove('hidden');
            } else {
                log.textContent += '\n❌ Fehler: ' + (data.error || 'Unbekannter Fehler');
                btn.disabled    = false;
                btn.textContent = 'Erneut versuchen';
                formWrap.classList.remove('hidden');
            }
        })
        .catch(err => {
            log.textContent += '\n❌ Netzwerkfehler: ' + err.message;
            btn.disabled    = false;
            btn.textContent = 'Erneut versuchen';
        });
}

function bibEscape(str) {
    return String(str).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
