<?php
/**
 * Ausleihstatistik — reports-lending.php
 * Kennzahlen, Monats-Trend (last 13 Monate), Top-Bücher,
 * Top-Nutzer, Überfälligkeits-Statistik.
 * Datenquellen: loans (aktiv) + loan_history (abgeschlossen).
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$page_title   = $lang['admin_lending_stats'] ?? 'Ausleihstatistik';

// ── Zeitraum-Filter ───────────────────────────────────────────────────────────
$period = $_GET['period'] ?? '12m';   // 12m | ytd | all
$allowed_periods = ['12m', 'ytd', 'all'];
if (!in_array($period, $allowed_periods, true)) $period = '12m';

$period_sql = match ($period) {
    'ytd'   => "AND loan_date >= MAKEDATE(YEAR(CURDATE()), 1)",
    'all'   => "",
    default => "AND loan_date >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)",
};
$period_label = match ($period) {
    'ytd'   => $lang['current_year'] ?? 'Laufendes Jahr',
    'all'   => $lang['total'] ?? 'Gesamt',
    default => $lang['last_12_months'] ?? 'Letzte 12 Monate',
};

// ── Hilfsfunktion: Loans + Loan_History kombiniert ────────────────────────────
// Gibt eine kombinierte Zahl zurück
function combined_count(string $extra_where = '', array $params = []): int {
    $sql = "SELECT COUNT(*) FROM (
                SELECT id FROM loans WHERE 1=1 $extra_where
                UNION ALL
                SELECT id FROM loan_history WHERE 1=1 $extra_where
            ) _c";
    return (int)db_get_var($sql, array_merge($params, $params));
}

// ── Statistik-Kacheln ─────────────────────────────────────────────────────────
$stat_total = combined_count($period_sql === '' ? '' : str_replace('AND loan_date', 'AND loan_date', $period_sql));
$stat_active    = (int)db_get_var("SELECT COUNT(*) FROM loans WHERE return_date IS NULL");
$stat_overdue   = (int)db_get_var("SELECT COUNT(*) FROM loans WHERE return_date IS NULL AND due_date < CURDATE()");
$stat_today     = (int)db_get_var(
    "SELECT COUNT(*) FROM (
        SELECT id FROM loans WHERE DATE(loan_date) = CURDATE()
        UNION ALL
        SELECT id FROM loan_history WHERE DATE(loan_date) = CURDATE()
    ) _c"
);
$stat_this_month = (int)db_get_var(
    "SELECT COUNT(*) FROM (
        SELECT id FROM loans
        WHERE YEAR(loan_date) = YEAR(CURDATE()) AND MONTH(loan_date) = MONTH(CURDATE())
        UNION ALL
        SELECT id FROM loan_history
        WHERE YEAR(loan_date) = YEAR(CURDATE()) AND MONTH(loan_date) = MONTH(CURDATE())
    ) _c"
);
$avg_days = (int)db_get_var(
    "SELECT ROUND(AVG(DATEDIFF(return_date, loan_date)))
     FROM loan_history WHERE return_date IS NOT NULL $period_sql"
);
$overdue_rate = $stat_total > 0
    ? round((int)db_get_var("SELECT COUNT(*) FROM loan_history WHERE was_overdue = 1 $period_sql") / $stat_total * 100, 1)
    : 0;

// ── Monats-Trend (last 13 Monate) ────────────────────────────────────────────
$monthly = db_get_results(
    "SELECT ym,
            DATE_FORMAT(STR_TO_DATE(CONCAT(ym, '-01'), '%Y-%m-%d'), '%b %y') AS label,
            SUM(cnt) AS cnt
     FROM (
         SELECT DATE_FORMAT(loan_date, '%Y-%m') AS ym, COUNT(*) AS cnt
         FROM loans
         WHERE loan_date >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
         GROUP BY ym
         UNION ALL
         SELECT DATE_FORMAT(loan_date, '%Y-%m') AS ym, COUNT(*) AS cnt
         FROM loan_history
         WHERE loan_date >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
         GROUP BY ym
     ) combined
     GROUP BY ym
     ORDER BY ym"
);
$max_monthly = max(array_column($monthly, 'cnt') ?: [1]);

// ── Top 10 Bücher im Zeitraum ─────────────────────────────────────────────────
$top_books = db_get_results(
    "SELECT b.id, b.title, b.author, b.image_url, COUNT(*) AS cnt
     FROM (
         SELECT book_id FROM loan_history
         WHERE book_id IS NOT NULL $period_sql
         UNION ALL
         SELECT bc.book_id FROM loans l JOIN book_copies bc ON l.copy_id = bc.id
         WHERE 1=1 $period_sql
     ) combined
     JOIN books b ON combined.book_id = b.id
     GROUP BY b.id, b.title, b.author, b.image_url
     ORDER BY cnt DESC
     LIMIT 10"
);
$max_top = max(array_column($top_books, 'cnt') ?: [1]);

// ── Top 10 Nutzer ─────────────────────────────────────────────────────────────
$top_users = db_get_results(
    "SELECT user_name, SUM(cnt) AS cnt FROM (
         SELECT user_name, COUNT(*) AS cnt FROM loan_history
         WHERE 1=1 $period_sql AND user_name IS NOT NULL GROUP BY user_name
         UNION ALL
         SELECT CONCAT(u.first_name, ' ', u.last_name) AS user_name, COUNT(*) AS cnt
         FROM loans l JOIN users u ON l.user_id = u.id
         WHERE 1=1 $period_sql GROUP BY user_name
     ) combined
     GROUP BY user_name
     ORDER BY cnt DESC
     LIMIT 10"
);
$max_users = max(array_column($top_users, 'cnt') ?: [1]);

// ── Aktive Ausleihen nach Klasse ──────────────────────────────────────────────
$by_class = db_get_results(
    "SELECT COALESCE(u.class_name, '—') AS class_name, COUNT(*) AS cnt
     FROM loans l
     LEFT JOIN users u ON l.user_id = u.id
     WHERE l.return_date IS NULL
     GROUP BY class_name
     ORDER BY cnt DESC
     LIMIT 8"
);

include __DIR__ . '/includes/header.php';

$flash = get_flash();
?>

<?php if ($flash): ?>
    <?php $fc = $flash['type'] === 'success' ? 'green' : 'red'; ?>
    <div class="mb-4 p-4 rounded-lg bg-<?= $fc ?>-50 border border-<?= $fc ?>-200 text-<?= $fc ?>-800">
        <?= esc_html($flash['message']) ?>
    </div>
<?php endif; ?>

<!-- Zeitraum-Filter -->
<div class="flex items-center justify-between mb-6">
    <h2 class="text-sm text-gray-500"><?= $lang['period'] ?? 'Zeitraum' ?>: <strong class="text-gray-700"><?= $period_label ?></strong></h2>
    <div class="flex gap-2">
        <?php foreach ([
            '12m' => $lang['last_12_months'] ?? '12 Monate',
            'ytd' => $lang['current_year'] ?? 'Laufendes Jahr',
            'all' => $lang['total'] ?? 'Gesamt'
        ] as $p => $l): ?>
            <a href="?period=<?= $p ?>"
               class="px-3 py-1.5 text-sm rounded-lg border <?= $period === $p ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-600 border-gray-300 hover:bg-gray-50' ?>">
                <?= $l ?>
            </a>
        <?php endforeach; ?>
    </div>
</div>

<!-- ── Statistik-Kacheln ──────────────────────────────────────────────────── -->
<div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
    <?php
    $tiles = [
        ['label' => $lang['loans_total'] ?? 'Ausleihen gesamt',       'val' => number_format($stat_total,      0,',','.'), 'sub' => $period_label,  'color' => 'blue'],
        ['label' => $lang['loaned_today'] ?? 'Heute ausgeliehen',      'val' => number_format($stat_today,      0,',','.'), 'sub' => date('d.m.Y'),  'color' => 'indigo'],
        ['label' => $lang['this_month'] ?? 'Diesen Monat',             'val' => number_format($stat_this_month, 0,',','.'), 'sub' => date('m/Y'),    'color' => 'purple'],
        ['label' => $lang['currently_active'] ?? 'Derzeit aktiv',      'val' => number_format($stat_active,     0,',','.'), 'sub' => $lang['loaned'] ?? 'ausgeliehen',  'color' => 'green'],
        ['label' => $lang['currently_overdue'] ?? 'Derzeit überfällig','val' => number_format($stat_overdue,    0,',','.'), 'sub' => $stat_active > 0 ? round($stat_overdue/$stat_active*100).'%' : '0%', 'color' => $stat_overdue > 0 ? 'red' : 'green'],
        ['label' => $lang['avg_loan_duration'] ?? 'Ø Ausleihdauer',    'val' => $avg_days . ' ' . ($lang['days'] ?? 'Tage'), 'sub' => $lang['returned'] ?? 'zurückgegebene', 'color' => 'orange'],
        ['label' => $lang['overdue_rate'] ?? 'Überfälligkeitsrate',    'val' => $overdue_rate . '%',                          'sub' => $lang['of_all_loans'] ?? 'aller Ausleihen', 'color' => $overdue_rate > 10 ? 'red' : 'gray'],
    ];
    foreach ($tiles as $t): ?>
        <div class="bg-white rounded-lg shadow p-5">
            <p class="text-xs text-gray-500 mb-1"><?= $t['label'] ?></p>
            <p class="text-2xl font-bold text-<?= $t['color'] ?>-600"><?= $t['val'] ?></p>
            <p class="text-xs text-gray-400 mt-0.5"><?= $t['sub'] ?></p>
        </div>
    <?php endforeach; ?>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">

    <!-- ── Monats-Trend ────────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-5"><?= $lang['loans_per_month'] ?? 'Ausleihen pro Monat' ?></h2>
        <?php if (empty($monthly)): ?>
            <p class="text-sm text-gray-400 text-center py-8"><?= $lang['no_data'] ?? 'Keine Daten vorhanden.' ?></p>
        <?php else: ?>
            <div class="space-y-2">
                <?php foreach ($monthly as $m):
                    $pct = $max_monthly > 0 ? round($m['cnt'] / $max_monthly * 100) : 0;
                    $is_current = $m['ym'] === date('Y-m');
                ?>
                    <div class="flex items-center gap-3">
                        <span class="text-xs text-gray-500 w-14 text-right flex-shrink-0 <?= $is_current ? 'font-bold text-blue-600' : '' ?>">
                            <?= esc_html($m['label']) ?>
                        </span>
                        <div class="flex-1 bg-gray-100 rounded-full h-6 relative overflow-hidden">
                            <div class="<?= $is_current ? 'bg-blue-500' : 'bg-blue-300' ?> h-6 rounded-full transition-all"
                                 style="width:<?= $pct ?>%"></div>
                            <span class="absolute inset-0 flex items-center pl-2 text-xs font-medium <?= $pct > 30 ? 'text-white' : 'text-gray-700' ?>">
                                <?= number_format($m['cnt'], 0, ',', '.') ?>
                            </span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- ── Aktive Ausleihen nach Klasse ────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-1"><?= $lang['active_loans_by_class'] ?? 'Aktive Ausleihen nach Klasse' ?></h2>
        <p class="text-xs text-gray-400 mb-4"><?= $lang['only_current_loans'] ?? 'Nur derzeit laufende Ausleihen' ?></p>
        <?php if (empty($by_class)): ?>
            <p class="text-sm text-gray-400 text-center py-8"><?= $lang['no_active_loans'] ?? 'Keine aktiven Ausleihen.' ?></p>
        <?php else:
            $max_class = max(array_column($by_class, 'cnt'));
        ?>
            <div class="space-y-2">
                <?php foreach ($by_class as $cl):
                    $pct = $max_class > 0 ? round($cl['cnt'] / $max_class * 100) : 0; ?>
                    <div class="flex items-center gap-3">
                        <span class="text-xs text-gray-600 w-16 text-right flex-shrink-0 font-mono">
                            <?= esc_html($cl['class_name']) ?>
                        </span>
                        <div class="flex-1 bg-gray-100 rounded-full h-6 relative overflow-hidden">
                            <div class="bg-indigo-400 h-6 rounded-full" style="width:<?= $pct ?>%"></div>
                            <span class="absolute inset-0 flex items-center pl-2 text-xs font-medium <?= $pct > 30 ? 'text-white' : 'text-gray-700' ?>">
                                <?= $cl['cnt'] ?>
                            </span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">

    <!-- ── Top Bücher ──────────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900"><?= $lang['top_10_books'] ?? 'Top 10 Bücher' ?></h2>
            <p class="text-xs text-gray-400"><?= $period_label ?></p>
        </div>
        <?php if (empty($top_books)): ?>
            <p class="p-5 text-sm text-gray-400 text-center"><?= $lang['no_loan_data'] ?? 'Keine Ausleih-Daten vorhanden.' ?></p>
        <?php else: ?>
            <div class="divide-y divide-gray-100">
                <?php foreach ($top_books as $i => $bk):
                    $pct = $max_top > 0 ? round($bk['cnt'] / $max_top * 100) : 0; ?>
                    <div class="p-4 flex items-center gap-3">
                        <span class="text-lg font-bold text-gray-300 w-7 text-right flex-shrink-0"><?= $i + 1 ?></span>
                        <?php if ($bk['image_url']): ?>
                            <img src="<?= esc_url($bk['image_url']) ?>" alt=""
                                 class="w-8 h-10 object-cover rounded flex-shrink-0">
                        <?php else: ?>
                            <div class="w-8 h-10 bg-gray-100 rounded flex-shrink-0"></div>
                        <?php endif; ?>
                        <div class="flex-1 min-w-0">
                            <a href="<?= BASE_URL ?>/admin/books-edit.php?id=<?= (int)$bk['id'] ?>"
                               class="text-sm font-medium text-gray-900 hover:text-blue-600 line-clamp-1">
                                <?= esc_html($bk['title']) ?>
                            </a>
                            <div class="mt-1 bg-gray-100 rounded-full h-2">
                                <div class="bg-blue-400 h-2 rounded-full" style="width:<?= $pct ?>%"></div>
                            </div>
                        </div>
                        <span class="text-sm font-bold text-blue-600 flex-shrink-0"><?= $bk['cnt'] ?>×</span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- ── Top Nutzer ──────────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900"><?= $lang['top_10_users'] ?? 'Top 10 Nutzer' ?></h2>
            <p class="text-xs text-gray-400"><?= $period_label ?></p>
        </div>
        <?php if (empty($top_users)): ?>
            <p class="p-5 text-sm text-gray-400 text-center"><?= $lang['no_data'] ?? 'Keine Daten vorhanden.' ?></p>
        <?php else: ?>
            <div class="divide-y divide-gray-100">
                <?php foreach ($top_users as $i => $u):
                    $pct = $max_users > 0 ? round($u['cnt'] / $max_users * 100) : 0; ?>
                    <div class="p-4 flex items-center gap-3">
                        <span class="text-lg font-bold text-gray-300 w-7 text-right flex-shrink-0"><?= $i + 1 ?></span>
                        <div class="w-8 h-8 rounded-full bg-indigo-100 flex items-center justify-center flex-shrink-0">
                            <span class="text-xs font-bold text-indigo-600">
                                <?= esc_html(mb_substr($u['user_name'], 0, 1)) ?>
                            </span>
                        </div>
                        <div class="flex-1 min-w-0">
                            <p class="text-sm font-medium text-gray-900 truncate"><?= esc_html($u['user_name']) ?></p>
                            <div class="mt-1 bg-gray-100 rounded-full h-2">
                                <div class="bg-indigo-400 h-2 rounded-full" style="width:<?= $pct ?>%"></div>
                            </div>
                        </div>
                        <span class="text-sm font-bold text-indigo-600 flex-shrink-0"><?= $u['cnt'] ?>×</span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
