<?php
/**
 * Bestandsübersicht — reports-inventory.php
 * Vollständige Bestandsanalyse: Bücher nach Sprache, Kategorie,
 * Exemplar-Verfügbarkeit, Zugangstimeline, Problemfälle.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$page_title   = $lang['admin_inventory_overview'] ?? 'Bestandsübersicht';

// ── Gesamt-Kennzahlen ─────────────────────────────────────────────────────────
$total_books   = (int)db_get_var("SELECT COUNT(*) FROM books");
$total_copies  = (int)db_get_var("SELECT COUNT(*) FROM book_copies WHERE status != 'aussortiert'");
$avail_copies  = (int)db_get_var("SELECT COUNT(*) FROM book_copies WHERE status = 'verfügbar'");
$loaned_copies = (int)db_get_var("SELECT COUNT(*) FROM book_copies WHERE status = 'ausgeliehen'");
$lost_copies   = (int)db_get_var("SELECT COUNT(*) FROM book_copies WHERE status = 'verloren'");

$books_no_copies = (int)db_get_var(
    "SELECT COUNT(*) FROM books b
     WHERE NOT EXISTS (
         SELECT 1 FROM book_copies bc
         WHERE bc.book_id = b.id AND bc.status NOT IN ('aussortiert','verloren')
     )"
);
$books_fully_loaned = (int)db_get_var(
    "SELECT COUNT(DISTINCT b.id) FROM books b
     JOIN book_copies bc ON bc.book_id = b.id
     WHERE bc.status NOT IN ('aussortiert','verloren')
       AND NOT EXISTS (
           SELECT 1 FROM book_copies bc2
           WHERE bc2.book_id = b.id AND bc2.status = 'verfügbar'
       )"
);
$avg_copies_per_book = $total_books > 0
    ? round($total_copies / $total_books, 1)
    : 0;

// ── Sprach-Verteilung ─────────────────────────────────────────────────────────
$by_language = db_get_results(
    "SELECT COALESCE(language, 'unbekannt') AS language, COUNT(*) AS cnt
     FROM books
     GROUP BY language
     ORDER BY cnt DESC
     LIMIT 12"
);
$max_lang = max(array_column($by_language, 'cnt') ?: [1]);

// ── Kategorie-Auswertung (Top 15) ────────────────────────────────────────────
$by_cat = db_get_results(
    "SELECT bc.name, bc.code,
            COUNT(DISTINCT b.id)                                AS books,
            COUNT(bcp.id)                                       AS copies,
            COALESCE(SUM(bcp.status = 'verfügbar'),    0)       AS avail,
            COALESCE(SUM(bcp.status = 'ausgeliehen'),  0)       AS loaned
     FROM book_categories bc
     LEFT JOIN book_subcategories bs  ON bs.category_id  = bc.id
     LEFT JOIN books b                ON b.subcategory_id = bs.id
     LEFT JOIN book_copies bcp        ON bcp.book_id = b.id
                                     AND bcp.status NOT IN ('aussortiert','verloren')
     GROUP BY bc.id, bc.name, bc.code
     HAVING books > 0
     ORDER BY books DESC
     LIMIT 15"
);
$max_cat = max(array_column($by_cat, 'books') ?: [1]);

// ── Zugangstimeline (letzten 10 Jahre, nach Jahr) ─────────────────────────────
$by_year = db_get_results(
    "SELECT yr, SUM(cnt) AS cnt FROM (
         SELECT YEAR(acquired_at) AS yr, COUNT(*) AS cnt
         FROM book_copies WHERE acquired_at IS NOT NULL
         GROUP BY yr
         UNION ALL
         SELECT YEAR(acquisition_date) AS yr, COUNT(*) AS cnt
         FROM book_copies WHERE acquisition_date IS NOT NULL AND acquired_at IS NULL
         GROUP BY yr
     ) t
     WHERE yr IS NOT NULL AND yr >= YEAR(CURDATE()) - 9
     GROUP BY yr
     ORDER BY yr"
);
$max_year = max(array_column($by_year, 'cnt') ?: [1]);

// ── Bücher mit 0 verfügbaren Exemplaren (alle ausgeliehen/reserviert) ─────────
$fully_loaned_list = db_get_results(
    "SELECT b.id, b.title, b.author,
            COUNT(bc.id)                                         AS total_copies,
            COALESCE(SUM(bc.status = 'ausgeliehen'),0)           AS loaned,
            COALESCE(SUM(bc.status = 'reserviert'),0)            AS reserved,
            (SELECT COUNT(*) FROM reservations r
             WHERE r.book_id = b.id AND r.status IN ('aktiv','verfügbar')) AS waiting
     FROM books b
     JOIN book_copies bc ON bc.book_id = b.id
     WHERE bc.status NOT IN ('aussortiert','verloren')
     GROUP BY b.id, b.title, b.author
     HAVING SUM(bc.status = 'verfügbar') = 0
     ORDER BY (loaned + reserved) DESC
     LIMIT 15"
);

// ── Neueste Zugänge (letzte 20) ────────────────────────────────────────────────
$newest = db_get_results(
    "SELECT b.id, b.title, b.author, b.image_url, b.year,
            bc.book_number, bc.signature, bc.status,
            COALESCE(bc.acquired_at, bc.acquisition_date, bc.created_at) AS added_date
     FROM book_copies bc
     JOIN books b ON bc.book_id = b.id
     ORDER BY COALESCE(bc.acquired_at, bc.acquisition_date, bc.created_at) DESC
     LIMIT 20"
);

include __DIR__ . '/includes/header.php';
?>

<!-- ── Kennzahlen-Kacheln ─────────────────────────────────────────────────── -->
<div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
    <?php
    $tiles = [
        ['label' => $lang['book_titles'] ?? 'Buchtitel',                    'val' => number_format($total_books,   0,',','.'), 'color' => 'blue'],
        ['label' => $lang['copies_active'] ?? 'Exemplare (aktiv)',           'val' => number_format($total_copies,  0,',','.'), 'color' => 'indigo'],
        ['label' => $lang['avg_copies_per_title'] ?? 'Ø Exemplare/Titel',   'val' => $avg_copies_per_book,                     'color' => 'purple'],
        ['label' => $lang['available'] ?? 'Verfügbar',                      'val' => number_format($avail_copies,  0,',','.') . ' (' . ($total_copies > 0 ? round($avail_copies/$total_copies*100) : 0) . '%)', 'color' => 'green'],
        ['label' => $lang['loaned'] ?? 'Ausgeliehen',                       'val' => number_format($loaned_copies, 0,',','.'), 'color' => 'orange'],
        ['label' => $lang['lost'] ?? 'Verloren',                            'val' => number_format($lost_copies,   0,',','.'), 'color' => $lost_copies > 0 ? 'red' : 'gray'],
        ['label' => $lang['titles_without_copy'] ?? 'Titel ohne Exemplar',  'val' => number_format($books_no_copies,     0,',','.'), 'color' => $books_no_copies > 0 ? 'yellow' : 'gray'],
        ['label' => $lang['titles_fully_loaned'] ?? 'Titel voll ausgeliehen','val' => number_format($books_fully_loaned, 0,',','.'), 'color' => 'orange'],
    ];
    foreach ($tiles as $t): ?>
        <div class="bg-white rounded-lg shadow p-5">
            <p class="text-xs text-gray-500 mb-1"><?= $t['label'] ?></p>
            <p class="text-2xl font-bold text-<?= $t['color'] ?>-600"><?= $t['val'] ?></p>
        </div>
    <?php endforeach; ?>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">

    <!-- ── Sprach-Verteilung ────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-4"><?= $lang['books_by_language'] ?? 'Bücher nach Sprache' ?></h2>
        <?php if (empty($by_language)): ?>
            <p class="text-sm text-gray-400 text-center py-6"><?= $lang['no_data'] ?? 'Keine Daten.' ?></p>
        <?php else: ?>
            <div class="space-y-2">
                <?php foreach ($by_language as $l):
                    $pct = $max_lang > 0 ? round($l['cnt'] / $max_lang * 100) : 0; ?>
                    <div class="flex items-center gap-3">
                        <span class="text-xs text-gray-600 w-24 flex-shrink-0 truncate"><?= esc_html($l['language']) ?></span>
                        <div class="flex-1 bg-gray-100 rounded-full h-6 relative overflow-hidden">
                            <div class="bg-blue-400 h-6 rounded-full" style="width:<?= $pct ?>%"></div>
                            <span class="absolute inset-0 flex items-center pl-2 text-xs font-medium <?= $pct > 25 ? 'text-white' : 'text-gray-700' ?>">
                                <?= number_format($l['cnt'], 0, ',', '.') ?>
                            </span>
                        </div>
                        <span class="text-xs text-gray-500 w-10 text-right flex-shrink-0">
                            <?= $total_books > 0 ? round($l['cnt'] / $total_books * 100) : 0 ?>%
                        </span>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- ── Zugangstimeline ──────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-4"><?= $lang['acquisitions_by_year'] ?? 'Zugänge nach Jahr' ?></h2>
        <?php if (empty($by_year)): ?>
            <p class="text-sm text-gray-400 text-center py-6"><?= $lang['no_acquisition_data'] ?? 'Keine Zugangsdaten.' ?></p>
        <?php else: ?>
            <div class="space-y-2">
                <?php foreach ($by_year as $yr):
                    $pct = $max_year > 0 ? round($yr['cnt'] / $max_year * 100) : 0;
                    $is_current = (int)$yr['yr'] === (int)date('Y'); ?>
                    <div class="flex items-center gap-3">
                        <span class="text-xs w-12 text-right flex-shrink-0 <?= $is_current ? 'font-bold text-blue-600' : 'text-gray-600' ?>">
                            <?= (int)$yr['yr'] ?>
                        </span>
                        <div class="flex-1 bg-gray-100 rounded-full h-6 relative overflow-hidden">
                            <div class="<?= $is_current ? 'bg-green-500' : 'bg-green-300' ?> h-6 rounded-full" style="width:<?= $pct ?>%"></div>
                            <span class="absolute inset-0 flex items-center pl-2 text-xs font-medium <?= $pct > 25 ? 'text-white' : 'text-gray-700' ?>">
                                <?= number_format($yr['cnt'], 0, ',', '.') ?>
                            </span>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- ── Kategorie-Tabelle ──────────────────────────────────────────────────── -->
<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-5 border-b">
        <h2 class="text-lg font-semibold text-gray-900"><?= $lang['stock_by_main_category'] ?? 'Bestand nach Hauptkategorie' ?></h2>
    </div>
    <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
            <thead class="bg-gray-50 text-gray-500 text-xs uppercase">
                <tr>
                    <th class="px-5 py-3 text-left font-medium"><?= $lang['book_category'] ?? 'Kategorie' ?></th>
                    <th class="px-4 py-3 text-right font-medium"><?= $lang['titles'] ?? 'Titel' ?></th>
                    <th class="px-4 py-3 text-right font-medium"><?= $lang['copies'] ?? 'Exemplare' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-green-700"><?= $lang['available'] ?? 'Verfügbar' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-orange-700"><?= $lang['loaned'] ?? 'Ausgeliehen' ?></th>
                    <th class="px-4 py-3 text-right font-medium"><?= $lang['utilization'] ?? 'Auslastung' ?></th>
                    <th class="px-4 py-3 font-medium"><?= $lang['stock_share'] ?? 'Anteil Bestand' ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
                <?php foreach ($by_cat as $cat):
                    $utilization = $cat['copies'] > 0 ? round($cat['loaned'] / $cat['copies'] * 100) : 0;
                    $share       = $total_books  > 0 ? round($cat['books']  / $total_books  * 100) : 0;
                    $share_max   = max(array_column($by_cat, 'books'));
                    $share_pct   = $share_max > 0 ? round($cat['books'] / $share_max * 100) : 0;
                ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-5 py-3 font-medium text-gray-900">
                            <span class="inline-block bg-gray-100 text-gray-500 text-xs font-mono px-1.5 py-0.5 rounded mr-1"><?= esc_html($cat['code']) ?></span>
                            <?= esc_html($cat['name']) ?>
                        </td>
                        <td class="px-4 py-3 text-right font-medium"><?= number_format($cat['books'], 0, ',', '.') ?></td>
                        <td class="px-4 py-3 text-right text-gray-600"><?= number_format($cat['copies'], 0, ',', '.') ?></td>
                        <td class="px-4 py-3 text-right text-green-700"><?= $cat['avail'] ?></td>
                        <td class="px-4 py-3 text-right text-orange-700"><?= $cat['loaned'] ?></td>
                        <td class="px-4 py-3 text-right text-sm">
                            <span class="<?= $utilization >= 70 ? 'text-red-600 font-semibold' : ($utilization >= 40 ? 'text-orange-500' : 'text-green-600') ?>">
                                <?= $utilization ?>%
                            </span>
                        </td>
                        <td class="px-4 py-3">
                            <div class="flex items-center gap-2">
                                <div class="w-24 bg-gray-100 rounded-full h-3">
                                    <div class="bg-indigo-400 h-3 rounded-full" style="width:<?= $share_pct ?>%"></div>
                                </div>
                                <span class="text-xs text-gray-500"><?= $share ?>%</span>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">

    <!-- ── Voll ausgeliehene Titel ──────────────────────────────────────────── -->
    <?php if (!empty($fully_loaned_list)): ?>
    <div class="bg-white rounded-lg shadow">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900"><?= $lang['fully_loaned_titles'] ?? 'Vollständig ausgeliehene Titel' ?></h2>
            <p class="text-xs text-gray-400"><?= $lang['no_copy_available'] ?? 'Kein Exemplar verfügbar' ?></p>
        </div>
        <div class="divide-y divide-gray-100 max-h-80 overflow-y-auto">
            <?php foreach ($fully_loaned_list as $bk): ?>
                <div class="px-5 py-3 flex items-start justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <a href="<?= BASE_URL ?>/admin/books-edit.php?id=<?= (int)$bk['id'] ?>"
                           class="text-sm font-medium text-gray-900 hover:text-blue-600 line-clamp-1">
                            <?= esc_html($bk['title']) ?>
                        </a>
                        <p class="text-xs text-gray-400 truncate"><?= esc_html($bk['author'] ?? '') ?></p>
                    </div>
                    <div class="flex gap-2 flex-shrink-0 text-xs">
                        <span class="px-1.5 py-0.5 bg-orange-100 text-orange-700 rounded">
                            <?= $bk['loaned'] ?>× <?= $lang['loaned'] ?? 'ausgeliehen' ?>
                        </span>
                        <?php if ($bk['waiting'] > 0): ?>
                            <span class="px-1.5 py-0.5 bg-blue-100 text-blue-700 rounded">
                                <?= $bk['waiting'] ?> <?= $lang['waiting'] ?? 'wartend' ?>
                            </span>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- ── Neueste Zugänge ──────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900"><?= $lang['newest_acquisitions'] ?? 'Neueste Zugänge' ?></h2>
            <p class="text-xs text-gray-400"><?= $lang['last_20_copies_by_date'] ?? 'Letzte 20 Exemplare nach Zugangsdatum' ?></p>
        </div>
        <div class="divide-y divide-gray-100 max-h-80 overflow-y-auto">
            <?php foreach ($newest as $ex): ?>
                <div class="px-5 py-3 flex items-start justify-between gap-3">
                    <div class="flex-1 min-w-0">
                        <a href="<?= BASE_URL ?>/admin/books-edit.php?id=<?= (int)$ex['id'] ?>"
                           class="text-sm font-medium text-gray-900 hover:text-blue-600 line-clamp-1">
                            <?= esc_html($ex['title']) ?>
                        </a>
                        <p class="text-xs text-gray-400 font-mono"><?= esc_html($ex['book_number'] ?? '') ?> &bull; <?= esc_html($ex['signature'] ?? '') ?></p>
                    </div>
                    <div class="flex-shrink-0 text-xs text-gray-500">
                        <?= $ex['added_date'] ? date('d.m.Y', strtotime($ex['added_date'])) : '–' ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
