<?php
/**
 * Berichtsgenerator — reports-generator.php
 * Flexible Berichterstellung: aktive Ausleihen, überfällige Ausleihen,
 * Bestandsliste, Ausleih-Historie — als HTML-Tabelle oder CSV-Export.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();

// ── Parameter einlesen & validieren ───────────────────────────────────────────
$report_type = $_GET['report'] ?? '';
$date_from   = $_GET['date_from'] ?? '';
$date_to     = $_GET['date_to'] ?? '';
$subcat_id   = (int)($_GET['subcat'] ?? 0);
$export_csv  = isset($_GET['export']) && $_GET['export'] === 'csv';

$allowed_reports = ['active_loans', 'overdue_loans', 'inventory_list', 'loan_history'];
if ($report_type !== '' && !in_array($report_type, $allowed_reports, true)) {
    $report_type = '';
}

// Datum-Validierung
$date_from_ts = $date_from !== '' ? strtotime($date_from) : false;
$date_to_ts   = $date_to   !== '' ? strtotime($date_to)   : false;
if (!$date_from_ts) $date_from = '';
if (!$date_to_ts)   $date_to   = '';

// ── Hilfsfunktion: Subcategories für Filter ───────────────────────────────────
global $pdo;

$subcategories = db_get_results(
    'SELECT bs.id, bs.code, bs.name, bc.code AS cat_code, bc.name AS cat_name
     FROM book_subcategories bs
     JOIN book_categories bc ON bs.category_id = bc.id
     WHERE bs.active = 1 AND bc.active = 1
     ORDER BY bc.sort_order, bc.code, bs.sort_order, bs.code'
);

// ── Bericht-Labels ─────────────────────────────────────────────────────────────
$report_labels = [
    'active_loans'   => $lang['active_loans_label'] ?? 'Aktive Ausleihen',
    'overdue_loans'  => $lang['overdue_loans_label'] ?? 'Überfällige Ausleihen',
    'inventory_list' => $lang['inventory_list_label'] ?? 'Bestandsliste',
    'loan_history'   => $lang['loan_history_label'] ?? 'Ausleih-Historie',
];

// ── Daten abfragen (nur wenn Berichtstyp gewählt) ────────────────────────────
$rows    = [];
$columns = [];
$total   = 0;

if ($report_type !== '') {

    // ── gemeinsame Filter-Fragmente ───────────────────────────────────────────
    $params = [];

    $date_col_loans   = 'l.loan_date';
    $date_col_history = 'lh.loan_date';

    $where_date_loans = '';
    if ($date_from !== '') {
        $where_date_loans .= " AND {$date_col_loans} >= ?";
        $params[] = $date_from . ' 00:00:00';
    }
    if ($date_to !== '') {
        $where_date_loans .= " AND {$date_col_loans} <= ?";
        $params[] = $date_to . ' 23:59:59';
    }

    $where_date_history = '';
    $params_history = [];
    if ($date_from !== '') {
        $where_date_history .= " AND {$date_col_history} >= ?";
        $params_history[] = $date_from . ' 00:00:00';
    }
    if ($date_to !== '') {
        $where_date_history .= " AND {$date_col_history} <= ?";
        $params_history[] = $date_to . ' 23:59:59';
    }

    $where_subcat_bc = '';
    $params_subcat   = [];
    if ($subcat_id > 0) {
        $where_subcat_bc = 'AND b.subcategory_id = ?';
        $params_subcat[] = $subcat_id;
    }

    // ═══════════════════════════════════════════════════════════════════════════
    if ($report_type === 'active_loans') {
        $columns = [
            $lang['loan_id_col'] ?? 'Ausleih-ID',
            $lang['username_col'] ?? 'Benutzername',
            $lang['class_name'] ?? 'Klasse',
            $lang['book_title'] ?? 'Titel',
            $lang['signature'] ?? 'Signatur',
            $lang['book_number_col'] ?? 'Buchungsnr.',
            $lang['loaned_on_col'] ?? 'Ausgeliehen am',
            $lang['due_on_col'] ?? 'Fällig am',
            $lang['days_open_col'] ?? 'Tage offen',
        ];

        $sql_params = array_merge($params_subcat, $params);
        $sql = "
            SELECT
                l.id            AS loan_id,
                CONCAT(u.last_name, ', ', u.first_name) AS user_name,
                u.class_name,
                b.title,
                bc.signature,
                bc.book_number,
                DATE_FORMAT(l.loan_date, '%d.%m.%Y') AS loan_date_fmt,
                DATE_FORMAT(l.due_date,  '%d.%m.%Y') AS due_date_fmt,
                DATEDIFF(CURDATE(), l.due_date)       AS days_open
            FROM loans l
            JOIN book_copies bc ON bc.id = l.copy_id
            JOIN books b        ON b.id  = bc.book_id
            JOIN users u        ON u.id  = l.user_id
            WHERE l.return_date IS NULL
              {$where_subcat_bc}
              {$where_date_loans}
            ORDER BY l.due_date ASC
        ";
        $rows = db_get_results($sql, $sql_params);
        $total = count($rows);

    // ═══════════════════════════════════════════════════════════════════════════
    } elseif ($report_type === 'overdue_loans') {
        $columns = [
            $lang['loan_id_col'] ?? 'Ausleih-ID',
            $lang['username_col'] ?? 'Benutzername',
            $lang['class_name'] ?? 'Klasse',
            $lang['book_title'] ?? 'Titel',
            $lang['signature'] ?? 'Signatur',
            $lang['book_number_col'] ?? 'Buchungsnr.',
            $lang['loaned_on_col'] ?? 'Ausgeliehen am',
            $lang['due_on_col'] ?? 'Fällig am',
            $lang['days_overdue_col'] ?? 'Tage überfällig',
        ];

        $sql_params = array_merge($params_subcat, $params);
        $sql = "
            SELECT
                l.id            AS loan_id,
                CONCAT(u.last_name, ', ', u.first_name) AS user_name,
                u.class_name,
                b.title,
                bc.signature,
                bc.book_number,
                DATE_FORMAT(l.loan_date, '%d.%m.%Y') AS loan_date_fmt,
                DATE_FORMAT(l.due_date,  '%d.%m.%Y') AS due_date_fmt,
                DATEDIFF(CURDATE(), l.due_date)       AS days_overdue
            FROM loans l
            JOIN book_copies bc ON bc.id = l.copy_id
            JOIN books b        ON b.id  = bc.book_id
            JOIN users u        ON u.id  = l.user_id
            WHERE l.return_date IS NULL
              AND l.due_date < CURDATE()
              {$where_subcat_bc}
              {$where_date_loans}
            ORDER BY l.due_date ASC
        ";
        $rows = db_get_results($sql, $sql_params);
        $total = count($rows);

    // ═══════════════════════════════════════════════════════════════════════════
    } elseif ($report_type === 'inventory_list') {
        $columns = [
            $lang['copy_id_col'] ?? 'Exemplar-ID',
            $lang['book_number_col'] ?? 'Buchungsnr.',
            $lang['signature'] ?? 'Signatur',
            $lang['book_title'] ?? 'Titel',
            $lang['book_author'] ?? 'Autor',
            $lang['book_year'] ?? 'Jahr',
            $lang['book_language'] ?? 'Sprache',
            $lang['status'] ?? 'Status',
            $lang['location_col'] ?? 'Standort',
            $lang['acquisition_date_col'] ?? 'Zugangsdatum',
        ];

        $sql_params = $params_subcat;
        $sql = "
            SELECT
                bc.id           AS copy_id,
                bc.book_number,
                bc.signature,
                b.title,
                b.author,
                b.year,
                b.language,
                bc.status,
                bc.location,
                COALESCE(
                    DATE_FORMAT(bc.acquired_at,      '%d.%m.%Y'),
                    DATE_FORMAT(bc.acquisition_date, '%d.%m.%Y'),
                    ''
                )               AS access_date
            FROM book_copies bc
            JOIN books b ON b.id = bc.book_id
            WHERE bc.status NOT IN ('aussortiert', 'verloren')
              {$where_subcat_bc}
            ORDER BY bc.signature ASC, bc.book_number ASC
        ";
        $rows = db_get_results($sql, $sql_params);
        $total = count($rows);

    // ═══════════════════════════════════════════════════════════════════════════
    } elseif ($report_type === 'loan_history') {
        $columns = [
            'ID',
            $lang['username_col'] ?? 'Benutzername',
            $lang['class_name'] ?? 'Klasse',
            $lang['book_title'] ?? 'Titel',
            $lang['book_number_col'] ?? 'Buchungsnr.',
            $lang['loaned_on_col'] ?? 'Ausgeliehen am',
            $lang['returned_on_col'] ?? 'Zurückgegeben am',
            $lang['duration_days_col'] ?? 'Dauer (Tage)',
            $lang['was_overdue_col'] ?? 'Überfällig',
        ];

        // loan_history: lh.user_name (denorm), lh.book_title (denorm)
        // Klasse über JOIN users (FK gesetzt)
        $sql_params = array_merge($params_subcat, $params_history);

        $where_subcat_lh = '';
        if ($subcat_id > 0) {
            // loan_history hat book_id → JOIN books
            $where_subcat_lh = 'AND b.subcategory_id = ?';
        }

        $sql = "
            SELECT
                lh.id,
                lh.user_name,
                u.class_name,
                lh.book_title       AS title,
                bc.book_number,
                DATE_FORMAT(lh.loan_date,   '%d.%m.%Y') AS loan_date_fmt,
                DATE_FORMAT(lh.return_date, '%d.%m.%Y') AS return_date_fmt,
                DATEDIFF(lh.return_date, lh.loan_date)  AS duration_days,
                IF(lh.was_overdue, 'Ja', 'Nein')        AS was_overdue_label
            FROM loan_history lh
            LEFT JOIN users u       ON u.id  = lh.user_id
            LEFT JOIN book_copies bc ON bc.id = lh.copy_id
            LEFT JOIN books b       ON b.id  = lh.book_id
            WHERE 1=1
              {$where_subcat_lh}
              {$where_date_history}
            ORDER BY lh.return_date DESC
            LIMIT 2000
        ";
        $rows = db_get_results($sql, $sql_params);
        $total = count($rows);
    }
}

// ── CSV-Export: vor HTML-Ausgabe senden und beenden ───────────────────────────
if ($export_csv && $report_type !== '' && !empty($columns)) {
    $filename = 'bericht_' . $report_type . '_' . date('Y-m-d') . '.csv';
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');

    $out = fopen('php://output', 'w');
    // BOM für Excel-UTF-8-Kompatibilität
    fputs($out, "\xEF\xBB\xBF");
    fputcsv($out, $columns, ';');

    foreach ($rows as $row) {
        fputcsv($out, array_values((array)$row), ';');
    }
    fclose($out);
    exit;
}

// ── Seite ausgeben ────────────────────────────────────────────────────────────
$page_title = $lang['admin_report_generator'] ?? 'Berichtsgenerator';
include __DIR__ . '/includes/header.php';
?>

<div class="p-6 max-w-7xl mx-auto">

  <!-- Seitenheader -->
  <div class="flex flex-wrap items-center justify-between gap-4 mb-6">
    <div>
      <h1 class="text-2xl font-bold text-gray-800"><?= $lang['admin_report_generator'] ?? 'Berichtsgenerator' ?></h1>
      <p class="text-sm text-gray-500 mt-1"><?= $lang['report_generator_subtitle'] ?? 'Wählen Sie Berichtstyp und Filter — Darstellung als Tabelle oder CSV-Download.' ?></p>
    </div>
    <div class="flex gap-2 text-sm">
      <a href="reports-lending.php" class="px-3 py-1.5 rounded border border-gray-300 hover:bg-gray-50 text-gray-600"><?= $lang['admin_lending_stats'] ?? 'Ausleihstatistik' ?></a>
      <a href="reports-inventory.php" class="px-3 py-1.5 rounded border border-gray-300 hover:bg-gray-50 text-gray-600"><?= $lang['admin_inventory_overview'] ?? 'Bestandsübersicht' ?></a>
    </div>
  </div>

  <!-- Filterformular -->
  <form method="get" class="bg-white rounded-xl shadow p-5 mb-6">
    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">

      <!-- Berichtstyp -->
      <div class="lg:col-span-2">
        <label class="block text-xs font-semibold text-gray-500 uppercase tracking-wide mb-1"><?= $lang['report_type'] ?? 'Berichtstyp' ?></label>
        <div class="grid grid-cols-2 gap-2">
          <?php foreach ($report_labels as $key => $label): ?>
          <label class="flex items-center gap-2 cursor-pointer p-2.5 rounded-lg border-2 transition-colors
            <?= $report_type === $key ? 'border-indigo-500 bg-indigo-50' : 'border-gray-200 hover:border-gray-300' ?>">
            <input type="radio" name="report" value="<?= esc_attr($key) ?>"
              <?= $report_type === $key ? 'checked' : '' ?>
              class="text-indigo-600 focus:ring-indigo-500">
            <span class="text-sm font-medium text-gray-700"><?= esc_html($label) ?></span>
          </label>
          <?php endforeach; ?>
        </div>
      </div>

      <!-- Rechte Spalte: Datums- & Kategoriefilter -->
      <div class="lg:col-span-2 space-y-3">

        <div class="grid grid-cols-2 gap-3">
          <div>
            <label class="block text-xs font-semibold text-gray-500 uppercase tracking-wide mb-1"><?= $lang['date_from'] ?? 'Von' ?></label>
            <input type="date" name="date_from" value="<?= esc_attr($date_from) ?>"
              class="w-full rounded-lg border-gray-300 shadow-sm text-sm focus:ring-indigo-500 focus:border-indigo-500 px-3 py-2 border">
          </div>
          <div>
            <label class="block text-xs font-semibold text-gray-500 uppercase tracking-wide mb-1"><?= $lang['date_until'] ?? 'Bis' ?></label>
            <input type="date" name="date_to" value="<?= esc_attr($date_to) ?>"
              class="w-full rounded-lg border-gray-300 shadow-sm text-sm focus:ring-indigo-500 focus:border-indigo-500 px-3 py-2 border">
          </div>
        </div>

        <div>
          <label class="block text-xs font-semibold text-gray-500 uppercase tracking-wide mb-1"><?= $lang['signature_group_optional'] ?? 'Signaturgruppe (optional)' ?></label>
          <select name="subcat"
            class="w-full rounded-lg border-gray-300 shadow-sm text-sm focus:ring-indigo-500 focus:border-indigo-500 px-3 py-2 border">
            <option value="0"><?= $lang['all_signature_groups'] ?? 'Alle Signaturgruppen' ?></option>
            <?php
            $current_cat = '';
            foreach ($subcategories as $sc):
                if ($sc['cat_code'] !== $current_cat) {
                    if ($current_cat !== '') echo '</optgroup>';
                    echo '<optgroup label="' . esc_attr($sc['cat_name']) . '">';
                    $current_cat = $sc['cat_code'];
                }
            ?>
            <option value="<?= esc_attr($sc['id']) ?>" <?= $subcat_id === (int)$sc['id'] ? 'selected' : '' ?>>
              <?= esc_html($sc['code'] . ' — ' . $sc['name']) ?>
            </option>
            <?php endforeach; if ($current_cat !== '') echo '</optgroup>'; ?>
          </select>
        </div>

        <div class="flex gap-2 pt-1">
          <button type="submit"
            class="flex-1 bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-semibold py-2.5 rounded-lg transition-colors">
            <?= $lang['show_report'] ?? 'Bericht anzeigen' ?>
          </button>
          <?php if ($report_type !== '' && !empty($rows)): ?>
          <a href="?<?= esc_attr(http_build_query(array_merge($_GET, ['export' => 'csv']))) ?>"
            class="flex items-center gap-1.5 px-4 py-2.5 bg-emerald-600 hover:bg-emerald-700 text-white text-sm font-semibold rounded-lg transition-colors">
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"/>
            </svg>
            CSV
          </a>
          <?php endif; ?>
        </div>
      </div>

    </div>
  </form>

  <?php if ($report_type === ''): ?>
  <!-- Kein Bericht gewählt -->
  <div class="bg-white rounded-xl shadow p-12 text-center text-gray-400">
    <svg class="w-16 h-16 mx-auto mb-4 text-gray-200" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
        d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414A1 1 0 0119 9.414V19a2 2 0 01-2 2z"/>
    </svg>
    <p class="text-lg font-medium text-gray-500"><?= $lang['select_report_type_hint'] ?? 'Wählen Sie einen Berichtstyp und klicken Sie auf "Bericht anzeigen".' ?></p>
  </div>

  <?php elseif (empty($rows)): ?>
  <!-- Keine Daten -->
  <div class="bg-white rounded-xl shadow p-10 text-center">
    <svg class="w-12 h-12 mx-auto mb-3 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
    </svg>
    <p class="text-gray-500 font-medium"><?= $lang['no_entries_for_filter'] ?? 'Keine Einträge für die gewählten Filter gefunden.' ?></p>
  </div>

  <?php else: ?>
  <!-- Ergebnistabelle -->
  <div class="bg-white rounded-xl shadow overflow-hidden">

    <!-- Tabellenkopf -->
    <div class="px-5 py-4 border-b border-gray-100 flex flex-wrap items-center justify-between gap-2">
      <div>
        <h2 class="text-base font-semibold text-gray-800">
          <?= esc_html($report_labels[$report_type]) ?>
        </h2>
        <p class="text-xs text-gray-400 mt-0.5">
          <?= number_format($total, 0, ',', '.') ?> <?= $lang['entries'] ?? 'Einträge' ?>
          <?php if ($date_from || $date_to): ?>
            · <?= $date_from ? esc_html(date('d.m.Y', strtotime($date_from))) : '–' ?>
            <?= $lang['date_until'] ?? 'bis' ?> <?= $date_to ? esc_html(date('d.m.Y', strtotime($date_to))) : ($lang['today'] ?? 'heute') ?>
          <?php endif; ?>
          <?php if ($subcat_id > 0):
            $subcat_label = '';
            foreach ($subcategories as $sc) {
                if ((int)$sc['id'] === $subcat_id) { $subcat_label = $sc['code'] . ' ' . $sc['name']; break; }
            }
          ?>
            · <?= esc_html($subcat_label) ?>
          <?php endif; ?>
        </p>
      </div>
      <a href="?<?= esc_attr(http_build_query(array_merge($_GET, ['export' => 'csv']))) ?>"
        class="flex items-center gap-1.5 px-3 py-1.5 bg-emerald-600 hover:bg-emerald-700 text-white text-xs font-semibold rounded-lg transition-colors">
        <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
            d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"/>
        </svg>
        <?= $lang['export_as_csv'] ?? 'Als CSV exportieren' ?>
      </a>
    </div>

    <!-- Scrollbare Tabelle -->
    <div class="overflow-x-auto">
      <table class="min-w-full text-sm">
        <thead class="bg-gray-50 border-b border-gray-200">
          <tr>
            <?php foreach ($columns as $col): ?>
            <th class="px-4 py-3 text-left text-xs font-semibold text-gray-500 uppercase tracking-wide whitespace-nowrap">
              <?= esc_html($col) ?>
            </th>
            <?php endforeach; ?>
          </tr>
        </thead>
        <tbody class="divide-y divide-gray-100">
          <?php foreach ($rows as $i => $row):
            $row_arr = (array)$row;
            $values  = array_values($row_arr);

            // Zeile einfärben: überfällig-Markierung für Ausleih-Berichte
            $is_overdue = false;
            if (in_array($report_type, ['active_loans', 'overdue_loans'], true)) {
                $days = (int)end($values);
                $is_overdue = ($days > 0);
            }
          ?>
          <tr class="<?= ($i % 2 === 1) ? 'bg-gray-50/50' : '' ?> <?= $is_overdue ? 'bg-red-50/40' : '' ?> hover:bg-indigo-50/30 transition-colors">
            <?php foreach ($values as $idx => $val):
              $cell = (string)($val ?? '');

              // Statusfelder farbig darstellen
              $status_classes = [
                  'verfügbar'      => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800',
                  'ausgeliehen'    => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800',
                  'reserviert'     => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800',
                  'in_bearbeitung' => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-orange-100 text-orange-800',
                  'verloren'       => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800',
                  'aussortiert'    => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-600',
                  'Ja'             => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-700',
                  'Nein'           => 'inline-flex px-2 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-500',
              ];

              // Tageszahl für Überfälligkeitsfelder: letzte Spalte in Ausleih-Berichten
              $is_days_col = $is_overdue && ($idx === count($values) - 1);
            ?>
            <td class="px-4 py-2.5 whitespace-nowrap text-gray-700">
              <?php if (isset($status_classes[$cell])): ?>
                <span class="<?= $status_classes[$cell] ?>"><?= esc_html($cell) ?></span>
              <?php elseif ($is_days_col && (int)$cell > 0): ?>
                <span class="font-semibold text-red-600"><?= esc_html($cell) ?> <?= $lang['days'] ?? 'Tage' ?></span>
              <?php elseif ($is_days_col && (int)$cell <= 0): ?>
                <span class="text-gray-400">–</span>
              <?php else: ?>
                <?= esc_html($cell) ?>
              <?php endif; ?>
            </td>
            <?php endforeach; ?>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>

    <?php if ($total >= 2000 && $report_type === 'loan_history'): ?>
    <div class="px-5 py-3 bg-amber-50 border-t border-amber-200 text-xs text-amber-700 text-center">
      <?= $lang['history_limit_notice'] ?? 'Anzeige auf 2.000 Einträge begrenzt. Für vollständige Daten CSV exportieren.' ?>
    </div>
    <?php endif; ?>

  </div>
  <?php endif; ?>

</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
