<?php
/**
 * Rückgabe buchen — lending-return.php
 * Single-File, AJAX-gesteuert: Ausleihe suchen → Rückgabe bestätigen
 *
 * AJAX-Actions (GET, kein CSRF): search_loan, get_loan
 * AJAX-Actions (POST, CSRF):     process_return
 *
 * Trigger pflegen current_loans / overdue_loans automatisch → NICHT manuell updaten!
 * return_date IS NULL = Ausleihe aktiv
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = $current_user['id'];

// ─── AJAX-Dispatcher ──────────────────────────────────────────────────────────

$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($action !== '') {
    header('Content-Type: application/json; charset=utf-8');

    switch ($action) {

        // ── search_loan (GET) ────────────────────────────────────────────────
        case 'search_loan':
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }

            // Numerisch (4-10 Ziffern) → Exact Match zuerst
            if (preg_match('/^\d{4,10}$/', $q)) {
                $exact = db_get_results(
                    "SELECT l.id AS loan_id, l.loan_date, l.due_date, l.status AS loan_status,
                            l.loan_note, l.extensions,
                            u.id AS user_id, u.first_name, u.last_name, u.class_name,
                            bc.id AS copy_id, bc.book_number, bc.accession_number, bc.signature,
                            b.id AS book_id, b.title, b.author,
                            DATEDIFF(CURDATE(), l.due_date) AS days_overdue
                     FROM loans l
                     JOIN users u ON l.user_id = u.id
                     JOIN book_copies bc ON l.copy_id = bc.id
                     JOIN books b ON bc.book_id = b.id
                     WHERE l.return_date IS NULL
                       AND (bc.book_number = ? OR bc.accession_number = ?)
                     ORDER BY l.due_date ASC
                     LIMIT 10",
                    [$q, $q]
                );
                if (!empty($exact)) { echo json_encode($exact); exit; }
            }

            // LIKE-Suche: Signatur, Titel, Autor, Name, Klasse
            $like = '%' . $q . '%';
            $rows = db_get_results(
                "SELECT l.id AS loan_id, l.loan_date, l.due_date, l.status AS loan_status,
                        l.loan_note, l.extensions,
                        u.id AS user_id, u.first_name, u.last_name, u.class_name,
                        bc.id AS copy_id, bc.book_number, bc.accession_number, bc.signature,
                        b.id AS book_id, b.title, b.author,
                        DATEDIFF(CURDATE(), l.due_date) AS days_overdue
                 FROM loans l
                 JOIN users u ON l.user_id = u.id
                 JOIN book_copies bc ON l.copy_id = bc.id
                 JOIN books b ON bc.book_id = b.id
                 WHERE l.return_date IS NULL
                   AND (bc.book_number LIKE ? OR bc.accession_number LIKE ?
                        OR bc.signature LIKE ? OR b.title LIKE ? OR b.author LIKE ?
                        OR u.first_name LIKE ? OR u.last_name LIKE ?
                        OR CONCAT(u.first_name,' ',u.last_name) LIKE ?
                        OR CONCAT(u.last_name,' ',u.first_name) LIKE ?
                        OR u.class_name LIKE ?)
                 ORDER BY l.due_date ASC
                 LIMIT 30",
                [$like, $like, $like, $like, $like, $like, $like, $like, $like, $like]
            );
            echo json_encode($rows);
            exit;

        // ── get_loan (GET) ───────────────────────────────────────────────────
        case 'get_loan':
            $loan_id = (int)($_GET['id'] ?? 0);
            if (!$loan_id) { echo json_encode(['error' => 'Keine ID']); exit; }
            $loan = db_get_row(
                "SELECT l.id AS loan_id, l.loan_date, l.due_date, l.status AS loan_status,
                        l.loan_note, l.extensions,
                        u.id AS user_id, u.first_name, u.last_name, u.class_name,
                        u.overdue_loans, u.current_loans,
                        ug.max_loans,
                        bc.id AS copy_id, bc.book_number, bc.accession_number, bc.signature,
                        b.id AS book_id, b.title, b.author, b.isbn,
                        DATEDIFF(CURDATE(), l.due_date) AS days_overdue
                 FROM loans l
                 JOIN users u ON l.user_id = u.id
                 LEFT JOIN user_groups ug ON u.user_group_id = ug.id
                 JOIN book_copies bc ON l.copy_id = bc.id
                 JOIN books b ON bc.book_id = b.id
                 WHERE l.id = ? AND l.return_date IS NULL",
                [$loan_id]
            );
            if (!$loan) { echo json_encode(['error' => 'Ausleihe nicht gefunden oder bereits zurückgegeben']); exit; }
            echo json_encode($loan);
            exit;

        // ── process_return (POST, CSRF, Transaction) ─────────────────────────
        case 'process_return':
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                echo json_encode(['error' => 'Methode nicht erlaubt']); exit;
            }
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            handle_return($admin_id);
            exit;

        default:
            http_response_code(400);
            echo json_encode(['error' => 'Unbekannte Aktion']);
            exit;
    }
}

// ─── handle_return() ──────────────────────────────────────────────────────────

function handle_return(int $admin_id): void {
    global $pdo;

    $loan_id     = (int)($_POST['loan_id'] ?? 0);
    $return_note = trim($_POST['return_note'] ?? '');

    if (!$loan_id) { echo json_encode(['error' => 'Keine Ausleihe-ID']); return; }

    // Ausleihe + Exemplar + Buch + Benutzer laden
    $loan = db_get_row(
        "SELECT l.id, l.user_id, l.copy_id, l.loan_date, l.due_date, l.status AS loan_status,
                u.first_name, u.last_name,
                bc.book_number, bc.signature,
                b.id AS book_id, b.title
         FROM loans l
         JOIN users u ON l.user_id = u.id
         JOIN book_copies bc ON l.copy_id = bc.id
         JOIN books b ON bc.book_id = b.id
         WHERE l.id = ? AND l.return_date IS NULL",
        [$loan_id]
    );
    if (!$loan) {
        echo json_encode(['error' => 'Ausleihe nicht gefunden oder bereits zurückgegeben']); return;
    }

    $was_overdue = (strtotime($loan['due_date']) < strtotime(date('Y-m-d')));

    try {
        $pdo->beginTransaction();

        // Ausleihe als zurückgegeben markieren
        // Trigger update_current_loans_after_return: current_loans - 1
        // Trigger update_overdue_count_after_loan_update: overdue_loans - 1 (wenn war überfällig)
        $stmt = $pdo->prepare(
            "UPDATE loans
             SET return_date  = NOW(),
                 status       = 'zurückgegeben',
                 returned_by  = ?,
                 return_note  = ?
             WHERE id = ? AND return_date IS NULL"
        );
        $stmt->execute([$admin_id, $return_note ?: null, $loan_id]);

        if ($stmt->rowCount() !== 1) {
            $pdo->rollBack();
            echo json_encode(['error' => 'Doppelte Rückgabe oder Datenbankfehler (rowCount).']); return;
        }

        // Pruefen ob eine aktive Vormerkung fuer dieses Buch existiert
        $reservation = db_get_row(
            "SELECT id, user_id FROM reservations
             WHERE book_id = ? AND status = 'aktiv'
             ORDER BY reserved_date ASC LIMIT 1",
            [$loan['book_id']]
        );

        $new_copy_status = $reservation ? 'reserviert' : 'verfügbar';

        // Exemplar-Status aktualisieren
        $pdo->prepare("UPDATE book_copies SET status = ? WHERE id = ?")
            ->execute([$new_copy_status, $loan['copy_id']]);

        // Wenn Vormerkung vorhanden: auf 'verfügbar' setzen + Exemplar zuweisen
        if ($reservation) {
            $pdo->prepare(
                "UPDATE reservations
                 SET status = 'verfügbar', copy_id = ?, notified_date = NOW()
                 WHERE id = ?"
            )->execute([$loan['copy_id'], $reservation['id']]);
        }

        // Loan-History-Eintrag
        $days_overdue = $was_overdue
            ? max(0, (int)((strtotime(date('Y-m-d')) - strtotime($loan['due_date'])) / 86400))
            : 0;

        $pdo->prepare(
            "INSERT INTO loan_history
             (user_id, book_id, copy_id, loan_date, return_date,
              was_overdue, days_overdue, user_name, book_title)
             VALUES (?, ?, ?, ?, NOW(), ?, ?, ?, ?)"
        )->execute([
            $loan['user_id'],
            $loan['book_id'],
            $loan['copy_id'],
            $loan['loan_date'],
            $was_overdue ? 1 : 0,
            $days_overdue,
            $loan['first_name'] . ' ' . $loan['last_name'],
            $loan['title'],
        ]);

        $pdo->commit();

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log('Return-Fehler: ' . $e->getMessage());
        echo json_encode(['error' => 'Datenbankfehler: ' . $e->getMessage()]); return;
    }

    log_activity($admin_id, 'loan_returned', 'loans', $loan_id,
        json_encode([
            'user'      => $loan['first_name'] . ' ' . $loan['last_name'],
            'book'      => $loan['title'],
            'signature' => $loan['signature'],
            'overdue'   => $was_overdue,
        ])
    );

    $response = [
        'success'          => true,
        'loan_id'          => $loan_id,
        'user_name'        => $loan['first_name'] . ' ' . $loan['last_name'],
        'book_title'       => $loan['title'],
        'signature'        => $loan['signature'],
        'book_number'      => $loan['book_number'],
        'was_overdue'      => $was_overdue,
        'days_overdue'     => $days_overdue ?? 0,
        'new_copy_status'  => $new_copy_status,
        'reservation_user' => null,
    ];

    if ($reservation) {
        $res_user = db_get_row(
            "SELECT first_name, last_name, class_name FROM users WHERE id = ?",
            [$reservation['user_id']]
        );
        $response['reservation_user'] = $res_user
            ? $res_user['first_name'] . ' ' . $res_user['last_name']
              . ($res_user['class_name'] ? ' (' . $res_user['class_name'] . ')' : '')
            : null;
    }

    echo json_encode($response);
}

// ─── Seiten-Setup ─────────────────────────────────────────────────────────────

$page_title = $lang['process_return'] ?? 'Rückgabe buchen';
$csrf_token = generate_csrf_token();

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['process_return'] ?? 'Rückgabe buchen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['return_workflow_hint'] ?? 'Ausleihe suchen und zurückbuchen'; ?></p>
        </div>
        <div class="flex gap-3 text-sm">
            <a href="lending-checkout.php" class="text-gray-500 hover:text-gray-700">&larr; <?php echo $lang['new_checkout'] ?? 'Neue Ausleihe'; ?></a>
            <a href="lending-active.php" class="text-gray-500 hover:text-gray-700"><?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?></a>
        </div>
    </div>

    <!-- ═══════════════════════════════════════ PANEL 1: Ausleihe suchen -->
    <div id="panel-search" class="bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900">
                <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-orange-500 text-white text-sm font-bold mr-2">1</span>
                <?php echo $lang['search_loan'] ?? 'Ausleihe suchen'; ?>
            </h2>
        </div>
        <div class="p-5">
            <div class="flex gap-3 mb-4">
                <input type="text" id="searchInput"
                       placeholder="<?php echo $lang['loan_search_placeholder'] ?? 'Buchungsnr., Signatur, Titel, Autor oder Benutzername (mind. 2 Zeichen)...'; ?>"
                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-orange-500 focus:border-orange-500">
                <button onclick="searchLoan()"
                        class="px-4 py-2 bg-orange-500 text-white text-sm rounded-md hover:bg-orange-600">
                    <?php echo $lang['search'] ?? 'Suchen'; ?>
                </button>
            </div>
            <div id="search-results"></div>
        </div>
    </div>

    <!-- ══════════════════════════════════════ PANEL 2: Rückgabe bestätigen -->
    <div id="panel-confirm" class="hidden bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b flex items-center justify-between">
            <h2 class="text-lg font-semibold text-gray-900">
                <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-orange-500 text-white text-sm font-bold mr-2">2</span>
                <?php echo $lang['confirm_return'] ?? 'Rückgabe bestätigen'; ?>
            </h2>
            <button onclick="resetConfirm()" class="text-sm text-gray-400 hover:text-gray-600">
                &times; <?php echo $lang['close'] ?? 'Schließen'; ?>
            </button>
        </div>
        <div class="p-5">

            <!-- Ausleihe-Details -->
            <div id="loan-details" class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-5"></div>

            <!-- Überfällig-Warnung -->
            <div id="overdue-warning" class="hidden mb-4 bg-red-50 border border-red-300 rounded-lg p-4">
                <div class="flex items-start gap-3">
                    <svg class="w-5 h-5 text-red-500 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                              d="M12 9v2m0 4h.01M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                    </svg>
                    <div>
                        <p class="font-semibold text-red-800" id="overdue-text"></p>
                        <p class="text-sm text-red-600 mt-0.5"><?php echo $lang['overdue_check_reminder_hint'] ?? 'Bitte prüfen ob eine Mahnung erforderlich ist.'; ?></p>
                    </div>
                </div>
            </div>

            <!-- Notiz -->
            <div class="mb-5">
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['return_note_optional'] ?? 'Rückgabe-Notiz (optional)'; ?></label>
                <input type="text" id="return-note" maxlength="200"
                       placeholder="<?php echo $lang['return_note_placeholder'] ?? 'z.B. Buchzustand, Schäden...'; ?>"
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-orange-500">
            </div>

            <!-- Fehler -->
            <div id="confirm-error" class="hidden mb-4 bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-700"></div>

            <!-- Buttons -->
            <div class="flex gap-3">
                <button onclick="submitReturn()" id="btn-return"
                        class="px-6 py-2 bg-orange-500 text-white rounded-md hover:bg-orange-600 font-medium">
                    <?php echo $lang['process_return'] ?? 'Rückgabe buchen'; ?>
                </button>
                <button onclick="resetConfirm()"
                        class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 text-sm">
                    <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                </button>
            </div>
        </div>
    </div>

    <!-- ════════════════════════════════════════════ SUCCESS-PANEL -->
    <div id="panel-success" class="hidden bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b bg-green-50 rounded-t-lg">
            <h2 class="text-lg font-semibold text-green-800">
                <svg class="inline w-5 h-5 mr-1 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                </svg>
                <?php echo $lang['return_success'] ?? 'Rückgabe erfolgreich gebucht'; ?>
            </h2>
        </div>
        <div class="p-5">
            <div id="success-details" class="bg-gray-50 rounded-lg p-4 mb-4 text-sm text-gray-800 space-y-1"></div>
            <div id="success-reservation" class="hidden mb-4 bg-blue-50 border border-blue-200 rounded-lg p-3 text-sm text-blue-800"></div>
            <div id="success-overdue-note" class="hidden mb-4 bg-orange-50 border border-orange-200 rounded-lg p-3 text-sm text-orange-800"></div>
            <div class="flex gap-3">
                <button onclick="nextReturn()"
                        class="px-5 py-2 bg-orange-500 text-white rounded-md hover:bg-orange-600 font-medium">
                    <?php echo $lang['next_return'] ?? 'Nächste Rückgabe'; ?>
                </button>
                <button onclick="window.location.href='lending-active.php'"
                        class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 text-sm">
                    <?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?>
                </button>
            </div>
        </div>
    </div>

</div>
</div>

<!-- ─── JavaScript ──────────────────────────────────────────────────────────── -->
<script>
const CSRF_TOKEN = '<?php echo esc_js($csrf_token); ?>';
let selectedLoan = null;

// i18n strings for JS
const i18n = {
    col_user:           '<?php echo esc_js($lang['users'] ?? 'Benutzer'); ?>',
    col_class:          '<?php echo esc_js($lang['class_name'] ?? 'Klasse'); ?>',
    col_title:          '<?php echo esc_js($lang['book_title'] ?? 'Titel'); ?>',
    col_signature:      '<?php echo esc_js($lang['signature'] ?? 'Signatur'); ?>',
    col_due:            '<?php echo esc_js($lang['loan_due_date'] ?? 'Fällig'); ?>',
    col_status:         '<?php echo esc_js($lang['status'] ?? 'Status'); ?>',
    col_action:         '<?php echo esc_js($lang['actions'] ?? 'Aktion'); ?>',
    btn_return:         '<?php echo esc_js($lang['return_book'] ?? 'Rückgabe'); ?>',
    process_return:     '<?php echo esc_js($lang['process_return'] ?? 'Rückgabe buchen'); ?>',
    processing:         '<?php echo esc_js($lang['processing'] ?? 'Wird gebucht...'); ?>',
    network_error:      '<?php echo esc_js($lang['network_error'] ?? 'Netzwerkfehler'); ?>',
    searching:          '<?php echo esc_js($lang['searching'] ?? 'Suche läuft...'); ?>',
    no_loan_found:      '<?php echo esc_js($lang['no_loan_found'] ?? 'Keine aktive Ausleihe gefunden.'); ?>',
    loans_found:        '<?php echo esc_js($lang['loans_found'] ?? 'aktive Ausleihe(n) gefunden'); ?>',
    user_label:         '<?php echo esc_js($lang['users'] ?? 'Benutzer'); ?>',
    book_copy_label:    '<?php echo esc_js($lang['book_copy_label'] ?? 'Buch / Exemplar'); ?>',
    loan_period_label:  '<?php echo esc_js($lang['loan_period'] ?? 'Ausleih-Zeitraum'); ?>',
    copy_number_label:  '<?php echo esc_js($lang['copy_number'] ?? 'Buchungsnummer'); ?>',
    loaned_on:          '<?php echo esc_js($lang['loan_loaned_on'] ?? 'Ausgeliehen'); ?>',
    due_label:          '<?php echo esc_js($lang['loan_due_date'] ?? 'Fällig'); ?>',
    extended_label:     '<?php echo esc_js($lang['extended_x_times'] ?? 'verlängert'); ?>',
    loan_id_label:      '<?php echo esc_js($lang['loan_id'] ?? 'Ausleihe-ID'); ?>',
    copy_status_label:  '<?php echo esc_js($lang['copy_status'] ?? 'Exemplar-Status'); ?>',
    reservation_hint:   '<?php echo esc_js($lang['reservation_hint'] ?? 'Vormerkung'); ?>',
    res_set_reserved:   '<?php echo esc_js($lang['res_set_reserved'] ?? 'Das Exemplar wurde auf "reserviert" gesetzt.'); ?>',
    res_reserved_for:   '<?php echo esc_js($lang['res_reserved_for'] ?? 'Dieses Buch ist vorgemerkt für'); ?>',
    overdue_days_note:  '<?php echo esc_js($lang['overdue_days_note'] ?? 'Tag(e) überfällig. Ggf. Mahnung prüfen.'); ?>',
    overdue_was:        '<?php echo esc_js($lang['overdue_was'] ?? 'Das Buch war'); ?>',
    overdue_days_ago:   '<?php echo esc_js($lang['overdue_days_ago'] ?? 'Tag(e) überfällig (fällig war'); ?>',
    signature_label:    '<?php echo esc_js($lang['signature'] ?? 'Signatur'); ?>',
    copy_no_label:      '<?php echo esc_js($lang['copy_number'] ?? 'Buchungsnr.'); ?>',
    book_label:         '<?php echo esc_js($lang['book_title'] ?? 'Buch'); ?>',
};

function escHtml(str) {
    if (str === null || str === undefined) return '';
    const d = document.createElement('div');
    d.textContent = String(str);
    return d.innerHTML;
}

// ─── Debounce ───────────────────────────────────────────────────────────────
let debounceTimer;
document.getElementById('searchInput').addEventListener('input', () => {
    clearTimeout(debounceTimer);
    debounceTimer = setTimeout(searchLoan, 400);
});
document.getElementById('searchInput').addEventListener('keydown', e => {
    if (e.key === 'Enter') { clearTimeout(debounceTimer); searchLoan(); }
});

// ─── Panel 1: Suche ─────────────────────────────────────────────────────────
function searchLoan() {
    const q = document.getElementById('searchInput').value.trim();
    const el = document.getElementById('search-results');
    if (q.length < 2) { el.innerHTML = ''; return; }

    el.innerHTML = spinner();

    fetch('lending-return.php?action=search_loan&q=' + encodeURIComponent(q))
        .then(r => r.json())
        .then(data => {
            if (data.error) { el.innerHTML = errorBox(data.error); return; }
            if (!data.length) {
                el.innerHTML = '<p class="text-sm text-gray-500 py-2">' + i18n.no_loan_found + '</p>';
                return;
            }
            // Genau 1 Treffer bei numerischer Suche → sofort auswählen
            const q2 = document.getElementById('searchInput').value.trim();
            if (data.length === 1 && /^\d{4,10}$/.test(q2)) {
                el.innerHTML = '';
                openConfirm(data[0]);
                return;
            }
            el.innerHTML = renderLoanTable(data);
        })
        .catch(err => { el.innerHTML = errorBox(i18n.network_error + ': ' + err.message); });
}

function renderLoanTable(loans) {
    let h = '<div class="overflow-x-auto"><table class="min-w-full text-sm border-collapse">';
    h += '<thead class="bg-gray-100"><tr>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_user + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_class + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_title + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_signature + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_due + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_status + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_action + '</th>';
    h += '</tr></thead><tbody>';

    loans.forEach(l => {
        const overdue = (l.days_overdue || 0) > 0;
        const dueFmt  = l.due_date ? l.due_date.split('-').reverse().join('.') : '-';
        h += '<tr class="hover:bg-gray-50' + (overdue ? ' bg-red-50' : '') + '">';
        h += '<td class="border px-3 py-2 font-medium">' + escHtml(l.last_name + ', ' + l.first_name) + '</td>';
        h += '<td class="border px-3 py-2">' + escHtml(l.class_name || '-') + '</td>';
        h += '<td class="border px-3 py-2 max-w-xs">';
        h += '<span class="line-clamp-2">' + escHtml(l.title) + '</span>';
        if (l.signature) h += '<span class="block text-xs text-gray-400 font-mono">' + escHtml(l.signature) + '</span>';
        h += '</td>';
        h += '<td class="border px-3 py-2 font-mono text-xs">' + escHtml(l.book_number || '-') + '</td>';
        h += '<td class="border px-3 py-2 text-center text-xs">' + escHtml(dueFmt) + '</td>';
        h += '<td class="border px-3 py-2 text-center">';
        if (overdue) {
            h += '<span class="px-2 py-0.5 text-xs font-medium rounded-full bg-red-100 text-red-800">';
            h += '+' + escHtml(l.days_overdue) + 'd überfällig</span>';
        } else {
            h += '<span class="px-2 py-0.5 text-xs font-medium rounded-full bg-green-100 text-green-800">aktiv</span>';
        }
        h += '</td>';
        h += '<td class="border px-3 py-2 text-center">';
        h += '<button onclick=\'openConfirm(' + JSON.stringify(l) + ')\' ';
        h += 'class="px-3 py-1 bg-orange-500 text-white text-xs rounded hover:bg-orange-600">' + i18n.btn_return + '</button>';
        h += '</td>';
        h += '</tr>';
    });

    h += '</tbody></table></div>';
    h += '<p class="mt-2 text-xs text-gray-500">' + loans.length + ' ' + i18n.loans_found + '</p>';
    return h;
}

// ─── Panel 2: Bestätigen ────────────────────────────────────────────────────
function openConfirm(loan) {
    selectedLoan = loan;

    const dueFmt  = loan.due_date  ? loan.due_date.split('-').reverse().join('.')  : '-';
    const loanFmt = loan.loan_date ? loan.loan_date.substring(0, 10).split('-').reverse().join('.') : '-';
    const overdue = (parseInt(loan.days_overdue) || 0) > 0;

    // Detailkarten
    let det = '';
    det += '<div class="bg-gray-50 rounded-lg p-4">';
    det += '<p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2">' + i18n.user_label + '</p>';
    det += '<p class="font-semibold text-gray-900">' + escHtml(loan.first_name + ' ' + loan.last_name) + '</p>';
    if (loan.class_name) det += '<p class="text-sm text-gray-500">' + escHtml(loan.class_name) + '</p>';
    det += '</div>';

    det += '<div class="bg-gray-50 rounded-lg p-4">';
    det += '<p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2">' + i18n.book_copy_label + '</p>';
    det += '<p class="font-semibold text-gray-900">' + escHtml(loan.title) + '</p>';
    if (loan.author) det += '<p class="text-sm text-gray-500">' + escHtml(loan.author) + '</p>';
    if (loan.signature) det += '<p class="text-xs font-mono text-gray-400 mt-1">' + escHtml(loan.signature) + '</p>';
    det += '</div>';

    det += '<div class="bg-gray-50 rounded-lg p-4">';
    det += '<p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2">' + i18n.loan_period_label + '</p>';
    det += '<p class="text-sm text-gray-700"><span class="font-medium">' + i18n.loaned_on + ':</span> ' + escHtml(loanFmt) + '</p>';
    det += '<p class="text-sm ' + (overdue ? 'text-red-700 font-semibold' : 'text-gray-700') + '">';
    det += '<span class="font-medium">' + i18n.due_label + ':</span> ' + escHtml(dueFmt);
    if (overdue) det += ' <span class="text-red-600">(+' + escHtml(loan.days_overdue) + ' Tage)</span>';
    det += '</p>';
    if (loan.extensions > 0) {
        det += '<p class="text-xs text-gray-500 mt-1">' + escHtml(loan.extensions) + '&times; ' + i18n.extended_label + '</p>';
    }
    det += '</div>';

    det += '<div class="bg-gray-50 rounded-lg p-4">';
    det += '<p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2">' + i18n.copy_number_label + '</p>';
    det += '<p class="font-mono text-gray-900">' + escHtml(loan.book_number || '-') + '</p>';
    det += '<p class="text-xs text-gray-400 mt-1">' + i18n.loan_id_label + ' #' + escHtml(loan.loan_id) + '</p>';
    det += '</div>';

    document.getElementById('loan-details').innerHTML = det;

    // Überfällig-Warnung
    const warnEl = document.getElementById('overdue-warning');
    const warnTxt = document.getElementById('overdue-text');
    if (overdue) {
        warnTxt.textContent = loan.days_overdue + ' ' + i18n.overdue_days_ago + ' ' + dueFmt + ')';
        warnEl.classList.remove('hidden');
    } else {
        warnEl.classList.add('hidden');
    }

    document.getElementById('return-note').value = '';
    document.getElementById('confirm-error').classList.add('hidden');
    document.getElementById('btn-return').disabled = false;
    document.getElementById('btn-return').textContent = i18n.process_return;

    document.getElementById('panel-confirm').classList.remove('hidden');
    document.getElementById('panel-success').classList.add('hidden');
    document.getElementById('panel-confirm').scrollIntoView({behavior: 'smooth', block: 'start'});
}

function resetConfirm() {
    selectedLoan = null;
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.add('hidden');
}

// ─── Rückgabe absenden ───────────────────────────────────────────────────────
function submitReturn() {
    if (!selectedLoan) return;

    const errEl = document.getElementById('confirm-error');
    errEl.classList.add('hidden');

    const btn = document.getElementById('btn-return');
    btn.disabled = true;
    btn.textContent = i18n.processing;

    const fd = new FormData();
    fd.append('action',      'process_return');
    fd.append('csrf_token',  CSRF_TOKEN);
    fd.append('loan_id',     selectedLoan.loan_id);
    fd.append('return_note', document.getElementById('return-note').value);

    fetch('lending-return.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            btn.disabled = false;
            btn.textContent = i18n.process_return;
            if (data.error) {
                errEl.textContent = data.error;
                errEl.classList.remove('hidden');
                return;
            }
            showSuccess(data);
        })
        .catch(err => {
            btn.disabled = false;
            btn.textContent = i18n.process_return;
            errEl.textContent = i18n.network_error + ': ' + err.message;
            errEl.classList.remove('hidden');
        });
}

// ─── Success-Panel ───────────────────────────────────────────────────────────
function showSuccess(data) {
    const dueFmt = data.due_date
        ? (typeof data.due_date === 'string'
            ? data.due_date.substring(0, 10).split('-').reverse().join('.')
            : data.due_date)
        : (selectedLoan && selectedLoan.due_date
            ? selectedLoan.due_date.substring(0, 10).split('-').reverse().join('.')
            : '-');

    let det = '';
    det += '<p><strong>' + i18n.col_user + ':</strong> ' + escHtml(data.user_name) + '</p>';
    det += '<p><strong>' + i18n.book_label + ':</strong> ' + escHtml(data.book_title) + '</p>';
    det += '<p><strong>' + i18n.signature_label + ':</strong> ' + escHtml(data.signature || '-') + '</p>';
    det += '<p><strong>' + i18n.copy_no_label + ':</strong> ' + escHtml(data.book_number || '-') + '</p>';
    det += '<p><strong>' + i18n.loan_id_label + ':</strong> #' + escHtml(data.loan_id) + '</p>';
    det += '<p><strong>' + i18n.copy_status_label + ':</strong> '
        + (data.new_copy_status === 'reserviert'
            ? '<span class="text-orange-600 font-medium">reserviert</span>'
            : '<span class="text-green-600 font-medium">verfügbar</span>')
        + '</p>';
    document.getElementById('success-details').innerHTML = det;

    // Vormerkungshinweis
    const resEl = document.getElementById('success-reservation');
    if (data.reservation_user) {
        resEl.innerHTML = '<strong>' + i18n.reservation_hint + ':</strong> ' + i18n.res_reserved_for + ' '
            + escHtml(data.reservation_user) + '. ' + i18n.res_set_reserved;
        resEl.classList.remove('hidden');
    } else {
        resEl.classList.add('hidden');
    }

    // Überfällig-Hinweis
    const overdueEl = document.getElementById('success-overdue-note');
    if (data.was_overdue) {
        overdueEl.textContent = i18n.overdue_was + ' ' + (data.days_overdue || '?')
            + ' ' + i18n.overdue_days_note;
        overdueEl.classList.remove('hidden');
    } else {
        overdueEl.classList.add('hidden');
    }

    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.remove('hidden');
    document.getElementById('panel-success').scrollIntoView({behavior: 'smooth', block: 'start'});
    selectedLoan = null;
}

function nextReturn() {
    document.getElementById('panel-success').classList.add('hidden');
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('search-results').innerHTML = '';
    document.getElementById('searchInput').value = '';
    document.getElementById('searchInput').focus();
    window.scrollTo({top: 0, behavior: 'smooth'});
}

// ─── UI-Hilfsfunktionen ──────────────────────────────────────────────────────
function spinner() {
    return '<div class="flex items-center gap-2 py-3 text-sm text-gray-500">'
         + '<div class="animate-spin rounded-full h-5 w-5 border-b-2 border-orange-500"></div>'
         + '<span>' + i18n.searching + '</span></div>';
}

function errorBox(msg) {
    return '<div class="bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-700">'
         + escHtml(msg) + '</div>';
}

// Fokus beim Laden; optional ?q= aus lending-active.php vorausfuellen
document.getElementById('searchInput').focus();
(function() {
    const urlQ = new URLSearchParams(window.location.search).get('q');
    if (urlQ && urlQ.length >= 2) {
        document.getElementById('searchInput').value = urlQ;
        searchLoan();
    }
})();
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
