<?php
/**
 * Vormerkungen verwalten — lending-reservations.php
 * Übersicht, Stornierung, Ablaufmarkierung, Ausleihe aus Vormerkung heraus.
 *
 * AJAX-Actions (GET, kein CSRF): search_user, search_book
 * AJAX-Actions (POST, CSRF):     cancel_reservation, expire_reservation,
 *                                checkout_reservation, create_reservation
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = $current_user['id'];

// ─── AJAX-Dispatcher ──────────────────────────────────────────────────────────

$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($action !== '') {
    header('Content-Type: application/json; charset=utf-8');

    switch ($action) {

        // ── search_user (GET) — fuer Neue-Vormerkung-Panel ──────────────────
        case 'search_user':
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }
            $like = '%' . $q . '%';
            $rows = db_get_results(
                "SELECT u.id, u.first_name, u.last_name, u.class_name, u.student_id, u.status
                 FROM users u
                 WHERE u.status != 'inaktiv'
                   AND (u.first_name LIKE ? OR u.last_name LIKE ?
                        OR CONCAT(u.last_name,' ',u.first_name) LIKE ?
                        OR CONCAT(u.first_name,' ',u.last_name) LIKE ?
                        OR u.class_name LIKE ? OR u.student_id LIKE ?)
                 ORDER BY u.last_name, u.first_name
                 LIMIT 20",
                [$like, $like, $like, $like, $like, $like]
            );
            echo json_encode($rows);
            exit;

        // ── search_book (GET) — fuer Neue-Vormerkung-Panel ──────────────────
        case 'search_book':
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }
            $like = '%' . $q . '%';
            // Buchtitel-Suche (Titel-Ebene, nicht Exemplar-Ebene)
            $rows = db_get_results(
                "SELECT b.id AS book_id, b.title, b.author, b.isbn,
                        (SELECT COUNT(*) FROM book_copies bc
                         WHERE bc.book_id = b.id AND bc.status = 'verfügbar') AS copies_available,
                        (SELECT COUNT(*) FROM book_copies bc
                         WHERE bc.book_id = b.id AND bc.status != 'aussortiert') AS copies_total
                 FROM books b
                 WHERE b.title LIKE ? OR b.author LIKE ? OR b.isbn LIKE ?
                 ORDER BY b.title
                 LIMIT 20",
                [$like, $like, $like]
            );
            echo json_encode($rows);
            exit;

        // ── create_reservation (POST, CSRF) ──────────────────────────────────
        case 'create_reservation':
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            $user_id = (int)($_POST['user_id'] ?? 0);
            $book_id = (int)($_POST['book_id'] ?? 0);
            if (!$user_id || !$book_id) {
                echo json_encode(['error' => 'Benutzer und Buch erforderlich']); exit;
            }
            // Duplikat-Prüfung
            $existing = db_get_row(
                "SELECT id FROM reservations
                 WHERE user_id = ? AND book_id = ? AND status IN ('aktiv','verfügbar')",
                [$user_id, $book_id]
            );
            if ($existing) {
                echo json_encode(['error' => 'Dieser Benutzer hat bereits eine aktive Vormerkung für dieses Buch.']); exit;
            }

            // Ist bereits ein Exemplar verfügbar? → Direkt 'verfügbar' setzen
            $avail_copy = db_get_row(
                "SELECT id FROM book_copies WHERE book_id = ? AND status = 'verfügbar' LIMIT 1",
                [$book_id]
            );
            $status   = $avail_copy ? 'verfügbar' : 'aktiv';
            $copy_id  = $avail_copy ? $avail_copy['id'] : null;

            if ($avail_copy) {
                db_update('book_copies', ['status' => 'reserviert'], 'id = ?', [$avail_copy['id']]);
            }

            $res_id = db_insert('reservations', [
                'book_id'       => $book_id,
                'user_id'       => $user_id,
                'reserved_date' => date('Y-m-d H:i:s'),
                'status'        => $status,
                'copy_id'       => $copy_id,
                'notified_date' => $status === 'verfügbar' ? date('Y-m-d H:i:s') : null,
            ]);
            if (!$res_id) { echo json_encode(['error' => 'Datenbankfehler']); exit; }

            log_activity($admin_id, 'reservation_created', 'reservations', $res_id,
                json_encode(['user_id' => $user_id, 'book_id' => $book_id, 'status' => $status]));

            echo json_encode(['success' => true, 'reservation_id' => $res_id, 'status' => $status]);
            exit;

        // ── cancel_reservation (POST, CSRF) ──────────────────────────────────
        case 'cancel_reservation':
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            $res_id = (int)($_POST['reservation_id'] ?? 0);
            if (!$res_id) { echo json_encode(['error' => 'Keine ID']); exit; }
            $res = db_get_row(
                "SELECT id, book_id, copy_id, status FROM reservations WHERE id = ?", [$res_id]
            );
            if (!$res || !in_array($res['status'], ['aktiv', 'verfügbar'], true)) {
                echo json_encode(['error' => 'Vormerkung nicht gefunden oder nicht aktiv']); exit;
            }
            db_update('reservations', ['status' => 'storniert'], 'id = ?', [$res_id]);
            release_copy_or_forward($res['copy_id'], $res['book_id']);
            log_activity($admin_id, 'reservation_cancelled', 'reservations', $res_id, '');
            echo json_encode(['success' => true]);
            exit;

        // ── expire_reservation (POST, CSRF) ──────────────────────────────────
        case 'expire_reservation':
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            $res_id = (int)($_POST['reservation_id'] ?? 0);
            if (!$res_id) { echo json_encode(['error' => 'Keine ID']); exit; }
            $res = db_get_row(
                "SELECT id, book_id, copy_id, status FROM reservations WHERE id = ?", [$res_id]
            );
            if (!$res || !in_array($res['status'], ['aktiv', 'verfügbar'], true)) {
                echo json_encode(['error' => 'Vormerkung nicht gefunden oder nicht aktiv']); exit;
            }
            db_update('reservations', ['status' => 'abgelaufen'], 'id = ?', [$res_id]);
            release_copy_or_forward($res['copy_id'], $res['book_id']);
            log_activity($admin_id, 'reservation_expired', 'reservations', $res_id, '');
            echo json_encode(['success' => true]);
            exit;

        // ── checkout_reservation (POST, CSRF, Transaction) ───────────────────
        case 'checkout_reservation':
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            handle_reservation_checkout($admin_id);
            exit;

        default:
            http_response_code(400);
            echo json_encode(['error' => 'Unbekannte Aktion']);
            exit;
    }
}

// ─── Hilfsfunktion: Exemplar freigeben oder Folge-Vormerkung bedienen ─────────

/**
 * Wenn eine Vormerkung mit zugewiesenem Exemplar storniert/abläuft:
 * Prüfe ob es eine nächste wartende Vormerkung für dasselbe Buch gibt.
 * Wenn ja → Exemplar an diese Vormerkung übergeben.
 * Wenn nein → Exemplar auf 'verfügbar' setzen.
 */
function release_copy_or_forward(?int $copy_id, int $book_id): void {
    if (!$copy_id) return;

    // Nur handeln wenn das Exemplar aktuell 'reserviert' ist
    $copy = db_get_row("SELECT status FROM book_copies WHERE id = ?", [$copy_id]);
    if (!$copy || $copy['status'] !== 'reserviert') return;

    // Nächste wartende Vormerkung für dieses Buch
    $next = db_get_row(
        "SELECT id, user_id FROM reservations
         WHERE book_id = ? AND status = 'aktiv'
         ORDER BY reserved_date ASC LIMIT 1",
        [$book_id]
    );

    if ($next) {
        db_update('reservations', [
            'status'        => 'verfügbar',
            'copy_id'       => $copy_id,
            'notified_date' => date('Y-m-d H:i:s'),
        ], 'id = ?', [$next['id']]);
        // Exemplar bleibt 'reserviert' für die nächste Vormerkung
    } else {
        db_update('book_copies', ['status' => 'verfügbar'], 'id = ?', [$copy_id]);
    }
}

// ─── Ausleihe aus Vormerkung ──────────────────────────────────────────────────

function handle_reservation_checkout(int $admin_id): void {
    global $pdo;

    $res_id = (int)($_POST['reservation_id'] ?? 0);
    if (!$res_id) { echo json_encode(['error' => 'Keine Reservation-ID']); return; }

    $res = db_get_row(
        "SELECT r.id, r.book_id, r.user_id, r.copy_id, r.status,
                u.first_name, u.last_name, u.status AS user_status,
                ug.loan_duration_days,
                bc.book_number, bc.signature, bc.status AS copy_status,
                b.title
         FROM reservations r
         JOIN users u ON r.user_id = u.id
         LEFT JOIN user_groups ug ON u.user_group_id = ug.id
         JOIN books b ON r.book_id = b.id
         LEFT JOIN book_copies bc ON r.copy_id = bc.id
         WHERE r.id = ?",
        [$res_id]
    );

    if (!$res) { echo json_encode(['error' => 'Vormerkung nicht gefunden']); return; }
    if ($res['status'] !== 'verfügbar') {
        echo json_encode(['error' => 'Exemplar noch nicht bereit (Status: ' . $res['status'] . ')']); return;
    }
    if (!$res['copy_id']) {
        echo json_encode(['error' => 'Kein Exemplar zugewiesen']); return;
    }
    if ($res['user_status'] === 'inaktiv') {
        echo json_encode(['error' => 'Inaktive Benutzer können nicht ausleihen']); return;
    }
    if ($res['copy_status'] !== 'reserviert') {
        echo json_encode(['error' => 'Exemplar ist nicht mehr reserviert (Status: ' . $res['copy_status'] . ')']); return;
    }

    $loan_days = (int)($res['loan_duration_days'] ?? 0) ?: (int)getSetting('loan_duration_days', 28);
    $due_date  = date('Y-m-d', strtotime('+' . $loan_days . ' days'));
    $note      = trim($_POST['loan_note'] ?? '');

    try {
        $pdo->beginTransaction();

        $loan_id = db_insert('loans', [
            'copy_id'   => $res['copy_id'],
            'user_id'   => $res['user_id'],
            'loan_date' => date('Y-m-d H:i:s'),
            'due_date'  => $due_date,
            'status'    => 'aktiv',
            'loan_note' => $note ?: null,
            'loaned_by' => $admin_id,
        ]);
        if (!$loan_id) { $pdo->rollBack(); echo json_encode(['error' => 'Loan-Insert fehlgeschlagen']); return; }

        $stmt = $pdo->prepare(
            "UPDATE book_copies SET status = 'ausgeliehen'
             WHERE id = ? AND status = 'reserviert'"
        );
        $stmt->execute([$res['copy_id']]);
        if ($stmt->rowCount() !== 1) {
            $pdo->rollBack();
            echo json_encode(['error' => 'Exemplar konnte nicht auf ausgeliehen gesetzt werden (Race Condition).']); return;
        }

        $pdo->prepare("UPDATE reservations SET status = 'abgeholt' WHERE id = ?")
            ->execute([$res_id]);

        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['error' => 'Datenbankfehler: ' . $e->getMessage()]); return;
    }

    log_activity($admin_id, 'reservation_checkout', 'loans', $loan_id,
        json_encode(['reservation_id' => $res_id, 'user' => $res['first_name'] . ' ' . $res['last_name'],
                     'book' => $res['title'], 'due_date' => $due_date]));

    echo json_encode([
        'success'   => true,
        'loan_id'   => $loan_id,
        'user_name' => $res['first_name'] . ' ' . $res['last_name'],
        'book_title'=> $res['title'],
        'due_date'  => $due_date,
        'due_fmt'   => date('d.m.Y', strtotime($due_date)),
    ]);
}

// ─── Statistiken ──────────────────────────────────────────────────────────────

$stat_aktiv    = (int)db_get_var("SELECT COUNT(*) FROM reservations WHERE status = 'aktiv'");
$stat_ready    = (int)db_get_var("SELECT COUNT(*) FROM reservations WHERE status = 'verfügbar'");
$stat_history  = (int)db_get_var("SELECT COUNT(*) FROM reservations WHERE status IN ('abgeholt','storniert','abgelaufen')");

// ─── Filter + Haupt-Query ─────────────────────────────────────────────────────

$filter = in_array($_GET['filter'] ?? '', ['all', 'active', 'ready', 'history']) ? $_GET['filter'] : 'all';

$where_status = match ($filter) {
    'active'  => "r.status = 'aktiv'",
    'ready'   => "r.status = 'verfügbar'",
    'history' => "r.status IN ('abgeholt','storniert','abgelaufen')",
    default   => "r.status IN ('aktiv','verfügbar')",
};

$reservations = db_get_results(
    "SELECT r.id AS reservation_id, r.book_id, r.user_id, r.reserved_date,
            r.valid_until, r.notified_date, r.status, r.copy_id,
            u.first_name, u.last_name, u.class_name, u.email,
            b.title, b.author,
            bc.book_number, bc.signature,
            (SELECT COUNT(*) FROM reservations r2
             WHERE r2.book_id = r.book_id AND r2.status = 'aktiv'
               AND r2.reserved_date < r.reserved_date) + 1 AS queue_position
     FROM reservations r
     JOIN users u ON r.user_id = u.id
     JOIN books b ON r.book_id = b.id
     LEFT JOIN book_copies bc ON r.copy_id = bc.id
     WHERE $where_status
     ORDER BY
       FIELD(r.status,'verfügbar','aktiv','abgeholt','storniert','abgelaufen'),
       r.reserved_date ASC"
);

// ─── Default Loan Days ────────────────────────────────────────────────────────
$default_loan_days = (int)getSetting('loan_duration_days', 28);

// ─── Seiten-Setup ─────────────────────────────────────────────────────────────

$lang = load_language();

$page_title = $lang['reservations_title'] ?? 'Vormerkungen';
$csrf_token = generate_csrf_token();

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['reservations_title'] ?? 'Vormerkungen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['reservations_subtitle'] ?? 'Buchvormerkungen verwalten und bearbeiten'; ?></p>
        </div>
        <div class="flex gap-3">
            <button onclick="toggleNewPanel()"
                    class="px-4 py-2 bg-blue-600 text-white text-sm rounded-md hover:bg-blue-700 font-medium">
                <?php echo $lang['new_reservation_btn'] ?? '+ Neue Vormerkung'; ?>
            </button>
            <a href="lending-active.php" class="px-4 py-2 border border-gray-300 text-gray-700 text-sm rounded-md hover:bg-gray-50">
                <?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?>
            </a>
        </div>
    </div>

    <!-- ═══════════════════════════════ Neue Vormerkung anlegen (Panel) -->
    <div id="newResPanel" class="hidden bg-white rounded-lg shadow mb-6 border border-blue-200">
        <div class="p-5 border-b bg-blue-50 flex items-center justify-between">
            <h2 class="font-semibold text-blue-900"><?php echo $lang['new_reservation_panel_title'] ?? 'Neue Vormerkung anlegen'; ?></h2>
            <button onclick="toggleNewPanel()" class="text-blue-400 hover:text-blue-600 text-xl leading-none">&times;</button>
        </div>
        <div class="p-5 grid grid-cols-1 md:grid-cols-2 gap-6">

            <!-- Benutzersuche -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['res_user_label'] ?? 'Benutzer'; ?></label>
                <div class="relative">
                    <input type="text" id="newUserInput" placeholder="<?php echo esc_attr($lang['res_user_placeholder'] ?? 'Name, Klasse oder Ausweis-Nr....'); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                </div>
                <div id="newUserResults" class="mt-2"></div>
                <div id="newUserSelected" class="hidden mt-2 bg-blue-50 border border-blue-200 rounded p-2 text-sm flex items-center justify-between">
                    <span id="newUserName" class="font-medium text-blue-900"></span>
                    <button onclick="clearUser()" class="text-blue-400 hover:text-blue-600 text-xs ml-2"><?php echo $lang['res_switch_user'] ?? 'wechseln'; ?></button>
                </div>
                <input type="hidden" id="newUserId">
            </div>

            <!-- Buchsuche -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['res_book_label'] ?? 'Buch'; ?></label>
                <div class="relative">
                    <input type="text" id="newBookInput" placeholder="<?php echo esc_attr($lang['res_book_placeholder'] ?? 'Titel, Autor oder ISBN...'); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                </div>
                <div id="newBookResults" class="mt-2"></div>
                <div id="newBookSelected" class="hidden mt-2 bg-blue-50 border border-blue-200 rounded p-2 text-sm flex items-center justify-between">
                    <span id="newBookName" class="font-medium text-blue-900"></span>
                    <button onclick="clearBook()" class="text-blue-400 hover:text-blue-600 text-xs ml-2"><?php echo $lang['res_switch_book'] ?? 'wechseln'; ?></button>
                </div>
                <input type="hidden" id="newBookId">
            </div>

        </div>
        <div class="px-5 pb-5 flex items-center gap-3">
            <button onclick="submitNewReservation()" id="btnCreateRes"
                    class="px-5 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium text-sm disabled:opacity-50"
                    disabled>
                <?php echo $lang['res_create_btn'] ?? 'Vormerkung anlegen'; ?>
            </button>
            <div id="newResError" class="text-sm text-red-600 hidden"></div>
            <div id="newResSuccess" class="text-sm text-green-700 hidden"></div>
        </div>
    </div>

    <!-- ═══════════════════════════════════════════ Statistik-Kacheln -->
    <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 mb-6">

        <a href="?filter=all"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'all' ? 'ring-2 ring-blue-500' : ''; ?>">
            <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold <?php echo $stat_aktiv > 0 ? 'text-blue-600' : 'text-gray-900'; ?>">
                    <?php echo $stat_aktiv; ?>
                </p>
                <p class="text-sm text-gray-500"><?php echo $lang['res_stat_active_waiting'] ?? 'Aktiv wartend'; ?></p>
            </div>
        </a>

        <a href="?filter=ready"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'ready' ? 'ring-2 ring-green-500' : ''; ?>">
            <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold <?php echo $stat_ready > 0 ? 'text-green-600' : 'text-gray-900'; ?>">
                    <?php echo $stat_ready; ?>
                </p>
                <p class="text-sm text-gray-500"><?php echo $lang['res_stat_ready_pickup'] ?? 'Bereit zur Abholung'; ?></p>
            </div>
        </a>

        <a href="?filter=history"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'history' ? 'ring-2 ring-gray-500' : ''; ?>">
            <div class="w-12 h-12 bg-gray-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold text-gray-900"><?php echo $stat_history; ?></p>
                <p class="text-sm text-gray-500"><?php echo $lang['res_stat_completed'] ?? 'Abgeschlossen'; ?></p>
            </div>
        </a>

    </div>

    <!-- ═══════════════════════════════════════════ Tabelle -->
    <div class="bg-white rounded-lg shadow overflow-hidden">

        <!-- Filter-Tabs -->
        <div class="px-4 pt-4 border-b flex gap-2 flex-wrap">
            <?php
            $tabs = [
                'all'     => ($lang['res_tab_all']     ?? 'Aktiv & Bereit') . ' (' . ($stat_aktiv + $stat_ready) . ')',
                'active'  => ($lang['res_tab_waiting'] ?? 'Wartend')        . ' (' . $stat_aktiv . ')',
                'ready'   => ($lang['res_tab_ready']   ?? 'Bereit')         . ' (' . $stat_ready . ')',
                'history' => ($lang['res_tab_history'] ?? 'Verlauf')        . ' (' . $stat_history . ')',
            ];
            foreach ($tabs as $key => $label):
            ?>
                <a href="?filter=<?php echo $key; ?>"
                   class="pb-3 px-1 text-sm font-medium border-b-2 transition
                          <?php echo $filter === $key
                              ? 'border-blue-600 text-blue-600'
                              : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?>">
                    <?php echo esc_html($label); ?>
                </a>
            <?php endforeach; ?>
        </div>

        <?php if (empty($reservations)): ?>
            <div class="text-center py-16">
                <svg class="mx-auto h-12 w-12 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                          d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                </svg>
                <p class="mt-3 text-gray-500 font-medium"><?php echo $lang['res_none_found'] ?? 'Keine Vormerkungen gefunden.'; ?></p>
            </div>

        <?php else: ?>
        <div class="overflow-x-auto">
        <table class="min-w-full">
            <thead>
                <tr class="bg-gray-700 text-white text-sm">
                    <th class="px-4 py-3 text-left font-medium"><?php echo $lang['res_col_user'] ?? 'Benutzer'; ?></th>
                    <th class="px-4 py-3 text-left font-medium"><?php echo $lang['res_col_book'] ?? 'Buch'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['res_col_reserved_date'] ?? 'Vorgemerkt'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['status'] ?? 'Status'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['res_col_position'] ?? 'Position / Notiz'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['actions'] ?? 'Aktionen'; ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
            <?php foreach ($reservations as $res):
                $status   = $res['status'];
                $res_fmt  = $res['reserved_date']
                    ? date('d.m.Y', strtotime($res['reserved_date'])) : '-';
                $notif_fmt = $res['notified_date']
                    ? date('d.m.Y', strtotime($res['notified_date'])) : null;

                $row_bg = match ($status) {
                    'verfügbar' => 'bg-green-50 hover:bg-green-100',
                    'aktiv'     => 'hover:bg-gray-50',
                    'abgelaufen', 'storniert' => 'bg-gray-50',
                    default     => 'hover:bg-gray-50',
                };
            ?>
                <tr class="<?php echo $row_bg; ?> transition-colors" id="res-<?php echo $res['reservation_id']; ?>">

                    <!-- Benutzer -->
                    <td class="px-4 py-3">
                        <p class="font-medium text-sm text-gray-900">
                            <?php echo esc_html($res['last_name'] . ', ' . $res['first_name']); ?>
                        </p>
                        <?php if ($res['class_name']): ?>
                            <span class="inline-block px-1.5 py-0.5 bg-gray-100 text-gray-600 text-xs rounded mt-0.5">
                                <?php echo esc_html($res['class_name']); ?>
                            </span>
                        <?php endif; ?>
                        <?php if ($res['email']): ?>
                            <p class="text-xs text-gray-400 mt-0.5 truncate max-w-[160px]"
                               title="<?php echo esc_attr($res['email']); ?>">
                                <?php echo esc_html($res['email']); ?>
                            </p>
                        <?php endif; ?>
                    </td>

                    <!-- Buch -->
                    <td class="px-4 py-3">
                        <p class="text-sm font-medium text-gray-900 truncate max-w-xs"
                           title="<?php echo esc_attr($res['title']); ?>">
                            <?php echo esc_html($res['title']); ?>
                        </p>
                        <?php if ($res['author']): ?>
                            <p class="text-xs text-gray-500"><?php echo esc_html($res['author']); ?></p>
                        <?php endif; ?>
                        <?php if ($res['signature']): ?>
                            <p class="text-xs font-mono text-gray-400 mt-0.5"><?php echo esc_html($res['signature']); ?></p>
                        <?php endif; ?>
                    </td>

                    <!-- Vorgemerkt am -->
                    <td class="px-4 py-3 text-center text-sm text-gray-600 whitespace-nowrap">
                        <?php echo esc_html($res_fmt); ?>
                        <?php if ($notif_fmt && $status === 'verfügbar'): ?>
                            <p class="text-xs text-green-600 mt-0.5"><?php echo $lang['res_notified_label'] ?? 'Benachrichtigt:'; ?> <?php echo esc_html($notif_fmt); ?></p>
                        <?php endif; ?>
                    </td>

                    <!-- Status-Badge -->
                    <td class="px-4 py-3 text-center">
                        <?php
                        $badge = match ($status) {
                            'aktiv'      => ['bg-blue-100 text-blue-800',   $lang['res_badge_waiting']   ?? 'wartend'],
                            'verfügbar'  => ['bg-green-100 text-green-800', $lang['res_badge_ready']     ?? 'bereit'],
                            'abgeholt'   => ['bg-gray-100 text-gray-600',   $lang['res_badge_picked_up'] ?? 'abgeholt'],
                            'abgelaufen' => ['bg-red-100 text-red-700',     $lang['res_badge_expired']   ?? 'abgelaufen'],
                            'storniert'  => ['bg-gray-100 text-gray-500',   $lang['res_badge_cancelled'] ?? 'storniert'],
                            default      => ['bg-gray-100 text-gray-600',   $status],
                        };
                        ?>
                        <span class="px-2 py-1 text-xs font-medium rounded-full <?php echo $badge[0]; ?>">
                            <?php echo esc_html($badge[1]); ?>
                        </span>
                    </td>

                    <!-- Position / Info -->
                    <td class="px-4 py-3 text-center">
                        <?php if ($status === 'aktiv'): ?>
                            <span class="text-sm font-semibold text-blue-700">
                                <?php echo $lang['res_position_label'] ?? 'Position'; ?> <?php echo esc_html($res['queue_position']); ?>
                            </span>
                        <?php elseif ($status === 'verfügbar'): ?>
                            <p class="text-xs text-green-700 font-medium">
                                <?php echo esc_html($res['book_number'] ?? ($lang['res_copy_assigned'] ?? 'Exemplar zugewiesen')); ?>
                            </p>
                        <?php else: ?>
                            <span class="text-xs text-gray-400">—</span>
                        <?php endif; ?>
                    </td>

                    <!-- Aktionen -->
                    <td class="px-4 py-3" id="actions-<?php echo $res['reservation_id']; ?>">
                        <div class="flex flex-col gap-1 items-center">

                        <?php if ($status === 'verfügbar'): ?>
                            <!-- Bereit zur Abholung: Ausleihe buchen -->
                            <button onclick="checkoutReservation(<?php echo $res['reservation_id']; ?>, '<?php echo esc_js($res['first_name'] . ' ' . $res['last_name']); ?>', '<?php echo esc_js($res['title']); ?>')"
                                    class="px-3 py-1 bg-green-600 text-white text-xs rounded hover:bg-green-700 whitespace-nowrap w-full">
                                <?php echo $lang['res_checkout_btn'] ?? 'Ausleihe buchen'; ?>
                            </button>
                            <button onclick="expireReservation(<?php echo $res['reservation_id']; ?>)"
                                    class="px-3 py-1 bg-gray-200 text-gray-700 text-xs rounded hover:bg-gray-300 whitespace-nowrap w-full">
                                <?php echo $lang['res_not_picked_up_btn'] ?? 'Nicht abgeholt'; ?>
                            </button>

                        <?php elseif ($status === 'aktiv'): ?>
                            <!-- Wartend: Stornieren -->
                            <button onclick="cancelReservation(<?php echo $res['reservation_id']; ?>)"
                                    class="px-3 py-1 bg-red-100 text-red-700 text-xs rounded hover:bg-red-200 whitespace-nowrap w-full">
                                <?php echo $lang['reservation_cancel'] ?? 'Stornieren'; ?>
                            </button>

                        <?php else: ?>
                            <span class="text-xs text-gray-400"><?php echo $lang['res_completed_label'] ?? 'abgeschlossen'; ?></span>
                        <?php endif; ?>

                        </div>
                    </td>

                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        </div>

        <div class="px-4 py-3 border-t bg-gray-50 text-sm text-gray-600">
            <?php echo count($reservations); ?> <?php echo count($reservations) !== 1 ? ($lang['res_count_plural'] ?? 'Vormerkungen angezeigt') : ($lang['res_count_singular'] ?? 'Vormerkung angezeigt'); ?>
        </div>
        <?php endif; ?>
    </div>

</div>
</div>

<!-- ─── Ausleihe-Bestätigungs-Modal ─────────────────────────────────────────── -->
<div id="checkoutModal" class="hidden fixed inset-0 bg-black bg-opacity-40 z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl shadow-xl p-6 max-w-md w-full mx-4">
        <h3 class="text-lg font-bold text-gray-900 mb-1"><?php echo $lang['res_modal_title'] ?? 'Ausleihe aus Vormerkung'; ?></h3>
        <p class="text-sm text-gray-500 mb-4" id="checkoutModalDesc"></p>

        <div class="mb-4">
            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['res_modal_note_label'] ?? 'Notiz (optional)'; ?></label>
            <input type="text" id="checkoutNote" maxlength="200" placeholder="<?php echo esc_attr($lang['res_modal_note_placeholder'] ?? 'z.B. Buchzustand...'); ?>"
                   class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-green-500">
        </div>

        <div id="checkoutModalError" class="hidden mb-3 text-sm text-red-700 bg-red-50 border border-red-200 rounded p-2"></div>
        <div id="checkoutModalSuccess" class="hidden mb-3 text-sm text-green-700 bg-green-50 border border-green-200 rounded p-2"></div>

        <div class="flex gap-3">
            <button onclick="confirmCheckout()" id="btnConfirmCheckout"
                    class="px-5 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 font-medium text-sm">
                <?php echo $lang['res_checkout_btn'] ?? 'Ausleihe buchen'; ?>
            </button>
            <button onclick="closeCheckoutModal()"
                    class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 text-sm">
                <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
            </button>
        </div>
    </div>
</div>

<!-- ─── JavaScript ──────────────────────────────────────────────────────────── -->
<script>
const CSRF_TOKEN        = '<?php echo esc_js($csrf_token); ?>';
const DEFAULT_LOAN_DAYS = <?php echo $default_loan_days; ?>;

// Translated strings for JS
const LANG = {
    confirmCancel:   '<?php echo addslashes($lang['res_js_confirm_cancel']  ?? 'Vormerkung wirklich stornieren?'); ?>',
    confirmExpire:   '<?php echo addslashes($lang['res_js_confirm_expire']  ?? 'Vormerkung als "nicht abgeholt" / abgelaufen markieren?'); ?>',
    errorPrefix:     '<?php echo addslashes($lang['res_js_error_prefix']    ?? 'Fehler:'); ?>',
    networkError:    '<?php echo addslashes($lang['res_js_network_error']   ?? 'Netzwerkfehler:'); ?>',
    checkoutBtn:     '<?php echo addslashes($lang['res_checkout_btn']       ?? 'Ausleihe buchen'); ?>',
    bookingProgress: '<?php echo addslashes($lang['res_js_booking']         ?? 'Wird gebucht...'); ?>',
    searching:       '<?php echo addslashes($lang['res_js_searching']       ?? 'Suche...'); ?>',
    noResults:       '<?php echo addslashes($lang['res_js_no_results']      ?? 'Keine Treffer'); ?>',
    availableCount:  '<?php echo addslashes($lang['res_js_available_count'] ?? 'verfügbar'); ?>',
    allLoaned:       '<?php echo addslashes($lang['res_js_all_loaned']      ?? 'alle ausgeliehen'); ?>',
    bookedMsg:       '<?php echo addslashes($lang['res_js_booked']          ?? 'Ausleihe #%s gebucht – Rückgabe bis %s'); ?>',
    successAvailable:'<?php echo addslashes($lang['res_success_created_available'] ?? 'Vormerkung angelegt – Exemplar sofort verfügbar!'); ?>',
    successWaiting:  '<?php echo addslashes($lang['res_success_created_waiting']   ?? 'Vormerkung angelegt – Benutzer wartet in der Warteschlange.'); ?>',
};

let pendingCheckoutId = null;

function escHtml(str) {
    if (str === null || str === undefined) return '';
    const d = document.createElement('div');
    d.textContent = String(str);
    return d.innerHTML;
}

// ─── Aktionen: Stornieren / Abgelaufen ──────────────────────────────────────
function cancelReservation(resId) {
    if (!confirm(LANG.confirmCancel)) return;
    postAction('cancel_reservation', {reservation_id: resId}, resId);
}

function expireReservation(resId) {
    if (!confirm(LANG.confirmExpire)) return;
    postAction('expire_reservation', {reservation_id: resId}, resId);
}

function postAction(action, extra, resId) {
    const fd = new FormData();
    fd.append('action',     action);
    fd.append('csrf_token', CSRF_TOKEN);
    for (const [k, v] of Object.entries(extra)) fd.append(k, v);

    fetch('lending-reservations.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.error) { alert(LANG.errorPrefix + ' ' + data.error); return; }
            const row = document.getElementById('res-' + resId);
            if (row) {
                row.style.opacity = '0.4';
                row.style.transition = 'opacity 0.4s';
                setTimeout(() => row.remove(), 500);
            }
        })
        .catch(err => alert(LANG.networkError + ' ' + err.message));
}

// ─── Ausleihe aus Vormerkung ─────────────────────────────────────────────────
function checkoutReservation(resId, userName, bookTitle) {
    pendingCheckoutId = resId;
    document.getElementById('checkoutModalDesc').textContent =
        userName + ' · ' + bookTitle;
    document.getElementById('checkoutNote').value = '';
    document.getElementById('checkoutModalError').classList.add('hidden');
    document.getElementById('checkoutModalSuccess').classList.add('hidden');
    document.getElementById('btnConfirmCheckout').disabled = false;
    document.getElementById('btnConfirmCheckout').textContent = LANG.checkoutBtn;
    document.getElementById('checkoutModal').classList.remove('hidden');
}

function closeCheckoutModal() {
    document.getElementById('checkoutModal').classList.add('hidden');
    pendingCheckoutId = null;
}

function confirmCheckout() {
    if (!pendingCheckoutId) return;
    const btn    = document.getElementById('btnConfirmCheckout');
    const errEl  = document.getElementById('checkoutModalError');
    const succEl = document.getElementById('checkoutModalSuccess');
    errEl.classList.add('hidden');
    btn.disabled = true;
    btn.textContent = LANG.bookingProgress;

    const fd = new FormData();
    fd.append('action',          'checkout_reservation');
    fd.append('csrf_token',      CSRF_TOKEN);
    fd.append('reservation_id',  pendingCheckoutId);
    fd.append('loan_note',       document.getElementById('checkoutNote').value);

    fetch('lending-reservations.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            btn.disabled = false;
            btn.textContent = LANG.checkoutBtn;
            if (data.error) {
                errEl.textContent = data.error;
                errEl.classList.remove('hidden');
                return;
            }
            succEl.textContent = LANG.bookedMsg.replace('%s', data.loan_id).replace('%s', data.due_fmt);
            succEl.classList.remove('hidden');
            // Zeile als abgeholt markieren
            const row = document.getElementById('res-' + pendingCheckoutId);
            if (row) {
                row.style.opacity = '0.4';
                row.style.transition = 'opacity 0.4s';
                setTimeout(() => row.remove(), 800);
            }
            setTimeout(closeCheckoutModal, 2000);
        })
        .catch(err => {
            btn.disabled = false;
            btn.textContent = LANG.checkoutBtn;
            errEl.textContent = LANG.networkError + ' ' + err.message;
            errEl.classList.remove('hidden');
        });
}

// ─── Neue Vormerkung anlegen ─────────────────────────────────────────────────
function toggleNewPanel() {
    const p = document.getElementById('newResPanel');
    p.classList.toggle('hidden');
    if (!p.classList.contains('hidden')) document.getElementById('newUserInput').focus();
}

// Debounce
let uTimer, bTimer;
document.getElementById('newUserInput').addEventListener('input', function() {
    clearTimeout(uTimer); uTimer = setTimeout(() => searchForNew('user', this.value), 400);
});
document.getElementById('newBookInput').addEventListener('input', function() {
    clearTimeout(bTimer); bTimer = setTimeout(() => searchForNew('book', this.value), 400);
});

function searchForNew(type, q) {
    const el = document.getElementById(type === 'user' ? 'newUserResults' : 'newBookResults');
    if (q.trim().length < 2) { el.innerHTML = ''; return; }

    const url = 'lending-reservations.php?action=search_' + type + '&q=' + encodeURIComponent(q);
    el.innerHTML = '<p class="text-xs text-gray-400 py-1">' + LANG.searching + '</p>';
    fetch(url)
        .then(r => r.json())
        .then(data => {
            if (!data.length) { el.innerHTML = '<p class="text-xs text-gray-400 py-1">' + LANG.noResults + '</p>'; return; }
            let h = '<div class="border border-gray-200 rounded text-sm divide-y max-h-40 overflow-y-auto">';
            data.forEach(item => {
                if (type === 'user') {
                    h += '<div class="px-3 py-2 hover:bg-blue-50 cursor-pointer flex justify-between items-center"'
                       + ' onclick=\'selectUser(' + JSON.stringify(item) + ')\'>'
                       + '<span>' + escHtml(item.last_name + ', ' + item.first_name)
                       + (item.class_name ? ' <span class="text-gray-400 text-xs">(' + escHtml(item.class_name) + ')</span>' : '')
                       + '</span>'
                       + '<span class="text-xs px-1.5 py-0.5 rounded-full '
                       + (item.status === 'aktiv' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700')
                       + '">' + escHtml(item.status) + '</span>'
                       + '</div>';
                } else {
                    const avail = parseInt(item.copies_available) || 0;
                    h += '<div class="px-3 py-2 hover:bg-blue-50 cursor-pointer"'
                       + ' onclick=\'selectBook(' + JSON.stringify(item) + ')\'>'
                       + '<p class="font-medium">' + escHtml(item.title) + '</p>'
                       + '<p class="text-xs text-gray-500">' + escHtml(item.author || '-')
                       + ' <span class="ml-2 ' + (avail > 0 ? 'text-green-600' : 'text-orange-500') + '">'
                       + (avail > 0 ? avail + ' ' + LANG.availableCount : LANG.allLoaned) + '</span></p>'
                       + '</div>';
                }
            });
            h += '</div>';
            el.innerHTML = h;
        });
}

function selectUser(u) {
    document.getElementById('newUserId').value = u.id;
    document.getElementById('newUserName').textContent = u.last_name + ', ' + u.first_name + (u.class_name ? ' (' + u.class_name + ')' : '');
    document.getElementById('newUserSelected').classList.remove('hidden');
    document.getElementById('newUserResults').innerHTML = '';
    document.getElementById('newUserInput').value = '';
    checkCreateReady();
}
function clearUser() {
    document.getElementById('newUserId').value = '';
    document.getElementById('newUserSelected').classList.add('hidden');
    checkCreateReady();
}
function selectBook(b) {
    document.getElementById('newBookId').value = b.book_id;
    document.getElementById('newBookName').textContent = b.title;
    document.getElementById('newBookSelected').classList.remove('hidden');
    document.getElementById('newBookResults').innerHTML = '';
    document.getElementById('newBookInput').value = '';
    checkCreateReady();
}
function clearBook() {
    document.getElementById('newBookId').value = '';
    document.getElementById('newBookSelected').classList.add('hidden');
    checkCreateReady();
}
function checkCreateReady() {
    const btn = document.getElementById('btnCreateRes');
    btn.disabled = !(document.getElementById('newUserId').value && document.getElementById('newBookId').value);
}

function submitNewReservation() {
    const userId = document.getElementById('newUserId').value;
    const bookId = document.getElementById('newBookId').value;
    const errEl  = document.getElementById('newResError');
    const succEl = document.getElementById('newResSuccess');
    errEl.classList.add('hidden');
    succEl.classList.add('hidden');

    const fd = new FormData();
    fd.append('action',     'create_reservation');
    fd.append('csrf_token', CSRF_TOKEN);
    fd.append('user_id',    userId);
    fd.append('book_id',    bookId);

    fetch('lending-reservations.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                errEl.textContent = data.error;
                errEl.classList.remove('hidden');
                return;
            }
            const statusMsg = data.status === 'verfügbar'
                ? LANG.successAvailable
                : LANG.successWaiting;
            succEl.textContent = statusMsg;
            succEl.classList.remove('hidden');
            clearUser(); clearBook();
            setTimeout(() => window.location.reload(), 1800);
        })
        .catch(err => {
            errEl.textContent = LANG.networkError + ' ' + err.message;
            errEl.classList.remove('hidden');
        });
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
