<?php
/**
 * Mahnungen versenden — lending-reminders.php
 * Zeigt alle überfälligen Ausleihen; ermöglicht Einzel- und Sammel-Mahnungen per E-Mail.
 *
 * AJAX-Actions (POST, CSRF): send_reminders
 * E-Mail: PHPMailer (SMTP) wenn vendor/autoload.php vorhanden + smtp_enabled=true,
 *         sonst PHP mail() als Fallback.
 * Protokoll: Jede gesendete Mahnung wird in der Tabelle `reminders` geloggt.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = $current_user['id'];

// ─── E-Mail-Hilfsfunktion ─────────────────────────────────────────────────────

/**
 * Sendet eine Mahnung. Gibt [success: bool, error: string] zurück.
 * Benutzt PHPMailer (SMTP) wenn konfiguriert, sonst PHP mail().
 */
function send_reminder_email(
    string $to,
    string $subject,
    string $body_plain,
    string $from_email,
    string $from_name
): array {
    $autoload = __DIR__ . '/../vendor/autoload.php';
    $smtp_enabled = getSetting('smtp_enabled', 'false') === 'true';

    if ($smtp_enabled && file_exists($autoload)) {
        require_once $autoload;
        try {
            $mail = new \PHPMailer\PHPMailer\PHPMailer(true);
            $mail->isSMTP();
            $mail->Host       = getSetting('smtp_host', '');
            $mail->SMTPAuth   = true;
            $mail->Username   = getSetting('smtp_username', '');
            $mail->Password   = getSetting('smtp_password', '');
            $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            $mail->Port       = (int)getSetting('smtp_port', '587');
            $mail->CharSet    = 'UTF-8';

            $mail->setFrom($from_email, $from_name);
            $mail->addAddress($to);
            $mail->Subject = $subject;
            $mail->Body    = $body_plain;
            $mail->isHTML(false);
            $mail->send();
            return ['success' => true, 'error' => ''];
        } catch (\Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // Fallback: PHP mail()
    $headers  = 'From: ' . $from_name . ' <' . $from_email . ">\r\n";
    $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
    $headers .= "Content-Transfer-Encoding: 8bit\r\n";
    $ok = @mail($to, '=?UTF-8?B?' . base64_encode($subject) . '?=', $body_plain, $headers);
    return $ok
        ? ['success' => true,  'error' => '']
        : ['success' => false, 'error' => 'PHP mail() fehlgeschlagen (Serverconfig prüfen)'];
}

/**
 * Erstellt den deutschen Mahnungs-E-Mail-Text.
 */
function build_reminder_body(array $loan): string {
    $library_name = getSetting('library_name', 'Schulbibliothek');
    $library_email = getSetting('library_email', '');
    $due_fmt  = date('d.m.Y', strtotime($loan['due_date']));
    $days     = max(1, (int)$loan['days_overdue']);
    $salutation = 'Sehr geehrte' . ('' /* kein Geschlecht bekannt */) . 'r ' . $loan['first_name'] . ' ' . $loan['last_name'];

    $body  = $salutation . ",\n\n";
    $body .= "hiermit erinnern wir Sie daran, dass das folgende Buch der {$library_name}\n";
    $body .= "bereits seit {$days} " . ($days === 1 ? 'Tag' : 'Tagen') . " überfällig ist.\n\n";
    $body .= "Wir bitten Sie, das Buch baldmöglichst in der Schulbibliothek abzugeben.\n\n";
    $body .= str_repeat('─', 46) . "\n";
    $body .= "AUSGELIEHENES BUCH\n";
    $body .= str_repeat('─', 46) . "\n";
    $body .= sprintf("%-16s %s\n", 'Titel:',       $loan['title']);
    if (!empty($loan['author'])) {
        $body .= sprintf("%-16s %s\n", 'Autor:',    $loan['author']);
    }
    if (!empty($loan['signature'])) {
        $body .= sprintf("%-16s %s\n", 'Signatur:', $loan['signature']);
    }
    if (!empty($loan['book_number'])) {
        $body .= sprintf("%-16s %s\n", 'Buchungsnr.:', $loan['book_number']);
    }
    $body .= sprintf("%-16s %s\n", 'Fällig am:',  $due_fmt);
    $body .= sprintf("%-16s %s\n", 'Überfällig:',
        $days . ' ' . ($days === 1 ? 'Tag' : 'Tage'));
    $body .= str_repeat('─', 46) . "\n\n";
    $body .= "Sollten Sie das Buch bereits zurückgegeben haben, betrachten Sie\n";
    $body .= "diese Nachricht bitte als gegenstandslos.\n\n";
    $body .= "Mit freundlichen Grüßen,\n";
    $body .= $library_name . "\n";
    if ($library_email) $body .= $library_email . "\n";

    return $body;
}

// ─── AJAX-Dispatcher ──────────────────────────────────────────────────────────

$action = $_POST['action'] ?? '';

if ($action === 'send_reminders') {
    header('Content-Type: application/json; charset=utf-8');

    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
    }

    $loan_ids = array_map('intval', (array)($_POST['loan_ids'] ?? []));
    $loan_ids = array_filter($loan_ids);

    if (empty($loan_ids)) {
        echo json_encode(['error' => 'Keine Ausleihen ausgewählt']); exit;
    }

    $from_email = getSetting('smtp_from_email') ?: getSetting('library_email', 'noreply@schule.at');
    $from_name  = getSetting('smtp_from_name') ?: getSetting('library_name', 'Schulbibliothek');
    $subject    = getSetting('library_name', 'Schulbibliothek') . ' – Mahnung: Buch bitte zurückgeben';

    $results = [];

    foreach ($loan_ids as $loan_id) {
        // Ausleihe mit Benutzer + Buch laden
        $loan = db_get_row(
            "SELECT l.id AS loan_id, l.loan_date, l.due_date,
                    DATEDIFF(CURDATE(), l.due_date) AS days_overdue,
                    u.id AS user_id, u.first_name, u.last_name, u.email,
                    bc.book_number, bc.signature,
                    b.title, b.author
             FROM loans l
             JOIN users u    ON l.user_id  = u.id
             JOIN book_copies bc ON l.copy_id = bc.id
             JOIN books b    ON bc.book_id  = b.id
             WHERE l.id = ? AND l.return_date IS NULL AND l.due_date < CURDATE()",
            [$loan_id]
        );

        if (!$loan) {
            $results[] = [
                'loan_id' => $loan_id,
                'success' => false,
                'email'   => '',
                'error'   => 'Ausleihe nicht gefunden oder nicht überfällig',
            ];
            continue;
        }

        if (empty($loan['email'])) {
            // Kein E-Mail → als nicht gesendet protokollieren
            $reminder_id = db_insert('reminders', [
                'loan_id'       => $loan_id,
                'user_id'       => $loan['user_id'],
                'sent_date'     => date('Y-m-d H:i:s'),
                'email_to'      => '',
                'email_subject' => $subject,
                'email_body'    => '(Keine E-Mail-Adresse vorhanden)',
                'status'        => 'fehlgeschlagen',
                'error_message' => 'Keine E-Mail-Adresse hinterlegt',
            ]);
            $results[] = [
                'loan_id'   => $loan_id,
                'success'   => false,
                'email'     => '',
                'error'     => 'Keine E-Mail-Adresse hinterlegt',
                'user_name' => $loan['first_name'] . ' ' . $loan['last_name'],
            ];
            continue;
        }

        $body   = build_reminder_body($loan);
        $result = send_reminder_email($loan['email'], $subject, $body, $from_email, $from_name);

        $status = $result['success'] ? 'versendet' : 'fehlgeschlagen';
        $reminder_id = db_insert('reminders', [
            'loan_id'       => $loan_id,
            'user_id'       => $loan['user_id'],
            'sent_date'     => date('Y-m-d H:i:s'),
            'email_to'      => $loan['email'],
            'email_subject' => $subject,
            'email_body'    => $body,
            'status'        => $status,
            'error_message' => $result['error'] ?: null,
        ]);

        log_activity($admin_id, 'reminder_sent', 'loans', $loan_id,
            json_encode(['user' => $loan['first_name'] . ' ' . $loan['last_name'],
                         'email' => $loan['email'], 'status' => $status]));

        $results[] = [
            'loan_id'    => $loan_id,
            'success'    => $result['success'],
            'email'      => $loan['email'],
            'error'      => $result['error'],
            'user_name'  => $loan['first_name'] . ' ' . $loan['last_name'],
            'sent_date'  => date('d.m.Y H:i'),
        ];
    }

    echo json_encode(['results' => $results]);
    exit;
}

// ─── Statistik-Kacheln ────────────────────────────────────────────────────────

$stat_total = (int)db_get_var(
    "SELECT COUNT(*) FROM loans WHERE return_date IS NULL AND due_date < CURDATE()"
);
$stat_never = (int)db_get_var(
    "SELECT COUNT(*) FROM loans l
     WHERE l.return_date IS NULL AND l.due_date < CURDATE()
       AND NOT EXISTS (
           SELECT 1 FROM reminders r
           WHERE r.loan_id = l.id AND r.status = 'versendet'
       )"
);
$stat_old = (int)db_get_var(
    "SELECT COUNT(DISTINCT l.id) FROM loans l
     JOIN reminders r ON r.loan_id = l.id AND r.status = 'versendet'
     WHERE l.return_date IS NULL AND l.due_date < CURDATE()
       AND r.sent_date < DATE_SUB(NOW(), INTERVAL 7 DAY)
       AND NOT EXISTS (
           SELECT 1 FROM reminders r2
           WHERE r2.loan_id = l.id AND r2.status = 'versendet'
             AND r2.sent_date >= DATE_SUB(NOW(), INTERVAL 7 DAY)
       )"
);
$stat_ready = $stat_never + $stat_old;

// ─── Filter + Haupt-Query ─────────────────────────────────────────────────────

$filter = in_array($_GET['filter'] ?? '', ['all', 'no_reminder', 'needed']) ? $_GET['filter'] : 'all';

$having = match ($filter) {
    'no_reminder' => 'HAVING reminder_count = 0',
    'needed'      => 'HAVING reminder_count = 0 OR (last_reminder_date IS NOT NULL AND last_reminder_date < DATE_SUB(NOW(), INTERVAL 7 DAY))',
    default       => '',
};

$loans = db_get_results(
    "SELECT l.id AS loan_id, l.loan_date, l.due_date,
            DATEDIFF(CURDATE(), l.due_date) AS days_overdue,
            u.id AS user_id, u.first_name, u.last_name, u.class_name, u.email,
            bc.id AS copy_id, bc.book_number, bc.signature,
            b.id AS book_id, b.title, b.author,
            (SELECT MAX(r.sent_date) FROM reminders r
             WHERE r.loan_id = l.id AND r.status = 'versendet') AS last_reminder_date,
            (SELECT COUNT(*) FROM reminders r
             WHERE r.loan_id = l.id AND r.status = 'versendet') AS reminder_count
     FROM loans l
     JOIN users u    ON l.user_id  = u.id
     JOIN book_copies bc ON l.copy_id = bc.id
     JOIN books b    ON bc.book_id  = b.id
     WHERE l.return_date IS NULL AND l.due_date < CURDATE()
     $having
     ORDER BY l.due_date ASC"
);

// ─── E-Mail-Konfigurationsstatus ──────────────────────────────────────────────

$smtp_enabled   = getSetting('smtp_enabled', 'false') === 'true';
$smtp_host      = getSetting('smtp_host', '');
$from_email_cfg = getSetting('smtp_from_email') ?: getSetting('library_email', '');
$mail_configured = ($smtp_enabled && $smtp_host && $from_email_cfg)
                || (!$smtp_enabled && $from_email_cfg);
$has_vendor = file_exists(__DIR__ . '/../vendor/autoload.php');

// ─── Seiten-Setup ─────────────────────────────────────────────────────────────

$page_title = $lang['admin_reminders'] ?? 'Mahnungen versenden';
$csrf_token = generate_csrf_token();

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['admin_reminders'] ?? 'Mahnungen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['reminders_subtitle'] ?? 'Überfällige Ausleihen per E-Mail mahnen'; ?></p>
        </div>
        <div class="flex gap-3 text-sm">
            <a href="lending-active.php" class="text-gray-500 hover:text-gray-700">&larr; <?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?></a>
            <a href="lending-return.php" class="px-4 py-2 bg-orange-500 text-white rounded-md hover:bg-orange-600 font-medium">
                <?php echo $lang['process_return'] ?? 'Rückgabe buchen'; ?>
            </a>
        </div>
    </div>

    <!-- E-Mail-Konfigurations-Warnung -->
    <?php if (!$mail_configured): ?>
    <div class="mb-6 bg-yellow-50 border border-yellow-300 rounded-lg p-4 flex items-start gap-3">
        <svg class="w-5 h-5 text-yellow-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                  d="M12 9v2m0 4h.01M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
        </svg>
        <div>
            <p class="font-semibold text-yellow-800"><?php echo $lang['email_not_configured'] ?? 'E-Mail nicht konfiguriert'; ?></p>
            <p class="text-sm text-yellow-700 mt-0.5">
                <?php echo $lang['reminders_email_not_configured_hint'] ?? 'Mahnungen können nicht gesendet werden. Bitte'; ?>
                <a href="settings-system.php#smtp" class="underline font-medium"><?php echo $lang['configure_email_smtp'] ?? 'E-Mail/SMTP konfigurieren'; ?></a>.
            </p>
        </div>
    </div>
    <?php elseif ($smtp_enabled && !$has_vendor): ?>
    <div class="mb-6 bg-blue-50 border border-blue-200 rounded-lg p-4 flex items-start gap-3">
        <svg class="w-5 h-5 text-blue-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
        </svg>
        <div>
            <p class="font-semibold text-blue-800"><?php echo $lang['smtp_active_phpmailer_missing'] ?? 'SMTP aktiv – PHPMailer nicht installiert'; ?></p>
            <p class="text-sm text-blue-700 mt-0.5">
                <?php echo $lang['smtp_fallback_hint'] ?? 'Fallback auf PHP'; ?> <code class="bg-blue-100 px-1 rounded">mail()</code>.
                <?php echo $lang['composer_install_hint'] ?? 'Für SMTP bitte'; ?> <code class="bg-blue-100 px-1 rounded">composer install</code> <?php echo $lang['run_it'] ?? 'ausführen'; ?>.
            </p>
        </div>
    </div>
    <?php endif; ?>

    <!-- ════════════════════════════════════════ Statistik-Kacheln -->
    <div class="grid grid-cols-2 sm:grid-cols-4 gap-4 mb-6">

        <a href="?filter=all"
           class="bg-white rounded-lg shadow p-4 flex flex-col items-center hover:bg-gray-50 transition
                  <?php echo $filter === 'all' ? 'ring-2 ring-red-500' : ''; ?>">
            <p class="text-3xl font-bold <?php echo $stat_total > 0 ? 'text-red-600' : 'text-gray-900'; ?>">
                <?php echo $stat_total; ?>
            </p>
            <p class="text-xs text-gray-500 mt-1 text-center"><?php echo $lang['overdue_total'] ?? 'Überfällig gesamt'; ?></p>
        </a>

        <a href="?filter=no_reminder"
           class="bg-white rounded-lg shadow p-4 flex flex-col items-center hover:bg-gray-50 transition
                  <?php echo $filter === 'no_reminder' ? 'ring-2 ring-orange-500' : ''; ?>">
            <p class="text-3xl font-bold <?php echo $stat_never > 0 ? 'text-orange-600' : 'text-gray-900'; ?>">
                <?php echo $stat_never; ?>
            </p>
            <p class="text-xs text-gray-500 mt-1 text-center"><?php echo $lang['no_reminder_yet'] ?? 'Noch keine Mahnung'; ?></p>
        </a>

        <a href="?filter=needed"
           class="bg-white rounded-lg shadow p-4 flex flex-col items-center hover:bg-gray-50 transition
                  <?php echo $filter === 'needed' ? 'ring-2 ring-yellow-500' : ''; ?>">
            <p class="text-3xl font-bold <?php echo $stat_old > 0 ? 'text-yellow-600' : 'text-gray-900'; ?>">
                <?php echo $stat_old; ?>
            </p>
            <p class="text-xs text-gray-500 mt-1 text-center"><?php echo $lang['reminder_needed_again'] ?? 'Erneut fällig (>7d)'; ?></p>
        </a>

        <div class="bg-white rounded-lg shadow p-4 flex flex-col items-center">
            <p class="text-3xl font-bold <?php echo $stat_ready > 0 ? 'text-blue-600' : 'text-gray-900'; ?>">
                <?php echo $stat_ready; ?>
            </p>
            <p class="text-xs text-gray-500 mt-1 text-center"><?php echo $lang['ready_to_send'] ?? 'Bereit zum Senden'; ?></p>
        </div>

    </div>

    <!-- ════════════════════════════════════════ Tabelle + Bulk-Aktion -->
    <div class="bg-white rounded-lg shadow overflow-hidden">

        <!-- Tabellen-Header mit Bulk-Aktionen -->
        <div class="px-4 py-3 border-b flex items-center justify-between flex-wrap gap-3 bg-gray-50">

            <!-- Filter-Tabs -->
            <div class="flex gap-2 text-sm">
                <a href="?filter=all"
                   class="px-3 py-1.5 rounded-md font-medium transition
                          <?php echo $filter === 'all' ? 'bg-gray-700 text-white' : 'text-gray-600 hover:bg-gray-200'; ?>">
                    <?php echo $lang['all_overdue'] ?? 'Alle überfälligen'; ?> (<?php echo $stat_total; ?>)
                </a>
                <a href="?filter=no_reminder"
                   class="px-3 py-1.5 rounded-md font-medium transition
                          <?php echo $filter === 'no_reminder' ? 'bg-gray-700 text-white' : 'text-gray-600 hover:bg-gray-200'; ?>">
                    <?php echo $lang['no_reminder_yet'] ?? 'Noch keine Mahnung'; ?> (<?php echo $stat_never; ?>)
                </a>
                <a href="?filter=needed"
                   class="px-3 py-1.5 rounded-md font-medium transition
                          <?php echo $filter === 'needed' ? 'bg-gray-700 text-white' : 'text-gray-600 hover:bg-gray-200'; ?>">
                    <?php echo $lang['reminder_needed'] ?? 'Mahnung nötig'; ?> (<?php echo $stat_never + $stat_old; ?>)
                </a>
            </div>

            <!-- Bulk-Aktionen -->
            <div class="flex items-center gap-3">
                <label class="flex items-center gap-2 text-sm text-gray-600 cursor-pointer select-none">
                    <input type="checkbox" id="selectAll" onchange="toggleAll(this)"
                           class="w-4 h-4 rounded border-gray-300 text-blue-600">
                    <?php echo $lang['select_all'] ?? 'Alle auswählen'; ?>
                </label>
                <button id="btnBulkSend" onclick="bulkSend()"
                        class="px-4 py-1.5 bg-red-600 text-white text-sm rounded-md hover:bg-red-700 font-medium
                               opacity-50 cursor-not-allowed"
                        disabled
                        <?php echo !$mail_configured ? 'title="' . esc_attr($lang['email_not_configured'] ?? 'E-Mail nicht konfiguriert') . '"' : ''; ?>>
                    <span id="bulkBtnLabel"><?php echo $lang['send_reminders_count'] ?? 'Mahnungen senden'; ?> (0)</span>
                </button>
            </div>
        </div>

        <?php if (empty($loans)): ?>
            <div class="text-center py-16">
                <svg class="mx-auto h-12 w-12 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                          d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
                <p class="mt-3 text-gray-500 font-medium"><?php echo $lang['no_overdue_loans'] ?? 'Keine überfälligen Ausleihen'; ?><?php echo $filter !== 'all' ? ' ' . ($lang['with_this_filter'] ?? 'mit diesem Filter') : ''; ?>.</p>
                <?php if ($filter !== 'all'): ?>
                    <a href="?filter=all" class="mt-2 inline-block text-sm text-blue-600 hover:underline"><?php echo $lang['show_all'] ?? 'Alle anzeigen'; ?></a>
                <?php endif; ?>
            </div>

        <?php else: ?>
        <div class="overflow-x-auto">
        <table class="min-w-full">
            <thead>
                <tr class="bg-gray-700 text-white text-sm">
                    <th class="px-4 py-3 text-center w-10">
                        <span class="sr-only"><?php echo $lang['selection'] ?? 'Auswahl'; ?></span>
                    </th>
                    <th class="px-4 py-3 text-left font-medium"><?php echo $lang['users'] ?? 'Benutzer'; ?></th>
                    <th class="px-4 py-3 text-left font-medium"><?php echo ($lang['book_title'] ?? 'Titel') . ' / ' . ($lang['signature'] ?? 'Signatur'); ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['loan_due_date'] ?? 'Fällig'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['overdue'] ?? 'Überfällig'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['last_reminder'] ?? 'Letzte Mahnung'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['actions'] ?? 'Aktion'; ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100" id="remindersTable">
            <?php foreach ($loans as $loan):
                $has_email   = !empty($loan['email']);
                $days_od     = (int)$loan['days_overdue'];
                $due_fmt     = date('d.m.Y', strtotime($loan['due_date']));
                $last_fmt    = $loan['last_reminder_date']
                    ? date('d.m.Y', strtotime($loan['last_reminder_date']))
                    : null;
                $count       = (int)$loan['reminder_count'];
                // Mehr als 7 Tage → erneut nötig; oder noch nie gesendet
                $recently_reminded = $loan['last_reminder_date']
                    && (strtotime($loan['last_reminder_date']) > strtotime('-7 days'));
                $row_bg = $days_od > 14 ? 'bg-red-50 hover:bg-red-100' : 'hover:bg-gray-50';
            ?>
                <tr class="<?php echo $row_bg; ?> transition-colors" id="row-<?php echo $loan['loan_id']; ?>">

                    <!-- Checkbox -->
                    <td class="px-4 py-3 text-center">
                        <input type="checkbox"
                               class="loan-cb w-4 h-4 rounded border-gray-300 text-red-600"
                               value="<?php echo $loan['loan_id']; ?>"
                               <?php echo (!$has_email || !$mail_configured) ? 'disabled title="' . (!$has_email ? esc_attr($lang['no_email_address'] ?? 'Keine E-Mail') : esc_attr($lang['email_not_configured'] ?? 'E-Mail nicht konfiguriert')) . '"' : ''; ?>
                               onchange="updateBulkBtn()">
                    </td>

                    <!-- Benutzer -->
                    <td class="px-4 py-3">
                        <p class="font-medium text-sm text-gray-900">
                            <?php echo esc_html($loan['last_name'] . ', ' . $loan['first_name']); ?>
                        </p>
                        <?php if ($loan['class_name']): ?>
                            <span class="inline-block px-1.5 py-0.5 bg-gray-100 text-gray-600 text-xs rounded mt-0.5">
                                <?php echo esc_html($loan['class_name']); ?>
                            </span>
                        <?php endif; ?>
                        <p class="text-xs mt-0.5">
                            <?php if ($has_email): ?>
                                <span class="text-green-600" title="<?php echo esc_attr($loan['email']); ?>">
                                    &#9993; <?php echo esc_html($loan['email']); ?>
                                </span>
                            <?php else: ?>
                                <span class="text-red-500">&#9993; <?php echo $lang['no_email_address'] ?? 'Keine E-Mail-Adresse'; ?></span>
                            <?php endif; ?>
                        </p>
                    </td>

                    <!-- Titel / Signatur -->
                    <td class="px-4 py-3">
                        <p class="text-sm font-medium text-gray-900 max-w-xs truncate"
                           title="<?php echo esc_attr($loan['title']); ?>">
                            <?php echo esc_html($loan['title']); ?>
                        </p>
                        <?php if ($loan['signature']): ?>
                            <p class="text-xs font-mono text-gray-400 mt-0.5"><?php echo esc_html($loan['signature']); ?></p>
                        <?php endif; ?>
                    </td>

                    <!-- Fällig am -->
                    <td class="px-4 py-3 text-center whitespace-nowrap text-sm text-gray-700">
                        <?php echo esc_html($due_fmt); ?>
                    </td>

                    <!-- Überfällig -->
                    <td class="px-4 py-3 text-center">
                        <span class="px-2 py-1 text-xs font-bold rounded-full
                            <?php echo $days_od > 14 ? 'bg-red-100 text-red-800' : 'bg-orange-100 text-orange-800'; ?>">
                            +<?php echo esc_html($days_od); ?> <?php echo $days_od === 1 ? ($lang['day_singular'] ?? 'Tag') : ($lang['days_label'] ?? 'Tage'); ?>
                        </span>
                    </td>

                    <!-- Letzte Mahnung -->
                    <td class="px-4 py-3 text-center" id="lastrem-<?php echo $loan['loan_id']; ?>">
                        <?php if ($last_fmt): ?>
                            <p class="text-xs font-medium <?php echo $recently_reminded ? 'text-green-700' : 'text-gray-600'; ?>">
                                <?php echo esc_html($last_fmt); ?>
                            </p>
                            <p class="text-xs text-gray-400"><?php echo esc_html($count); ?>&times; <?php echo $lang['sent_label'] ?? 'gesendet'; ?></p>
                        <?php else: ?>
                            <span class="text-xs text-gray-400">—</span>
                        <?php endif; ?>
                    </td>

                    <!-- Aktion -->
                    <td class="px-4 py-3 text-center" id="action-<?php echo $loan['loan_id']; ?>">
                        <?php if ($mail_configured && $has_email): ?>
                            <button onclick="sendSingle(<?php echo $loan['loan_id']; ?>)"
                                    class="px-3 py-1 bg-red-600 text-white text-xs rounded hover:bg-red-700 whitespace-nowrap">
                                <?php echo $lang['send_reminder'] ?? 'Mahnung senden'; ?>
                            </button>
                        <?php elseif (!$has_email): ?>
                            <span class="text-xs text-gray-400"><?php echo $lang['no_email_address'] ?? 'Keine E-Mail'; ?></span>
                        <?php else: ?>
                            <span class="text-xs text-gray-400"><?php echo $lang['email_not_configured'] ?? 'E-Mail nicht konfiguriert'; ?></span>
                        <?php endif; ?>
                    </td>

                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        </div>

        <div class="px-4 py-3 border-t bg-gray-50 text-sm text-gray-600">
            <?php echo count($loans); ?> <?php echo $lang['overdue_loans_displayed'] ?? 'überfällige Ausleihe(n) angezeigt'; ?>
            <?php if ($filter !== 'all'): ?>
                · <a href="?filter=all" class="text-blue-600 hover:underline"><?php echo $lang['show_all'] ?? 'Alle'; ?> <?php echo $stat_total; ?> <?php echo $lang['show_label'] ?? 'anzeigen'; ?></a>
            <?php endif; ?>
        </div>

        <?php endif; ?>
    </div>

</div>
</div>

<!-- ─── Sende-Fortschritts-Overlay ──────────────────────────────────────────── -->
<div id="sendOverlay" class="hidden fixed inset-0 bg-black bg-opacity-40 z-50 flex items-center justify-center">
    <div class="bg-white rounded-xl shadow-xl p-6 max-w-sm w-full mx-4">
        <h3 class="text-lg font-bold text-gray-900 mb-1"><?php echo $lang['sending_reminders'] ?? 'Mahnungen werden gesendet…'; ?></h3>
        <p class="text-sm text-gray-500 mb-4" id="overlayStatus"><?php echo $lang['please_wait'] ?? 'Bitte warten.'; ?></p>
        <div class="w-full bg-gray-200 rounded-full h-2">
            <div id="overlayBar" class="bg-red-500 h-2 rounded-full transition-all duration-300" style="width:0%"></div>
        </div>
    </div>
</div>

<!-- ─── JavaScript ──────────────────────────────────────────────────────────── -->
<script>
const CSRF_TOKEN = '<?php echo esc_js($csrf_token); ?>';

// i18n strings for JS
const i18n = {
    send_reminders_count: '<?php echo esc_js($lang['send_reminders_count'] ?? 'Mahnungen senden'); ?>',
    confirm_send:         '<?php echo esc_js($lang['confirm_send_reminders'] ?? 'Wirklich'); ?>',
    reminder_singular:    '<?php echo esc_js($lang['reminder_singular'] ?? 'Mahnung'); ?>',
    reminder_plural:      '<?php echo esc_js($lang['reminder_plural'] ?? 'Mahnungen'); ?>',
    confirm_send_q:       '<?php echo esc_js($lang['confirm_send_question'] ?? 'senden?'); ?>',
    sent_progress:        '<?php echo esc_js($lang['sent_progress'] ?? 'gesendet'); ?>',
    failed_label:         '<?php echo esc_js($lang['failed_label'] ?? 'fehlgeschlagen'); ?>',
    network_error:        '<?php echo esc_js($lang['network_error'] ?? 'Netzwerkfehler'); ?>',
    error_label:          '<?php echo esc_js($lang['error'] ?? 'Fehler'); ?>',
    sent_badge:           '<?php echo esc_js($lang['sent_badge'] ?? 'Gesendet'); ?>',
    error_badge:          '<?php echo esc_js($lang['error_badge'] ?? 'Fehler'); ?>',
    just_sent:            '<?php echo esc_js($lang['just_sent'] ?? 'eben gesendet'); ?>',
};

function escHtml(str) {
    if (str === null || str === undefined) return '';
    const d = document.createElement('div');
    d.textContent = String(str);
    return d.innerHTML;
}

// ─── Checkbox-Logik ─────────────────────────────────────────────────────────
function getCheckedIds() {
    return [...document.querySelectorAll('.loan-cb:checked')].map(c => c.value);
}

function updateBulkBtn() {
    const ids = getCheckedIds();
    const btn  = document.getElementById('btnBulkSend');
    const lbl  = document.getElementById('bulkBtnLabel');
    lbl.textContent = i18n.send_reminders_count + ' (' + ids.length + ')';
    if (ids.length > 0) {
        btn.disabled = false;
        btn.classList.remove('opacity-50', 'cursor-not-allowed');
    } else {
        btn.disabled = true;
        btn.classList.add('opacity-50', 'cursor-not-allowed');
    }
    // Globale Checkbox anpassen
    const allCbs = [...document.querySelectorAll('.loan-cb:not([disabled])')];
    document.getElementById('selectAll').checked = allCbs.length > 0 && allCbs.every(c => c.checked);
    document.getElementById('selectAll').indeterminate = ids.length > 0 && !allCbs.every(c => c.checked);
}

function toggleAll(master) {
    document.querySelectorAll('.loan-cb:not([disabled])').forEach(cb => {
        cb.checked = master.checked;
    });
    updateBulkBtn();
}

// ─── Einzelne Mahnung senden ─────────────────────────────────────────────────
function sendSingle(loanId) {
    sendReminders([String(loanId)]);
}

// ─── Sammel-Mahnung senden ───────────────────────────────────────────────────
function bulkSend() {
    const ids = getCheckedIds();
    if (!ids.length) return;
    const label = ids.length === 1 ? i18n.reminder_singular : i18n.reminder_plural;
    if (!confirm(i18n.confirm_send + ' ' + ids.length + ' ' + label + ' ' + i18n.confirm_send_q)) return;
    sendReminders(ids);
}

// ─── Kern-Sende-Funktion ─────────────────────────────────────────────────────
function sendReminders(loanIds) {
    const overlay   = document.getElementById('sendOverlay');
    const statusEl  = document.getElementById('overlayStatus');
    const barEl     = document.getElementById('overlayBar');

    overlay.classList.remove('hidden');
    statusEl.textContent = '0 / ' + loanIds.length + ' ' + i18n.sent_progress + '…';
    barEl.style.width = '0%';

    const fd = new FormData();
    fd.append('action',     'send_reminders');
    fd.append('csrf_token', CSRF_TOKEN);
    loanIds.forEach(id => fd.append('loan_ids[]', id));

    fetch('lending-reminders.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            overlay.classList.add('hidden');

            if (data.error) {
                alert(i18n.error_label + ': ' + data.error);
                return;
            }

            let sent = 0, failed = 0;
            (data.results || []).forEach(res => {
                res.success ? sent++ : failed++;
                updateRow(res);
            });

            // Fortschrittsbalken kurz anzeigen und wieder ausblenden
            barEl.style.width = '100%';
            setTimeout(() => { barEl.style.width = '0%'; }, 800);

            // Checkboxen deaktivieren (schon gesendet)
            loanIds.forEach(id => {
                const cb = document.querySelector('.loan-cb[value="' + id + '"]');
                if (cb) { cb.checked = false; cb.disabled = true; }
            });
            updateBulkBtn();

            const msg = sent + ' ' + (sent !== 1 ? i18n.reminder_plural : i18n.reminder_singular) + ' ' + i18n.sent_progress
                + (failed > 0 ? ', ' + failed + ' ' + i18n.failed_label + '.' : '.');
            statusEl.textContent = msg;
            overlay.classList.remove('hidden');
            setTimeout(() => overlay.classList.add('hidden'), 2500);
        })
        .catch(err => {
            overlay.classList.add('hidden');
            alert(i18n.network_error + ': ' + err.message);
        });
}

// ─── Tabellenzeile nach dem Senden aktualisieren ─────────────────────────────
function updateRow(res) {
    const actionCell  = document.getElementById('action-'  + res.loan_id);
    const lastremCell = document.getElementById('lastrem-' + res.loan_id);
    if (!actionCell) return;

    if (res.success) {
        actionCell.innerHTML =
            '<span class="px-2 py-1 text-xs font-medium rounded-full bg-green-100 text-green-800">&#10003; ' + i18n.sent_badge + '</span>';
        if (lastremCell) {
            const now = new Date();
            const fmt = now.toLocaleDateString('de-AT', {day:'2-digit', month:'2-digit', year:'numeric'});
            lastremCell.innerHTML =
                '<p class="text-xs font-medium text-green-700">' + escHtml(fmt) + '</p>'
              + '<p class="text-xs text-gray-400">' + i18n.just_sent + '</p>';
        }
    } else {
        actionCell.innerHTML =
            '<span class="px-2 py-1 text-xs font-medium rounded-full bg-red-100 text-red-800" title="'
            + escHtml(res.error) + '">&#10007; ' + i18n.error_badge + '</span>';
    }
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
