<?php
/**
 * Ausleihe buchen — lending-checkout.php
 * Single-File, AJAX-gesteuert: Benutzer suchen → Buch suchen → Ausleihe buchen
 *
 * AJAX-Actions (GET, kein CSRF): search_user, get_user, ldap_search, search_book
 * AJAX-Actions (POST, CSRF):     create_and_select_user, checkout
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = $current_user['id'];

// ─── AJAX-Dispatcher ──────────────────────────────────────────────────────────

$action = $_GET['action'] ?? $_POST['action'] ?? '';

if ($action !== '') {
    header('Content-Type: application/json; charset=utf-8');

    switch ($action) {

        // ── search_user (GET) ────────────────────────────────────────────────
        case 'search_user':
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }
            $like = '%' . $q . '%';
            $rows = db_get_results(
                "SELECT u.id, u.first_name, u.last_name, u.class_name, u.student_id,
                        u.status, u.current_loans, u.overdue_loans,
                        ug.name AS user_group, ug.max_loans
                 FROM users u
                 LEFT JOIN user_groups ug ON u.user_group_id = ug.id
                 WHERE u.first_name LIKE ? OR u.last_name LIKE ?
                    OR CONCAT(u.first_name,' ',u.last_name) LIKE ?
                    OR CONCAT(u.last_name,' ',u.first_name) LIKE ?
                    OR u.class_name LIKE ? OR u.student_id LIKE ?
                 ORDER BY u.last_name, u.first_name
                 LIMIT 30",
                [$like, $like, $like, $like, $like, $like]
            );
            echo json_encode($rows);
            exit;

        // ── get_user (GET) ───────────────────────────────────────────────────
        case 'get_user':
            $uid = (int)($_GET['id'] ?? 0);
            if (!$uid) { echo json_encode(['error' => 'Keine ID']); exit; }
            $user = db_get_row(
                "SELECT u.id, u.first_name, u.last_name, u.class_name, u.student_id,
                        u.status, u.blocked_reason, u.current_loans, u.overdue_loans,
                        ug.name AS user_group, ug.max_loans, ug.loan_duration_days
                 FROM users u
                 LEFT JOIN user_groups ug ON u.user_group_id = ug.id
                 WHERE u.id = ?",
                [$uid]
            );
            if (!$user) { echo json_encode(['error' => 'Benutzer nicht gefunden']); exit; }

            // Aktive Ausleihen (return_date IS NULL = aktiv)
            $loans = db_get_results(
                "SELECT l.id, l.loan_date, l.due_date, b.title, bc.book_number, bc.signature
                 FROM loans l
                 JOIN book_copies bc ON l.copy_id = bc.id
                 JOIN books b ON bc.book_id = b.id
                 WHERE l.user_id = ? AND l.return_date IS NULL
                 ORDER BY l.loan_date DESC
                 LIMIT 5",
                [$uid]
            );
            $user['active_loans'] = $loans;
            echo json_encode($user);
            exit;

        // ── ldap_search (GET) ────────────────────────────────────────────────
        case 'ldap_search':
            if (getSetting('ldap_enabled') !== '1') {
                echo json_encode(['error' => 'LDAP ist nicht aktiviert']); exit;
            }
            if (!extension_loaded('ldap')) {
                echo json_encode(['error' => 'PHP LDAP-Extension nicht geladen']); exit;
            }
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }

            $server        = getSetting('ldap_host');
            $port          = (int)getSetting('ldap_port', '389');
            $base_dn       = getSetting('ldap_base_dn');
            $bind_dn       = getSetting('ldap_bind_dn');
            $bind_password = getSetting('ldap_bind_password');
            $usernameAttr  = getSetting('ldap_username_attribute', 'sAMAccountName');
            $emailAttr     = getSetting('ldap_email_attribute', 'mail');
            $firstAttr     = getSetting('ldap_firstname_attribute', 'givenName');
            $lastAttr      = getSetting('ldap_lastname_attribute', 'sn');

            if (!$server || !$base_dn) {
                echo json_encode(['error' => 'LDAP nicht konfiguriert']); exit;
            }
            try {
                $ldap = @ldap_connect($server, $port);
                if (!$ldap) { echo json_encode(['error' => 'LDAP-Verbindung fehlgeschlagen']); exit; }
                ldap_set_option($ldap, LDAP_OPT_PROTOCOL_VERSION, 3);
                ldap_set_option($ldap, LDAP_OPT_REFERRALS, 0);
                ldap_set_option($ldap, LDAP_OPT_NETWORK_TIMEOUT, 5);
                if (!@ldap_bind($ldap, $bind_dn, $bind_password)) {
                    ldap_close($ldap); echo json_encode(['error' => 'LDAP-Bind fehlgeschlagen']); exit;
                }
                $escaped = ldap_escape($q, '', LDAP_ESCAPE_FILTER);
                $filter  = '(&(objectClass=user)(|(sn=*' . $escaped . '*)(givenName=*' . $escaped . '*)))';
                $search  = @ldap_search($ldap, $base_dn, $filter,
                    ['dn', $usernameAttr, $emailAttr, $firstAttr, $lastAttr], 0, 50);
                $results = [];
                if ($search) {
                    $entries = ldap_get_entries($ldap, $search);
                    for ($i = 0; $i < $entries['count']; $i++) {
                        $e = $entries[$i];
                        $results[] = [
                            'username'   => $e[strtolower($usernameAttr)][0] ?? '',
                            'email'      => $e[strtolower($emailAttr)][0] ?? '',
                            'first_name' => $e[strtolower($firstAttr)][0] ?? '',
                            'last_name'  => $e[strtolower($lastAttr)][0] ?? '',
                            'ldap_dn'    => $e['dn'] ?? '',
                        ];
                    }
                }
                ldap_close($ldap);
                echo json_encode($results);
            } catch (Exception $e) {
                if (isset($ldap)) @ldap_close($ldap);
                echo json_encode(['error' => 'LDAP-Fehler: ' . $e->getMessage()]);
            }
            exit;

        // ── create_and_select_user (POST, CSRF) ─────────────────────────────
        case 'create_and_select_user':
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                echo json_encode(['error' => 'Methode nicht erlaubt']); exit;
            }
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            $username   = trim($_POST['username'] ?? '');
            $email      = trim($_POST['email'] ?? '');
            $first_name = trim($_POST['first_name'] ?? '');
            $last_name  = trim($_POST['last_name'] ?? '');
            $ldap_dn    = trim($_POST['ldap_dn'] ?? '');
            $group_id   = (int)($_POST['user_group_id'] ?? 2);

            if (empty($username) || empty($first_name) || empty($last_name)) {
                echo json_encode(['error' => 'Vorname, Nachname und Benutzername sind Pflichtfelder']); exit;
            }

            // Sokrates-Kandidat: gleicher Name, kein email, kein username
            $candidate = db_get_row(
                "SELECT id FROM users
                 WHERE first_name = ? AND last_name = ?
                   AND (email IS NULL OR email = '')
                   AND (username IS NULL OR username = '')
                 LIMIT 1",
                [$first_name, $last_name]
            );

            if ($candidate) {
                // MERGE: bestehenden Sokrates-User mit LDAP-Daten anreichern
                db_update('users', [
                    'user_group_id' => $group_id,
                    'username'      => $username,
                    'email'         => $email ?: null,
                    'ldap_dn'       => $ldap_dn ?: null,
                    'auth_method'   => 'ldap',
                    'status'        => 'aktiv',
                ], 'id = ?', [$candidate['id']]);
                $new_id = $candidate['id'];
                log_activity($admin_id, 'ldap_user_merged', 'users', $new_id, "LDAP merge: $username");
            } else {
                // Pruefen ob Username schon vergeben
                if (db_get_row("SELECT id FROM users WHERE username = ?", [$username])) {
                    echo json_encode(['error' => 'Benutzername bereits vergeben']); exit;
                }
                $new_id = db_insert('users', [
                    'user_group_id' => $group_id,
                    'username'      => $username,
                    'email'         => $email ?: null,
                    'first_name'    => $first_name,
                    'last_name'     => $last_name,
                    'ldap_dn'       => $ldap_dn ?: null,
                    'auth_method'   => 'ldap',
                    'status'        => 'aktiv',
                    'created_at'    => date('Y-m-d H:i:s'),
                ]);
                if (!$new_id) { echo json_encode(['error' => 'Datenbankfehler beim Anlegen']); exit; }
                log_activity($admin_id, 'ldap_user_created', 'users', $new_id, "LDAP create: $username");
            }

            // Vollstaendigen User inkl. Gruppe zurueckgeben
            $user = db_get_row(
                "SELECT u.id, u.first_name, u.last_name, u.class_name, u.student_id,
                        u.status, u.blocked_reason, u.current_loans, u.overdue_loans,
                        ug.name AS user_group, ug.max_loans, ug.loan_duration_days
                 FROM users u
                 LEFT JOIN user_groups ug ON u.user_group_id = ug.id
                 WHERE u.id = ?",
                [$new_id]
            );
            $user['active_loans'] = [];
            echo json_encode(['success' => true, 'user' => $user]);
            exit;

        // ── search_book (GET) ────────────────────────────────────────────────
        case 'search_book':
            $q = trim($_GET['q'] ?? '');
            if (mb_strlen($q) < 2) { echo json_encode([]); exit; }

            // Numerisch (4-10 Ziffern) → Exact Match zuerst
            if (preg_match('/^\d{4,10}$/', $q)) {
                $exact = db_get_results(
                    "SELECT bc.id AS copy_id, bc.book_number, bc.accession_number,
                            bc.signature, bc.status,
                            b.id AS book_id, b.title, b.author, b.year
                     FROM book_copies bc
                     JOIN books b ON bc.book_id = b.id
                     WHERE bc.status IN ('verfügbar','reserviert')
                       AND (bc.book_number = ? OR bc.accession_number = ?)
                     ORDER BY bc.status = 'verfügbar' DESC
                     LIMIT 25",
                    [$q, $q]
                );
                if (!empty($exact)) { echo json_encode($exact); exit; }
            }

            // LIKE-Suche: book_number, accession_number, signature, title, author
            $like = '%' . $q . '%';
            $rows = db_get_results(
                "SELECT bc.id AS copy_id, bc.book_number, bc.accession_number,
                        bc.signature, bc.status,
                        b.id AS book_id, b.title, b.author, b.year
                 FROM book_copies bc
                 JOIN books b ON bc.book_id = b.id
                 WHERE bc.status IN ('verfügbar','reserviert')
                   AND (bc.book_number LIKE ? OR bc.accession_number LIKE ?
                        OR bc.signature LIKE ? OR b.title LIKE ? OR b.author LIKE ?)
                 ORDER BY bc.status = 'verfügbar' DESC, b.title
                 LIMIT 25",
                [$like, $like, $like, $like, $like]
            );
            echo json_encode($rows);
            exit;

        // ── checkout (POST, CSRF, Transaction) ──────────────────────────────
        case 'checkout':
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                echo json_encode(['error' => 'Methode nicht erlaubt']); exit;
            }
            if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
                echo json_encode(['error' => 'Sicherheitsfehler (CSRF)']); exit;
            }
            handle_checkout($admin_id);
            exit;

        default:
            http_response_code(400);
            echo json_encode(['error' => 'Unbekannte Aktion']);
            exit;
    }
}

// ─── handle_checkout() ────────────────────────────────────────────────────────

function handle_checkout(int $admin_id): void {
    global $pdo;

    $user_id  = (int)($_POST['user_id'] ?? 0);
    $copy_id  = (int)($_POST['copy_id'] ?? 0);
    $due_date = trim($_POST['due_date'] ?? '');
    $note     = trim($_POST['loan_note'] ?? '');

    if (!$user_id || !$copy_id || !$due_date) {
        echo json_encode(['error' => 'Pflichtfelder fehlen']); return;
    }
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $due_date) || $due_date < date('Y-m-d')) {
        echo json_encode(['error' => 'Rückgabedatum muss heute oder in der Zukunft liegen']); return;
    }

    // Benutzer pruefen
    $user = db_get_row(
        "SELECT u.id, u.first_name, u.last_name, u.status, u.current_loans, u.overdue_loans,
                ug.max_loans
         FROM users u
         LEFT JOIN user_groups ug ON u.user_group_id = ug.id
         WHERE u.id = ?",
        [$user_id]
    );
    if (!$user) { echo json_encode(['error' => 'Benutzer nicht gefunden']); return; }
    if ($user['status'] === 'inaktiv') {
        echo json_encode(['error' => 'Inaktive Benutzer koennen nicht ausleihen']); return;
    }

    // Exemplar pruefen
    $copy = db_get_row(
        "SELECT bc.id, bc.status, bc.book_number, bc.signature,
                b.id AS book_id, b.title, b.author
         FROM book_copies bc
         JOIN books b ON bc.book_id = b.id
         WHERE bc.id = ?",
        [$copy_id]
    );
    if (!$copy) { echo json_encode(['error' => 'Exemplar nicht gefunden']); return; }
    if (!in_array($copy['status'], ['verfügbar', 'reserviert'], true)) {
        echo json_encode(['error' => 'Exemplar ist nicht verfuegbar (Status: ' . $copy['status'] . ')']); return;
    }

    // Warnings sammeln (kein Block)
    $warnings = [];
    if ($user['status'] === 'gesperrt') {
        $warnings[] = 'Benutzer ist gesperrt.';
    }
    if (($user['overdue_loans'] ?? 0) > 0) {
        $warnings[] = 'Benutzer hat ' . $user['overdue_loans'] . ' ueberfaellige Ausleihe(n).';
    }
    if (($user['current_loans'] ?? 0) >= ($user['max_loans'] ?? 5)) {
        $warnings[] = 'Ausleih-Limit erreicht (' . $user['current_loans'] . '/' . $user['max_loans'] . ').';
    }

    // Transaction
    try {
        $pdo->beginTransaction();

        $loan_id = db_insert('loans', [
            'copy_id'   => $copy_id,
            'user_id'   => $user_id,
            'loan_date' => date('Y-m-d H:i:s'),
            'due_date'  => $due_date,
            'status'    => 'aktiv',
            'loan_note' => $note ?: null,
            'loaned_by' => $admin_id,
        ]);

        if (!$loan_id) {
            $pdo->rollBack();
            echo json_encode(['error' => 'Datenbankfehler beim Anlegen der Ausleihe']); return;
        }

        // Atomisches UPDATE mit Race-Condition-Schutz
        $stmt = $pdo->prepare(
            "UPDATE book_copies SET status = 'ausgeliehen'
             WHERE id = ? AND status IN ('verfügbar', 'reserviert')"
        );
        $stmt->execute([$copy_id]);

        if ($stmt->rowCount() !== 1) {
            $pdo->rollBack();
            echo json_encode(['error' => 'Exemplar wurde zwischenzeitlich von jemand anderem ausgeliehen (Race Condition).']); return;
        }

        $pdo->commit();

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log('Checkout-Fehler: ' . $e->getMessage());
        echo json_encode(['error' => 'Datenbankfehler: ' . $e->getMessage()]); return;
    }

    log_activity($admin_id, 'loan_created', 'loans', $loan_id,
        json_encode(['user_id' => $user_id, 'copy_id' => $copy_id, 'due_date' => $due_date,
                     'book' => $copy['title'], 'signature' => $copy['signature']]));

    echo json_encode([
        'success'    => true,
        'loan_id'    => $loan_id,
        'user_name'  => $user['first_name'] . ' ' . $user['last_name'],
        'book_title' => $copy['title'],
        'signature'  => $copy['signature'],
        'book_number'=> $copy['book_number'],
        'due_date'   => $due_date,
        'warnings'   => $warnings,
    ]);
}

// ─── Seiten-Setup ─────────────────────────────────────────────────────────────

$page_title    = $lang['process_checkout'] ?? 'Ausleihe buchen';
$ldap_enabled  = getSetting('ldap_enabled') === '1';
$csrf_token    = generate_csrf_token();
$default_loan_days = (int)getSetting('loan_duration_days', 28);

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['process_checkout'] ?? 'Ausleihe buchen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500"><?php echo $lang['checkout_workflow_hint'] ?? 'Benutzer suchen &rarr; Buch suchen &rarr; Ausleihe buchen'; ?></p>
        </div>
        <a href="lending-active.php" class="text-sm text-gray-500 hover:text-gray-700">&rarr; <?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?></a>
    </div>

    <!-- ═══════════════════════════════════════════ PANEL 1: Nutzer suchen -->
    <div id="panel-user" class="bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b flex items-center justify-between">
            <h2 class="text-lg font-semibold text-gray-900">
                <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-blue-600 text-white text-sm font-bold mr-2">1</span>
                <?php echo $lang['search_user'] ?? 'Benutzer suchen'; ?>
            </h2>
            <button id="btn-reset-user" onclick="resetAll()" class="hidden text-sm text-gray-400 hover:text-gray-600">
                &times; <?php echo $lang['reset'] ?? 'Zurücksetzen'; ?>
            </button>
        </div>

        <!-- Suchzeile -->
        <div id="user-search-area" class="p-5">
            <div class="flex gap-3">
                <input type="text" id="userSearchInput"
                       placeholder="<?php echo $lang['user_search_placeholder'] ?? 'Name, Klasse oder Ausweis-Nr. (mind. 2 Zeichen)...'; ?>"
                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                <button onclick="searchUser()" class="px-4 py-2 bg-blue-600 text-white text-sm rounded-md hover:bg-blue-700">
                    <?php echo $lang['search'] ?? 'Suchen'; ?>
                </button>
            </div>
            <div id="user-results" class="mt-4"></div>

            <!-- LDAP-Fallback -->
            <div id="ldap-fallback" class="hidden mt-3">
                <button onclick="showLdapSearch()"
                        class="text-sm text-blue-600 hover:text-blue-800 underline">
                    <?php echo $lang['search_in_active_directory'] ?? 'Im Active Directory suchen'; ?>
                </button>
            </div>

            <!-- LDAP-Suchbereich -->
            <div id="ldap-search-area" class="hidden mt-4 border-t pt-4">
                <p class="text-sm text-gray-600 mb-3 font-medium"><?php echo $lang['active_directory_search'] ?? 'Active Directory Suche'; ?></p>
                <div class="flex gap-3 mb-4">
                    <input type="text" id="ldapSearchInput"
                           placeholder="<?php echo $lang['ldap_search_placeholder'] ?? 'Name im Active Directory suchen...'; ?>"
                           class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                    <button onclick="runLdapSearch()" class="px-4 py-2 bg-indigo-600 text-white text-sm rounded-md hover:bg-indigo-700">
                        <?php echo $lang['search_ad'] ?? 'AD suchen'; ?>
                    </button>
                </div>
                <div id="ldap-results"></div>

                <!-- LDAP Anlegen-Panel -->
                <div id="ldap-create-panel" class="hidden border border-blue-200 bg-blue-50 rounded-lg p-4 mt-3">
                    <p class="text-sm font-medium text-blue-800 mb-3">
                        <?php echo $lang['create_user_from_ad'] ?? 'Benutzer aus Active Directory anlegen'; ?>
                        <span class="font-normal text-blue-600">(<?php echo $lang['ldap_merge_hint'] ?? 'LDAP-Merge mit Sokrates-Eintrag wird geprueft'; ?>)</span>
                    </p>
                    <div class="grid grid-cols-2 gap-3 mb-3">
                        <div>
                            <label class="block text-xs font-medium text-gray-700 mb-1"><?php echo $lang['first_name'] ?? 'Vorname'; ?></label>
                            <input type="text" id="lcFirstName" readonly
                                   class="w-full px-2 py-1.5 text-sm border border-gray-200 bg-white rounded">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-700 mb-1"><?php echo $lang['last_name'] ?? 'Nachname'; ?></label>
                            <input type="text" id="lcLastName" readonly
                                   class="w-full px-2 py-1.5 text-sm border border-gray-200 bg-white rounded">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-700 mb-1"><?php echo $lang['username_ad'] ?? 'Benutzername (AD)'; ?></label>
                            <input type="text" id="lcUsername"
                                   class="w-full px-2 py-1.5 text-sm border border-gray-300 rounded focus:ring-1 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-700 mb-1"><?php echo $lang['email'] ?? 'E-Mail'; ?></label>
                            <input type="email" id="lcEmail"
                                   class="w-full px-2 py-1.5 text-sm border border-gray-300 rounded focus:ring-1 focus:ring-blue-500">
                        </div>
                    </div>
                    <input type="hidden" id="lcDn">
                    <div id="ldap-create-error" class="hidden mb-3 text-sm text-red-700 bg-red-50 border border-red-200 rounded p-2"></div>
                    <div class="flex gap-2">
                        <button onclick="createAndSelectLdapUser()"
                                class="px-4 py-1.5 bg-green-600 text-white text-sm rounded hover:bg-green-700 font-medium">
                            <?php echo $lang['create_and_select'] ?? 'Anlegen &amp; Auswählen'; ?>
                        </button>
                        <button onclick="document.getElementById('ldap-create-panel').classList.add('hidden')"
                                class="px-4 py-1.5 border border-gray-300 text-gray-700 text-sm rounded hover:bg-gray-50">
                            <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Ausgewählter Benutzer (nur sichtbar wenn selectedUser gesetzt) -->
        <div id="selected-user-bar" class="hidden px-5 pb-5">
            <div class="flex items-center gap-3 bg-blue-50 border border-blue-200 rounded-lg p-3">
                <div id="user-avatar" class="w-10 h-10 rounded-full bg-blue-600 text-white flex items-center justify-center font-bold text-sm flex-shrink-0"></div>
                <div class="flex-1 min-w-0">
                    <p id="user-name-display" class="font-semibold text-gray-900 text-sm"></p>
                    <p id="user-meta-display" class="text-xs text-gray-500"></p>
                </div>
                <button onclick="resetAll()" class="text-xs text-blue-600 hover:text-blue-800 underline flex-shrink-0">
                    <?php echo $lang['change_user'] ?? 'Wechseln'; ?>
                </button>
            </div>
        </div>
    </div>

    <!-- ══════════════════════════════════════ PANEL 2: Buch suchen -->
    <div id="panel-book" class="hidden bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b flex items-center justify-between">
            <h2 class="text-lg font-semibold text-gray-900">
                <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-blue-600 text-white text-sm font-bold mr-2">2</span>
                <?php echo $lang['select_book'] ?? 'Buch auswählen'; ?>
            </h2>
        </div>

        <!-- Benutzer-Karte -->
        <div id="user-card" class="p-5 border-b bg-gray-50"></div>

        <!-- Buchsuche -->
        <div class="p-5">
            <div class="flex gap-3 mb-4">
                <input type="text" id="bookSearchInput"
                       placeholder="<?php echo $lang['book_search_placeholder'] ?? 'Barcode, Signatur oder Titel / Autor (mind. 2 Zeichen)...'; ?>"
                       class="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                <button onclick="searchBook()" class="px-4 py-2 bg-blue-600 text-white text-sm rounded-md hover:bg-blue-700">
                    <?php echo $lang['search'] ?? 'Suchen'; ?>
                </button>
            </div>
            <div id="book-results"></div>
        </div>
    </div>

    <!-- ════════════════════════════════════ PANEL 3: Bestätigung -->
    <div id="panel-confirm" class="hidden bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b">
            <h2 class="text-lg font-semibold text-gray-900">
                <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-blue-600 text-white text-sm font-bold mr-2">3</span>
                <?php echo $lang['confirm_checkout'] ?? 'Ausleihe bestätigen'; ?>
            </h2>
        </div>
        <div class="p-5">

            <!-- Zusammenfassung -->
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                <div class="bg-gray-50 rounded-lg p-4">
                    <p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2"><?php echo $lang['users'] ?? 'Benutzer'; ?></p>
                    <p id="confirm-user" class="font-semibold text-gray-900"></p>
                    <p id="confirm-user-meta" class="text-sm text-gray-500 mt-0.5"></p>
                </div>
                <div class="bg-gray-50 rounded-lg p-4">
                    <p class="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2"><?php echo $lang['book_copy_label'] ?? 'Buch / Exemplar'; ?></p>
                    <p id="confirm-book" class="font-semibold text-gray-900"></p>
                    <p id="confirm-signature" class="text-sm text-gray-500 font-mono mt-0.5"></p>
                </div>
            </div>

            <!-- Datumsfelder + Notiz -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['loan_start'] ?? 'Ausleihbeginn'; ?></label>
                    <input type="text" id="confirm-loan-date" readonly
                           class="w-full px-3 py-2 border border-gray-200 bg-gray-50 rounded-md text-sm text-gray-600">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['return_by'] ?? 'Rückgabe bis'; ?> <span class="text-red-500">*</span></label>
                    <input type="date" id="confirm-due-date"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['note_optional'] ?? 'Notiz (optional)'; ?></label>
                    <input type="text" id="confirm-note" maxlength="200" placeholder="<?php echo $lang['note_placeholder'] ?? 'z.B. Buchzustand, Bemerkung...'; ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                </div>
            </div>

            <!-- Warn-Bereich -->
            <div id="confirm-warnings" class="hidden mb-4 bg-yellow-50 border border-yellow-200 rounded-lg p-3">
                <p class="text-sm font-medium text-yellow-800 mb-1"><?php echo $lang['notices'] ?? 'Hinweise:'; ?></p>
                <ul id="confirm-warnings-list" class="text-sm text-yellow-700 list-disc list-inside space-y-0.5"></ul>
            </div>

            <!-- Fehler -->
            <div id="confirm-error" class="hidden mb-4 bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-700"></div>

            <!-- Buttons -->
            <div class="flex gap-3">
                <button onclick="submitCheckout()" id="btn-checkout"
                        class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 font-medium">
                    <?php echo $lang['process_checkout'] ?? 'Ausleihe buchen'; ?>
                </button>
                <button onclick="resetBook()"
                        class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 text-sm">
                    <?php echo $lang['different_book'] ?? 'Anderes Buch'; ?>
                </button>
                <button onclick="resetAll()"
                        class="px-4 py-2 border border-gray-300 text-gray-500 rounded-md hover:bg-gray-50 text-sm">
                    <?php echo $lang['restart'] ?? 'Neu starten'; ?>
                </button>
            </div>
        </div>
    </div>

    <!-- ═══════════════════════════════════════════ SUCCESS-PANEL -->
    <div id="panel-success" class="hidden bg-white rounded-lg shadow mb-6">
        <div class="p-5 border-b bg-green-50 rounded-t-lg">
            <h2 class="text-lg font-semibold text-green-800">
                <svg class="inline w-5 h-5 mr-1 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                </svg>
                <?php echo $lang['checkout_success'] ?? 'Ausleihe erfolgreich gebucht'; ?>
            </h2>
        </div>
        <div class="p-5">
            <div id="success-details" class="bg-gray-50 rounded-lg p-4 mb-4 text-sm text-gray-800 space-y-1"></div>
            <div id="success-warnings" class="hidden mb-4 bg-yellow-50 border border-yellow-200 rounded-lg p-3 text-sm text-yellow-700"></div>
            <div class="flex gap-3">
                <button onclick="nextLoan()"
                        class="px-5 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium">
                    <?php echo $lang['next_checkout'] ?? 'Nächste Ausleihe'; ?>
                </button>
                <button onclick="resetAll()"
                        class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50 text-sm">
                    <?php echo $lang['done'] ?? 'Fertig'; ?>
                </button>
            </div>
        </div>
    </div>

</div>
</div>

<!-- ─── JavaScript ──────────────────────────────────────────────────────────── -->
<script>
// State
let selectedUser = null;
let selectedCopy = null;
const CSRF_TOKEN  = '<?php echo esc_js($csrf_token); ?>';
const LDAP_ENABLED = <?php echo $ldap_enabled ? 'true' : 'false'; ?>;
const DEFAULT_LOAN_DAYS = <?php echo $default_loan_days; ?>;

// i18n strings for JS
const i18n = {
    search_user:          '<?php echo esc_js($lang['search_user'] ?? 'Benutzer suchen'); ?>',
    search_book:          '<?php echo esc_js($lang['search_book'] ?? 'Buch suchen'); ?>',
    no_db_results:        '<?php echo esc_js($lang['no_db_results'] ?? 'Keine Treffer in der Datenbank.'); ?>',
    no_copy_found:        '<?php echo esc_js($lang['no_copy_found'] ?? 'Kein verfügbares Exemplar gefunden.'); ?>',
    no_ad_results:        '<?php echo esc_js($lang['no_ad_results'] ?? 'Keine Treffer im Active Directory.'); ?>',
    network_error:        '<?php echo esc_js($lang['network_error'] ?? 'Netzwerkfehler'); ?>',
    searching:            '<?php echo esc_js($lang['searching'] ?? 'Suche läuft...'); ?>',
    col_name:             '<?php echo esc_js($lang['name'] ?? 'Name'); ?>',
    col_class:            '<?php echo esc_js($lang['class_name'] ?? 'Klasse'); ?>',
    col_card_no:          '<?php echo esc_js($lang['student_id'] ?? 'Ausweis-Nr.'); ?>',
    col_status:           '<?php echo esc_js($lang['status'] ?? 'Status'); ?>',
    col_loans:            '<?php echo esc_js($lang['current_loans'] ?? 'Ausleihen'); ?>',
    col_action:           '<?php echo esc_js($lang['actions'] ?? 'Aktion'); ?>',
    col_title:            '<?php echo esc_js($lang['book_title'] ?? 'Titel'); ?>',
    col_author:           '<?php echo esc_js($lang['book_author'] ?? 'Autor'); ?>',
    col_signature:        '<?php echo esc_js($lang['signature'] ?? 'Signatur'); ?>',
    col_copy_number:      '<?php echo esc_js($lang['copy_number'] ?? 'Buchungsnr.'); ?>',
    col_username:         '<?php echo esc_js($lang['username'] ?? 'Benutzername'); ?>',
    col_email:            '<?php echo esc_js($lang['email'] ?? 'E-Mail'); ?>',
    btn_select:           '<?php echo esc_js($lang['select'] ?? 'Auswählen'); ?>',
    btn_checkout:         '<?php echo esc_js($lang['checkout'] ?? 'Ausleihen'); ?>',
    btn_checkout_anyway:  '<?php echo esc_js($lang['checkout_anyway'] ?? 'Trotzdem ausleihen'); ?>',
    btn_create_select:    '<?php echo esc_js($lang['create_and_select'] ?? 'Anlegen & Auswählen'); ?>',
    process_checkout:     '<?php echo esc_js($lang['process_checkout'] ?? 'Ausleihe buchen'); ?>',
    processing:           '<?php echo esc_js($lang['processing'] ?? 'Wird gebucht...'); ?>',
    notices:              '<?php echo esc_js($lang['notices'] ?? 'Hinweise:'); ?>',
    user_label:           '<?php echo esc_js($lang['users'] ?? 'Benutzer'); ?>',
    book_label:           '<?php echo esc_js($lang['book_copy_label'] ?? 'Buch / Exemplar'); ?>',
    loan_id_label:        '<?php echo esc_js($lang['loan_id'] ?? 'Ausleihnummer'); ?>',
    copy_no_label:        '<?php echo esc_js($lang['copy_number'] ?? 'Buchungsnr.'); ?>',
    signature_label:      '<?php echo esc_js($lang['signature'] ?? 'Signatur'); ?>',
    return_by_label:      '<?php echo esc_js($lang['return_by'] ?? 'Rückgabe bis'); ?>',
    overdue_warning:      '<?php echo esc_js($lang['overdue_warning'] ?? 'Achtung'); ?>',
    overdue_loans_warn:   '<?php echo esc_js($lang['overdue_loans_warn'] ?? 'überfällige Ausleihe(n)!'); ?>',
    loan_limit_reached:   '<?php echo esc_js($lang['loan_limit_reached'] ?? 'Ausleih-Limit erreicht'); ?>',
    loan_limit_note:      '<?php echo esc_js($lang['loan_limit_note'] ?? 'Ausleihe trotzdem möglich.'); ?>',
    loans_label:          '<?php echo esc_js($lang['loans_label'] ?? 'Ausleihen'); ?>',
    return_date_required: '<?php echo esc_js($lang['return_date_required'] ?? 'Bitte Rückgabedatum eingeben.'); ?>',
    enter_min_chars:      '<?php echo esc_js($lang['enter_min_chars'] ?? 'Mind. 2 Zeichen eingeben.'); ?>',
    inactive_label:       '<?php echo esc_js($lang['status_inactive'] ?? 'inaktiv'); ?>',
};

// XSS-sichere String-Escaping-Funktion
function escHtml(str) {
    if (str === null || str === undefined) return '';
    const d = document.createElement('div');
    d.textContent = String(str);
    return d.innerHTML;
}

// ─── Debounce ───────────────────────────────────────────────────────────────
function debounce(fn, delay) {
    let t;
    return function(...args) { clearTimeout(t); t = setTimeout(() => fn.apply(this, args), delay); };
}

const debouncedUserSearch = debounce(searchUser, 400);
const debouncedBookSearch = debounce(searchBook, 400);

document.getElementById('userSearchInput').addEventListener('input', debouncedUserSearch);
document.getElementById('userSearchInput').addEventListener('keydown', e => { if (e.key === 'Enter') searchUser(); });
document.getElementById('bookSearchInput').addEventListener('input', debouncedBookSearch);
document.getElementById('bookSearchInput').addEventListener('keydown', e => { if (e.key === 'Enter') searchBook(); });
document.getElementById('ldapSearchInput').addEventListener('keydown', e => { if (e.key === 'Enter') runLdapSearch(); });

// ─── Panel 1: Benutzersuche ──────────────────────────────────────────────────
function searchUser() {
    const q = document.getElementById('userSearchInput').value.trim();
    const el = document.getElementById('user-results');
    if (q.length < 2) { el.innerHTML = ''; return; }

    el.innerHTML = spinner();
    document.getElementById('ldap-fallback').classList.add('hidden');
    document.getElementById('ldap-search-area').classList.add('hidden');

    fetch('lending-checkout.php?action=search_user&q=' + encodeURIComponent(q))
        .then(r => r.json())
        .then(data => {
            if (data.error) { el.innerHTML = errorBox(data.error); return; }
            if (!data.length) {
                el.innerHTML = '<p class="text-sm text-gray-500 py-2">' + i18n.no_db_results + '</p>';
                if (LDAP_ENABLED) document.getElementById('ldap-fallback').classList.remove('hidden');
                return;
            }
            el.innerHTML = renderUserTable(data);
        })
        .catch(err => { el.innerHTML = errorBox(i18n.network_error + ': ' + err.message); });
}

function renderUserTable(users) {
    let h = '<div class="overflow-x-auto"><table class="min-w-full text-sm border-collapse">';
    h += '<thead class="bg-gray-100"><tr>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_name + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_class + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_card_no + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_status + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_loans + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_action + '</th>';
    h += '</tr></thead><tbody>';

    users.forEach(u => {
        const inactive = u.status === 'inaktiv';
        const statusBadge = statusBadgeHtml(u.status);
        const loans = (u.current_loans || 0) + '/' + (u.max_loans || '?');
        h += '<tr class="hover:bg-gray-50">';
        h += '<td class="border px-3 py-2 font-medium">' + escHtml(u.last_name + ', ' + u.first_name) + '</td>';
        h += '<td class="border px-3 py-2">' + escHtml(u.class_name || '-') + '</td>';
        h += '<td class="border px-3 py-2 font-mono text-xs">' + escHtml(u.student_id || '-') + '</td>';
        h += '<td class="border px-3 py-2 text-center">' + statusBadge + '</td>';
        h += '<td class="border px-3 py-2 text-center text-xs">' + escHtml(loans) + '</td>';
        h += '<td class="border px-3 py-2 text-center">';
        if (inactive) {
            h += '<span class="text-xs text-gray-400">' + i18n.inactive_label + '</span>';
        } else {
            h += '<button onclick=\'selectUser(' + u.id + ')\' ';
            h += 'class="px-3 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700">' + i18n.btn_select + '</button>';
        }
        h += '</td>';
        h += '</tr>';
    });

    h += '</tbody></table></div>';
    return h;
}

function statusBadgeHtml(status) {
    const map = {
        'aktiv':    'bg-green-100 text-green-800',
        'gesperrt': 'bg-red-100 text-red-800',
        'inaktiv':  'bg-gray-100 text-gray-600',
    };
    const cls = map[status] || 'bg-gray-100 text-gray-600';
    return '<span class="px-2 py-0.5 text-xs font-medium rounded-full ' + cls + '">' + escHtml(status) + '</span>';
}

function selectUser(userId) {
    fetch('lending-checkout.php?action=get_user&id=' + userId)
        .then(r => r.json())
        .then(user => {
            if (user.error) { alert(user.error); return; }
            selectedUser = user;
            showUserSelected();
            showPanel2();
        });
}

function showUserSelected() {
    const u = selectedUser;
    const initials = (u.first_name ? u.first_name[0] : '') + (u.last_name ? u.last_name[0] : '');
    document.getElementById('user-avatar').textContent = initials.toUpperCase();
    document.getElementById('user-name-display').textContent = u.last_name + ', ' + u.first_name;
    document.getElementById('user-meta-display').textContent =
        [u.class_name, u.user_group, u.student_id ? ('ID: ' + u.student_id) : ''].filter(Boolean).join(' · ');
    document.getElementById('btn-reset-user').classList.remove('hidden');
    document.getElementById('user-search-area').classList.add('hidden');
    document.getElementById('selected-user-bar').classList.remove('hidden');
}

// ─── LDAP-Fallback ──────────────────────────────────────────────────────────
function showLdapSearch() {
    document.getElementById('ldap-search-area').classList.remove('hidden');
    document.getElementById('ldap-fallback').classList.add('hidden');
    document.getElementById('ldapSearchInput').focus();
}

function runLdapSearch() {
    const q = document.getElementById('ldapSearchInput').value.trim();
    const el = document.getElementById('ldap-results');
    if (q.length < 2) { el.innerHTML = '<p class="text-sm text-yellow-600">' + i18n.enter_min_chars + '</p>'; return; }

    el.innerHTML = spinner();
    document.getElementById('ldap-create-panel').classList.add('hidden');

    fetch('lending-checkout.php?action=ldap_search&q=' + encodeURIComponent(q))
        .then(r => r.json())
        .then(data => {
            if (data.error) { el.innerHTML = errorBox(data.error); return; }
            if (!data.length) { el.innerHTML = '<p class="text-sm text-gray-500">' + i18n.no_ad_results + '</p>'; return; }

            let h = '<div class="overflow-x-auto mt-2"><table class="min-w-full text-sm border-collapse">';
            h += '<thead class="bg-indigo-50"><tr>';
            h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_name + '</th>';
            h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_username + '</th>';
            h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_email + '</th>';
            h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_action + '</th>';
            h += '</tr></thead><tbody>';
            data.forEach(u => {
                h += '<tr class="hover:bg-gray-50">';
                h += '<td class="border px-3 py-2">' + escHtml(u.last_name + ', ' + u.first_name) + '</td>';
                h += '<td class="border px-3 py-2 font-mono text-xs">' + escHtml(u.username || '-') + '</td>';
                h += '<td class="border px-3 py-2 text-xs">' + escHtml(u.email || '-') + '</td>';
                h += '<td class="border px-3 py-2 text-center">';
                h += '<button onclick=\'openLdapCreate(' + JSON.stringify(u) + ')\' ';
                h += 'class="px-3 py-1 bg-indigo-600 text-white text-xs rounded hover:bg-indigo-700">' + i18n.btn_create_select + '</button>';
                h += '</td></tr>';
            });
            h += '</tbody></table></div>';
            el.innerHTML = h;
        })
        .catch(err => { el.innerHTML = errorBox(i18n.network_error + ': ' + err.message); });
}

function openLdapCreate(user) {
    document.getElementById('lcFirstName').value = user.first_name || '';
    document.getElementById('lcLastName').value  = user.last_name  || '';
    document.getElementById('lcUsername').value  = user.username   || '';
    document.getElementById('lcEmail').value     = user.email      || '';
    document.getElementById('lcDn').value        = user.ldap_dn    || '';
    document.getElementById('ldap-create-error').classList.add('hidden');
    document.getElementById('ldap-create-panel').classList.remove('hidden');
    document.getElementById('ldap-create-panel').scrollIntoView({behavior: 'smooth', block: 'nearest'});
}

function createAndSelectLdapUser() {
    const fd = new FormData();
    fd.append('action',        'create_and_select_user');
    fd.append('csrf_token',    CSRF_TOKEN);
    fd.append('first_name',    document.getElementById('lcFirstName').value);
    fd.append('last_name',     document.getElementById('lcLastName').value);
    fd.append('username',      document.getElementById('lcUsername').value);
    fd.append('email',         document.getElementById('lcEmail').value);
    fd.append('ldap_dn',       document.getElementById('lcDn').value);
    fd.append('user_group_id', '2');  // Standard: Schueler-Gruppe

    const errEl = document.getElementById('ldap-create-error');
    errEl.classList.add('hidden');

    fetch('lending-checkout.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                errEl.textContent = data.error;
                errEl.classList.remove('hidden');
                return;
            }
            selectedUser = data.user;
            showUserSelected();
            showPanel2();
        })
        .catch(err => {
            errEl.textContent = i18n.network_error + ': ' + err.message;
            errEl.classList.remove('hidden');
        });
}

// ─── Panel 2: Buch suchen ───────────────────────────────────────────────────
function showPanel2() {
    document.getElementById('panel-book').classList.remove('hidden');
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.add('hidden');
    renderUserCard();
    document.getElementById('bookSearchInput').value = '';
    document.getElementById('book-results').innerHTML = '';
    document.getElementById('bookSearchInput').focus();
    document.getElementById('panel-book').scrollIntoView({behavior: 'smooth', block: 'start'});
}

function renderUserCard() {
    const u = selectedUser;
    const current = parseInt(u.current_loans) || 0;
    const max = parseInt(u.max_loans) || 5;
    const pct = max > 0 ? Math.min(100, Math.round(current / max * 100)) : 0;
    const barColor = current >= max ? 'bg-red-500' : (current >= max * 0.8 ? 'bg-yellow-400' : 'bg-green-500');

    let html = '<div class="flex items-start gap-4">';
    const initials = ((u.first_name || '')[0] || '') + ((u.last_name || '')[0] || '');
    html += '<div class="w-12 h-12 rounded-full bg-blue-600 text-white flex items-center justify-center font-bold text-base flex-shrink-0">';
    html += escHtml(initials.toUpperCase()) + '</div>';
    html += '<div class="flex-1">';
    html += '<div class="flex items-center gap-2 mb-1">';
    html += '<p class="font-semibold text-gray-900">' + escHtml(u.first_name + ' ' + u.last_name) + '</p>';
    html += statusBadgeHtml(u.status || 'aktiv');
    html += '</div>';
    html += '<p class="text-sm text-gray-500 mb-2">' + escHtml([u.class_name, u.user_group].filter(Boolean).join(' · ')) + '</p>';

    // Ueberfaellig-Warnung
    if ((u.overdue_loans || 0) > 0) {
        html += '<div class="mb-2 text-xs bg-red-50 text-red-700 border border-red-200 rounded px-2 py-1">';
        html += '<strong>' + i18n.overdue_warning + ':</strong> ' + escHtml(u.overdue_loans) + ' ' + i18n.overdue_loans_warn + '</div>';
    }

    // Limit-Soft-Warnung
    if (current >= max) {
        html += '<div class="mb-2 text-xs bg-yellow-50 text-yellow-700 border border-yellow-200 rounded px-2 py-1">';
        html += i18n.loan_limit_reached + ' (' + escHtml(current) + '/' + escHtml(max) + '). ' + i18n.loan_limit_note + '</div>';
    }

    // Fortschrittsbalken
    html += '<div class="flex items-center gap-2">';
    html += '<div class="flex-1 bg-gray-200 rounded-full h-2">';
    html += '<div class="' + barColor + ' h-2 rounded-full transition-all" style="width:' + pct + '%"></div>';
    html += '</div>';
    html += '<span class="text-xs text-gray-500 flex-shrink-0">' + escHtml(current) + '/' + escHtml(max) + ' ' + i18n.loans_label + '</span>';
    html += '</div>';
    html += '</div></div>';

    document.getElementById('user-card').innerHTML = html;
}

function searchBook() {
    const q = document.getElementById('bookSearchInput').value.trim();
    const el = document.getElementById('book-results');
    if (q.length < 2) { el.innerHTML = ''; return; }

    el.innerHTML = spinner();

    fetch('lending-checkout.php?action=search_book&q=' + encodeURIComponent(q))
        .then(r => r.json())
        .then(data => {
            if (data.error) { el.innerHTML = errorBox(data.error); return; }
            if (!data.length) {
                el.innerHTML = '<p class="text-sm text-gray-500 py-2">' + i18n.no_copy_found + '</p>';
                return;
            }
            el.innerHTML = renderBookTable(data);
        })
        .catch(err => { el.innerHTML = errorBox(i18n.network_error + ': ' + err.message); });
}

function renderBookTable(copies) {
    let h = '<div class="overflow-x-auto"><table class="min-w-full text-sm border-collapse">';
    h += '<thead class="bg-gray-100"><tr>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_title + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_author + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_signature + '</th>';
    h += '<th class="border px-3 py-2 text-left font-medium text-gray-700">' + i18n.col_copy_number + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_status + '</th>';
    h += '<th class="border px-3 py-2 text-center font-medium text-gray-700">' + i18n.col_action + '</th>';
    h += '</tr></thead><tbody>';

    copies.forEach(c => {
        const isReserved = c.status === 'reserviert';
        h += '<tr class="hover:bg-gray-50">';
        h += '<td class="border px-3 py-2 max-w-xs"><span class="line-clamp-2">' + escHtml(c.title) + '</span></td>';
        h += '<td class="border px-3 py-2 text-xs">' + escHtml(c.author || '-') + '</td>';
        h += '<td class="border px-3 py-2 font-mono text-xs">' + escHtml(c.signature || '-') + '</td>';
        h += '<td class="border px-3 py-2 font-mono text-xs">' + escHtml(c.book_number || '-') + '</td>';
        h += '<td class="border px-3 py-2 text-center">';
        if (isReserved) {
            h += '<span class="px-2 py-0.5 text-xs font-medium rounded-full bg-orange-100 text-orange-800">reserviert</span>';
        } else {
            h += '<span class="px-2 py-0.5 text-xs font-medium rounded-full bg-green-100 text-green-800">verfügbar</span>';
        }
        h += '</td>';
        h += '<td class="border px-3 py-2 text-center">';
        h += '<button onclick=\'selectCopy(' + JSON.stringify(c) + ')\' ';
        h += 'class="px-3 py-1 text-white text-xs rounded hover:opacity-80 ';
        h += isReserved ? 'bg-orange-500 hover:bg-orange-600' : 'bg-green-600 hover:bg-green-700';
        h += '">' + (isReserved ? i18n.btn_checkout_anyway : i18n.btn_checkout) + '</button>';
        h += '</td>';
        h += '</tr>';
    });

    h += '</tbody></table></div>';
    return h;
}

function selectCopy(copy) {
    selectedCopy = copy;
    showPanel3();
}

// ─── Panel 3: Bestätigung ────────────────────────────────────────────────────
function showPanel3() {
    const u = selectedUser;
    const c = selectedCopy;

    // Datum berechnen
    const today = new Date();
    const todayStr = today.toISOString().slice(0, 10);
    const loanDays = (u.loan_duration_days > 0) ? parseInt(u.loan_duration_days) : DEFAULT_LOAN_DAYS;
    const dueDate = new Date(today);
    dueDate.setDate(dueDate.getDate() + loanDays);
    const dueDateStr = dueDate.toISOString().slice(0, 10);

    document.getElementById('confirm-user').textContent = u.first_name + ' ' + u.last_name;
    document.getElementById('confirm-user-meta').textContent =
        [u.class_name, u.user_group].filter(Boolean).join(' · ');
    document.getElementById('confirm-book').textContent = c.title + (c.author ? ' — ' + c.author : '');
    document.getElementById('confirm-signature').textContent =
        [c.signature, c.book_number ? 'Nr. ' + c.book_number : ''].filter(Boolean).join(' · ');
    document.getElementById('confirm-loan-date').value = todayStr;
    document.getElementById('confirm-due-date').value  = dueDateStr;
    document.getElementById('confirm-due-date').min    = todayStr;
    document.getElementById('confirm-note').value = '';
    document.getElementById('confirm-error').classList.add('hidden');

    // Warnungen anzeigen
    const warnings = [];
    if (u.status === 'gesperrt') warnings.push('Benutzer ist gesperrt.');
    if ((u.overdue_loans || 0) > 0) warnings.push('Benutzer hat ' + u.overdue_loans + ' überfällige Ausleihe(n).');
    if ((u.current_loans || 0) >= (u.max_loans || 5)) warnings.push('Ausleih-Limit ist erreicht (' + u.current_loans + '/' + u.max_loans + ').');
    if (c.status === 'reserviert') warnings.push('Dieses Exemplar ist vorgemerkt.');

    const warnEl = document.getElementById('confirm-warnings');
    const warnList = document.getElementById('confirm-warnings-list');
    if (warnings.length) {
        warnList.innerHTML = warnings.map(w => '<li>' + escHtml(w) + '</li>').join('');
        warnEl.classList.remove('hidden');
    } else {
        warnEl.classList.add('hidden');
    }

    document.getElementById('panel-confirm').classList.remove('hidden');
    document.getElementById('panel-success').classList.add('hidden');
    document.getElementById('panel-confirm').scrollIntoView({behavior: 'smooth', block: 'start'});
}

function submitCheckout() {
    const dueDate = document.getElementById('confirm-due-date').value;
    const note    = document.getElementById('confirm-note').value;
    const errEl   = document.getElementById('confirm-error');
    errEl.classList.add('hidden');

    if (!dueDate) { errEl.textContent = i18n.return_date_required; errEl.classList.remove('hidden'); return; }

    const btn = document.getElementById('btn-checkout');
    btn.disabled = true;
    btn.textContent = i18n.processing;

    const fd = new FormData();
    fd.append('action',     'checkout');
    fd.append('csrf_token', CSRF_TOKEN);
    fd.append('user_id',    selectedUser.id);
    fd.append('copy_id',    selectedCopy.copy_id);
    fd.append('due_date',   dueDate);
    fd.append('loan_note',  note);

    fetch('lending-checkout.php', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            btn.disabled = false;
            btn.textContent = i18n.process_checkout;
            if (data.error) {
                errEl.textContent = data.error;
                errEl.classList.remove('hidden');
                return;
            }
            showSuccess(data);
        })
        .catch(err => {
            btn.disabled = false;
            btn.textContent = i18n.process_checkout;
            errEl.textContent = i18n.network_error + ': ' + err.message;
            errEl.classList.remove('hidden');
        });
}

function showSuccess(data) {
    const dueFormatted = data.due_date
        ? data.due_date.split('-').reverse().join('.')
        : '-';

    let details = '<p><strong>' + i18n.user_label + ':</strong> ' + escHtml(data.user_name) + '</p>';
    details += '<p><strong>' + i18n.col_title + ':</strong> ' + escHtml(data.book_title) + '</p>';
    details += '<p><strong>' + i18n.signature_label + ':</strong> ' + escHtml(data.signature || '-') + '</p>';
    details += '<p><strong>' + i18n.copy_no_label + ':</strong> ' + escHtml(data.book_number || '-') + '</p>';
    details += '<p><strong>' + i18n.loan_id_label + ':</strong> #' + escHtml(data.loan_id) + '</p>';
    details += '<p><strong>' + i18n.return_by_label + ':</strong> ' + escHtml(dueFormatted) + '</p>';

    document.getElementById('success-details').innerHTML = details;

    const warnEl = document.getElementById('success-warnings');
    if (data.warnings && data.warnings.length) {
        warnEl.innerHTML = '<strong>' + i18n.notices + '</strong><ul class="list-disc list-inside mt-1">'
            + data.warnings.map(w => '<li>' + escHtml(w) + '</li>').join('') + '</ul>';
        warnEl.classList.remove('hidden');
    } else {
        warnEl.classList.add('hidden');
    }

    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.remove('hidden');
    document.getElementById('panel-success').scrollIntoView({behavior: 'smooth', block: 'start'});

    // current_loans im State aktualisieren (fuer naechste Ausleihe)
    if (selectedUser) selectedUser.current_loans = (parseInt(selectedUser.current_loans) || 0) + 1;
}

// ─── Reset-Hilfsfunktionen ───────────────────────────────────────────────────
function nextLoan() {
    // Benutzer behalten, Buch-Panel zuruecksetzen
    selectedCopy = null;
    document.getElementById('panel-book').classList.remove('hidden');
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.add('hidden');
    renderUserCard();
    document.getElementById('bookSearchInput').value = '';
    document.getElementById('book-results').innerHTML = '';
    document.getElementById('bookSearchInput').focus();
    document.getElementById('panel-book').scrollIntoView({behavior: 'smooth', block: 'start'});
}

function resetBook() {
    selectedCopy = null;
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('bookSearchInput').value = '';
    document.getElementById('book-results').innerHTML = '';
    document.getElementById('panel-book').scrollIntoView({behavior: 'smooth', block: 'start'});
    document.getElementById('bookSearchInput').focus();
}

function resetAll() {
    selectedUser = null;
    selectedCopy = null;

    document.getElementById('userSearchInput').value = '';
    document.getElementById('user-results').innerHTML = '';
    document.getElementById('user-search-area').classList.remove('hidden');
    document.getElementById('selected-user-bar').classList.add('hidden');
    document.getElementById('btn-reset-user').classList.add('hidden');
    document.getElementById('ldap-fallback').classList.add('hidden');
    document.getElementById('ldap-search-area').classList.add('hidden');
    document.getElementById('ldap-results').innerHTML = '';
    document.getElementById('ldap-create-panel').classList.add('hidden');

    document.getElementById('panel-book').classList.add('hidden');
    document.getElementById('panel-confirm').classList.add('hidden');
    document.getElementById('panel-success').classList.add('hidden');

    document.getElementById('userSearchInput').focus();
    window.scrollTo({top: 0, behavior: 'smooth'});
}

// ─── UI-Hilfsfunktionen ──────────────────────────────────────────────────────
function spinner() {
    return '<div class="flex items-center gap-2 py-3 text-sm text-gray-500">'
         + '<div class="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>'
         + '<span>' + i18n.searching + '</span></div>';
}

function errorBox(msg) {
    return '<div class="bg-red-50 border border-red-200 rounded-lg p-3 text-sm text-red-700">'
         + escHtml(msg) + '</div>';
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
