<?php
/**
 * Aktive Ausleihen — lending-active.php
 * Server-gerendertes Dashboard: Statistik-Kacheln, Filter, sortierbare Tabelle, Pagination
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();

// ─── GET-Parameter (validiert) ────────────────────────────────────────────────

$filter     = in_array($_GET['filter'] ?? '', ['all', 'overdue', 'today']) ? $_GET['filter'] : 'all';
$class_filter = trim($_GET['class'] ?? '');
$q          = trim($_GET['q'] ?? '');
$page       = max(1, (int)($_GET['page'] ?? 1));
$per_page   = 25;
$offset     = ($page - 1) * $per_page;

// Whitelist fuer Sortierung
$sort_allowed = ['due_date', 'loan_date', 'last_name', 'title'];
$sort = in_array($_GET['sort'] ?? '', $sort_allowed) ? $_GET['sort'] : 'due_date';
$dir  = ($_GET['dir'] ?? 'asc') === 'desc' ? 'DESC' : 'ASC';

// Gegenrichtung fuer Toggle-Links
$toggle_dir = $dir === 'ASC' ? 'desc' : 'asc';

// ─── WHERE-Bedingungen aufbauen ───────────────────────────────────────────────

$where  = ['l.return_date IS NULL'];
$params = [];

if ($filter === 'overdue') {
    $where[] = 'l.due_date < CURDATE()';
} elseif ($filter === 'today') {
    $where[] = 'l.due_date = CURDATE()';
}

if ($class_filter !== '') {
    $where[] = 'u.class_name = ?';
    $params[] = $class_filter;
}

if (mb_strlen($q) >= 2) {
    $like = '%' . $q . '%';
    $where[] = '(u.first_name LIKE ? OR u.last_name LIKE ?
                 OR CONCAT(u.last_name,\' \',u.first_name) LIKE ?
                 OR b.title LIKE ? OR b.author LIKE ?
                 OR bc.book_number LIKE ? OR bc.signature LIKE ?)';
    $params = array_merge($params, [$like, $like, $like, $like, $like, $like, $like]);
}

$where_sql = 'WHERE ' . implode(' AND ', $where);

// Spaltenname fuer ORDER BY (Whitelist bereits geprüft oben)
$sort_col = match ($sort) {
    'loan_date'  => 'l.loan_date',
    'last_name'  => 'u.last_name',
    'title'      => 'b.title',
    default      => 'l.due_date',
};

// ─── Statistik-Kacheln (Gesamt-Counts, keine Filter) ─────────────────────────

$stat_total   = (int)db_get_var("SELECT COUNT(*) FROM loans WHERE return_date IS NULL");
$stat_overdue = (int)db_get_var("SELECT COUNT(*) FROM loans WHERE return_date IS NULL AND due_date < CURDATE()");
$stat_today   = (int)db_get_var("SELECT COUNT(*) FROM loans WHERE return_date IS NULL AND due_date = CURDATE()");

// ─── Klassen-Dropdown (alle Klassen mit aktiven Ausleihen) ───────────────────

$classes = db_get_results(
    "SELECT DISTINCT u.class_name
     FROM loans l JOIN users u ON l.user_id = u.id
     WHERE l.return_date IS NULL AND u.class_name IS NOT NULL AND u.class_name != ''
     ORDER BY u.class_name"
);

// ─── Haupt-Query: Ausleihen mit Filter + Pagination ──────────────────────────

$data_sql = "
    SELECT l.id AS loan_id, l.loan_date, l.due_date, l.extensions,
           CASE
               WHEN l.due_date < CURDATE() THEN 'überfällig'
               WHEN l.due_date = CURDATE() THEN 'heute_fällig'
               ELSE 'aktiv'
           END AS loan_status,
           DATEDIFF(CURDATE(), l.due_date) AS days_overdue,
           u.id AS user_id, u.first_name, u.last_name, u.class_name,
           b.id AS book_id, b.title, b.author,
           bc.id AS copy_id, bc.book_number, bc.signature
    FROM loans l
    JOIN users u    ON l.user_id  = u.id
    JOIN book_copies bc ON l.copy_id = bc.id
    JOIN books b    ON bc.book_id  = b.id
    $where_sql
    ORDER BY $sort_col $dir, l.loan_date ASC
    LIMIT $per_page OFFSET $offset";

$loans = db_get_results($data_sql, $params);

// Gesamt-Count fuer Pagination
$count_sql   = "SELECT COUNT(*) FROM loans l
                JOIN users u ON l.user_id = u.id
                JOIN book_copies bc ON l.copy_id = bc.id
                JOIN books b ON bc.book_id = b.id
                $where_sql";
$total_rows  = (int)db_get_var($count_sql, $params);
$total_pages = max(1, (int)ceil($total_rows / $per_page));

// ─── Hilfsfunktionen (PHP) ───────────────────────────────────────────────────

function sort_link(string $col, string $label, string $current_sort, string $current_dir): string {
    $toggle = ($current_sort === $col && $current_dir === 'ASC') ? 'desc' : 'asc';
    $arrow  = '';
    if ($current_sort === $col) {
        $arrow = $current_dir === 'ASC' ? ' ↑' : ' ↓';
    }
    $params = array_merge($_GET, ['sort' => $col, 'dir' => $toggle, 'page' => 1]);
    $url = '?' . http_build_query($params);
    return '<a href="' . esc_attr($url) . '" class="hover:text-blue-200">'
         . esc_html($label . $arrow) . '</a>';
}

function page_url(int $p): string {
    $params = array_merge($_GET, ['page' => $p]);
    return '?' . http_build_query($params);
}

function filter_url(string $f): string {
    $params = array_merge($_GET, ['filter' => $f, 'page' => 1]);
    return '?' . http_build_query($params);
}

// ─── Seiten-Setup ─────────────────────────────────────────────────────────────

$page_title = $lang['admin_active_loans'] ?? 'Aktive Ausleihen';

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Seitenkopf -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?></h1>
            <p class="mt-1 text-sm text-gray-500">
                <?php echo $total_rows; ?> <?php echo $lang['loan_count_label'] ?? 'Ausleihe(n)'; ?>
                <?php if ($filter !== 'all' || $q !== '' || $class_filter !== ''): ?>
                    (<?php echo $lang['filtered'] ?? 'gefiltert'; ?>)
                <?php endif; ?>
            </p>
        </div>
        <div class="flex gap-3 text-sm">
            <a href="lending-checkout.php"
               class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 font-medium">
                + <?php echo $lang['new_checkout'] ?? 'Neue Ausleihe'; ?>
            </a>
            <a href="lending-return.php"
               class="px-4 py-2 bg-orange-500 text-white rounded-md hover:bg-orange-600 font-medium">
                <?php echo $lang['process_return'] ?? 'Rückgabe buchen'; ?>
            </a>
        </div>
    </div>

    <!-- ═══════════════════════════════════════════ Statistik-Kacheln -->
    <div class="grid grid-cols-1 sm:grid-cols-3 gap-4 mb-6">

        <a href="<?php echo esc_attr(filter_url('all')); ?>"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'all' ? 'ring-2 ring-blue-500' : ''; ?>">
            <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold text-gray-900"><?php echo $stat_total; ?></p>
                <p class="text-sm text-gray-500"><?php echo $lang['total_active_loans'] ?? 'Aktive Ausleihen gesamt'; ?></p>
            </div>
        </a>

        <a href="<?php echo esc_attr(filter_url('overdue')); ?>"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'overdue' ? 'ring-2 ring-red-500' : ''; ?>">
            <div class="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M12 9v2m0 4h.01M10.29 3.86L1.82 18a2 2 0 001.71 3h16.94a2 2 0 001.71-3L13.71 3.86a2 2 0 00-3.42 0z"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold <?php echo $stat_overdue > 0 ? 'text-red-600' : 'text-gray-900'; ?>">
                    <?php echo $stat_overdue; ?>
                </p>
                <p class="text-sm text-gray-500"><?php echo $lang['overdue'] ?? 'Überfällig'; ?></p>
            </div>
        </a>

        <a href="<?php echo esc_attr(filter_url('today')); ?>"
           class="bg-white rounded-lg shadow p-5 flex items-center gap-4 hover:bg-gray-50 transition
                  <?php echo $filter === 'today' ? 'ring-2 ring-yellow-500' : ''; ?>">
            <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center flex-shrink-0">
                <svg class="w-6 h-6 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div>
                <p class="text-2xl font-bold <?php echo $stat_today > 0 ? 'text-yellow-600' : 'text-gray-900'; ?>">
                    <?php echo $stat_today; ?>
                </p>
                <p class="text-sm text-gray-500"><?php echo $lang['due_today'] ?? 'Heute fällig'; ?></p>
            </div>
        </a>

    </div>

    <!-- ═══════════════════════════════════════════ Filter-Leiste -->
    <div class="bg-white rounded-lg shadow mb-4">
        <form method="GET" class="p-4 flex flex-wrap gap-3 items-end">
            <input type="hidden" name="filter" value="<?php echo esc_attr($filter); ?>">

            <!-- Suche -->
            <div class="flex-1 min-w-48">
                <label class="block text-xs font-medium text-gray-600 mb-1"><?php echo $lang['search'] ?? 'Suche'; ?></label>
                <input type="text" name="q" value="<?php echo esc_attr($q); ?>"
                       placeholder="<?php echo $lang['active_loans_search_placeholder'] ?? 'Name, Titel, Signatur, Buchungsnr....'; ?>"
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Klassen-Filter -->
            <?php if (!empty($classes)): ?>
            <div>
                <label class="block text-xs font-medium text-gray-600 mb-1"><?php echo $lang['class_name'] ?? 'Klasse'; ?></label>
                <select name="class"
                        class="px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-2 focus:ring-blue-500">
                    <option value=""><?php echo $lang['all_classes'] ?? 'Alle Klassen'; ?></option>
                    <?php foreach ($classes as $c): ?>
                        <option value="<?php echo esc_attr($c['class_name']); ?>"
                            <?php echo $class_filter === $c['class_name'] ? 'selected' : ''; ?>>
                            <?php echo esc_html($c['class_name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>

            <div class="flex gap-2">
                <button type="submit"
                        class="px-4 py-2 bg-blue-600 text-white text-sm rounded-md hover:bg-blue-700">
                    <?php echo $lang['filter'] ?? 'Filtern'; ?>
                </button>
                <?php if ($q !== '' || $class_filter !== '' || $filter !== 'all'): ?>
                <a href="lending-active.php"
                   class="px-4 py-2 border border-gray-300 text-gray-600 text-sm rounded-md hover:bg-gray-50">
                    <?php echo $lang['reset'] ?? 'Zurücksetzen'; ?>
                </a>
                <?php endif; ?>
            </div>
        </form>
    </div>

    <!-- ═══════════════════════════════════════════ Tabelle -->
    <div class="bg-white rounded-lg shadow overflow-hidden">

        <?php if (empty($loans)): ?>
            <div class="text-center py-16">
                <svg class="mx-auto h-12 w-12 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                          d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414A1 1 0 0119 9.414V19a2 2 0 01-2 2z"/>
                </svg>
                <p class="mt-3 text-gray-500 font-medium"><?php echo $lang['no_active_loans_found'] ?? 'Keine aktiven Ausleihen gefunden.'; ?></p>
                <?php if ($q !== '' || $class_filter !== '' || $filter !== 'all'): ?>
                    <a href="lending-active.php" class="mt-2 inline-block text-sm text-blue-600 hover:underline">
                        <?php echo $lang['reset_filter'] ?? 'Filter zurücksetzen'; ?>
                    </a>
                <?php endif; ?>
            </div>

        <?php else: ?>

        <div class="overflow-x-auto">
        <table class="min-w-full">
            <thead>
                <tr class="bg-gray-700 text-white text-sm">
                    <th class="px-4 py-3 text-left font-medium w-4"></th>
                    <th class="px-4 py-3 text-left font-medium">
                        <?php echo sort_link('last_name', $lang['users'] ?? 'Benutzer', $sort, $dir); ?>
                    </th>
                    <th class="px-4 py-3 text-left font-medium">
                        <?php echo sort_link('title', ($lang['book_title'] ?? 'Titel') . ' / ' . ($lang['book_author'] ?? 'Autor'), $sort, $dir); ?>
                    </th>
                    <th class="px-4 py-3 text-left font-medium hidden lg:table-cell"><?php echo ($lang['signature'] ?? 'Signatur') . ' / ' . ($lang['book_number'] ?? 'Nr.'); ?></th>
                    <th class="px-4 py-3 text-center font-medium hidden md:table-cell">
                        <?php echo sort_link('loan_date', $lang['loan_loaned_on'] ?? 'Ausgeliehen', $sort, $dir); ?>
                    </th>
                    <th class="px-4 py-3 text-center font-medium">
                        <?php echo sort_link('due_date', $lang['loan_due_date'] ?? 'Fällig', $sort, $dir); ?>
                    </th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['status'] ?? 'Status'; ?></th>
                    <th class="px-4 py-3 text-center font-medium"><?php echo $lang['actions'] ?? 'Aktion'; ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
            <?php foreach ($loans as $loan):
                $status   = $loan['loan_status'];
                $overdue  = $status === 'überfällig';
                $today    = $status === 'heute_fällig';
                $due_fmt  = date('d.m.Y', strtotime($loan['due_date']));
                $loan_fmt = date('d.m.Y', strtotime($loan['loan_date']));

                // Zeilenhintergrund
                $row_bg = $overdue ? 'bg-red-50 hover:bg-red-100' : ($today ? 'bg-yellow-50 hover:bg-yellow-100' : 'hover:bg-gray-50');
                // Linker Farbbalken
                $bar_color = $overdue ? 'bg-red-500' : ($today ? 'bg-yellow-400' : 'bg-green-400');
            ?>
                <tr class="<?php echo $row_bg; ?> transition-colors">

                    <!-- Farbbalken -->
                    <td class="px-0 py-0 w-1">
                        <div class="<?php echo $bar_color; ?> w-1 h-full min-h-[52px]"></div>
                    </td>

                    <!-- Benutzer -->
                    <td class="px-4 py-3">
                        <p class="font-medium text-gray-900 text-sm">
                            <?php echo esc_html($loan['last_name'] . ', ' . $loan['first_name']); ?>
                        </p>
                        <?php if ($loan['class_name']): ?>
                            <span class="inline-block mt-0.5 px-1.5 py-0.5 bg-gray-100 text-gray-600 text-xs rounded">
                                <?php echo esc_html($loan['class_name']); ?>
                            </span>
                        <?php endif; ?>
                    </td>

                    <!-- Titel / Autor -->
                    <td class="px-4 py-3 max-w-xs">
                        <p class="text-sm font-medium text-gray-900 truncate" title="<?php echo esc_attr($loan['title']); ?>">
                            <?php echo esc_html($loan['title']); ?>
                        </p>
                        <?php if ($loan['author']): ?>
                            <p class="text-xs text-gray-500 truncate"><?php echo esc_html($loan['author']); ?></p>
                        <?php endif; ?>
                    </td>

                    <!-- Signatur / Buchungsnr. -->
                    <td class="px-4 py-3 hidden lg:table-cell">
                        <?php if ($loan['signature']): ?>
                            <p class="text-xs font-mono text-gray-700"><?php echo esc_html($loan['signature']); ?></p>
                        <?php endif; ?>
                        <?php if ($loan['book_number']): ?>
                            <p class="text-xs font-mono text-gray-400"><?php echo esc_html($loan['book_number']); ?></p>
                        <?php endif; ?>
                    </td>

                    <!-- Ausgeliehen am -->
                    <td class="px-4 py-3 text-center text-sm text-gray-600 hidden md:table-cell whitespace-nowrap">
                        <?php echo esc_html($loan_fmt); ?>
                        <?php if ((int)$loan['extensions'] > 0): ?>
                            <span class="block text-xs text-gray-400">
                                <?php echo esc_html($loan['extensions']); ?>&times; <?php echo $lang['extended_abbr'] ?? 'verl.'; ?>
                            </span>
                        <?php endif; ?>
                    </td>

                    <!-- Fällig am -->
                    <td class="px-4 py-3 text-center whitespace-nowrap">
                        <p class="text-sm font-medium <?php echo $overdue ? 'text-red-700' : ($today ? 'text-yellow-700' : 'text-gray-700'); ?>">
                            <?php echo esc_html($due_fmt); ?>
                        </p>
                        <?php if ($overdue && $loan['days_overdue'] > 0): ?>
                            <p class="text-xs text-red-600 font-semibold">
                                +<?php echo esc_html($loan['days_overdue']); ?> <?php echo $lang['days_label'] ?? 'Tage'; ?>
                            </p>
                        <?php endif; ?>
                    </td>

                    <!-- Status-Badge -->
                    <td class="px-4 py-3 text-center">
                        <?php if ($overdue): ?>
                            <span class="px-2 py-1 text-xs font-medium rounded-full bg-red-100 text-red-800"><?php echo $lang['overdue'] ?? 'überfällig'; ?></span>
                        <?php elseif ($today): ?>
                            <span class="px-2 py-1 text-xs font-medium rounded-full bg-yellow-100 text-yellow-800"><?php echo $lang['due_today'] ?? 'heute fällig'; ?></span>
                        <?php else: ?>
                            <span class="px-2 py-1 text-xs font-medium rounded-full bg-green-100 text-green-800"><?php echo $lang['status_active_loan'] ?? 'aktiv'; ?></span>
                        <?php endif; ?>
                    </td>

                    <!-- Aktion -->
                    <td class="px-4 py-3 text-center">
                        <a href="lending-return.php?q=<?php echo urlencode($loan['book_number'] ?? ''); ?>"
                           class="inline-block px-3 py-1 bg-orange-500 text-white text-xs rounded hover:bg-orange-600 whitespace-nowrap">
                            <?php echo $lang['return_book'] ?? 'Rückgabe'; ?>
                        </a>
                    </td>

                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
        </div>

        <!-- Footer: Anzahl + Pagination -->
        <div class="px-4 py-3 border-t bg-gray-50 flex items-center justify-between flex-wrap gap-3">

            <p class="text-sm text-gray-600">
                <?php
                $from = $offset + 1;
                $to   = min($offset + $per_page, $total_rows);
                echo esc_html(
                    ($lang['showing'] ?? 'Zeige') . ' ' . $from . '–' . $to . ' ' .
                    ($lang['of'] ?? 'von') . ' ' . $total_rows . ' ' .
                    ($lang['loan_count_label'] ?? 'Ausleihe(n)')
                );
                ?>
            </p>

            <?php if ($total_pages > 1): ?>
            <nav class="flex items-center gap-1">
                <?php if ($page > 1): ?>
                    <a href="<?php echo esc_attr(page_url($page - 1)); ?>"
                       class="px-3 py-1.5 text-sm border border-gray-300 rounded hover:bg-gray-100">
                        &larr;
                    </a>
                <?php endif; ?>

                <?php
                // Seitenzahlen: erste, letzte, aktuelle ± 2
                $pages_to_show = [];
                for ($p = max(1, $page - 2); $p <= min($total_pages, $page + 2); $p++) {
                    $pages_to_show[] = $p;
                }
                if (!in_array(1, $pages_to_show)) {
                    echo '<a href="' . esc_attr(page_url(1)) . '" class="px-3 py-1.5 text-sm border border-gray-300 rounded hover:bg-gray-100">1</a>';
                    if ($pages_to_show[0] > 2) echo '<span class="px-1 text-gray-400">…</span>';
                }
                foreach ($pages_to_show as $p):
                ?>
                    <a href="<?php echo esc_attr(page_url($p)); ?>"
                       class="px-3 py-1.5 text-sm border rounded <?php echo $p === $page ? 'bg-blue-600 text-white border-blue-600' : 'border-gray-300 hover:bg-gray-100'; ?>">
                        <?php echo $p; ?>
                    </a>
                <?php endforeach;
                if (!in_array($total_pages, $pages_to_show)) {
                    if (end($pages_to_show) < $total_pages - 1) echo '<span class="px-1 text-gray-400">…</span>';
                    echo '<a href="' . esc_attr(page_url($total_pages)) . '" class="px-3 py-1.5 text-sm border border-gray-300 rounded hover:bg-gray-100">' . $total_pages . '</a>';
                }
                ?>

                <?php if ($page < $total_pages): ?>
                    <a href="<?php echo esc_attr(page_url($page + 1)); ?>"
                       class="px-3 py-1.5 text-sm border border-gray-300 rounded hover:bg-gray-100">
                        &rarr;
                    </a>
                <?php endif; ?>
            </nav>
            <?php endif; ?>

        </div>

        <?php endif; // empty($loans) ?>
    </div>

</div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
