<?php
/**
 * Inventur starten — inventory-start.php
 * Bestandsübersicht (Kacheln + Kategorie-Tabelle) und
 * Verwaltung der PHP-Session-basierten Inventur-Sitzung.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = (int)$current_user['id'];
$page_title   = $lang['admin_inventory_start'] ?? 'Inventur starten';

// ── POST-Handler ──────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf_token($_POST['csrf_token'] ?? '');
    $action = $_POST['action'] ?? '';

    if ($action === 'start_inventory') {
        $_SESSION['inventory'] = [
            'started_at' => time(),
            'admin_id'   => $admin_id,
            'checked'    => [],   // copy_id => [title, book_number, old_status, new_status, time]
        ];
        log_activity($admin_id, 'inventory_started', 'system', 0, null);
        header('Location: ' . BASE_URL . '/admin/inventory-ongoing.php');
        exit;
    }

    if ($action === 'clear_inventory') {
        unset($_SESSION['inventory']);
        set_flash('info', $lang['inventory_session_ended'] ?? 'Inventur-Sitzung beendet.');
        header('Location: ' . BASE_URL . '/admin/inventory-start.php');
        exit;
    }
}

// ── Statistiken ────────────────────────────────────────────────────────────────
$status_rows = db_get_results("SELECT status, COUNT(*) AS cnt FROM book_copies GROUP BY status");
$stats = [];
foreach ($status_rows as $r) {
    $stats[$r['status']] = (int)$r['cnt'];
}
$total_copies = array_sum($stats);
$total_books  = (int)db_get_var("SELECT COUNT(*) FROM books");
$total_titles_with_copies = (int)db_get_var(
    "SELECT COUNT(DISTINCT book_id) FROM book_copies WHERE status NOT IN ('aussortiert')"
);

// Kategorien-Aufschlüsselung
$by_category = db_get_results(
    "SELECT bc.id, bc.name, bc.code,
            COUNT(bcp.id)                                      AS total,
            COALESCE(SUM(bcp.status = 'verfügbar'),    0)      AS avail,
            COALESCE(SUM(bcp.status = 'ausgeliehen'),  0)      AS loaned,
            COALESCE(SUM(bcp.status = 'reserviert'),   0)      AS reserved,
            COALESCE(SUM(bcp.status = 'in_bearbeitung'),0)     AS processing,
            COALESCE(SUM(bcp.status = 'verloren'),     0)      AS lost,
            COALESCE(SUM(bcp.status = 'aussortiert'),  0)      AS discarded
     FROM book_categories bc
     LEFT JOIN book_subcategories bs  ON bs.category_id  = bc.id
     LEFT JOIN books b                ON b.subcategory_id = bs.id
     LEFT JOIN book_copies bcp        ON bcp.book_id      = b.id
     GROUP BY bc.id, bc.name, bc.code
     HAVING total > 0
     ORDER BY bc.sort_order, bc.name"
);

// Zugangsjahre (letzte 8 Jahre)
$by_year = db_get_results(
    "SELECT YEAR(COALESCE(acquired_at, acquisition_date, created_at)) AS yr,
            COUNT(*) AS cnt
     FROM book_copies
     WHERE COALESCE(acquired_at, acquisition_date, created_at) IS NOT NULL
     GROUP BY yr
     ORDER BY yr DESC
     LIMIT 8"
);

$inventory_active = isset($_SESSION['inventory']);
$inv_checked      = $inventory_active ? count($_SESSION['inventory']['checked']) : 0;
$inv_started_ago  = $inventory_active ? round((time() - $_SESSION['inventory']['started_at']) / 60) : 0;

$csrf_token = generate_csrf_token();

include __DIR__ . '/includes/header.php';

$flash = get_flash();
?>

<?php if ($flash): ?>
    <?php $fc = $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : ($flash['type'] === 'warning' ? 'yellow' : 'blue')); ?>
    <div class="mb-6 p-4 rounded-lg bg-<?= $fc ?>-50 border border-<?= $fc ?>-200 text-<?= $fc ?>-800">
        <?= esc_html($flash['message']) ?>
    </div>
<?php endif; ?>

<!-- Aktive Sitzungs-Banner -->
<?php if ($inventory_active): ?>
    <div class="mb-6 bg-yellow-50 border border-yellow-300 rounded-lg p-4 flex items-center justify-between">
        <div class="flex items-center gap-3">
            <svg class="h-6 w-6 text-yellow-500 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                      d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
            <div>
                <p class="font-semibold text-yellow-900"><?= sprintf($lang['inventory_running_since'] ?? 'Inventur läuft seit %d Minuten', $inv_started_ago) ?></p>
                <p class="text-sm text-yellow-700"><?= $inv_checked ?> <?= $lang['copies_already_checked'] ?? 'Exemplar(e) bereits geprüft' ?></p>
            </div>
        </div>
        <div class="flex gap-3">
            <a href="<?= BASE_URL ?>/admin/inventory-ongoing.php"
               class="px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700 text-sm font-medium">
                <?= $lang['continue_inventory'] ?? 'Weiter zur Inventur' ?>
            </a>
            <form method="POST" onsubmit="return confirm('<?= esc_attr($lang['confirm_end_session'] ?? 'Inventur-Sitzung wirklich beenden? Alle gespeicherten Prüfungen gehen verloren.') ?>')">
                <?= csrf_field() ?>
                <input type="hidden" name="action" value="clear_inventory">
                <button type="submit" class="px-4 py-2 bg-red-100 text-red-700 rounded-lg hover:bg-red-200 text-sm font-medium">
                    <?= $lang['end_session'] ?? 'Sitzung beenden' ?>
                </button>
            </form>
        </div>
    </div>
<?php endif; ?>

<!-- ── Statistik-Kacheln ──────────────────────────────────────────────────── -->
<div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">

    <?php
    $tiles = [
        ['label' => $lang['book_titles'] ?? 'Buchtitel',       'val' => number_format($total_books,  0, ',', '.'), 'color' => 'blue',   'icon' => 'M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253'],
        ['label' => $lang['copies'] ?? 'Exemplare',            'val' => number_format($total_copies, 0, ',', '.'), 'color' => 'indigo', 'icon' => 'M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10'],
        ['label' => $lang['available'] ?? 'Verfügbar',         'val' => number_format($stats['verfügbar'] ?? 0, 0, ',', '.'),  'color' => 'green',  'icon' => 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'],
        ['label' => $lang['loaned'] ?? 'Ausgeliehen',          'val' => number_format($stats['ausgeliehen'] ?? 0, 0, ',', '.'), 'color' => 'orange', 'icon' => 'M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z'],
        ['label' => $lang['reserved'] ?? 'Reserviert',         'val' => number_format($stats['reserviert'] ?? 0, 0, ',', '.'),  'color' => 'yellow', 'icon' => 'M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z'],
        ['label' => $lang['in_processing'] ?? 'In Bearbeitung','val' => number_format($stats['in_bearbeitung'] ?? 0, 0, ',', '.'), 'color' => 'blue', 'icon' => 'M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z'],
        ['label' => $lang['lost'] ?? 'Verloren',               'val' => number_format($stats['verloren'] ?? 0, 0, ',', '.'),    'color' => 'red',   'icon' => 'M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z'],
        ['label' => $lang['discarded'] ?? 'Aussortiert',       'val' => number_format($stats['aussortiert'] ?? 0, 0, ',', '.'), 'color' => 'gray',  'icon' => 'M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16'],
    ];
    foreach ($tiles as $t): ?>
        <div class="bg-white rounded-lg shadow p-5">
            <div class="flex items-center gap-3">
                <div class="flex-shrink-0 w-10 h-10 rounded-lg bg-<?= $t['color'] ?>-100 flex items-center justify-center">
                    <svg class="h-5 w-5 text-<?= $t['color'] ?>-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $t['icon'] ?>"/>
                    </svg>
                </div>
                <div>
                    <p class="text-xs text-gray-500"><?= $t['label'] ?></p>
                    <p class="text-xl font-bold text-gray-900"><?= $t['val'] ?></p>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
</div>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">

    <!-- Inventur starten -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-900 mb-4">
            <svg class="inline h-5 w-5 mr-1 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"/>
            </svg>
            <?= $lang['conduct_inventory'] ?? 'Inventur durchführen' ?>
        </h2>
        <p class="text-sm text-gray-600 mb-4">
            <?= $lang['inventory_description'] ?? 'Starten Sie eine Inventur-Sitzung, um Exemplare physisch zu prüfen und ihren Status (verfügbar, verloren, aussortiert) zu aktualisieren. Die Sitzung ist browserseitig gespeichert und bleibt bis zum manuellen Abschluss aktiv.' ?>
        </p>
        <?php if ($inventory_active): ?>
            <a href="<?= BASE_URL ?>/admin/inventory-ongoing.php"
               class="block w-full text-center px-4 py-2.5 bg-yellow-500 text-white rounded-lg hover:bg-yellow-600 font-medium">
                <?= $lang['resume_inventory'] ?? 'Laufende Inventur fortsetzen' ?>
            </a>
        <?php else: ?>
            <form method="POST">
                <?= csrf_field() ?>
                <input type="hidden" name="action" value="start_inventory">
                <button type="submit"
                        class="w-full px-4 py-2.5 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium">
                    <?= $lang['start_new_inventory'] ?? 'Neue Inventur starten' ?>
                </button>
            </form>
        <?php endif; ?>

        <div class="mt-4 pt-4 border-t">
            <a href="<?= BASE_URL ?>/admin/inventory-history.php"
               class="text-sm text-blue-600 hover:text-blue-800">
                <?= $lang['manage_lost_discarded'] ?? 'Verloren &amp; Aussortierte Exemplare verwalten' ?> &rarr;
            </a>
        </div>
    </div>

    <!-- Zugangsjahre -->
    <div class="bg-white rounded-lg shadow p-6 lg:col-span-2">
        <h2 class="text-lg font-semibold text-gray-900 mb-4"><?= $lang['acquisitions_by_year'] ?? 'Zugang nach Jahr' ?></h2>
        <?php if (empty($by_year)): ?>
            <p class="text-sm text-gray-500"><?= $lang['no_acquisition_data'] ?? 'Keine Zugangsdaten vorhanden.' ?></p>
        <?php else: ?>
            <?php
            $max_yr = max(array_column($by_year, 'cnt'));
            foreach ($by_year as $yr):
                $pct = $max_yr > 0 ? round($yr['cnt'] / $max_yr * 100) : 0;
            ?>
                <div class="flex items-center gap-3 mb-2">
                    <span class="text-sm text-gray-600 w-12 text-right flex-shrink-0"><?= (int)$yr['yr'] ?></span>
                    <div class="flex-1 bg-gray-100 rounded-full h-5 relative">
                        <div class="bg-blue-500 h-5 rounded-full" style="width:<?= $pct ?>%"></div>
                        <span class="absolute inset-0 flex items-center justify-end pr-2 text-xs font-medium text-gray-700">
                            <?= number_format($yr['cnt'], 0, ',', '.') ?>
                        </span>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
</div>

<!-- ── Kategorie-Tabelle ──────────────────────────────────────────────────── -->
<div class="bg-white rounded-lg shadow mb-8">
    <div class="p-5 border-b">
        <h2 class="text-lg font-semibold text-gray-900"><?= $lang['stock_by_category'] ?? 'Bestand nach Kategorie' ?></h2>
    </div>
    <div class="overflow-x-auto">
        <table class="min-w-full text-sm">
            <thead class="bg-gray-50 text-gray-500 text-xs uppercase">
                <tr>
                    <th class="px-5 py-3 text-left font-medium"><?= $lang['book_category'] ?? 'Kategorie' ?></th>
                    <th class="px-4 py-3 text-right font-medium"><?= $lang['total'] ?? 'Gesamt' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-green-700"><?= $lang['available'] ?? 'Verfügbar' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-orange-700"><?= $lang['loaned'] ?? 'Ausgeliehen' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-yellow-700"><?= $lang['reserved'] ?? 'Reserviert' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-blue-700"><?= $lang['processing_abbr'] ?? 'Bearb.' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-red-700"><?= $lang['lost'] ?? 'Verloren' ?></th>
                    <th class="px-4 py-3 text-right font-medium text-gray-500"><?= $lang['discarded'] ?? 'Aussortiert' ?></th>
                    <th class="px-4 py-3 text-right font-medium"><?= $lang['available_pct'] ?? 'Verfügbar %' ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
                <?php
                $totals = array_fill_keys(['total','avail','loaned','reserved','processing','lost','discarded'], 0);
                foreach ($by_category as $cat):
                    foreach ($totals as $k => $_) $totals[$k] += (int)$cat[$k];
                    $pct = $cat['total'] > 0 ? round($cat['avail'] / $cat['total'] * 100) : 0;
                ?>
                    <tr class="hover:bg-gray-50">
                        <td class="px-5 py-3 font-medium text-gray-900">
                            <span class="inline-block bg-gray-100 text-gray-600 text-xs font-mono px-1.5 py-0.5 rounded mr-2"><?= esc_html($cat['code']) ?></span>
                            <?= esc_html($cat['name']) ?>
                        </td>
                        <td class="px-4 py-3 text-right font-medium text-gray-900"><?= number_format($cat['total'], 0, ',', '.') ?></td>
                        <td class="px-4 py-3 text-right text-green-700"><?= $cat['avail'] ?></td>
                        <td class="px-4 py-3 text-right text-orange-700"><?= $cat['loaned'] ?></td>
                        <td class="px-4 py-3 text-right text-yellow-700"><?= $cat['reserved'] ?></td>
                        <td class="px-4 py-3 text-right text-blue-700"><?= $cat['processing'] ?: '–' ?></td>
                        <td class="px-4 py-3 text-right text-red-700"><?= $cat['lost'] ?: '–' ?></td>
                        <td class="px-4 py-3 text-right text-gray-500"><?= $cat['discarded'] ?: '–' ?></td>
                        <td class="px-4 py-3 text-right">
                            <div class="flex items-center justify-end gap-2">
                                <div class="w-16 bg-gray-200 rounded-full h-2">
                                    <div class="h-2 rounded-full <?= $pct >= 70 ? 'bg-green-500' : ($pct >= 40 ? 'bg-yellow-500' : 'bg-red-500') ?>"
                                         style="width:<?= $pct ?>%"></div>
                                </div>
                                <span class="text-xs text-gray-600 w-8 text-right"><?= $pct ?>%</span>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot class="bg-gray-50 font-semibold border-t-2 border-gray-200">
                <tr>
                    <td class="px-5 py-3 text-gray-700"><?= $lang['total'] ?? 'Gesamt' ?></td>
                    <td class="px-4 py-3 text-right"><?= number_format($totals['total'], 0, ',', '.') ?></td>
                    <td class="px-4 py-3 text-right text-green-700"><?= number_format($totals['avail'], 0, ',', '.') ?></td>
                    <td class="px-4 py-3 text-right text-orange-700"><?= number_format($totals['loaned'], 0, ',', '.') ?></td>
                    <td class="px-4 py-3 text-right text-yellow-700"><?= number_format($totals['reserved'], 0, ',', '.') ?></td>
                    <td class="px-4 py-3 text-right text-blue-700"><?= $totals['processing'] ?: '–' ?></td>
                    <td class="px-4 py-3 text-right text-red-700"><?= $totals['lost'] ?: '–' ?></td>
                    <td class="px-4 py-3 text-right text-gray-500"><?= $totals['discarded'] ?: '–' ?></td>
                    <td class="px-4 py-3 text-right text-xs text-gray-500">
                        <?= $totals['total'] > 0 ? round($totals['avail'] / $totals['total'] * 100) : 0 ?>%
                    </td>
                </tr>
            </tfoot>
        </table>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
