<?php
/**
 * Laufende Inventur — inventory-ongoing.php
 * AJAX-gesteuerte Exemplarprüfung. Suche per Buchnummer/Signatur,
 * Status-Änderung per AJAX-POST, Fortschritt in $_SESSION['inventory'].
 *
 * AJAX GET  ?action=search           → JSON: {copy} oder {error}
 * AJAX POST ?action=set_status       → JSON: {success, new_status, checked_count}
 * AJAX POST ?action=mark_checked     → JSON: {success, checked_count}
 * AJAX POST ?action=finish           → JSON: {success, redirect}
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = (int)$current_user['id'];
$page_title   = $lang['admin_inventory_ongoing'] ?? 'Laufende Inventur';

// Ohne aktive Sitzung zurück zu Start
if (!isset($_SESSION['inventory'])) {
    set_flash('warning', $lang['no_active_inventory_session'] ?? 'Keine aktive Inventur-Sitzung. Bitte zuerst starten.');
    header('Location: ' . BASE_URL . '/admin/inventory-start.php');
    exit;
}

$inventory = &$_SESSION['inventory'];

// ── AJAX GET: Exemplar suchen ─────────────────────────────────────────────────
if (isset($_GET['action'])) {
    header('Content-Type: application/json; charset=utf-8');

    if ($_GET['action'] === 'search') {
        $q = trim($_GET['q'] ?? '');
        if ($q === '') {
            echo json_encode(['error' => 'Suchbegriff fehlt.']);
            exit;
        }
        $copy = db_get_row(
            "SELECT bc.id, bc.book_number, bc.copy_index, bc.signature,
                    bc.status, bc.location, bc.book_id,
                    b.title, b.author, b.isbn, b.image_url,
                    bs.code AS subcategory_code
             FROM book_copies bc
             JOIN books b ON bc.book_id = b.id
             LEFT JOIN book_subcategories bs ON b.subcategory_id = bs.id
             WHERE bc.book_number = ? OR bc.accession_number = ? OR bc.signature = ?
             LIMIT 1",
            [$q, $q, $q]
        );
        if (!$copy) {
            echo json_encode(['error' => 'Kein Exemplar gefunden f&uuml;r: ' . esc_html($q)]);
            exit;
        }
        $copy['already_checked'] = isset($inventory['checked'][(int)$copy['id']]);
        echo json_encode(['copy' => $copy]);
        exit;
    }

    echo json_encode(['error' => 'Unbekannte Aktion']);
    exit;
}

// ── AJAX POST ─────────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json; charset=utf-8');
    verify_csrf_token($_POST['csrf_token'] ?? '');
    $action = $_POST['action'] ?? '';

    if ($action === 'set_status') {
        $copy_id    = (int)($_POST['copy_id'] ?? 0);
        $new_status = $_POST['status'] ?? '';
        $allowed    = ['verfügbar', 'verloren', 'aussortiert', 'in_bearbeitung'];

        if (!in_array($new_status, $allowed, true)) {
            echo json_encode(['error' => 'Ung&uuml;ltiger Status.']);
            exit;
        }

        $copy = db_get_row(
            "SELECT bc.*, b.title FROM book_copies bc JOIN books b ON bc.book_id = b.id WHERE bc.id = ?",
            [$copy_id]
        );
        if (!$copy) {
            echo json_encode(['error' => 'Exemplar nicht gefunden.']);
            exit;
        }
        // Aktiv ausgeliehene Exemplare nicht anfassen
        if ($copy['status'] === 'ausgeliehen') {
            $active = db_get_var(
                "SELECT COUNT(*) FROM loans WHERE copy_id = ? AND return_date IS NULL",
                [$copy_id]
            );
            if ($active > 0) {
                echo json_encode(['error' => 'Dieses Exemplar ist derzeit aktiv ausgeliehen und kann nicht ge&auml;ndert werden.']);
                exit;
            }
        }

        db_update('book_copies', ['status' => $new_status], ['id' => $copy_id]);
        log_activity($admin_id, 'inventory_status_changed', 'book_copies', $copy_id,
            json_encode(['old' => $copy['status'], 'new' => $new_status, 'title' => mb_substr($copy['title'], 0, 80)]));

        $inventory['checked'][$copy_id] = [
            'title'       => mb_substr($copy['title'], 0, 60),
            'book_number' => $copy['book_number'],
            'old_status'  => $copy['status'],
            'new_status'  => $new_status,
            'time'        => time(),
        ];

        echo json_encode([
            'success'       => true,
            'new_status'    => $new_status,
            'checked_count' => count($inventory['checked']),
        ]);
        exit;
    }

    if ($action === 'mark_checked') {
        $copy_id = (int)($_POST['copy_id'] ?? 0);
        $copy = db_get_row(
            "SELECT bc.*, b.title FROM book_copies bc JOIN books b ON bc.book_id = b.id WHERE bc.id = ?",
            [$copy_id]
        );
        if ($copy) {
            $inventory['checked'][$copy_id] = [
                'title'       => mb_substr($copy['title'], 0, 60),
                'book_number' => $copy['book_number'],
                'old_status'  => $copy['status'],
                'new_status'  => $copy['status'],
                'time'        => time(),
            ];
            echo json_encode(['success' => true, 'checked_count' => count($inventory['checked'])]);
        } else {
            echo json_encode(['error' => 'Exemplar nicht gefunden.']);
        }
        exit;
    }

    if ($action === 'finish') {
        $checked_count = count($inventory['checked']);
        $duration_min  = round((time() - $inventory['started_at']) / 60);
        log_activity($admin_id, 'inventory_finished', 'system', 0,
            json_encode(['checked_count' => $checked_count, 'duration_min' => $duration_min]));
        unset($_SESSION['inventory']);
        echo json_encode([
            'success'  => true,
            'redirect' => BASE_URL . '/admin/inventory-start.php',
            'summary'  => ['checked' => $checked_count, 'minutes' => $duration_min],
        ]);
        exit;
    }

    echo json_encode(['error' => 'Unbekannte Aktion.']);
    exit;
}

// ── Seitenaufruf: Fortschritts-Daten ─────────────────────────────────────────
$checked_count  = count($inventory['checked']);
$started_min    = max(0, round((time() - $inventory['started_at']) / 60));
// Die letzten 10 geprüften (neueste zuerst)
$recent_checked = array_slice(array_reverse($inventory['checked'], true), 0, 10, true);

$csrf_token = generate_csrf_token();

include __DIR__ . '/includes/header.php';
?>

<!-- ── Sitzungs-Statusbar ────────────────────────────────────────────────── -->
<div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6 flex items-center justify-between flex-wrap gap-3">
    <div class="flex items-center gap-4">
        <div class="text-center">
            <p class="text-2xl font-bold text-blue-700" id="checkedCount"><?= $checked_count ?></p>
            <p class="text-xs text-blue-600"><?= $lang['checked'] ?? 'Geprüft' ?></p>
        </div>
        <div class="text-center border-l border-blue-300 pl-4">
            <p class="text-2xl font-bold text-gray-700"><?= $started_min ?></p>
            <p class="text-xs text-gray-500"><?= $lang['minutes'] ?? 'Minuten' ?></p>
        </div>
        <div class="text-sm text-blue-700">
            <?= $lang['session_started'] ?? 'Sitzung gestartet' ?>: <?= date('d.m.Y H:i', $inventory['started_at']) ?>
        </div>
    </div>
    <button id="btnFinish"
            class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 font-medium text-sm"
            onclick="finishInventory()">
        <?= $lang['finish_inventory'] ?? 'Inventur abschließen' ?>
    </button>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">

    <!-- ── Suchbereich ──────────────────────────────────────────────────────── -->
    <div>
        <div class="bg-white rounded-lg shadow p-6 mb-4">
            <h2 class="text-lg font-semibold text-gray-900 mb-4"><?= $lang['search_scan_copy'] ?? 'Exemplar suchen / scannen' ?></h2>

            <div class="flex gap-2 mb-2">
                <input type="text" id="searchInput"
                       placeholder="<?= esc_attr($lang['scan_placeholder'] ?? 'Buchnummer oder Signatur (z.B. 13751 oder De Bro 13751)') ?>"
                       autofocus
                       class="flex-1 px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-sm font-mono">
                <button onclick="searchCopy()"
                        class="px-4 py-2.5 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm font-medium whitespace-nowrap">
                    <?= $lang['search'] ?? 'Suchen' ?>
                </button>
            </div>
            <p class="text-xs text-gray-500">
                <?= $lang['scan_hint'] ?? 'Exakte Eingabe der Buchnummer oder Signatur. Barcode-Scanner kann hier direkt eingeben.' ?>
            </p>
        </div>

        <!-- Suchergebnis -->
        <div id="resultArea"></div>
    </div>

    <!-- ── Letzte Prüfungen ──────────────────────────────────────────────────── -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-5 border-b flex justify-between items-center">
            <h2 class="text-lg font-semibold text-gray-900"><?= $lang['recent_checks'] ?? 'Letzte Prüfungen' ?></h2>
            <span id="checkedBadge"
                  class="px-2.5 py-0.5 bg-blue-100 text-blue-700 rounded-full text-sm font-medium">
                <?= $checked_count ?>
            </span>
        </div>
        <div id="recentList" class="divide-y divide-gray-100 max-h-96 overflow-y-auto">
            <?php if (empty($recent_checked)): ?>
                <p class="p-5 text-sm text-gray-400 text-center" id="emptyMsg"><?= $lang['no_copies_checked_yet'] ?? 'Noch keine Exemplare geprüft.' ?></p>
            <?php else: ?>
                <?php foreach ($recent_checked as $cid => $c): ?>
                    <?php
                    $changed = $c['old_status'] !== $c['new_status'];
                    $color   = $changed
                        ? ($c['new_status'] === 'verloren'    ? 'red'
                        : ($c['new_status'] === 'aussortiert' ? 'gray'
                        : 'yellow'))
                        : 'green';
                    ?>
                    <div class="p-4 flex items-start gap-3">
                        <div class="w-2 h-2 mt-2 rounded-full bg-<?= $color ?>-400 flex-shrink-0"></div>
                        <div class="flex-1 min-w-0">
                            <p class="text-sm font-medium text-gray-900 truncate"><?= esc_html($c['title']) ?></p>
                            <p class="text-xs text-gray-500 font-mono"><?= esc_html($c['book_number']) ?></p>
                            <?php if ($changed): ?>
                                <p class="text-xs text-<?= $color ?>-600 mt-0.5">
                                    <?= esc_html($c['old_status']) ?> → <?= esc_html($c['new_status']) ?>
                                </p>
                            <?php else: ?>
                                <p class="text-xs text-green-600 mt-0.5"><?= $lang['checked_label'] ?? 'geprüft' ?> (<?= esc_html($c['new_status']) ?>)</p>
                            <?php endif; ?>
                        </div>
                        <span class="text-xs text-gray-400 flex-shrink-0"><?= date('H:i', $c['time']) ?></span>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Ergebnis-Template (versteckt) -->
<template id="tplCopy">
    <div class="bg-white rounded-lg shadow p-5">
        <div class="flex gap-4 mb-4">
            <div id="resImg" class="flex-shrink-0 w-16"></div>
            <div class="flex-1 min-w-0">
                <h3 id="resTitle" class="font-semibold text-gray-900 leading-snug"></h3>
                <p id="resAuthor" class="text-sm text-gray-500 mt-0.5"></p>
                <p class="text-xs text-gray-400 mt-1 font-mono">
                    <span id="resNumber"></span>
                    &bull;
                    <span id="resSig"></span>
                </p>
                <div class="mt-2" id="resStatusWrap"></div>
            </div>
        </div>

        <div id="alreadyCheckedMsg"
             class="mb-3 px-3 py-2 bg-green-50 border border-green-200 rounded text-sm text-green-700"
             style="display:none">
            <?= $lang['already_checked_this_session'] ?? 'Dieses Exemplar wurde in dieser Sitzung bereits geprüft.' ?>
        </div>

        <div id="lockedMsg"
             class="mb-3 px-3 py-2 bg-orange-50 border border-orange-200 rounded text-sm text-orange-700"
             style="display:none">
            <?= $lang['copy_locked_active_loan'] ?? 'Dieses Exemplar ist aktiv ausgeliehen — Status kann nicht geändert werden.' ?>
        </div>

        <div id="actionBtns" class="flex flex-wrap gap-2">
            <button onclick="setStatus('verfügbar')"
                    class="px-3 py-1.5 text-sm rounded-lg border-2 border-green-500 text-green-700 hover:bg-green-50 font-medium">
                <?= $lang['present'] ?? 'Vorhanden' ?>
            </button>
            <button onclick="setStatus('in_bearbeitung')"
                    class="px-3 py-1.5 text-sm rounded-lg border-2 border-blue-400 text-blue-700 hover:bg-blue-50 font-medium">
                <?= $lang['in_processing'] ?? 'In Bearbeitung' ?>
            </button>
            <button onclick="setStatus('verloren')"
                    class="px-3 py-1.5 text-sm rounded-lg border-2 border-red-400 text-red-700 hover:bg-red-50 font-medium">
                <?= $lang['lost'] ?? 'Verloren' ?>
            </button>
            <button onclick="setStatus('aussortiert')"
                    class="px-3 py-1.5 text-sm rounded-lg border-2 border-gray-400 text-gray-600 hover:bg-gray-50 font-medium">
                <?= $lang['discard'] ?? 'Aussortieren' ?>
            </button>
            <button onclick="markChecked()"
                    class="ml-auto px-3 py-1.5 text-sm rounded-lg bg-gray-100 text-gray-600 hover:bg-gray-200 font-medium">
                <?= $lang['confirm_only'] ?? 'Nur bestätigen' ?>
            </button>
        </div>

        <p id="actionMsg" class="mt-3 text-sm font-medium" style="display:none"></p>
    </div>
</template>

<script>
const CSRF = <?= json_encode($csrf_token) ?>;
let currentCopy = null;

// ── Suche ────────────────────────────────────────────────────────────────────
document.getElementById('searchInput').addEventListener('keydown', e => {
    if (e.key === 'Enter') searchCopy();
});

function searchCopy() {
    const q = document.getElementById('searchInput').value.trim();
    if (!q) return;
    const area = document.getElementById('resultArea');
    area.innerHTML = '<p class="text-sm text-gray-400 p-4"><?= $lang['searching'] ?? 'Suche ...' ?></p>';

    fetch(`?action=search&q=${encodeURIComponent(q)}`)
        .then(r => r.json())
        .then(data => {
            if (data.error) {
                area.innerHTML = `<div class="bg-red-50 border border-red-200 rounded-lg p-4 text-sm text-red-700">${data.error}</div>`;
                return;
            }
            currentCopy = data.copy;
            renderCopy(data.copy);
        })
        .catch(() => {
            area.innerHTML = '<div class="bg-red-50 border border-red-200 rounded-lg p-4 text-sm text-red-700"><?= $lang['search_error'] ?? 'Fehler bei der Suche.' ?></div>';
        });
}

function renderCopy(c) {
    const tpl   = document.getElementById('tplCopy');
    const clone = tpl.content.cloneNode(true);
    const wrap  = document.createElement('div');
    wrap.appendChild(clone);

    wrap.querySelector('#resTitle').textContent  = c.title;
    wrap.querySelector('#resAuthor').textContent = c.author ?? '';
    wrap.querySelector('#resNumber').textContent = c.book_number ?? '';
    wrap.querySelector('#resSig').textContent    = c.signature ?? '';

    // Cover
    const imgWrap = wrap.querySelector('#resImg');
    if (c.image_url) {
        imgWrap.innerHTML = `<img src="${escHtml(c.image_url)}" class="w-16 h-20 object-cover rounded" alt="">`;
    } else {
        imgWrap.innerHTML = `<div class="w-16 h-20 bg-gray-100 rounded flex items-center justify-center text-gray-300">
            <svg class="h-8 w-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                    d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
            </svg></div>`;
    }

    // Status badge
    const statusColors = {
        'verfügbar': 'green', 'ausgeliehen': 'orange', 'reserviert': 'yellow',
        'in_bearbeitung': 'blue', 'verloren': 'red', 'aussortiert': 'gray'
    };
    const sc = statusColors[c.status] ?? 'gray';
    wrap.querySelector('#resStatusWrap').innerHTML =
        `<span class="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-${sc}-100 text-${sc}-800">${escHtml(c.status)}</span>`;

    // Already checked?
    if (c.already_checked) {
        wrap.querySelector('#alreadyCheckedMsg').style.display = '';
    }

    // Locked (ausgeliehen with active loan)?
    if (c.status === 'ausgeliehen') {
        wrap.querySelector('#lockedMsg').style.display = '';
        wrap.querySelector('#actionBtns').style.display = 'none';
    }

    document.getElementById('resultArea').innerHTML = '';
    document.getElementById('resultArea').appendChild(wrap.firstElementChild);

    // Auto-clear search input for next scan
    document.getElementById('searchInput').value = '';
    document.getElementById('searchInput').focus();
}

// ── Status setzen ─────────────────────────────────────────────────────────────
function setStatus(newStatus) {
    if (!currentCopy) return;

    if (newStatus === 'verloren' && !confirm(`<?= addslashes($lang['confirm_mark_lost'] ?? 'Exemplar wirklich als VERLOREN markieren?') ?> "${currentCopy.title}"`)) return;
    if (newStatus === 'aussortiert' && !confirm(`<?= addslashes($lang['confirm_discard'] ?? 'Exemplar wirklich AUSSONDERN?') ?> "${currentCopy.title}"`)) return;

    fetch('?', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: new URLSearchParams({
            action:   'set_status',
            copy_id:  currentCopy.id,
            status:   newStatus,
            csrf_token: CSRF,
        }),
    })
    .then(r => r.json())
    .then(data => {
        if (data.error) {
            showActionMsg(data.error, 'red');
            return;
        }
        showActionMsg('<?= addslashes($lang['status_updated'] ?? 'Status aktualisiert') ?>: ' + newStatus, 'green');
        updateCheckedCount(data.checked_count);
        addRecentRow(currentCopy, newStatus);
        currentCopy = null;
    });
}

function markChecked() {
    if (!currentCopy) return;
    fetch('?', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: new URLSearchParams({
            action:     'mark_checked',
            copy_id:    currentCopy.id,
            csrf_token: CSRF,
        }),
    })
    .then(r => r.json())
    .then(data => {
        if (data.error) { showActionMsg(data.error, 'red'); return; }
        showActionMsg('<?= addslashes($lang['marked_as_checked'] ?? 'Als geprüft vermerkt.') ?>', 'green');
        updateCheckedCount(data.checked_count);
        addRecentRow(currentCopy, currentCopy.status);
        currentCopy = null;
    });
}

function showActionMsg(msg, color) {
    const el = document.querySelector('#actionMsg') || document.getElementById('resultArea').querySelector('#actionMsg');
    if (!el) return;
    el.textContent  = msg;
    el.className    = `mt-3 text-sm font-medium text-${color}-700`;
    el.style.display = '';
}

function updateCheckedCount(n) {
    document.getElementById('checkedCount').textContent = n;
    document.getElementById('checkedBadge').textContent = n;
}

function addRecentRow(copy, newStatus) {
    const emptyMsg = document.getElementById('emptyMsg');
    if (emptyMsg) emptyMsg.remove();

    const changed = copy.status !== newStatus;
    const color   = changed
        ? (newStatus === 'verloren' ? 'red' : (newStatus === 'aussortiert' ? 'gray' : 'yellow'))
        : 'green';
    const checkedLabel = '<?= addslashes($lang['checked_label'] ?? 'geprüft') ?>';
    const statusTxt = changed ? `${copy.status} → ${newStatus}` : `${checkedLabel} (${newStatus})`;

    const div = document.createElement('div');
    div.className = 'p-4 flex items-start gap-3 border-b border-gray-100';
    div.innerHTML = `
        <div class="w-2 h-2 mt-2 rounded-full bg-${color}-400 flex-shrink-0"></div>
        <div class="flex-1 min-w-0">
            <p class="text-sm font-medium text-gray-900 truncate">${escHtml(copy.title)}</p>
            <p class="text-xs text-gray-500 font-mono">${escHtml(copy.book_number ?? '')}</p>
            <p class="text-xs text-${color}-600 mt-0.5">${escHtml(statusTxt)}</p>
        </div>
        <span class="text-xs text-gray-400 flex-shrink-0">${new Date().toLocaleTimeString('de-AT',{hour:'2-digit',minute:'2-digit'})}</span>
    `;

    const list = document.getElementById('recentList');
    list.insertBefore(div, list.firstChild);

    // Max 10 rows
    while (list.children.length > 10) list.removeChild(list.lastChild);
}

// ── Inventur abschließen ──────────────────────────────────────────────────────
function finishInventory() {
    const count = parseInt(document.getElementById('checkedCount').textContent);
    const confirmMsg = `<?= addslashes($lang['confirm_finish_inventory'] ?? 'Inventur abschließen?') ?>\n${count} <?= addslashes($lang['copies_checked_msg'] ?? 'Exemplar(e) wurden geprüft.') ?>\n<?= addslashes($lang['session_will_end'] ?? 'Die Sitzung wird beendet.') ?>`;
    if (!confirm(confirmMsg)) return;

    fetch('?', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: new URLSearchParams({action: 'finish', csrf_token: CSRF}),
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) window.location.href = data.redirect;
    });
}

function escHtml(s) {
    return String(s ?? '').replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
