<?php
/**
 * Inventurhistorie — inventory-history.php
 * Verwaltung aller Exemplare mit Problemstatus:
 * verloren, aussortiert, in_bearbeitung.
 * Aktionen: Status ändern, permanent löschen.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../includes/auth.php';

require_admin();

$current_user = get_logged_in_user();
$admin_id     = (int)$current_user['id'];
$page_title   = $lang['admin_inventory_history'] ?? 'Inventurhistorie';

// ── POST-Handler ──────────────────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf_token($_POST['csrf_token'] ?? '');
    $action  = $_POST['action']  ?? '';
    $copy_id = (int)($_POST['copy_id'] ?? 0);

    if ($copy_id <= 0) {
        set_flash('error', $lang['invalid_copy_id'] ?? 'Ungültige Exemplar-ID.');
        header('Location: ' . $_SERVER['REQUEST_URI']);
        exit;
    }

    $copy = db_get_row(
        "SELECT bc.*, b.title FROM book_copies bc JOIN books b ON bc.book_id = b.id WHERE bc.id = ?",
        [$copy_id]
    );
    if (!$copy) {
        set_flash('error', $lang['copy_not_found'] ?? 'Exemplar nicht gefunden.');
        header('Location: ' . $_SERVER['REQUEST_URI']);
        exit;
    }

    if ($action === 'set_status') {
        $new_status = $_POST['status'] ?? '';
        $allowed    = ['verfügbar', 'verloren', 'aussortiert', 'in_bearbeitung'];
        if (!in_array($new_status, $allowed, true)) {
            set_flash('error', $lang['invalid_status'] ?? 'Ungültiger Status.');
            header('Location: ' . $_SERVER['REQUEST_URI']);
            exit;
        }
        db_update('book_copies', ['status' => $new_status], ['id' => $copy_id]);
        log_activity($admin_id, 'copy_status_changed', 'book_copies', $copy_id,
            json_encode(['old' => $copy['status'], 'new' => $new_status, 'title' => mb_substr($copy['title'], 0, 80)]));
        set_flash('success', sprintf('"%s" (Nr. %s) → %s: %s',
            mb_substr($copy['title'], 0, 40), $copy['book_number'],
            $lang['status'] ?? 'Status', $new_status));
    }

    if ($action === 'delete_copy') {
        // Safety check: not actively loaned
        $active_loan = db_get_var(
            "SELECT COUNT(*) FROM loans WHERE copy_id = ? AND return_date IS NULL",
            [$copy_id]
        );
        if ($active_loan > 0) {
            set_flash('error', $lang['cannot_delete_active_loan'] ?? 'Exemplar kann nicht gelöscht werden – es ist noch aktiv ausgeliehen.');
            header('Location: ' . $_SERVER['REQUEST_URI']);
            exit;
        }
        // Cancel open reservations for this copy
        db_query(
            "UPDATE reservations SET status = 'storniert', copy_id = NULL WHERE copy_id = ? AND status IN ('aktiv','verfügbar')",
            [$copy_id]
        );
        db_delete('book_copies', ['id' => $copy_id]);
        log_activity($admin_id, 'copy_deleted', 'book_copies', $copy_id,
            json_encode(['title' => mb_substr($copy['title'], 0, 80), 'book_number' => $copy['book_number'], 'status' => $copy['status']]));
        set_flash('success', sprintf($lang['copy_deleted_success'] ?? 'Exemplar "%s" (Nr. %s) wurde dauerhaft gelöscht.',
            mb_substr($copy['title'], 0, 40), $copy['book_number']));
    }

    header('Location: ' . $_SERVER['REQUEST_URI']);
    exit;
}

// ── Filter ────────────────────────────────────────────────────────────────────
$filter   = $_GET['filter'] ?? 'all';
$allowed_filters = ['all', 'verloren', 'aussortiert', 'in_bearbeitung'];
if (!in_array($filter, $allowed_filters, true)) $filter = 'all';

$q    = trim($_GET['q'] ?? '');
$page = max(1, (int)($_GET['page'] ?? 1));
$per  = 30;

// ── WHERE aufbauen ────────────────────────────────────────────────────────────
$where  = [];
$params = [];

if ($filter === 'all') {
    $where[]  = "bc.status IN ('verloren','aussortiert','in_bearbeitung')";
} else {
    $where[]  = "bc.status = ?";
    $params[] = $filter;
}

if ($q !== '') {
    $like     = '%' . $q . '%';
    $where[]  = "(b.title LIKE ? OR bc.book_number LIKE ? OR bc.signature LIKE ? OR b.author LIKE ?)";
    array_push($params, $like, $like, $like, $like);
}

$where_sql = 'WHERE ' . implode(' AND ', $where);

// ── Anzahl ────────────────────────────────────────────────────────────────────
$total      = (int)db_get_var("SELECT COUNT(*) FROM book_copies bc JOIN books b ON bc.book_id = b.id $where_sql", $params);
$total_pages = max(1, (int)ceil($total / $per));
$page        = min($page, $total_pages);
$offset      = ($page - 1) * $per;

// ── Exemplare laden ────────────────────────────────────────────────────────────
$copies = db_get_results(
    "SELECT bc.id, bc.book_number, bc.copy_index, bc.signature,
            bc.status, bc.location, bc.book_id, bc.acquired_at,
            b.title, b.author, b.isbn, b.image_url,
            bs.code AS subcategory_code
     FROM book_copies bc
     JOIN books b ON bc.book_id = b.id
     LEFT JOIN book_subcategories bs ON b.subcategory_id = bs.id
     $where_sql
     ORDER BY bc.status, b.title_sort ASC, b.title ASC
     LIMIT $per OFFSET $offset",
    $params
);

// ── Zähl-Kacheln ─────────────────────────────────────────────────────────────
$counts = db_get_results(
    "SELECT status, COUNT(*) AS cnt FROM book_copies
     WHERE status IN ('verloren','aussortiert','in_bearbeitung')
     GROUP BY status"
);
$cnt = [];
foreach ($counts as $r) $cnt[$r['status']] = (int)$r['cnt'];
$cnt_all = array_sum($cnt);

$csrf_token = generate_csrf_token();

// ── Paginierungs-URL ──────────────────────────────────────────────────────────
$pag_url = function(int $p) use ($filter, $q): string {
    return BASE_URL . '/admin/inventory-history.php?' . http_build_query(array_filter([
        'filter' => $filter !== 'all' ? $filter : '',
        'q'      => $q,
        'page'   => $p,
    ]));
};

include __DIR__ . '/includes/header.php';

$flash = get_flash();
?>

<?php if ($flash): ?>
    <?php $fc = $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'yellow'); ?>
    <div class="mb-6 p-4 rounded-lg bg-<?= $fc ?>-50 border border-<?= $fc ?>-200 text-<?= $fc ?>-800 flash-message">
        <?= esc_html($flash['message']) ?>
    </div>
<?php endif; ?>

<!-- ── Stat-Kacheln + Filter-Tabs ────────────────────────────────────────── -->
<div class="flex flex-wrap gap-3 mb-6">
    <?php
    $tabs = [
        ['filter' => 'all',            'label' => $lang['all'] ?? 'Alle',                      'cnt' => $cnt_all,                   'color' => 'gray'],
        ['filter' => 'verloren',       'label' => $lang['lost'] ?? 'Verloren',                  'cnt' => $cnt['verloren'] ?? 0,       'color' => 'red'],
        ['filter' => 'aussortiert',    'label' => $lang['discarded'] ?? 'Aussortiert',           'cnt' => $cnt['aussortiert'] ?? 0,    'color' => 'gray'],
        ['filter' => 'in_bearbeitung', 'label' => $lang['in_processing'] ?? 'In Bearbeitung',   'cnt' => $cnt['in_bearbeitung'] ?? 0, 'color' => 'blue'],
    ];
    foreach ($tabs as $t):
        $active = $filter === $t['filter'];
        $url    = BASE_URL . '/admin/inventory-history.php?filter=' . $t['filter'] . ($q ? '&q=' . urlencode($q) : '');
    ?>
        <a href="<?= $url ?>"
           class="flex items-center gap-2 px-4 py-2 rounded-lg border-2 text-sm font-medium transition
               <?= $active
                   ? "border-{$t['color']}-500 bg-{$t['color']}-50 text-{$t['color']}-700"
                   : 'border-gray-200 bg-white text-gray-600 hover:border-gray-300' ?>">
            <?= $t['label'] ?>
            <span class="px-1.5 py-0.5 rounded text-xs
                <?= $active ? "bg-{$t['color']}-100 text-{$t['color']}-800" : 'bg-gray-100 text-gray-500' ?>">
                <?= number_format($t['cnt'], 0, ',', '.') ?>
            </span>
        </a>
    <?php endforeach; ?>

    <!-- Suchfeld -->
    <form method="GET" class="flex gap-2 ml-auto">
        <input type="hidden" name="filter" value="<?= esc_attr($filter) ?>">
        <input type="text" name="q" value="<?= esc_attr($q) ?>"
               placeholder="<?= esc_attr($lang['search_copy_placeholder'] ?? 'Titel, Buchnr., Signatur, Autor ...') ?>"
               class="px-3 py-1.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 w-64">
        <button type="submit" class="px-3 py-1.5 bg-gray-700 text-white rounded-lg text-sm hover:bg-gray-800"><?= $lang['search'] ?? 'Suchen' ?></button>
        <?php if ($q): ?>
            <a href="<?= BASE_URL ?>/admin/inventory-history.php?filter=<?= esc_attr($filter) ?>"
               class="px-3 py-1.5 bg-gray-100 text-gray-600 rounded-lg text-sm hover:bg-gray-200">×</a>
        <?php endif; ?>
    </form>
</div>

<!-- Ergebnis-Info -->
<div class="flex items-center justify-between mb-3 text-sm text-gray-500">
    <span>
        <strong class="text-gray-900"><?= number_format($total, 0, ',', '.') ?></strong>
        <?= $lang['copies_label'] ?? 'Exemplar(e)' ?>
    </span>
    <?php if ($total_pages > 1): ?>
        <span><?= $lang['page'] ?? 'Seite' ?> <?= $page ?> / <?= $total_pages ?></span>
    <?php endif; ?>
</div>

<!-- ── Exemplar-Tabelle ───────────────────────────────────────────────────── -->
<?php if (empty($copies)): ?>
    <div class="bg-white rounded-lg shadow p-12 text-center text-gray-400">
        <svg class="mx-auto h-16 w-16 mb-4 text-gray-200" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                  d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
        </svg>
        <p class="text-lg font-medium"><?= $lang['no_copies_found'] ?? 'Keine Exemplare gefunden' ?></p>
        <?php if ($filter !== 'all'): ?>
            <p class="text-sm mt-1"><?= $lang['all_copies_good_status'] ?? 'Alle Exemplare sind in einem guten Status.' ?></p>
        <?php endif; ?>
    </div>
<?php else: ?>
    <div class="bg-white rounded-lg shadow overflow-hidden mb-6">
        <div class="overflow-x-auto">
            <table class="min-w-full text-sm">
                <thead class="bg-gray-50 text-gray-500 text-xs uppercase">
                    <tr>
                        <th class="px-5 py-3 text-left font-medium w-10"></th>
                        <th class="px-5 py-3 text-left font-medium"><?= $lang['book_title'] ?? 'Buch' ?></th>
                        <th class="px-4 py-3 text-left font-medium"><?= $lang['copy_label'] ?? 'Exemplar' ?></th>
                        <th class="px-4 py-3 text-left font-medium"><?= $lang['signature'] ?? 'Signatur' ?></th>
                        <th class="px-4 py-3 text-left font-medium"><?= $lang['status'] ?? 'Status' ?></th>
                        <th class="px-4 py-3 text-left font-medium"><?= $lang['acquisition'] ?? 'Zugang' ?></th>
                        <th class="px-4 py-3 text-right font-medium"><?= $lang['actions'] ?? 'Aktionen' ?></th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-100">
                    <?php foreach ($copies as $copy):
                        $status_color = [
                            'verloren'       => 'red',
                            'aussortiert'    => 'gray',
                            'in_bearbeitung' => 'blue',
                        ][$copy['status']] ?? 'gray';
                    ?>
                        <tr class="hover:bg-gray-50" id="row-<?= (int)$copy['id'] ?>">
                            <!-- Mini-Cover -->
                            <td class="px-3 py-2">
                                <div class="w-8 h-10 rounded overflow-hidden bg-gray-100 flex-shrink-0">
                                    <?php if ($copy['image_url']): ?>
                                        <img src="<?= esc_url($copy['image_url']) ?>"
                                             alt="" class="w-full h-full object-cover">
                                    <?php else: ?>
                                        <div class="w-full h-full flex items-center justify-center">
                                            <svg class="h-4 w-4 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                      d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
                                            </svg>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <!-- Buch -->
                            <td class="px-5 py-3">
                                <a href="<?= BASE_URL ?>/admin/books-edit.php?id=<?= (int)$copy['book_id'] ?>"
                                   class="font-medium text-gray-900 hover:text-blue-600 line-clamp-1">
                                    <?= esc_html($copy['title']) ?>
                                </a>
                                <p class="text-xs text-gray-400 truncate"><?= esc_html($copy['author'] ?? '') ?></p>
                            </td>
                            <!-- Exemplar -->
                            <td class="px-4 py-3 font-mono text-gray-600">
                                <?= esc_html($copy['book_number'] ?? '–') ?>
                                <?php if ((int)$copy['copy_index'] > 1): ?>
                                    <span class="text-xs text-gray-400">(Ex.<?= (int)$copy['copy_index'] ?>)</span>
                                <?php endif; ?>
                            </td>
                            <!-- Signatur -->
                            <td class="px-4 py-3 text-gray-600 font-mono text-xs">
                                <?= esc_html($copy['signature'] ?? '–') ?>
                            </td>
                            <!-- Status -->
                            <td class="px-4 py-3">
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                    bg-<?= $status_color ?>-100 text-<?= $status_color ?>-800">
                                    <?= esc_html($copy['status']) ?>
                                </span>
                            </td>
                            <!-- Zugang -->
                            <td class="px-4 py-3 text-gray-400 text-xs">
                                <?= $copy['acquired_at'] ? date('d.m.Y', strtotime($copy['acquired_at'])) : '–' ?>
                            </td>
                            <!-- Aktionen -->
                            <td class="px-4 py-3 text-right">
                                <div class="flex items-center justify-end gap-2 flex-wrap">
                                    <?php if ($copy['status'] !== 'verfügbar'): ?>
                                        <!-- Wiederherstellen / Als gefunden markieren -->
                                        <form method="POST" class="inline">
                                            <?= csrf_field() ?>
                                            <input type="hidden" name="action"  value="set_status">
                                            <input type="hidden" name="copy_id" value="<?= (int)$copy['id'] ?>">
                                            <input type="hidden" name="status"  value="verfügbar">
                                            <button type="submit"
                                                    class="px-2.5 py-1 text-xs rounded border border-green-400 text-green-700 hover:bg-green-50">
                                                <?= $copy['status'] === 'verloren' ? ($lang['found'] ?? 'Gefunden') : ($lang['restore'] ?? 'Wiederherstellen') ?>
                                            </button>
                                        </form>
                                    <?php endif; ?>

                                    <?php if ($copy['status'] !== 'in_bearbeitung'): ?>
                                        <form method="POST" class="inline">
                                            <?= csrf_field() ?>
                                            <input type="hidden" name="action"  value="set_status">
                                            <input type="hidden" name="copy_id" value="<?= (int)$copy['id'] ?>">
                                            <input type="hidden" name="status"  value="in_bearbeitung">
                                            <button type="submit"
                                                    class="px-2.5 py-1 text-xs rounded border border-blue-300 text-blue-600 hover:bg-blue-50">
                                                <?= $lang['in_processing_abbr'] ?? 'In Bearb.' ?>
                                            </button>
                                        </form>
                                    <?php endif; ?>

                                    <?php if ($copy['status'] !== 'verloren'): ?>
                                        <form method="POST" class="inline">
                                            <?= csrf_field() ?>
                                            <input type="hidden" name="action"  value="set_status">
                                            <input type="hidden" name="copy_id" value="<?= (int)$copy['id'] ?>">
                                            <input type="hidden" name="status"  value="verloren">
                                            <button type="submit"
                                                    class="px-2.5 py-1 text-xs rounded border border-red-300 text-red-600 hover:bg-red-50">
                                                <?= $lang['lost'] ?? 'Verloren' ?>
                                            </button>
                                        </form>
                                    <?php endif; ?>

                                    <?php if ($copy['status'] !== 'aussortiert'): ?>
                                        <form method="POST" class="inline">
                                            <?= csrf_field() ?>
                                            <input type="hidden" name="action"  value="set_status">
                                            <input type="hidden" name="copy_id" value="<?= (int)$copy['id'] ?>">
                                            <input type="hidden" name="status"  value="aussortiert">
                                            <button type="submit"
                                                    class="px-2.5 py-1 text-xs rounded border border-gray-300 text-gray-500 hover:bg-gray-50">
                                                <?= $lang['discard'] ?? 'Aussondern' ?>
                                            </button>
                                        </form>
                                    <?php endif; ?>

                                    <!-- Permanent löschen -->
                                    <form method="POST" class="inline"
                                          onsubmit="return confirm('<?= esc_attr(($lang['confirm_delete_copy'] ?? 'Exemplar DAUERHAFT löschen?') . '\nBuchnr.: ' . ($copy['book_number'] ?? '') . '\n' . mb_substr($copy['title'], 0, 50) . '\n\n' . ($lang['cannot_be_undone'] ?? 'Dieser Vorgang kann nicht rückgängig gemacht werden!')) ?>')">
                                        <?= csrf_field() ?>
                                        <input type="hidden" name="action"  value="delete_copy">
                                        <input type="hidden" name="copy_id" value="<?= (int)$copy['id'] ?>">
                                        <button type="submit"
                                                class="px-2.5 py-1 text-xs rounded bg-red-600 text-white hover:bg-red-700">
                                            <?= $lang['delete'] ?? 'Löschen' ?>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
        <nav class="flex justify-center">
            <div class="flex items-center gap-1">
                <?php if ($page > 1): ?>
                    <a href="<?= $pag_url($page - 1) ?>"
                       class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-l-md hover:bg-gray-50">&laquo;</a>
                <?php endif; ?>

                <?php for ($p = 1; $p <= $total_pages; $p++):
                    if ($p > 1 && $p < $total_pages && abs($p - $page) > 2):
                        if ($p === 2 || $p === $total_pages - 1):
                            echo '<span class="px-3 py-2 text-sm text-gray-400">...</span>';
                        endif;
                        continue;
                    endif; ?>
                    <a href="<?= $pag_url($p) ?>"
                       class="px-3 py-2 text-sm font-medium border <?= $p === $page ? 'bg-blue-600 text-white border-blue-600' : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50' ?>">
                        <?= $p ?>
                    </a>
                <?php endfor; ?>

                <?php if ($page < $total_pages): ?>
                    <a href="<?= $pag_url($page + 1) ?>"
                       class="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-r-md hover:bg-gray-50">&raquo;</a>
                <?php endif; ?>
            </div>
        </nav>
    <?php endif; ?>
<?php endif; ?>

<?php include __DIR__ . '/includes/footer.php'; ?>
