<?php
/**
 * Admin Dashboard
 * Main dashboard for administrators
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

// Require login + admin
require_login();
if (!is_admin()) {
    header('Location: ' . BASE_URL . '/dashboard.php');
    exit;
}

$page_title = 'Admin Dashboard';
$current_user = get_logged_in_user();
$lang = load_language();

// Get system statistics
$stats = [
    'total_books' => db_get_var("SELECT COUNT(DISTINCT book_id) FROM book_copies"),
    'total_copies' => db_get_var("SELECT COUNT(*) FROM book_copies WHERE status != 'aussortiert'"),
    'available_copies' => db_get_var("SELECT COUNT(*) FROM book_copies WHERE status = 'verfügbar'"),
    'active_loans' => db_get_var("SELECT COUNT(*) FROM loans WHERE status IN ('aktiv', 'überfällig')"),
    'overdue_loans' => db_get_var("SELECT COUNT(*) FROM loans WHERE status = 'überfällig'"),
    'reservations' => db_get_var("SELECT COUNT(*) FROM reservations WHERE status IN ('aktiv', 'verfügbar')"),
    'total_users' => db_get_var("SELECT COUNT(*) FROM users WHERE status = 'aktiv'"),
];

// Recent activity (last 10 actions)
$recent_activity = db_get_results("
    SELECT al.*, u.first_name, u.last_name
    FROM activity_log al
    LEFT JOIN users u ON al.user_id = u.id
    ORDER BY al.created_at DESC
    LIMIT 10
");

// Overdue loans (top 5)
$overdue_loans = db_get_results("
    SELECT l.*, bc.book_number, bc.signature,
           b.title, b.author,
           u.first_name, u.last_name, u.email,
           DATEDIFF(CURDATE(), l.due_date) as days_overdue
    FROM loans l
    JOIN book_copies bc ON l.copy_id = bc.id
    JOIN books b ON bc.book_id = b.id
    JOIN users u ON l.user_id = u.id
    WHERE l.status = 'überfällig'
    ORDER BY l.due_date ASC
    LIMIT 5
");

// Pending reservations (top 5)
$pending_reservations = db_get_results("
    SELECT r.*, b.title, b.author,
           u.first_name, u.last_name, u.email,
           DATEDIFF(CURDATE(), r.reserved_date) as days_waiting
    FROM reservations r
    JOIN books b ON r.book_id = b.id
    JOIN users u ON r.user_id = u.id
    WHERE r.status IN ('aktiv', 'verfügbar')
    ORDER BY r.status DESC, r.reserved_date ASC
    LIMIT 5
");

include __DIR__ . '/includes/header.php';
?>

<!-- Statistics Grid -->
<div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4 mb-8">
    <!-- Total Books -->
    <a href="<?php echo BASE_URL; ?>/admin/reports-inventory.php"
       class="bg-white rounded-lg shadow p-5 flex items-center hover:shadow-md transition-shadow">
        <div class="flex-shrink-0 bg-blue-500 rounded-md p-3">
            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6.253v13m0-13C10.832 5.477 9.246 5 7.5 5S4.168 5.477 3 6.253v13C4.168 18.477 5.754 18 7.5 18s3.332.477 4.5 1.253m0-13C13.168 5.477 14.754 5 16.5 5c1.747 0 3.332.477 4.5 1.253v13C19.832 18.477 18.247 18 16.5 18c-1.746 0-3.332.477-4.5 1.253"/>
            </svg>
        </div>
        <div class="ml-4">
            <p class="text-sm font-medium text-gray-500"><?php echo $lang['admin_total_books'] ?? 'Bücher Gesamt'; ?></p>
            <div class="flex items-baseline flex-wrap gap-x-1">
                <p class="text-2xl font-bold text-gray-900"><?php echo $stats['total_books']; ?></p>
                <p class="text-xs text-gray-500"><?php echo $stats['total_copies']; ?> <?php echo $lang['copies_abbr'] ?? 'Ex.'; ?></p>
            </div>
        </div>
    </a>

    <!-- Available Copies -->
    <a href="<?php echo BASE_URL; ?>/admin/inventory-start.php"
       class="bg-white rounded-lg shadow p-5 flex items-center hover:shadow-md transition-shadow">
        <div class="flex-shrink-0 bg-green-500 rounded-md p-3">
            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
        </div>
        <div class="ml-4">
            <p class="text-sm font-medium text-gray-500"><?php echo $lang['admin_available'] ?? 'Verfügbar'; ?></p>
            <p class="text-2xl font-bold text-gray-900"><?php echo $stats['available_copies']; ?></p>
        </div>
    </a>

    <!-- Active Loans -->
    <a href="<?php echo BASE_URL; ?>/admin/lending-active.php"
       class="bg-white rounded-lg shadow p-5 flex items-center hover:shadow-md transition-shadow">
        <div class="flex-shrink-0 bg-yellow-500 rounded-md p-3">
            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
            </svg>
        </div>
        <div class="ml-4">
            <p class="text-sm font-medium text-gray-500"><?php echo $lang['admin_active_loans'] ?? 'Aktive Ausleihen'; ?></p>
            <p class="text-2xl font-bold text-gray-900"><?php echo $stats['active_loans']; ?></p>
        </div>
    </a>

    <!-- Overdue Loans -->
    <a href="<?php echo BASE_URL; ?>/admin/lending-active.php?filter=overdue"
       class="bg-white rounded-lg shadow p-5 flex items-center hover:shadow-md transition-shadow">
        <div class="flex-shrink-0 <?php echo $stats['overdue_loans'] > 0 ? 'bg-red-500' : 'bg-gray-400'; ?> rounded-md p-3">
            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
        </div>
        <div class="ml-4">
            <p class="text-sm font-medium text-gray-500"><?php echo $lang['admin_overdue'] ?? 'Überfällig'; ?></p>
            <p class="text-2xl font-bold <?php echo $stats['overdue_loans'] > 0 ? 'text-red-600' : 'text-gray-900'; ?>">
                <?php echo $stats['overdue_loans']; ?>
            </p>
        </div>
    </a>

    <!-- Reservations -->
    <a href="<?php echo BASE_URL; ?>/admin/lending-reservations.php"
       class="bg-white rounded-lg shadow p-5 flex items-center hover:shadow-md transition-shadow">
        <div class="flex-shrink-0 bg-purple-500 rounded-md p-3">
            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
            </svg>
        </div>
        <div class="ml-4">
            <p class="text-sm font-medium text-gray-500">Vormerkungen</p>
            <p class="text-2xl font-bold text-gray-900"><?php echo $stats['reservations']; ?></p>
        </div>
    </a>
</div>

<!-- Quick Actions -->
<div class="bg-white rounded-lg shadow mb-8 p-6">
    <h2 class="text-xl font-bold text-gray-900 mb-4">
        <svg class="inline h-6 w-6 mr-2 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
        </svg>
        <?php echo $lang['admin_quick_actions'] ?? 'Schnellzugriff'; ?>
    </h2>
    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
        <a href="<?php echo BASE_URL; ?>/admin/books-add.php"
           class="flex items-center p-4 bg-blue-50 hover:bg-blue-100 rounded-lg">
            <svg class="h-8 w-8 text-blue-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
            </svg>
            <div>
                <div class="font-medium text-gray-900"><?php echo $lang['admin_add_book'] ?? 'Buch erfassen'; ?></div>
                <div class="text-sm text-gray-500"><?php echo $lang['admin_add_book_desc'] ?? 'Neues Buch hinzufügen'; ?></div>
            </div>
        </a>

        <a href="<?php echo BASE_URL; ?>/admin/lending-checkout.php"
           class="flex items-center p-4 bg-green-50 hover:bg-green-100 rounded-lg">
            <svg class="h-8 w-8 text-green-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"/>
            </svg>
            <div>
                <div class="font-medium text-gray-900"><?php echo $lang['admin_checkout'] ?? 'Ausleihe'; ?></div>
                <div class="text-sm text-gray-500"><?php echo $lang['admin_checkout_desc'] ?? 'Buch ausleihen'; ?></div>
            </div>
        </a>

        <a href="<?php echo BASE_URL; ?>/admin/lending-return.php"
           class="flex items-center p-4 bg-purple-50 hover:bg-purple-100 rounded-lg">
            <svg class="h-8 w-8 text-purple-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"/>
            </svg>
            <div>
                <div class="font-medium text-gray-900"><?php echo $lang['admin_return'] ?? 'Rückgabe'; ?></div>
                <div class="text-sm text-gray-500"><?php echo $lang['admin_return_desc'] ?? 'Buch zurücknehmen'; ?></div>
            </div>
        </a>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
    <!-- Overdue Loans -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
                <?php echo $lang['admin_overdue_loans'] ?? 'Überfällige Ausleihen'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($overdue_loans)): ?>
                <div class="text-center py-8 text-gray-500">
                    <svg class="mx-auto h-12 w-12 mb-4 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <p><?php echo $lang['admin_no_overdue'] ?? 'Keine überfälligen Ausleihen'; ?></p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($overdue_loans as $loan): ?>
                        <div class="flex justify-between items-start p-4 bg-red-50 rounded-lg">
                            <div class="flex-1">
                                <p class="font-medium text-gray-900"><?php echo esc_html($loan['title']); ?></p>
                                <p class="text-sm text-gray-600"><?php echo esc_html($loan['first_name'] . ' ' . $loan['last_name']); ?></p>
                                <p class="text-xs text-gray-500 mt-1">Signatur: <?php echo esc_html($loan['signature']); ?></p>
                            </div>
                            <div class="text-right ml-4">
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                    <?php echo $loan['days_overdue']; ?> <?php echo $lang['days_overdue'] ?? 'Tage'; ?>
                                </span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="mt-4 text-center">
                    <a href="<?php echo BASE_URL; ?>/admin/lending-active.php?filter=overdue"
                       class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                        <?php echo $lang['view_all'] ?? 'Alle anzeigen'; ?> →
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Pending Reservations -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b">
            <h2 class="text-xl font-bold text-gray-900">
                <svg class="inline h-6 w-6 mr-2 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                </svg>
                <?php echo $lang['admin_pending_reservations'] ?? 'Wartende Vormerkungen'; ?>
            </h2>
        </div>
        <div class="p-6">
            <?php if (empty($pending_reservations)): ?>
                <div class="text-center py-8 text-gray-500">
                    <svg class="mx-auto h-12 w-12 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"/>
                    </svg>
                    <p><?php echo $lang['admin_no_reservations'] ?? 'Keine wartenden Vormerkungen'; ?></p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($pending_reservations as $reservation):
                        $is_ready = ($reservation['status'] === 'verfügbar');
                    ?>
                        <div class="flex justify-between items-start p-4 <?php echo $is_ready ? 'bg-green-50' : 'bg-yellow-50'; ?> rounded-lg">
                            <div class="flex-1">
                                <p class="font-medium text-gray-900"><?php echo esc_html($reservation['title']); ?></p>
                                <p class="text-sm text-gray-600"><?php echo esc_html($reservation['first_name'] . ' ' . $reservation['last_name']); ?></p>
                            </div>
                            <div class="text-right ml-4">
                                <?php if ($is_ready): ?>
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                        Abholbereit
                                    </span>
                                <?php else: ?>
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                        <?php echo $reservation['days_waiting']; ?> <?php echo $lang['days_waiting'] ?? 'Tage'; ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="mt-4 text-center">
                    <a href="<?php echo BASE_URL; ?>/admin/lending-reservations.php"
                       class="text-blue-600 hover:text-blue-800 text-sm font-medium">
                        <?php echo $lang['view_all'] ?? 'Alle anzeigen'; ?> →
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Recent Activity -->
<div class="bg-white rounded-lg shadow">
    <div class="p-6 border-b">
        <h2 class="text-xl font-bold text-gray-900">
            <svg class="inline h-6 w-6 mr-2 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
            <?php echo $lang['admin_recent_activity'] ?? 'Letzte Aktivitäten'; ?>
        </h2>
    </div>
    <div class="p-6">
        <?php if (empty($recent_activity)): ?>
            <p class="text-center py-8 text-gray-500"><?php echo $lang['admin_no_activity'] ?? 'Keine Aktivitäten'; ?></p>
        <?php else: ?>
            <div class="space-y-4">
                <?php foreach ($recent_activity as $activity): ?>
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <div class="h-8 w-8 rounded-full bg-gray-200 flex items-center justify-center">
                                <svg class="h-5 w-5 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                </svg>
                            </div>
                        </div>
                        <div class="ml-3 flex-1">
                            <p class="text-sm text-gray-900">
                                <span class="font-medium"><?php echo esc_html($activity['first_name'] . ' ' . $activity['last_name']); ?></span>
                                <span class="text-gray-600"> - <?php echo esc_html($activity['action']); ?></span>
                            </p>
                            <p class="text-xs text-gray-500 mt-1">
                                <?php echo date('d.m.Y H:i', strtotime($activity['created_at'])); ?>
                            </p>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
