<?php
/**
 * Bücher ausscheiden — books-remove.php
 *
 * Schritt 1: Buchungsnummern (kommagetrennt) eingeben → Vorschau
 * Schritt 2: Bestätigen → Exemplare auf Status "ausgeschieden" setzen
 *            (Buchsatz bleibt erhalten; kein hard delete)
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang = load_language();
$page_title   = $lang['remove_book'] ?? 'Bücher ausscheiden';
$current_user = get_logged_in_user();

// ─── POST: endgültig ausscheiden ──────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'confirm_remove') {
    $ids = array_filter(array_map('intval', explode(',', $_POST['copy_ids'] ?? '')));

    if (!$ids) {
        set_flash($lang['no_copies_selected'] ?? 'Keine Exemplare ausgewählt.', 'error');
        header('Location: books-remove.php');
        exit;
    }

    // Check none are currently loaned
    $placeholders = implode(',', array_fill(0, count($ids), '?'));
    $activeLoans  = (int) db_get_var(
        "SELECT COUNT(*) FROM loans WHERE copy_id IN ($placeholders) AND return_date IS NULL",
        $ids
    );

    if ($activeLoans > 0) {
        set_flash($activeLoans . ' ' . ($lang['copies_still_on_loan_err'] ?? 'Exemplar(e) sind noch ausgeliehen und können nicht ausgeschieden werden.'), 'error');
        header('Location: books-remove.php');
        exit;
    }

    db_query(
        "UPDATE book_copies SET status = 'ausgeschieden' WHERE id IN ($placeholders)",
        $ids
    );

    foreach ($ids as $copyId) {
        log_activity('copy_removed', 'book_copies', $copyId, json_encode(['action' => 'ausgeschieden']));
    }

    $n = count($ids);
    set_flash($n . ' ' . ($n !== 1 ? ($lang['copies_discard_plural'] ?? 'Exemplare') : ($lang['copies_discard_label'] ?? 'Exemplar')) . ' ' . ($lang['copies_removed_success'] ?? 'erfolgreich ausgeschieden.'), 'success');
    header('Location: books-remove.php');
    exit;
}

// ─── POST: Vorschau berechnen ─────────────────────────────────────────────────
$previewCopies  = [];
$previewErrors  = [];
$inputNumbers   = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'preview') {
    $inputNumbers = trim($_POST['book_numbers'] ?? '');
    $rawNumbers   = array_filter(array_map('trim', preg_split('/[\s,;]+/', $inputNumbers)));

    foreach ($rawNumbers as $num) {
        $copy = db_get_row("
            SELECT bc.id, bc.book_number, bc.signature, bc.status,
                   b.title, b.author
            FROM book_copies bc
            JOIN books b ON b.id = bc.book_id
            WHERE bc.book_number = ?
        ", [$num]);

        if (!$copy) {
            $previewErrors[] = ($lang['book_number_field'] ?? 'Buchungsnummer') . " '$num' " . ($lang['number_not_found'] ?? 'nicht gefunden.');
            continue;
        }
        if ($copy['status'] === 'ausgeschieden') {
            $previewErrors[] = ($lang['book_number_field'] ?? 'Buchungsnummer') . " '$num' " . ($lang['already_discarded'] ?? 'ist bereits ausgeschieden.');
            continue;
        }

        // Check active loan
        $loaned = (int) db_get_var(
            "SELECT COUNT(*) FROM loans WHERE copy_id = ? AND return_date IS NULL",
            [$copy['id']]
        );
        if ($loaned) {
            $previewErrors[] = ($lang['book_number_field'] ?? 'Buchungsnummer') . " '$num' " . ($lang['currently_on_loan'] ?? 'ist derzeit ausgeliehen.');
            continue;
        }

        $previewCopies[] = $copy;
    }
}

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Zurück -->
    <div class="mb-6">
        <a href="books-edit.php" class="text-sm text-gray-600 hover:text-gray-900">← <?php echo $lang['to_book_list'] ?? 'Zur Bücherliste'; ?></a>
    </div>

    <!-- Flash -->
    <?php
    $flash = get_flash();
    if ($flash):
    ?>
        <div class="mb-6 px-4 py-3 rounded-lg border-l-4 text-sm
            <?php echo $flash['type'] === 'success'
                ? 'bg-green-50 border-green-400 text-green-800'
                : 'bg-red-50 border-red-400 text-red-700'; ?>">
            <?php echo esc_html($flash['message']); ?>
        </div>
    <?php endif; ?>

    <!-- ── Schritt 1: Eingabe ─────────────────────────────────────────────── -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
        <h2 class="font-semibold text-gray-800 mb-1"><?php echo $lang['enter_book_numbers'] ?? 'Buchungsnummern eingeben'; ?></h2>
        <p class="text-sm text-gray-500 mb-4">
            <?php echo $lang['enter_book_numbers_help'] ?? 'Tragen Sie die Buchungsnummern der auszuscheidenden Exemplare ein — mehrere Nummern durch Komma, Strichpunkt oder Leerzeichen trennen.'; ?>
        </p>

        <form method="POST" class="space-y-4">
            <input type="hidden" name="action" value="preview">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">
                    <?php echo $lang['book_numbers_label'] ?? 'Buchungsnummern'; ?> <span class="text-red-500">*</span>
                </label>
                <textarea name="book_numbers" rows="3" required
                          placeholder="z.B. 14613, 14613_2, 8001"
                          class="w-full px-3 py-2 border border-gray-300 rounded-lg font-mono text-sm focus:ring-2 focus:ring-blue-500"
                          ><?php echo esc_html($inputNumbers); ?></textarea>
            </div>
            <button type="submit"
                    class="px-5 py-2 bg-blue-600 text-white font-semibold rounded-lg hover:bg-blue-700 text-sm">
                <?php echo $lang['show_preview'] ?? 'Vorschau anzeigen'; ?>
            </button>
        </form>
    </div>

    <!-- ── Schritt 2: Vorschau / Bestätigung ─────────────────────────────── -->
    <?php if (!empty($previewErrors) || !empty($previewCopies)): ?>

        <?php if (!empty($previewErrors)): ?>
            <div class="mb-4 bg-red-50 border-l-4 border-red-400 p-4 rounded space-y-1">
                <?php foreach ($previewErrors as $err): ?>
                    <p class="text-sm text-red-700"><?php echo esc_html($err); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (!empty($previewCopies)): ?>
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                <h2 class="font-semibold text-gray-800 mb-1">
                    <?php echo $lang['preview_heading'] ?? 'Vorschau'; ?>:
                    <?php echo count($previewCopies); ?>
                    <?php echo count($previewCopies) !== 1
                        ? ($lang['copies_discard_plural'] ?? 'Exemplare ausscheiden')
                        : ($lang['copies_discard_label'] ?? 'Exemplar ausscheiden'); ?>
                </h2>
                <p class="text-sm text-gray-500 mb-5">
                    <?php echo $lang['status_discarded'] ?? 'ausgeschieden'; ?>.
                    <?php echo $lang['book_data_preserved'] ?? 'Die Buchdaten bleiben erhalten.'; ?>
                </p>

                <!-- Exemplar-Tabelle -->
                <div class="overflow-x-auto mb-6">
                    <table class="w-full text-sm">
                        <thead>
                            <tr class="border-b border-gray-200 text-left text-xs text-gray-500 uppercase">
                                <th class="pb-2 pr-4"><?php echo $lang['book_nr_col'] ?? 'Buchungsnr.'; ?></th>
                                <th class="pb-2 pr-4"><?php echo $lang['signature'] ?? 'Signatur'; ?></th>
                                <th class="pb-2 pr-4"><?php echo $lang['title_col'] ?? 'Titel'; ?></th>
                                <th class="pb-2"><?php echo $lang['author_col'] ?? 'Autor'; ?></th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            <?php foreach ($previewCopies as $copy): ?>
                                <tr>
                                    <td class="py-2 pr-4">
                                        <span class="px-2 py-0.5 bg-gray-200 text-gray-700 rounded font-mono text-xs">
                                            <?php echo esc_html($copy['book_number']); ?>
                                        </span>
                                    </td>
                                    <td class="py-2 pr-4 font-mono text-xs text-gray-600">
                                        <?php echo esc_html($copy['signature'] ?? '—'); ?>
                                    </td>
                                    <td class="py-2 pr-4 text-gray-900">
                                        <?php echo esc_html($copy['title']); ?>
                                    </td>
                                    <td class="py-2 text-gray-600">
                                        <?php echo esc_html($copy['author'] ?? '—'); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Warnhinweis -->
                <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded mb-5">
                    <div class="flex items-start gap-3">
                        <svg class="w-5 h-5 text-yellow-500 mt-0.5 shrink-0" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                        </svg>
                        <div>
                            <p class="font-bold text-yellow-800 text-sm"><?php echo $lang['check_list_warning'] ?? 'Bitte prüfen Sie die Liste sorgfältig.'; ?></p>
                            <p class="text-yellow-700 text-sm mt-0.5">
                                <?php echo $lang['discarded_cannot_loan'] ?? 'Ausgeschiedene Exemplare können nicht mehr ausgeliehen werden. Die Buchungsnummern bleiben in der Datenbank erhalten.'; ?>
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Bestätigen -->
                <form method="POST" class="flex gap-3">
                    <input type="hidden" name="action" value="confirm_remove">
                    <input type="hidden" name="copy_ids"
                           value="<?php echo esc_attr(implode(',', array_column($previewCopies, 'id'))); ?>">
                    <button type="submit"
                            class="px-5 py-2.5 bg-red-600 text-white font-semibold rounded-lg hover:bg-red-700 text-sm">
                        <?php echo $lang['confirm_remove'] ?? 'Endgültig ausscheiden'; ?>
                    </button>
                    <a href="books-remove.php"
                       class="px-5 py-2.5 bg-gray-100 text-gray-700 font-medium rounded-lg hover:bg-gray-200 text-sm">
                        <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                    </a>
                </form>
            </div>
        <?php endif; ?>

    <?php endif; ?>

</div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
