<?php
/**
 * Littera Import System
 * Import books and copies from Littera database CSV export (mdbtools)
 *
 * Reads from: uploads/littera/csv_export/
 * Required files: titel.csv, exemplar.csv
 * Lookup files: verlag.csv, sprache.csv, medienart.csv
 * All CSV files are Windows-1252 encoded (mdbtools export)
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang = load_language();
$page_title = $lang['littera_import'] ?? 'Littera-Import';
$current_user = get_logged_in_user();

// CSV base directory
define('CSV_DIR', __DIR__ . '/../uploads/littera/csv_export/');

// Initialize import session
if (!isset($_SESSION['littera_import'])) {
    $_SESSION['littera_import'] = [
        'step' => 1,
        'stats' => []
    ];
}

$import = &$_SESSION['littera_import'];

// Check if required CSV files exist
$requiredFiles = ['titel.csv', 'exemplar.csv', 'verlag.csv', 'sprache.csv', 'medienart.csv'];
$missingFiles = [];
foreach ($requiredFiles as $f) {
    if (!file_exists(CSV_DIR . $f)) $missingFiles[] = $f;
}
$csvReady = empty($missingFiles);

// ─── File Upload Handler (regular POST, not AJAX) ────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'upload_csv') {
    // Create directory if needed
    if (!is_dir(CSV_DIR)) {
        mkdir(CSV_DIR, 0755, true);
    }

    $allowedNames = array_flip($requiredFiles);
    $uploaded = [];
    $errors   = [];

    foreach ($_FILES as $key => $file) {
        if ($file['error'] === UPLOAD_ERR_NO_FILE) continue;
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $errors[] = 'Upload-Fehler bei ' . esc_html($key) . ' (Code ' . $file['error'] . ')';
            continue;
        }
        $name = basename($file['name']);
        if (!isset($allowedNames[$name])) {
            $errors[] = 'Unbekannte Datei: ' . esc_html($name) . ' (erlaubt: ' . implode(', ', $requiredFiles) . ')';
            continue;
        }
        if (move_uploaded_file($file['tmp_name'], CSV_DIR . $name)) {
            $uploaded[] = $name;
        } else {
            $errors[] = 'Konnte ' . esc_html($name) . ' nicht speichern';
        }
    }

    if ($uploaded) {
        set_flash('Hochgeladen: ' . implode(', ', $uploaded), 'success');
    }
    if ($errors) {
        set_flash(implode('; ', $errors), 'error');
    }
    header('Location: books-littera-import.php#analyze');
    exit;
}

// AJAX Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');

    try {
        switch ($_POST['action']) {
            case 'analyze':
                handleAnalyze();
                break;
            case 'preview':
                handlePreview();
                break;
            case 'import':
                handleImport();
                break;
            case 'reset':
                // Clear imported data
                handleReset();
                break;
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// ============================================================
// CSV Reading Functions
// ============================================================

/**
 * Read a CSV file with Windows-1252 to UTF-8 conversion
 * Uses fgetcsv for proper CSV parsing (handles quoted fields)
 */
function readLitteraCsv($filename, $limit = null) {
    $filepath = CSV_DIR . $filename;
    if (!file_exists($filepath)) {
        throw new Exception("Datei nicht gefunden: $filename");
    }

    $handle = fopen($filepath, 'r');
    if (!$handle) throw new Exception("Kann $filename nicht öffnen");

    // Detect encoding: read first 4KB and check if valid UTF-8
    $sample = fread($handle, 4096);
    $isUtf8 = mb_check_encoding($sample, 'UTF-8');
    rewind($handle);

    // Read header
    $headerRaw = fgetcsv($handle);
    if (!$headerRaw) {
        fclose($handle);
        throw new Exception("Leere Datei: $filename");
    }
    $headers = array_map(function($v) use ($isUtf8) {
        return trim($isUtf8 ? $v : iconv('Windows-1252', 'UTF-8//IGNORE', $v));
    }, $headerRaw);

    $data = [];
    $count = 0;

    while (($row = fgetcsv($handle)) !== false) {
        // Convert encoding only if not already UTF-8
        if (!$isUtf8) {
            $row = array_map(function($v) {
                return iconv('Windows-1252', 'UTF-8//IGNORE', $v);
            }, $row);
        }

        // Skip rows with wrong column count
        if (count($row) !== count($headers)) continue;

        $assoc = array_combine($headers, $row);
        $data[] = $assoc;
        $count++;

        if ($limit !== null && $count >= $limit) break;
    }

    fclose($handle);
    return $data;
}

/**
 * Count data rows in a CSV file (excluding header)
 */
function countCsvRows($filename) {
    $filepath = CSV_DIR . $filename;
    if (!file_exists($filepath)) return 0;

    $handle = fopen($filepath, 'r');
    fgetcsv($handle); // skip header

    $count = 0;
    while (fgetcsv($handle) !== false) {
        $count++;
    }
    fclose($handle);
    return $count;
}

/**
 * Load a lookup table: Buchungsnummer → field value
 */
function loadLookup($filename, $valueField) {
    $lookup = [];
    try {
        $data = readLitteraCsv($filename);
        foreach ($data as $row) {
            $key = $row['Buchungsnummer'] ?? '';
            $val = $row[$valueField] ?? '';
            if ($key !== '') {
                $lookup[$key] = $val;
            }
        }
    } catch (Exception $e) {
        // Lookup file missing is non-fatal
    }
    return $lookup;
}

// ============================================================
// Data Cleaning Functions
// ============================================================

/**
 * Clean title: remove ¬ sort markers, normalize whitespace
 */
function cleanTitle($title) {
    // Remove ¬ markers (Littera sort helpers like "¬Das¬ verlorene Symbol")
    $title = preg_replace('/¬/', '', $title);
    // The iconv conversion may produce Â before ¬, clean that too
    $title = str_replace(['Â¬', 'Â'], '', $title);
    $title = trim($title);
    return preg_replace('/\s+/', ' ', $title);
}

/**
 * Clean author: trim, normalize whitespace, keep "Nachname, Vorname" format
 * (Bibliotheksstandard: Nachname, Vorname)
 */
function cleanAuthor($author) {
    $author = trim($author);
    if (empty($author)) return null;
    return preg_replace('/\s+/', ' ', $author) ?: null;
}

/**
 * Extract short author from author string (first 3 letters of last name)
 */
function extractAuthorShort($haupteintrag) {
    $haupteintrag = trim($haupteintrag);
    if (empty($haupteintrag)) return null;

    // "Nachname, Vorname" -> take first 3 chars of Nachname
    $parts = explode(',', $haupteintrag, 2);
    $lastName = trim($parts[0]);
    if (empty($lastName)) return null;

    // Use mb_substr for multibyte safety
    return mb_substr($lastName, 0, 3, 'UTF-8');
}

/**
 * Normalize ISBN: remove all non-digit/X characters
 */
function normalizeIsbn($isbn) {
    $isbn = preg_replace('/[^0-9Xx]/', '', strtoupper(trim($isbn)));
    // Accept ISBN-10 (10 chars) and ISBN-13 (13 chars)
    if (strlen($isbn) === 10 || strlen($isbn) === 13) {
        return $isbn;
    }
    // If it's close, still keep it
    if (strlen($isbn) >= 9) {
        return $isbn;
    }
    return null;
}

/**
 * Parse Littera date "MM/DD/YY HH:MM:SS" to MySQL DATE
 */
function parseLitteraDate($dateStr) {
    $dateStr = trim($dateStr);
    if (empty($dateStr)) return null;

    $date = DateTime::createFromFormat('m/d/y H:i:s', $dateStr);
    if ($date) return $date->format('Y-m-d');

    $date = DateTime::createFromFormat('m/d/y', substr($dateStr, 0, 8));
    if ($date) return $date->format('Y-m-d');

    return null;
}

// ============================================================
// AJAX Handlers
// ============================================================

function handleAnalyze() {
    global $import;

    // Count rows
    $titelCount = countCsvRows('titel.csv');
    $exemplarCount = countCsvRows('exemplar.csv');

    // Check lookup files
    $verlagExists = file_exists(CSV_DIR . 'verlag.csv');
    $spracheExists = file_exists(CSV_DIR . 'sprache.csv');
    $medienartExists = file_exists(CSV_DIR . 'medienart.csv');

    // Count existing books in DB
    $existingBooks = db_get_var("SELECT COUNT(*) FROM books");
    $existingCopies = db_get_var("SELECT COUNT(*) FROM book_copies");

    $import['step'] = 2;

    echo json_encode([
        'success' => true,
        'stats' => [
            'titel' => $titelCount,
            'exemplar' => $exemplarCount,
            'verlag' => $verlagExists ? countCsvRows('verlag.csv') : 0,
            'sprache' => $spracheExists ? countCsvRows('sprache.csv') : 0,
            'lookup_verlag' => $verlagExists,
            'lookup_sprache' => $spracheExists,
            'lookup_medienart' => $medienartExists,
            'db_books' => $existingBooks,
            'db_copies' => $existingCopies
        ]
    ]);
}

function handlePreview() {
    // Load lookups for resolved preview
    $verlage = loadLookup('verlag.csv', 'Verlag');
    $sprachen = loadLookup('sprache.csv', 'Sprache');

    // Read first 15 titles
    $titel = readLitteraCsv('titel.csv', 15);
    $exemplar = readLitteraCsv('exemplar.csv', 15);

    // Enrich title data for preview
    $titelPreview = [];
    foreach ($titel as $row) {
        $titelPreview[] = [
            'buchungsnummer' => $row['Buchungsnummer'],
            'titel' => cleanTitle($row['Haupttitel'] ?? ''),
            'untertitel' => cleanTitle($row['Untertitel'] ?? ''),
            'autor' => cleanAuthor($row['Haupteintrag'] ?? ''),
            'isbn_original' => $row['ISBN'] ?? '',
            'isbn_normalisiert' => normalizeIsbn($row['ISBN'] ?? ''),
            'verlag_id' => $row['Verlag'] ?? '',
            'verlag_name' => $verlage[$row['Verlag'] ?? ''] ?? '?',
            'sprache_id' => $row['Sprache'] ?? '',
            'sprache_name' => $sprachen[$row['Sprache'] ?? ''] ?? '?',
            'jahr' => $row['Erscheinungsjahr'] ?? '',
        ];
    }

    // Enrich exemplar data for preview
    $exemplarPreview = [];
    foreach ($exemplar as $row) {
        $sig1 = $row['Sig1'] ?? '';
        $sig2 = $row['Sig2'] ?? '';
        $exNr = $row['Exemplarnummer'] ?? '';
        $signature = trim("$sig1 $sig2 $exNr");

        $exemplarPreview[] = [
            'exemplar_id' => $row['Buchungsnummer'],
            'titel_fk' => $row['Titel'],
            'exemplarnummer' => $exNr,
            'sig1' => $sig1,
            'sig2' => $sig2,
            'signature' => $signature,
            'status' => $row['Status'] ?? '',
            'zugangsdatum' => parseLitteraDate($row['Zugangsdatum'] ?? '') ?? '-',
        ];
    }

    echo json_encode([
        'success' => true,
        'titel' => $titelPreview,
        'exemplar' => $exemplarPreview,
        'counts' => [
            'titel' => countCsvRows('titel.csv'),
            'exemplar' => countCsvRows('exemplar.csv')
        ]
    ]);
}

function handleImport() {
    global $pdo, $import;

    set_time_limit(0);
    ini_set('memory_limit', '512M');

    $limit = isset($_POST['limit']) ? intval($_POST['limit']) : null;

    $stats = [
        'titel' => ['total' => 0, 'imported' => 0, 'errors' => 0, 'duplicates' => 0],
        'exemplar' => ['total' => 0, 'imported' => 0, 'skipped' => 0, 'errors' => 0, 'no_title' => 0],
        'errors' => [],
        'start_time' => microtime(true)
    ];

    try {
        // Load lookup tables
        $verlage = loadLookup('verlag.csv', 'Verlag');
        $sprachen = loadLookup('sprache.csv', 'Sprache');

        $pdo->beginTransaction();

        // Phase 1: Import titles
        $titelMapping = importTitles($verlage, $sprachen, $stats, $limit);

        // Phase 2: Import exemplars
        importExemplare($titelMapping, $stats);

        // Update next_book_number setting based on highest imported book_number
        $maxBookNr = db_get_var("SELECT MAX(CAST(book_number AS UNSIGNED)) FROM book_copies WHERE book_number REGEXP '^[0-9]+$'");
        if ($maxBookNr) {
            $nextNr = intval($maxBookNr) + 1;
            db_query("INSERT INTO settings (setting_key, setting_value) VALUES ('next_book_number', ?)
                      ON DUPLICATE KEY UPDATE setting_value = ?", [$nextNr, $nextNr]);
        }

        $pdo->commit();

        $stats['duration'] = round(microtime(true) - $stats['start_time'], 2);
        // Keep only first 20 errors for response
        if (count($stats['errors']) > 20) {
            $stats['error_total'] = count($stats['errors']);
            $stats['errors'] = array_slice($stats['errors'], 0, 20);
        }

        $import['step'] = 4;
        $import['stats'] = $stats;

        log_activity('littera_import', 'system', 0, json_encode([
            'titel_imported' => $stats['titel']['imported'],
            'exemplar_imported' => $stats['exemplar']['imported'],
            'duration' => $stats['duration']
        ]));

        echo json_encode(['success' => true, 'stats' => $stats]);

    } catch (Exception $e) {
        $pdo->rollBack();
        $stats['duration'] = round(microtime(true) - $stats['start_time'], 2);
        echo json_encode(['success' => false, 'error' => $e->getMessage(), 'stats' => $stats]);
    }
}

function handleReset() {
    global $pdo;

    // Delete all imported copies and books
    $pdo->exec("DELETE FROM book_copies");
    $pdo->exec("DELETE FROM books");
    $pdo->exec("ALTER TABLE book_copies AUTO_INCREMENT = 1");
    $pdo->exec("ALTER TABLE books AUTO_INCREMENT = 1");

    $_SESSION['littera_import'] = ['step' => 1, 'stats' => []];

    echo json_encode(['success' => true, 'message' => 'Alle Daten gelöscht']);
}

// ============================================================
// Import Functions
// ============================================================

function importTitles($verlage, $sprachen, &$stats, $limit = null) {
    $mapping = []; // Littera Buchungsnummer => new books.id
    $data = readLitteraCsv('titel.csv', $limit);
    $stats['titel']['total'] = count($data);

    foreach ($data as $row) {
        try {
            $buchungsNr = $row['Buchungsnummer'] ?? '';
            if (empty($buchungsNr)) continue;

            // Clean title
            $title = cleanTitle($row['Haupttitel'] ?? '');
            $subtitle = cleanTitle($row['Untertitel'] ?? '');

            // Fallback for empty title
            if (empty($title)) {
                $title = cleanTitle($row['HaupttitelKurz'] ?? '');
            }
            if (empty($title) && !empty($subtitle)) {
                $title = $subtitle;
                $subtitle = '';
            }
            if (empty($title)) {
                $title = 'Buchungsnummer ' . $buchungsNr;
            }

            // Normalize ISBN
            $isbn = normalizeIsbn($row['ISBN'] ?? '');

            // Duplicate check by ISBN
            if (!empty($isbn)) {
                $existing = db_get_row("SELECT id FROM books WHERE isbn = ?", [$isbn]);
                if ($existing) {
                    $stats['titel']['duplicates']++;
                    $mapping[$buchungsNr] = $existing['id'];
                    continue;
                }
            }

            // Author
            $authorRaw = $row['Haupteintrag'] ?? '';
            $author = cleanAuthor($authorRaw);
            $authorShort = extractAuthorShort($authorRaw);

            // Duplicate check by title + author (for books without ISBN)
            if (empty($isbn) && !empty($title) && !empty($author)) {
                $existing = db_get_row(
                    "SELECT id FROM books WHERE title = ? AND author = ?",
                    [$title, $author]
                );
                if ($existing) {
                    $stats['titel']['duplicates']++;
                    $mapping[$buchungsNr] = $existing['id'];
                    continue;
                }
            }

            // Resolve publisher
            $verlagId = $row['Verlag'] ?? '';
            $publisher = $verlage[$verlagId] ?? null;

            // Resolve language
            $spracheId = $row['Sprache'] ?? '';
            $language = $sprachen[$spracheId] ?? null;
            // Capitalize first letter
            if ($language) {
                $language = mb_convert_case($language, MB_CASE_TITLE, 'UTF-8');
            }

            // Parse year
            $year = null;
            if (!empty($row['Erscheinungsjahr'])) {
                if (preg_match('/(\d{4})/', $row['Erscheinungsjahr'], $m)) {
                    $year = (int)$m[1];
                    if ($year < 1000 || $year > (date('Y') + 5)) $year = null;
                }
            }

            // Parse pages
            $pages = null;
            if (!empty($row['Umfang'])) {
                if (preg_match('/(\d{1,5})/', $row['Umfang'], $m)) {
                    $pages = (int)$m[1];
                    if ($pages < 1 || $pages > 9999) $pages = null;
                }
            }

            // Edition
            $edition = !empty($row['Auflage']) ? trim($row['Auflage']) : null;

            // Place of publication
            $place = !empty($row['Ort']) ? trim($row['Ort']) : null;

            // Series
            $series = null;
            if (!empty($row['Reihe'])) {
                // Reihe is an ID, but we don't resolve it for now — use Reihenvermerk
                $series = !empty($row['Reihenvermerk']) ? trim($row['Reihenvermerk']) : null;
            }

            // Description from Annotation/Bemerkung
            $description = '';
            if (!empty($row['Annotation'])) $description .= trim($row['Annotation']);
            if (!empty($row['Bemerkung'])) {
                if (!empty($description)) $description .= "\n";
                $description .= trim($row['Bemerkung']);
            }

            // Acquisition date
            $createdAt = parseLitteraDate($row['Zugangsdatum'] ?? '');

            $bookId = db_insert('books', [
                'isbn' => $isbn,
                'title' => $title,
                'subtitle' => !empty($subtitle) ? $subtitle : null,
                'author' => $author,
                'author_short' => $authorShort,
                'edition' => $edition,
                'place_of_publication' => $place,
                'publisher' => $publisher,
                'year' => $year,
                'pages' => $pages,
                'language' => $language ?: 'Deutsch',
                'description' => !empty($description) ? $description : null,
                'series' => $series,
                'created_by' => 1,
                'created_at' => $createdAt ? $createdAt . ' 00:00:00' : date('Y-m-d H:i:s'),
            ]);

            $mapping[$buchungsNr] = $bookId;
            $stats['titel']['imported']++;

        } catch (Exception $e) {
            $stats['titel']['errors']++;
            $stats['errors'][] = [
                'type' => 'Titel',
                'id' => $row['Buchungsnummer'] ?? '?',
                'error' => $e->getMessage()
            ];
        }
    }

    return $mapping;
}

function importExemplare($titelMapping, &$stats) {
    $data = readLitteraCsv('exemplar.csv');
    $stats['exemplar']['total'] = count($data);

    // Track copy index per book
    $copyCounter = [];

    foreach ($data as $row) {
        try {
            $titelFk = $row['Titel'] ?? '';

            if (!isset($titelMapping[$titelFk])) {
                $stats['exemplar']['no_title']++;
                continue;
            }

            $bookId = $titelMapping[$titelFk];
            $exemplarNr = trim($row['Exemplarnummer'] ?? '');
            $sig1 = trim($row['Sig1'] ?? '');
            $sig2 = trim($row['Sig2'] ?? '');

            // Check for duplicate by accession_number
            if (!empty($exemplarNr)) {
                $existing = db_get_row(
                    "SELECT id FROM book_copies WHERE accession_number = ?",
                    [$exemplarNr]
                );
                if ($existing) {
                    $stats['exemplar']['skipped']++;
                    continue;
                }
            }

            // Build signature: "Sig1 Sig2 Exemplarnummer"
            $signatureParts = [];
            if ($sig1 !== '') $signatureParts[] = $sig1;
            if ($sig2 !== '') $signatureParts[] = $sig2;
            if ($exemplarNr !== '') $signatureParts[] = $exemplarNr;
            $signature = implode(' ', $signatureParts);

            // Copy index per book (1, 2, 3, ...)
            if (!isset($copyCounter[$bookId])) {
                // Check existing copies for this book
                $maxIdx = db_get_var(
                    "SELECT MAX(copy_index) FROM book_copies WHERE book_id = ?",
                    [$bookId]
                );
                $copyCounter[$bookId] = ($maxIdx ?? 0) + 1;
            } else {
                $copyCounter[$bookId]++;
            }

            // Parse acquisition date
            $acquisitionDate = parseLitteraDate($row['Zugangsdatum'] ?? '');

            // Map Littera status: 1=verfügbar
            $status = 'verfügbar';
            $litteraStatus = trim($row['Status'] ?? '');
            if ($litteraStatus !== '' && $litteraStatus !== '1') {
                $status = 'aussortiert';
            }

            // Location = Sig1 (category like "De", "MEd")
            $location = !empty($sig1) ? $sig1 : null;

            // Condition note from Bemerkung
            $conditionNote = !empty(trim($row['Bemerkung'] ?? '')) ? trim($row['Bemerkung']) : null;

            db_insert('book_copies', [
                'book_id' => $bookId,
                'book_number' => !empty($exemplarNr) ? $exemplarNr : null,
                'copy_index' => $copyCounter[$bookId],
                'status' => $status,
                'location' => $location,
                'accession_number' => !empty($exemplarNr) ? $exemplarNr : null,
                'signature' => !empty($signature) ? $signature : null,
                'acquisition_date' => $acquisitionDate,
                'condition_note' => $conditionNote,
            ]);

            $stats['exemplar']['imported']++;

        } catch (Exception $e) {
            $stats['exemplar']['errors']++;
            $stats['errors'][] = [
                'type' => 'Exemplar',
                'id' => $row['Buchungsnummer'] ?? '?',
                'error' => $e->getMessage()
            ];
        }
    }
}

// ============================================================
// Page Output
// ============================================================

include __DIR__ . '/includes/header.php';

$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Page Header -->
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900"><?php echo $lang['littera_import'] ?? 'Littera-Import'; ?></h1>
            <p class="mt-2 text-sm text-gray-600">
                <?php echo $lang['littera_import_subtitle'] ?? 'Importieren Sie Ihren Altbestand aus der Littera-Datenbank (CSV-Export via mdbtools)'; ?>
            </p>
        </div>

        <!-- Flash Message -->
        <?php if ($flash): ?>
            <div class="mb-6 flash-message bg-<?php echo $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'blue'); ?>-50 border-l-4 border-<?php echo $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'blue'); ?>-400 p-4 rounded">
                <p class="text-sm text-<?php echo $flash['type'] === 'success' ? 'green' : ($flash['type'] === 'error' ? 'red' : 'blue'); ?>-700">
                    <?php echo esc_html($flash['message']); ?>
                </p>
            </div>
        <?php endif; ?>

        <!-- Progress Steps -->
        <div class="mb-8">
            <div class="flex items-center">
                <?php
                $steps = [
                    1 => ['label' => $lang['step_analysis'] ?? 'Analyse'],
                    2 => ['label' => $lang['step_preview'] ?? 'Vorschau'],
                    3 => ['label' => $lang['step_import']  ?? 'Import'],
                    4 => ['label' => $lang['step_result']  ?? 'Ergebnis'],
                ];
                foreach ($steps as $num => $step):
                    $active = $import['step'] >= $num;
                    $complete = $import['step'] > $num;
                ?>
                    <div class="flex-1 text-center">
                        <div class="w-10 h-10 mx-auto rounded-full <?php echo $active ? 'bg-blue-600' : 'bg-gray-300'; ?> text-white flex items-center justify-center font-bold text-sm">
                            <?php echo $complete ? '&#10003;' : $num; ?>
                        </div>
                        <p class="mt-2 text-xs font-medium <?php echo $active ? 'text-blue-600' : 'text-gray-500'; ?>">
                            <?php echo $step['label']; ?>
                        </p>
                    </div>
                    <?php if ($num < count($steps)): ?>
                        <div class="flex-1 border-t-2 <?php echo $import['step'] > $num ? 'border-blue-600' : 'border-gray-300'; ?> mx-2 mt-[-8px]"></div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Tabs -->
        <div class="mb-6 border-b border-gray-200">
            <nav class="-mb-px flex space-x-8">
                <a href="#analyze" class="tab-link border-b-2 py-4 px-1 text-sm font-medium cursor-pointer" data-tab="analyze">
                    <?php echo $lang['step_analysis'] ?? 'Analyse'; ?>
                </a>
                <a href="#preview" class="tab-link border-b-2 py-4 px-1 text-sm font-medium cursor-pointer" data-tab="preview">
                    <?php echo $lang['step_preview'] ?? 'Vorschau'; ?>
                </a>
                <a href="#import" class="tab-link border-b-2 py-4 px-1 text-sm font-medium cursor-pointer" data-tab="import">
                    <?php echo $lang['step_import'] ?? 'Import'; ?>
                </a>
                <a href="#result" class="tab-link border-b-2 py-4 px-1 text-sm font-medium cursor-pointer" data-tab="result">
                    <?php echo $lang['step_result'] ?? 'Ergebnis'; ?>
                </a>
            </nav>
        </div>

        <!-- ==================== ANALYZE TAB ==================== -->
        <div id="tab-analyze" class="tab-content">
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['step1_title'] ?? 'Schritt 1: Datenanalyse'; ?></h2>
                    <p class="mt-1 text-sm text-gray-500">
                        <?php echo $lang['step1_subtitle'] ?? 'CSV-Dateien aus'; ?> <code class="bg-gray-100 px-1 rounded">uploads/littera/csv_export/</code>
                    </p>
                </div>
                <div class="p-6">

                    <!-- Upload Form -->
                    <div class="mb-6 bg-gray-50 border border-gray-200 rounded-lg p-4">
                        <p class="text-sm font-semibold text-gray-700 mb-3"><?php echo $lang['upload_csv_files'] ?? 'CSV-Dateien hochladen'; ?></p>
                        <form method="POST" enctype="multipart/form-data" class="space-y-3">
                            <input type="hidden" name="action" value="upload_csv">
                            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3">
                                <?php foreach ($requiredFiles as $f):
                                    $exists = file_exists(CSV_DIR . $f);
                                ?>
                                    <div>
                                        <label class="block text-xs font-medium text-gray-600 mb-1">
                                            <?php echo esc_html($f); ?>
                                            <?php if ($exists): ?>
                                                <span class="text-green-600 ml-1">&#10003; <?php echo $lang['file_present'] ?? 'vorhanden'; ?></span>
                                            <?php else: ?>
                                                <span class="text-red-500 ml-1"><?php echo $lang['file_missing'] ?? 'fehlt'; ?></span>
                                            <?php endif; ?>
                                        </label>
                                        <input type="file" name="<?php echo esc_attr($f); ?>" accept=".csv"
                                               class="block w-full text-xs text-gray-500 file:mr-2 file:py-1 file:px-2 file:rounded file:border-0 file:text-xs file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <button type="submit"
                                    class="px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-md hover:bg-blue-700">
                                <?php echo $lang['upload_files_btn'] ?? 'Dateien hochladen'; ?>
                            </button>
                        </form>
                    </div>

                    <!-- File Status -->
                    <div class="mb-4">
                        <p class="text-sm font-medium text-gray-700 mb-2"><?php echo $lang['file_status_label'] ?? 'Status in'; ?> <code class="bg-gray-100 px-1 rounded">uploads/littera/csv_export/</code>:</p>
                        <div class="grid grid-cols-2 sm:grid-cols-5 gap-2">
                            <?php foreach ($requiredFiles as $f):
                                $exists = file_exists(CSV_DIR . $f);
                            ?>
                                <div class="p-2 rounded text-xs font-mono <?php echo $exists ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700'; ?>">
                                    <?php echo $exists ? '&#10003;' : '&#10007;'; ?> <?php echo esc_html($f); ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php if (!$csvReady): ?>
                        <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-4">
                            <p class="text-sm text-red-700 font-bold"><?php echo $lang['missing_files_label'] ?? 'Fehlende Dateien'; ?>: <?php echo esc_html(implode(', ', $missingFiles)); ?></p>
                            <p class="text-sm text-red-600 mt-1">
                                <?php echo $lang['missing_files_help'] ?? 'Dateien hochladen oder mit mdbtools exportieren und nach uploads/littera/csv_export/ kopieren.'; ?>
                            </p>
                        </div>
                    <?php endif; ?>

                    <div id="analyzeResult" class="mb-6"></div>

                    <div class="flex gap-3">
                        <button onclick="runAnalyze()" <?php echo !$csvReady ? 'disabled' : ''; ?>
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                            <?php echo $lang['analyze_start'] ?? 'Analyse starten'; ?>
                        </button>
                        <button onclick="switchTab('preview')"
                                class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                            <?php echo $lang['go_to_preview'] ?? 'Weiter zur Vorschau'; ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- ==================== PREVIEW TAB ==================== -->
        <div id="tab-preview" class="tab-content hidden">
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['step2_title'] ?? 'Schritt 2: Datenvorschau'; ?></h2>
                    <p class="mt-1 text-sm text-gray-500"><?php echo $lang['step2_subtitle'] ?? 'Erste 15 Einträge mit allen Transformationen'; ?></p>
                </div>
                <div class="p-6">
                    <div id="previewContainer">
                        <p class="text-gray-500"><?php echo $lang['click_load_preview'] ?? 'Klicken Sie "Vorschau laden" um die Daten zu sehen.'; ?></p>
                    </div>

                    <div class="flex gap-3 mt-6">
                        <button onclick="switchTab('analyze')"
                                class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                            <?php echo $lang['go_back_btn'] ?? 'Zurück'; ?>
                        </button>
                        <button onclick="loadPreview()"
                                class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            <?php echo $lang['load_preview_btn'] ?? 'Vorschau laden'; ?>
                        </button>
                        <button onclick="switchTab('import')"
                                class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                            <?php echo $lang['go_to_import_btn'] ?? 'Weiter zum Import'; ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- ==================== IMPORT TAB ==================== -->
        <div id="tab-import" class="tab-content hidden">
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['step3_title'] ?? 'Schritt 3: Import durchführen'; ?></h2>
                </div>
                <div class="p-6">

                    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                        <p class="text-sm text-yellow-800 font-bold"><?php echo $lang['import_warning_title'] ?? 'Hinweis'; ?>:</p>
                        <ul class="text-sm text-yellow-700 list-disc list-inside mt-1">
                            <li><?php echo $lang['import_duplicates_note'] ?? 'Duplikate werden per ISBN erkannt und übersprungen'; ?></li>
                            <li><?php echo $lang['import_no_isbn_note'] ?? 'Bei Bücher ohne ISBN: Duplikaterkennung über Titel + Autor'; ?></li>
                            <li><?php echo $lang['import_exemplar_note'] ?? 'Exemplare mit gleicher Inventarnummer werden übersprungen'; ?></li>
                        </ul>
                    </div>

                    <div class="grid grid-cols-2 gap-4 mb-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2"><?php echo $lang['import_mode'] ?? 'Import-Modus'; ?></label>
                            <select id="importMode" class="w-full border-gray-300 rounded-md shadow-sm p-2 border">
                                <option value="10"><?php echo $lang['import_mode_test'] ?? 'Test-Import (10 Titel)'; ?></option>
                                <option value="50"><?php echo $lang['import_mode_small'] ?? 'Kleiner Import (50 Titel)'; ?></option>
                                <option value="0" selected><?php echo $lang['import_mode_full'] ?? 'Vollimport (alle Titel)'; ?></option>
                            </select>
                        </div>
                    </div>

                    <div id="importProgress" class="hidden mb-6">
                        <div class="text-center py-8">
                            <div class="inline-block animate-spin rounded-full h-16 w-16 border-b-4 border-blue-600"></div>
                            <p class="mt-4 text-gray-700 font-medium text-lg"><?php echo $lang['import_running'] ?? 'Import läuft...'; ?></p>
                            <p class="text-sm text-gray-500 mt-2"><?php echo $lang['import_running_wait'] ?? 'Dies kann einige Minuten dauern. Bitte das Fenster nicht schließen.'; ?></p>
                        </div>
                    </div>

                    <div id="importButtons" class="flex gap-3">
                        <button onclick="switchTab('preview')"
                                class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                            <?php echo $lang['go_back_btn'] ?? 'Zurück'; ?>
                        </button>
                        <button onclick="startImport()" id="importBtn"
                                class="px-6 py-2 bg-green-600 text-white rounded-md hover:bg-green-700">
                            <?php echo $lang['import_start_btn'] ?? 'Import starten'; ?>
                        </button>
                        <button onclick="resetData()"
                                class="px-6 py-2 bg-red-600 text-white rounded-md hover:bg-red-700"
                                title="<?php echo esc_attr($lang['reset_data_btn'] ?? 'Daten zurücksetzen'); ?>">
                            <?php echo $lang['reset_data_btn'] ?? 'Daten zurücksetzen'; ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- ==================== RESULT TAB ==================== -->
        <div id="tab-result" class="tab-content hidden">
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['step4_title'] ?? 'Schritt 4: Ergebnis'; ?></h2>
                </div>
                <div class="p-6">
                    <div id="resultContainer">
                        <?php if (!empty($import['stats'])):
                            $st = $import['stats'];
                        ?>
                            <div class="grid grid-cols-2 gap-4 mb-6">
                                <div class="bg-blue-50 rounded-lg p-6">
                                    <h3 class="font-bold text-blue-900 mb-3"><?php echo $lang['titles_label'] ?? 'Titel'; ?></h3>
                                    <div class="space-y-2 text-sm">
                                        <p><?php echo $lang['total_label'] ?? 'Gesamt'; ?>: <strong><?php echo $st['titel']['total']; ?></strong></p>
                                        <p class="text-green-700"><?php echo $lang['imported_label'] ?? 'Importiert'; ?>: <strong><?php echo $st['titel']['imported']; ?></strong></p>
                                        <p class="text-yellow-700"><?php echo $lang['duplicates_label'] ?? 'Duplikate'; ?>: <strong><?php echo $st['titel']['duplicates']; ?></strong></p>
                                        <p class="text-red-700"><?php echo $lang['errors_label'] ?? 'Fehler'; ?>: <strong><?php echo $st['titel']['errors']; ?></strong></p>
                                    </div>
                                </div>
                                <div class="bg-green-50 rounded-lg p-6">
                                    <h3 class="font-bold text-green-900 mb-3"><?php echo $lang['copies_label'] ?? 'Exemplare'; ?></h3>
                                    <div class="space-y-2 text-sm">
                                        <p><?php echo $lang['total_label'] ?? 'Gesamt'; ?>: <strong><?php echo $st['exemplar']['total']; ?></strong></p>
                                        <p class="text-green-700"><?php echo $lang['imported_label'] ?? 'Importiert'; ?>: <strong><?php echo $st['exemplar']['imported']; ?></strong></p>
                                        <p class="text-yellow-700"><?php echo $lang['skipped_label'] ?? 'Übersprungen'; ?>: <strong><?php echo $st['exemplar']['skipped']; ?></strong></p>
                                        <p class="text-gray-600"><?php echo $lang['no_title_label'] ?? 'Ohne Titel'; ?>: <strong><?php echo $st['exemplar']['no_title']; ?></strong></p>
                                        <p class="text-red-700"><?php echo $lang['errors_label'] ?? 'Fehler'; ?>: <strong><?php echo $st['exemplar']['errors']; ?></strong></p>
                                    </div>
                                </div>
                            </div>
                            <p class="text-sm text-gray-600 mb-4"><?php echo $lang['duration_label'] ?? 'Dauer'; ?>: <?php echo $st['duration']; ?> <?php echo $lang['seconds_label'] ?? 'Sekunden'; ?></p>

                            <?php if (!empty($st['errors'])): ?>
                                <details class="mb-4">
                                    <summary class="cursor-pointer text-sm text-red-600 font-medium"><?php echo $lang['show_error_details'] ?? 'Fehlerdetails anzeigen'; ?></summary>
                                    <div class="mt-2 bg-red-50 rounded p-4 text-xs overflow-x-auto">
                                        <table class="min-w-full">
                                            <thead><tr>
                                                <th class="text-left px-2"><?php echo $lang['error_type_col'] ?? 'Typ'; ?></th>
                                                <th class="text-left px-2"><?php echo $lang['error_id_col'] ?? 'ID'; ?></th>
                                                <th class="text-left px-2"><?php echo $lang['error_msg_col'] ?? 'Fehler'; ?></th>
                                            </tr></thead>
                                            <tbody>
                                            <?php foreach ($st['errors'] as $err): ?>
                                                <tr><td class="px-2"><?php echo esc_html($err['type']); ?></td><td class="px-2"><?php echo esc_html($err['id']); ?></td><td class="px-2"><?php echo esc_html($err['error']); ?></td></tr>
                                            <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </details>
                            <?php endif; ?>

                            <div class="flex gap-3">
                                <a href="<?php echo BASE_URL; ?>/admin/index.php"
                                   class="px-6 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-center">
                                    <?php echo $lang['to_dashboard'] ?? 'Zum Dashboard'; ?>
                                </a>
                                <a href="<?php echo BASE_URL; ?>/admin/books-edit.php"
                                   class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 text-center">
                                    <?php echo $lang['show_book_list'] ?? 'Bücherliste anzeigen'; ?>
                                </a>
                                <button onclick="switchTab('import')"
                                        class="px-6 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300">
                                    <?php echo $lang['new_import'] ?? 'Neuer Import'; ?>
                                </button>
                            </div>
                        <?php else: ?>
                            <p class="text-gray-600"><?php echo $lang['no_import_yet'] ?? 'Noch kein Import durchgeführt. Starten Sie den Import im vorherigen Schritt.'; ?></p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

    </div>
</div>

<script>
// Tab switching
function switchTab(name) {
    document.querySelectorAll('.tab-content').forEach(el => el.classList.add('hidden'));
    document.getElementById('tab-' + name).classList.remove('hidden');

    document.querySelectorAll('.tab-link').forEach(link => {
        if (link.dataset.tab === name) {
            link.classList.remove('border-transparent', 'text-gray-500');
            link.classList.add('border-blue-500', 'text-blue-600');
        } else {
            link.classList.remove('border-blue-500', 'text-blue-600');
            link.classList.add('border-transparent', 'text-gray-500');
        }
    });

    window.location.hash = name;
}

// Initialize tab from hash or step
document.addEventListener('DOMContentLoaded', function() {
    const hash = window.location.hash.substring(1);
    const validTabs = ['analyze', 'preview', 'import', 'result'];
    if (hash && validTabs.includes(hash)) {
        switchTab(hash);
    } else {
        const step = <?php echo $import['step']; ?>;
        switch(step) {
            case 1: switchTab('analyze'); break;
            case 2: switchTab('preview'); break;
            case 3: switchTab('import'); break;
            case 4: switchTab('result'); break;
            default: switchTab('analyze');
        }
    }
});

// Tab click handlers
document.querySelectorAll('.tab-link').forEach(link => {
    link.addEventListener('click', function(e) {
        e.preventDefault();
        switchTab(this.dataset.tab);
    });
});

// Analyze
function runAnalyze() {
    const container = document.getElementById('analyzeResult');
    container.innerHTML = '<div class="text-center py-4"><div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div><p class="mt-2 text-sm text-gray-500">Analysiere...</p></div>';

    const fd = new FormData();
    fd.append('action', 'analyze');

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (!data.success) { alert('Fehler: ' + data.error); return; }

            const s = data.stats;
            let html = '<div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">';
            html += card('Titel', s.titel, 'blue');
            html += card('Exemplare', s.exemplar, 'green');
            html += card('Verlage', s.verlag, 'purple');
            html += card('Sprachen', s.sprache, 'yellow');
            html += '</div>';

            html += '<div class="grid grid-cols-3 gap-4 mb-4">';
            html += '<div class="p-3 rounded ' + (s.lookup_verlag ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700') + ' text-sm">verlag.csv: ' + (s.lookup_verlag ? 'OK' : 'fehlt') + '</div>';
            html += '<div class="p-3 rounded ' + (s.lookup_sprache ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700') + ' text-sm">sprache.csv: ' + (s.lookup_sprache ? 'OK' : 'fehlt') + '</div>';
            html += '<div class="p-3 rounded ' + (s.lookup_medienart ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700') + ' text-sm">medienart.csv: ' + (s.lookup_medienart ? 'OK' : 'fehlt') + '</div>';
            html += '</div>';

            if (s.db_books > 0 || s.db_copies > 0) {
                html += '<div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 text-sm text-yellow-800">';
                html += '<strong>Achtung:</strong> Die Datenbank enthaelt bereits ' + s.db_books + ' Buecher und ' + s.db_copies + ' Exemplare. Duplikate werden beim Import uebersprungen.';
                html += '</div>';
            }

            container.innerHTML = html;
        })
        .catch(err => { container.innerHTML = '<p class="text-red-600">Fehler: ' + err.message + '</p>'; });
}

function card(label, count, color) {
    return '<div class="bg-' + color + '-50 rounded-lg p-4 text-center"><p class="text-2xl font-bold text-' + color + '-700">' + (count || 0).toLocaleString() + '</p><p class="text-sm text-' + color + '-600">' + label + '</p></div>';
}

// Preview
function loadPreview() {
    const container = document.getElementById('previewContainer');
    container.innerHTML = '<div class="text-center py-4"><div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div><p class="mt-2 text-sm text-gray-500">Lade Vorschau...</p></div>';

    const fd = new FormData();
    fd.append('action', 'preview');

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (!data.success) { alert('Fehler: ' + data.error); return; }

            let html = '<div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">';
            html += '<p class="text-blue-900 font-medium">' + data.counts.titel.toLocaleString() + ' Titel | ' + data.counts.exemplar.toLocaleString() + ' Exemplare</p>';
            html += '</div>';

            // Titel table
            html += '<h3 class="font-bold text-gray-900 mb-3">Titel (erste 15)</h3>';
            html += '<div class="overflow-x-auto mb-6"><table class="min-w-full text-xs border-collapse">';
            html += '<thead class="bg-gray-100"><tr><th class="border px-2 py-1 text-left">Nr</th><th class="border px-2 py-1 text-left">Titel</th><th class="border px-2 py-1 text-left">Autor</th><th class="border px-2 py-1 text-left">ISBN</th><th class="border px-2 py-1 text-left">Verlag</th><th class="border px-2 py-1 text-left">Sprache</th><th class="border px-2 py-1 text-left">Jahr</th></tr></thead><tbody>';

            data.titel.forEach(t => {
                html += '<tr class="hover:bg-gray-50">';
                html += '<td class="border px-2 py-1 font-mono">' + t.buchungsnummer + '</td>';
                html += '<td class="border px-2 py-1">' + esc(t.titel) + (t.untertitel ? '<br><span class="text-gray-400">' + esc(t.untertitel) + '</span>' : '') + '</td>';
                html += '<td class="border px-2 py-1">' + esc(t.autor || '-') + '</td>';
                html += '<td class="border px-2 py-1 font-mono">' + esc(t.isbn_normalisiert || '-') + '</td>';
                html += '<td class="border px-2 py-1">' + esc(t.verlag_name) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(t.sprache_name) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(t.jahr) + '</td>';
                html += '</tr>';
            });

            html += '</tbody></table></div>';

            // Exemplar table
            html += '<h3 class="font-bold text-gray-900 mb-3">Exemplare (erste 15)</h3>';
            html += '<div class="overflow-x-auto"><table class="min-w-full text-xs border-collapse">';
            html += '<thead class="bg-gray-100"><tr><th class="border px-2 py-1 text-left">ExID</th><th class="border px-2 py-1 text-left">TitelFK</th><th class="border px-2 py-1 text-left">InvNr</th><th class="border px-2 py-1 text-left">Sig1</th><th class="border px-2 py-1 text-left">Sig2</th><th class="border px-2 py-1 text-left">Signatur</th><th class="border px-2 py-1 text-left">Status</th><th class="border px-2 py-1 text-left">Datum</th></tr></thead><tbody>';

            data.exemplar.forEach(e => {
                html += '<tr class="hover:bg-gray-50">';
                html += '<td class="border px-2 py-1 font-mono">' + e.exemplar_id + '</td>';
                html += '<td class="border px-2 py-1 font-mono">' + e.titel_fk + '</td>';
                html += '<td class="border px-2 py-1 font-bold">' + e.exemplarnummer + '</td>';
                html += '<td class="border px-2 py-1">' + esc(e.sig1) + '</td>';
                html += '<td class="border px-2 py-1">' + esc(e.sig2) + '</td>';
                html += '<td class="border px-2 py-1 font-bold text-blue-700">' + esc(e.signature) + '</td>';
                html += '<td class="border px-2 py-1">' + e.status + '</td>';
                html += '<td class="border px-2 py-1">' + e.zugangsdatum + '</td>';
                html += '</tr>';
            });

            html += '</tbody></table></div>';

            container.innerHTML = html;
        })
        .catch(err => { container.innerHTML = '<p class="text-red-600">Fehler: ' + err.message + '</p>'; });
}

function esc(str) {
    if (!str) return '';
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
}

// Import
function startImport() {
    if (!confirm('Import jetzt starten? Bestehende Daten mit gleicher ISBN/Inventarnummer werden uebersprungen.')) return;

    const btn = document.getElementById('importBtn');
    btn.disabled = true;
    btn.classList.add('opacity-50');
    document.getElementById('importProgress').classList.remove('hidden');
    document.getElementById('importButtons').classList.add('hidden');

    const mode = document.getElementById('importMode').value;
    const fd = new FormData();
    fd.append('action', 'import');
    if (mode !== '0') fd.append('limit', mode);

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                window.location.href = window.location.pathname + '#result';
                window.location.reload();
            } else {
                alert('Import fehlgeschlagen: ' + data.error);
                document.getElementById('importProgress').classList.add('hidden');
                document.getElementById('importButtons').classList.remove('hidden');
                btn.disabled = false;
                btn.classList.remove('opacity-50');
            }
        })
        .catch(err => {
            alert('Netzwerkfehler: ' + err.message);
            document.getElementById('importProgress').classList.add('hidden');
            document.getElementById('importButtons').classList.remove('hidden');
            btn.disabled = false;
            btn.classList.remove('opacity-50');
        });
}

// Reset
function resetData() {
    if (!confirm('ACHTUNG: Alle importierten Buecher und Exemplare werden unwiderruflich geloescht! Fortfahren?')) return;
    if (!confirm('Wirklich ALLE Daten loeschen? Dies kann nicht rueckgaengig gemacht werden!')) return;

    const fd = new FormData();
    fd.append('action', 'reset');

    fetch('', {method: 'POST', body: fd})
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                window.location.href = window.location.pathname + '#analyze';
                window.location.reload();
            } else {
                alert('Fehler: ' + data.error);
            }
        });
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
