<?php
/**
 * Barcode-Etikettendruck — books-label-print.php
 *
 * Druckt Signatur-Etiketten für Buchexemplare.
 * Unterstützte Formate: AVERY 3651, AVERY 3657, Brother QL-810W
 *
 * URL-Parameter (mind. einer erforderlich):
 *   ?book_id=X        → alle Exemplare von Buch X
 *   ?copy_id=X        → einzelnes Exemplar
 *   ?copy_ids=1,2,3   → kommagetrennte Exemplar-IDs
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';
require_once __DIR__ . '/../vendor/autoload.php';

use Picqer\Barcode\BarcodeGeneratorSVG;

require_admin();

$lang       = load_language();
$page_title = $lang['admin_print_labels'] ?? 'Etiketten drucken';

// ─── Parameter: welche Exemplare? ────────────────────────────────────────────
$copies = [];

if (!empty($_GET['copy_id']) && is_numeric($_GET['copy_id'])) {
    $copies = db_get_results(
        "SELECT bc.id, bc.book_number, bc.signature, bc.copy_index,
                b.id AS book_id, b.title, b.author
         FROM book_copies bc
         JOIN books b ON b.id = bc.book_id
         WHERE bc.id = ?",
        [intval($_GET['copy_id'])]
    );
} elseif (!empty($_GET['copy_ids'])) {
    $rawIds = array_values(array_filter(array_map('intval', explode(',', $_GET['copy_ids']))));
    if ($rawIds) {
        $ph     = implode(',', array_fill(0, count($rawIds), '?'));
        $copies = db_get_results(
            "SELECT bc.id, bc.book_number, bc.signature, bc.copy_index,
                    b.id AS book_id, b.title, b.author
             FROM book_copies bc
             JOIN books b ON b.id = bc.book_id
             WHERE bc.id IN ($ph)
             ORDER BY b.title, bc.copy_index",
            $rawIds
        );
    }
} elseif (!empty($_GET['book_id']) && is_numeric($_GET['book_id'])) {
    $copies = db_get_results(
        "SELECT bc.id, bc.book_number, bc.signature, bc.copy_index,
                b.id AS book_id, b.title, b.author
         FROM book_copies bc
         JOIN books b ON b.id = bc.book_id
         WHERE bc.book_id = ?
         ORDER BY bc.copy_index",
        [intval($_GET['book_id'])]
    );
}

// ─── Buchsuche (wenn keine Exemplare per URL-Parameter angegeben) ─────────────
$searchQuery   = '';
$searchResults = [];

if (empty($copies)) {
    // Direkt-Suche per Buchungsnummer
    if (!empty($_GET['book_number'])) {
        $bn = trim($_GET['book_number']);
        $found = db_get_results(
            "SELECT bc.id, bc.book_number, bc.signature, bc.copy_index,
                    b.id AS book_id, b.title, b.author
             FROM book_copies bc
             JOIN books b ON b.id = bc.book_id
             WHERE bc.book_number = ?",
            [$bn]
        );
        if ($found) {
            $copies = $found;
        }
    }

    // Buchsuche per Titel / Autor / ISBN
    if (empty($copies) && !empty($_GET['q'])) {
        $searchQuery = trim($_GET['q']);
        if (mb_strlen($searchQuery) >= 2) {
            $like = '%' . $searchQuery . '%';
            $searchResults = db_get_results(
                "SELECT b.id, b.title, b.author, b.isbn,
                        COUNT(bc.id) AS copies_count
                 FROM books b
                 LEFT JOIN book_copies bc ON bc.book_id = b.id AND bc.status != 'discarded'
                 WHERE b.title LIKE ? OR b.author LIKE ? OR b.isbn LIKE ?
                 GROUP BY b.id
                 ORDER BY b.title
                 LIMIT 25",
                [$like, $like, $like]
            );
        }
    }
}

// ─── Barcodes generieren ─────────────────────────────────────────────────────
$generator  = new BarcodeGeneratorSVG();
$labelsData = [];

foreach ($copies as $copy) {
    $num = $copy['book_number'] ?? '';
    $svg = '';

    if ($num) {
        $svg = $generator->getBarcode($num, BarcodeGeneratorSVG::TYPE_CODE_128, 1, 30);

        // viewBox ergänzen + feste Dimensionen entfernen damit das SVG skalierbar wird
        $svg = preg_replace_callback(
            '/<svg([^>]*)>/',
            function ($m) {
                $attrs = $m[1];
                if (preg_match('/width="(\d+)"/', $attrs, $mw)
                    && preg_match('/height="(\d+)"/', $attrs, $mh)) {
                    $vb    = 'viewBox="0 0 ' . $mw[1] . ' ' . $mh[1] . '"';
                    $attrs = preg_replace('/\s*width="\d+"/', '', $attrs);
                    $attrs = preg_replace('/\s*height="\d+"/', '', $attrs);
                    $attrs .= ' ' . $vb . ' preserveAspectRatio="none"';
                }
                return '<svg' . $attrs . '>';
            },
            $svg,
            1
        );
    }

    $titleShort = mb_strlen($copy['title']) > 30
        ? mb_substr($copy['title'], 0, 28) . '…'
        : $copy['title'];

    $labelsData[] = [
        'id'          => (int)$copy['id'],
        'book_number' => $num,
        'signature'   => $copy['signature'] ?? $num,
        'title'       => $titleShort,
        'barcode_svg' => $svg,
    ];
}

// Buch-Titel für Breadcrumb (nur wenn einzelnes Buch)
$breadcrumbBookTitle = '';
if (!empty($_GET['book_id']) && !empty($copies)) {
    $breadcrumbBookTitle = $copies[0]['title'] ?? '';
}

include __DIR__ . '/includes/header.php';
?>

<style>
/* ── Print Sheet (screen: hidden) ─────────────────────────────────────────── */
#print-sheet { display: none; }

/* ── Print mode ───────────────────────────────────────────────────────────── */
@media print {
    /* Alles ausblenden außer dem Print Sheet */
    body > * { display: none !important; }
    #print-sheet { display: block !important; }

    /* Label Basis */
    .print-label,
    .print-label-empty {
        display: inline-flex;
        flex-direction: column;
        align-items: center;
        justify-content: space-between;
        box-sizing: border-box;
        overflow: hidden;
        vertical-align: top;
        padding: 1.5mm 1mm 1.5mm 1mm;
        font-family: Arial, Helvetica, sans-serif;
    }
    .print-label-empty { visibility: hidden; }

    /* Barcode */
    .print-label .lbl-bcode { width: calc(100% - 8mm); margin: 0 auto; overflow: hidden; }
    .print-label .lbl-bcode svg { width: 100%; height: 100%; display: block; }

    /* Signatur */
    .print-label .lbl-sig {
        font-family: 'Courier New', monospace;
        font-weight: bold;
        text-align: center;
        line-height: 1.2;
        width: 100%;
    }

    /* Titel */
    .print-label .lbl-title {
        text-align: center;
        color: #333;
        line-height: 1.2;
        width: 100%;
        word-break: break-word;
    }

    /* ── AVERY 3651: 52.5 × 29.7 mm, 4 Spalten, kein Rand ── */
    body.fmt-avery3651 .print-label,
    body.fmt-avery3651 .print-label-empty { width: 52.5mm; height: 29.7mm; }
    body.fmt-avery3651 .print-label .lbl-bcode  { height: 14mm; }
    body.fmt-avery3651 .print-label .lbl-sig    { font-size: 14pt; }
    body.fmt-avery3651 .print-label .lbl-title  { font-size: 8pt; }
    body.fmt-avery3651 #print-sheet { width: 210mm; padding: 0; margin: 0; font-size: 0; }

    /* ── AVERY 3657: 48.5 × 25.4 mm, 4 Spalten, 8 mm Seitenrand ── */
    body.fmt-avery3657 .print-label,
    body.fmt-avery3657 .print-label-empty { width: 48.5mm; height: 25.4mm; }
    body.fmt-avery3657 .print-label .lbl-bcode  { height: 11mm; }
    body.fmt-avery3657 .print-label .lbl-sig    { font-size: 12pt; }
    body.fmt-avery3657 .print-label .lbl-title  { font-size: 7pt; }
    body.fmt-avery3657 #print-sheet { width: 210mm; padding: 0 8mm; margin: 0; font-size: 0; box-sizing: border-box; }

    /* ── Brother QL-810W: 62 × 29 mm, 1 Etikett pro Seite ── */
    body.fmt-brother_ql .print-label {
        width: 62mm;
        height: 29mm;
        display: flex;
        page-break-after: always;
    }
    body.fmt-brother_ql .print-label .lbl-bcode  { height: 13mm; }
    body.fmt-brother_ql .print-label .lbl-sig    { font-size: 14pt; }
    body.fmt-brother_ql .print-label .lbl-title  { font-size: 8pt; }
    body.fmt-brother_ql #print-sheet { width: 62mm; padding: 0; margin: 0; }
}
</style>

<div class="max-w-7xl mx-auto px-4 py-6">

    <!-- Breadcrumb -->
    <div class="mb-4 text-sm text-gray-500">
        <a href="books-edit.php" class="hover:text-blue-600"><?php echo esc_html($lang['books'] ?? 'Bücher'); ?></a>
        <?php if ($breadcrumbBookTitle): ?>
            <span class="mx-2">›</span>
            <a href="books-edit-detail.php?id=<?php echo esc_attr($_GET['book_id'] ?? ''); ?>"
               class="hover:text-blue-600"><?php echo esc_html(mb_substr($breadcrumbBookTitle, 0, 40)); ?></a>
        <?php endif; ?>
        <span class="mx-2">›</span>
        <span class="text-gray-800"><?php echo esc_html($lang['admin_print_labels'] ?? 'Etiketten drucken'); ?></span>
    </div>

    <?php if (empty($labelsData)): ?>
    <div class="max-w-xl">

        <!-- Buchsuche -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-4">
            <h2 class="text-base font-semibold text-gray-800 mb-3">
                📚 <?php echo esc_html($lang['label_search_heading'] ?? 'Buch suchen'); ?>
            </h2>
            <form method="get" action="" class="flex gap-2">
                <input type="text" name="q"
                       value="<?php echo esc_attr($searchQuery); ?>"
                       placeholder="<?php echo esc_attr($lang['label_search_placeholder'] ?? 'Titel, Autor oder ISBN…'); ?>"
                       class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500"
                       autofocus>
                <button type="submit"
                        class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm font-medium whitespace-nowrap">
                    🔍 <?php echo esc_html($lang['label_search_btn'] ?? 'Suchen'); ?>
                </button>
            </form>
        </div>

        <!-- Buchungsnummer direkt -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-4">
            <h2 class="text-base font-semibold text-gray-800 mb-3">
                🔢 <?php echo esc_html($lang['label_book_number_heading'] ?? 'oder Buchungsnummer direkt eingeben'); ?>
            </h2>
            <form method="get" action="" class="flex gap-2">
                <input type="text" name="book_number"
                       placeholder="<?php echo esc_attr($lang['label_book_number_placeholder'] ?? 'z.B. 14613…'); ?>"
                       class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 font-mono">
                <button type="submit"
                        class="px-4 py-2 bg-gray-700 text-white rounded-lg hover:bg-gray-800 text-sm font-medium whitespace-nowrap">
                    🖨️ <?php echo esc_html($lang['label_book_number_btn'] ?? 'Etikett drucken'); ?>
                </button>
            </form>
        </div>

        <?php if ($searchQuery && empty($searchResults)): ?>
        <!-- Keine Ergebnisse -->
        <div class="bg-yellow-50 border border-yellow-200 rounded-xl p-4 text-sm text-yellow-800">
            <?php echo esc_html($lang['label_search_no_results'] ?? 'Keine Bücher gefunden für'); ?>
            „<?php echo esc_html($searchQuery); ?>"
        </div>

        <?php elseif ($searchResults): ?>
        <!-- Suchergebnisse -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
            <?php foreach ($searchResults as $i => $book): ?>
            <div class="flex items-center justify-between px-4 py-3 <?php echo $i > 0 ? 'border-t border-gray-100' : ''; ?> hover:bg-gray-50">
                <div class="min-w-0 mr-3">
                    <p class="text-sm font-medium text-gray-900 truncate"><?php echo esc_html($book['title']); ?></p>
                    <p class="text-xs text-gray-500 truncate"><?php echo esc_html($book['author']); ?>
                        <?php if ($book['isbn']): ?>
                            · <span class="font-mono"><?php echo esc_html($book['isbn']); ?></span>
                        <?php endif; ?>
                        · <?php echo (int)$book['copies_count']; ?> <?php echo esc_html($lang['label_search_copies_count'] ?? 'Exemplar(e)'); ?>
                    </p>
                </div>
                <a href="?book_id=<?php echo (int)$book['id']; ?>"
                   class="shrink-0 px-3 py-1.5 bg-blue-600 text-white text-xs font-medium rounded-lg hover:bg-blue-700 whitespace-nowrap">
                    🖨️ <?php echo esc_html($lang['label_print_all_btn'] ?? 'Alle Etiketten'); ?>
                </a>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

    </div>
    <?php else: ?>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

        <!-- ── Config Panel ──────────────────────────────────────────────── -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
            <h2 class="text-lg font-bold text-gray-800 mb-5">⚙️ <?php echo esc_html($lang['label_settings_heading'] ?? 'Einstellungen'); ?></h2>

            <!-- Format -->
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo esc_html($lang['label_format'] ?? 'Etikettenformat'); ?></label>
                <select id="cfg-format" onchange="updateAll()"
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    <option value="avery3651"><?php echo esc_html($lang['label_format_avery3651'] ?? 'AVERY 3651 (52,5 × 29,7 mm · 4×10 / A4)'); ?></option>
                    <option value="avery3657"><?php echo esc_html($lang['label_format_avery3657'] ?? 'AVERY 3657 (48,5 × 25,4 mm · 4×12 / A4)'); ?></option>
                    <option value="brother_ql"><?php echo esc_html($lang['label_format_brother_ql'] ?? 'Brother QL-810W (62 × 29 mm · 1/Seite)'); ?></option>
                </select>
            </div>

            <!-- Startposition (nicht für Brother QL) -->
            <div class="mb-4" id="startpos-row">
                <label class="block text-sm font-medium text-gray-700 mb-1">
                    <?php echo esc_html($lang['label_start_position'] ?? 'Startposition'); ?>
                    <span class="text-xs text-gray-400 font-normal ml-1">(<?php echo esc_html($lang['label_start_position_empty_hint'] ?? 'leere Felder am Anfang'); ?>)</span>
                </label>
                <input type="number" id="cfg-startpos" value="0" min="0" max="39"
                       oninput="updateAll()"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                <p class="text-xs text-gray-400 mt-1"><?php echo esc_html($lang['label_start_position_used_sheets'] ?? 'Für bereits teilweise verwendete Bögen'); ?></p>
            </div>

            <!-- Kopien je Exemplar -->
            <div class="mb-5">
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo esc_html($lang['label_copies_per_exemplar'] ?? 'Kopien je Exemplar'); ?></label>
                <input type="number" id="cfg-copies" value="1" min="1" max="5"
                       oninput="updateAll()"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Exemplar-Auswahl -->
            <div class="mb-5">
                <div class="flex justify-between items-center mb-2">
                    <label class="block text-sm font-medium text-gray-700"><?php echo esc_html($lang['label_copies_selection'] ?? 'Exemplare'); ?></label>
                    <div class="flex gap-3 text-xs">
                        <button onclick="selectAll(true)"  class="text-blue-600 hover:underline"><?php echo esc_html($lang['label_select_all'] ?? 'Alle'); ?></button>
                        <button onclick="selectAll(false)" class="text-gray-500 hover:underline"><?php echo esc_html($lang['label_deselect_all'] ?? 'Keine'); ?></button>
                    </div>
                </div>
                <div class="space-y-2 max-h-52 overflow-y-auto pr-1" id="copy-list"></div>
            </div>

            <!-- Drucken -->
            <button onclick="printLabels()"
                    class="w-full px-4 py-3 bg-blue-600 text-white font-semibold rounded-lg hover:bg-blue-700 transition-colors">
                🖨️ <?php echo esc_html($lang['label_print_button'] ?? 'Drucken'); ?>
            </button>
            <p class="text-xs text-gray-400 mt-2 leading-relaxed" id="print-hint">
                <?php echo esc_html($lang['label_print_hint'] ?? 'Im Druckdialog "Tatsächliche Größe" (kein Skalieren) und Ränder = 0 wählen.'); ?>
            </p>
        </div>

        <!-- ── Preview Panel ─────────────────────────────────────────────── -->
        <div class="lg:col-span-2 bg-white rounded-xl shadow-sm border border-gray-200 p-5">
            <h2 class="text-lg font-bold text-gray-800 mb-1"><?php echo esc_html($lang['label_preview'] ?? 'Vorschau'); ?></h2>
            <p class="text-xs text-gray-400 mb-3"><?php echo esc_html($lang['label_preview_hint'] ?? 'Zeigt die ersten 2 Reihen (nicht maßstabsgetreu)'); ?></p>
            <div id="preview-wrap"
                 class="overflow-auto min-h-24 bg-gray-50 rounded-lg p-3 border border-dashed border-gray-300">
            </div>
            <p class="text-xs text-gray-400 mt-2" id="preview-count"></p>
        </div>

    </div>
    <?php endif; ?>
</div>

<!-- Print Sheet – JS verschiebt es vor dem Drucken als direktes body-Kind -->
<div id="print-sheet"></div>

<script>
const LABELS_DATA = <?php echo json_encode($labelsData, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_UNESCAPED_UNICODE); ?>;
const LANG = {
    none_selected:    '<?php echo addslashes($lang['label_js_none_selected'] ?? 'Keine Exemplare ausgewählt'); ?>',
    select_min_one:   '<?php echo addslashes($lang['label_js_select_min_one'] ?? 'Bitte mindestens ein Exemplar auswählen.'); ?>',
    print_hint:       '<?php echo addslashes($lang['label_print_hint'] ?? 'Im Druckdialog "Tatsächliche Größe" (kein Skalieren) und Ränder = 0 wählen.'); ?>',
    print_brother_hint: '<?php echo addslashes($lang['label_print_brother_hint'] ?? 'Im Druckdialog Papierformat 62 mm × 29 mm ohne Seitenränder wählen.'); ?>',
};

// ── Format-Konfiguration ──────────────────────────────────────────────────────
const FORMATS = {
    avery3651:  { cols: 4, perSheet: 40, wMm: 52.5, hMm: 29.7, pageCss: '210mm 297mm', cls: 'fmt-avery3651' },
    avery3657:  { cols: 4, perSheet: 48, wMm: 48.5, hMm: 25.4, pageCss: '210mm 297mm', cls: 'fmt-avery3657' },
    brother_ql: { cols: 1, perSheet: null, wMm: 62,  hMm: 29,  pageCss: '62mm 29mm',   cls: 'fmt-brother_ql' },
};

// Skalierung für Vorschau (Pixel pro mm auf dem Bildschirm)
const PX_PER_MM = 2.5;

let selectedIds = new Set(LABELS_DATA.map(l => l.id));

// ── HTML-Escape ───────────────────────────────────────────────────────────────
function esc(s) {
    return String(s ?? '').replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;');
}

// ── sprintf-style helper for LANG templates ───────────────────────────────────
function sprintfSimple(tpl) {
    const args = Array.prototype.slice.call(arguments, 1);
    let i = 0;
    return tpl.replace(/%d/g, function() { return args[i++]; });
}

// ── Copy-Auswahlliste aufbauen ────────────────────────────────────────────────
function buildCopyList() {
    const list = document.getElementById('copy-list');
    if (!list) return;
    list.innerHTML = LABELS_DATA.map(label => `
        <label class="flex items-start gap-2 cursor-pointer select-none">
            <input type="checkbox" class="copy-chk mt-0.5 shrink-0"
                   value="${label.id}"
                   ${selectedIds.has(label.id) ? 'checked' : ''}
                   onchange="onCopyToggle(${label.id}, this.checked)">
            <div class="min-w-0">
                <span class="font-mono font-bold text-xs text-gray-800">${esc(label.signature)}</span>
                <span class="block text-xs text-gray-500 truncate">${esc(label.title)}</span>
            </div>
        </label>`).join('');
}

function onCopyToggle(id, checked) {
    if (checked) selectedIds.add(id); else selectedIds.delete(id);
    updateAll();
}

function selectAll(v) {
    LABELS_DATA.forEach(l => v ? selectedIds.add(l.id) : selectedIds.delete(l.id));
    buildCopyList();
    updateAll();
}

// ── Geordnete Label-Liste aufbauen (Start-Platzhalter + Kopien berücksichtigen) ─
function buildLabelList() {
    const startPos  = Math.max(0, parseInt(document.getElementById('cfg-startpos')?.value) || 0);
    const copiesPer = Math.max(1, parseInt(document.getElementById('cfg-copies')?.value)   || 1);
    const selected  = LABELS_DATA.filter(l => selectedIds.has(l.id));
    const list      = [];
    for (let i = 0; i < startPos; i++) list.push(null);        // Platzhalter
    selected.forEach(l => { for (let c = 0; c < copiesPer; c++) list.push(l); });
    return list;
}

// ── Vorschau rendern ──────────────────────────────────────────────────────────
function updatePreview() {
    const fmt       = FORMATS[document.getElementById('cfg-format').value];
    const labelList = buildLabelList();
    const wrap      = document.getElementById('preview-wrap');
    const countEl   = document.getElementById('preview-count');
    if (!wrap) return;

    const activeCount = labelList.filter(Boolean).length;
    if (!activeCount) {
        wrap.innerHTML = '<p class="text-sm text-gray-400 p-4 text-center">' + esc(LANG.none_selected) + '</p>';
        if (countEl) countEl.textContent = '';
        return;
    }

    const pW          = fmt.wMm * PX_PER_MM;
    const pH          = fmt.hMm * PX_PER_MM;
    const barcodeH    = Math.round(pH * 0.48);
    const sigFontPx   = Math.round(14  / 3.78 * PX_PER_MM);
    const titleFontPx = Math.round(8   / 3.78 * PX_PER_MM);
    const previewRows = 2;
    const previewMax  = fmt.cols * previewRows;
    const previewItems = labelList.slice(0, previewMax);

    const labelStyle = `width:${pW}px;height:${pH}px;padding:2px 3px;box-sizing:border-box;`
                     + `overflow:hidden;display:inline-flex;flex-direction:column;`
                     + `align-items:center;justify-content:space-between;`
                     + `font-family:Arial,sans-serif;vertical-align:top;`;

    let html = '<div style="font-size:0;line-height:0;">';
    previewItems.forEach(label => {
        if (!label) {
            html += `<div style="${labelStyle}background:#f9f9f9;border:1px dashed #ccc;"></div>`;
        } else {
            html += `<div style="${labelStyle}background:#fff;border:1px solid #ccc;">
                <div style="width:calc(100% - ${Math.round(10*PX_PER_MM)}px);margin:0 auto;height:${barcodeH}px;overflow:hidden;">${label.barcode_svg}</div>
                <div style="font-family:'Courier New',monospace;font-weight:bold;font-size:${sigFontPx}px;text-align:center;line-height:1.2;width:100%;">${esc(label.signature)}</div>
                <div style="font-size:${titleFontPx}px;text-align:center;color:#555;line-height:1.2;width:100%;word-break:break-word;">${esc(label.title)}</div>
            </div>`;
        }
    });
    html += '</div>';

    if (countEl) {
        const total   = labelList.length;
        const showing = previewItems.length;
        const more    = total - showing;
        const countShowingTpl = '<?php echo addslashes($lang['label_js_count_showing'] ?? 'Zeige %d von %d Etiketten (%d Exemplare)'); ?>';
        const countTotalTpl   = '<?php echo addslashes($lang['label_js_count_total']   ?? '%d Etikett(en) (%d Exemplar(e))'); ?>';
        countEl.textContent = more > 0
            ? sprintfSimple(countShowingTpl, showing, total, activeCount)
            : sprintfSimple(countTotalTpl, total, activeCount);
    }

    wrap.innerHTML = html;
}

// ── UI-Hints & Einschränkungen aktualisieren ──────────────────────────────────
function updateHints() {
    const fmtKey = document.getElementById('cfg-format').value;
    const fmt    = FORMATS[fmtKey];

    // Startposition nur für Bogendrucker anzeigen
    const startRow = document.getElementById('startpos-row');
    if (startRow) startRow.style.display = (fmtKey === 'brother_ql') ? 'none' : '';

    // Startpos-Maximum anpassen
    const startInput = document.getElementById('cfg-startpos');
    if (startInput && fmt.perSheet) startInput.max = fmt.perSheet - 1;

    // Druckhinweis
    const hint = document.getElementById('print-hint');
    if (hint) {
        hint.textContent = (fmtKey === 'brother_ql')
            ? LANG.print_brother_hint
            : LANG.print_hint;
    }
}

// ── Drucken ───────────────────────────────────────────────────────────────────
function printLabels() {
    const fmtKey    = document.getElementById('cfg-format').value;
    const fmt       = FORMATS[fmtKey];
    const labelList = buildLabelList();

    if (!labelList.filter(Boolean).length) {
        alert(LANG.select_min_one);
        return;
    }

    // Format-Klasse an body setzen (aktiviert format-spezifisches Print-CSS)
    document.body.classList.remove('fmt-avery3651', 'fmt-avery3657', 'fmt-brother_ql');
    document.body.classList.add(fmt.cls);

    // @page-Regel dynamisch erzeugen
    let styleTag = document.getElementById('dyn-page-style');
    if (!styleTag) {
        styleTag = document.createElement('style');
        styleTag.id = 'dyn-page-style';
        document.head.appendChild(styleTag);
    }
    styleTag.textContent = `@media print { @page { size: ${fmt.pageCss}; margin: 0; } }`;

    // Print Sheet befüllen
    const sheet = document.getElementById('print-sheet');
    let html = '';
    labelList.forEach(label => {
        if (fmtKey === 'brother_ql' && !label) return; // Platzhalter bei QL überspringen
        if (!label) {
            html += '<span class="print-label-empty"></span>';
        } else {
            html += `<div class="print-label">
                <div class="lbl-bcode">${label.barcode_svg}</div>
                <div class="lbl-sig">${esc(label.signature)}</div>
                <div class="lbl-title">${esc(label.title)}</div>
            </div>`;
        }
    });
    sheet.innerHTML = html;

    // Print Sheet als direktes body-Kind verschieben,
    // damit @media print { body > * { display:none } } korrekt greift
    document.body.appendChild(sheet);

    window.print();
}

// ── Initialisierung ───────────────────────────────────────────────────────────
function updateAll() {
    updateHints();
    updatePreview();
}

buildCopyList();
updateAll();
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
