<?php
/**
 * Books - Liste aller Bücher mit Suche, Sortierung und Pagination
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$current_user = get_logged_in_user();
$lang = load_language();
$page_title = $lang['books'] ?? 'Bücher';

// ── Parameter ─────────────────────────────────────────────────────────────────
$searchQuery = trim($_GET['search'] ?? '');
$perPage     = in_array((int)($_GET['per_page'] ?? 50), [25, 50, 100]) ? (int)($_GET['per_page'] ?? 50) : 50;
$currentPage = max(1, (int)($_GET['page'] ?? 1));
$offset      = ($currentPage - 1) * $perPage;

$allowedSorts = ['title', 'subtitle', 'author', 'author_short', 'isbn', 'book_number', 'location', 'status'];
$sortBy      = in_array($_GET['sort'] ?? '', $allowedSorts) ? $_GET['sort'] : 'title';
$sortDir     = (($_GET['dir'] ?? 'asc') === 'desc') ? 'DESC' : 'ASC';
$sortDirFlip = $sortDir === 'ASC' ? 'desc' : 'asc';

$sortMap = [
    'title'        => 'b.title',
    'subtitle'     => 'b.subtitle',
    'author'       => 'b.author',
    'author_short' => 'b.author_short',
    'isbn'         => 'b.isbn',
    'book_number'  => 'MIN(CAST(bc.book_number AS UNSIGNED))',
    'location'     => 'MIN(bc.location)',
    'status'       => 'available_count',
];
$orderSql = ($sortMap[$sortBy] ?? 'b.title') . ' ' . $sortDir;

// ── Helper: URL mit aktuellen Parametern ──────────────────────────────────────
function bookUrl(array $overrides = []): string {
    global $searchQuery, $perPage, $sortBy, $sortDir;
    $params = array_filter([
        'search'   => $searchQuery ?: null,
        'per_page' => $perPage !== 50 ? $perPage : null,
        'sort'     => $sortBy !== 'title' ? $sortBy : null,
        'dir'      => $sortDir !== 'ASC' ? strtolower($sortDir) : null,
        'page'     => null,
    ], fn($v) => $v !== null);
    foreach ($overrides as $k => $v) {
        if ($v === null) {
            unset($params[$k]);
        } else {
            $params[$k] = $v;
        }
    }
    $qs = http_build_query($params);
    return '?' . ($qs ?: '');
}

// ── Helper: Sortierspalten-Link ───────────────────────────────────────────────
function sortLink(string $col, string $label): string {
    global $sortBy, $sortDir, $sortDirFlip;
    $isActive = $sortBy === $col;
    $dir      = $isActive ? $sortDirFlip : 'asc';
    $url      = bookUrl(['sort' => $col, 'dir' => $dir, 'page' => null]);
    if ($isActive) {
        $arrow = $sortDir === 'ASC'
            ? ' <span class="text-blue-600">&#8593;</span>'
            : ' <span class="text-blue-600">&#8595;</span>';
        $cls = 'text-blue-700 font-semibold';
    } else {
        $arrow = ' <span class="text-gray-300">&#8597;</span>';
        $cls   = 'text-gray-500 hover:text-gray-700';
    }
    return '<a href="' . esc_attr($url) . '" class="' . $cls . ' whitespace-nowrap">' . $label . $arrow . '</a>';
}

// ── Query ─────────────────────────────────────────────────────────────────────
$selectSql = "
    SELECT
        b.id,
        b.title,
        b.subtitle,
        b.author,
        b.author_short,
        b.isbn,
        MIN(bc.book_number)                                                      AS book_number,
        GROUP_CONCAT(DISTINCT bc.location ORDER BY bc.location SEPARATOR ', ')  AS locations,
        COUNT(bc.id)                                                             AS copy_count,
        COALESCE(SUM(bc.status = 'verfügbar'), 0)                               AS available_count,
        COALESCE(SUM(bc.status = 'ausgeliehen'), 0)                             AS loaned_count
";

$useFulltext  = false;
$searchParams = [];
$whereClause  = '';

if (!empty($searchQuery)) {
    // Buchnummer-Direktsuche: "#13751" oder "13751"
    if (preg_match('/^#?(\d+)$/', $searchQuery, $m)) {
        $bn = $m[1];
        $whereClause  = "WHERE bc.book_number = ? OR bc.accession_number = ? OR bc.signature LIKE ?";
        $searchParams = [$bn, $bn, '%' . $bn . '%'];
    }
    // Volltext (≥ 3 Zeichen)
    elseif (strlen($searchQuery) >= 3) {
        $useFulltext = true;
        $ftQuery     = '+' . implode('* +', preg_split('/\s+/', preg_quote($searchQuery, '/'))) . '*';
        $likeParam   = '%' . $searchQuery . '%';
        $whereClause = "
            WHERE (
                MATCH(b.title, b.subtitle, b.author, b.keywords, b.description) AGAINST (? IN BOOLEAN MODE)
                OR b.isbn LIKE ?
                OR b.id IN (
                    SELECT book_id FROM book_copies
                    WHERE book_number LIKE ? OR accession_number LIKE ? OR location LIKE ? OR signature LIKE ?
                )
            )
        ";
        $searchParams = [$ftQuery, $likeParam, $likeParam, $likeParam, $likeParam, $likeParam];
    }
    // Kurzer Begriff — LIKE
    else {
        $likeParam   = '%' . $searchQuery . '%';
        $whereClause = "
            WHERE b.title LIKE ? OR b.author LIKE ? OR b.isbn LIKE ? OR b.subtitle LIKE ?
               OR b.id IN (SELECT book_id FROM book_copies WHERE book_number LIKE ? OR location LIKE ?)
        ";
        $searchParams = [$likeParam, $likeParam, $likeParam, $likeParam, $likeParam, $likeParam];
    }
}

$fromJoin   = "FROM books b LEFT JOIN book_copies bc ON b.id = bc.book_id $whereClause";
$totalBooks = (int)db_get_var("SELECT COUNT(DISTINCT b.id) $fromJoin", $searchParams);
$books      = db_get_results(
    "$selectSql $fromJoin GROUP BY b.id ORDER BY $orderSql LIMIT ? OFFSET ?",
    array_merge($searchParams, [$perPage, $offset])
);
$totalPages = max(1, ceil($totalBooks / $perPage));

include __DIR__ . '/includes/header.php';
$flash = get_flash();
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 py-6">

    <!-- Page Header -->
    <div class="mb-6 flex items-center justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900"><?php echo $lang['books'] ?? 'Bücher'; ?></h1>
            <p class="mt-1 text-sm text-gray-500">
                <?php echo number_format($totalBooks, 0, ',', '.'); ?>
                <?php echo !empty($searchQuery)
                    ? ($lang['books_found'] ?? 'Bücher gefunden')
                    : ($lang['books_total'] ?? 'Bücher insgesamt'); ?>
            </p>
        </div>
        <a href="books-add.php" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm">
            + <?php echo $lang['add_new_book'] ?? 'Neues Buch'; ?>
        </a>
    </div>

    <!-- Flash Message -->
    <?php if ($flash): ?>
        <?php
        $fc = ['success' => 'green', 'error' => 'red', 'warning' => 'yellow', 'info' => 'blue'];
        $c  = $fc[$flash['type']] ?? 'blue';
        ?>
        <div class="mb-4 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo $flash['message']; ?></p>
        </div>
    <?php endif; ?>

    <!-- Search & Controls -->
    <div class="bg-white rounded-lg shadow mb-4 p-4">
        <form method="GET" class="flex gap-3 items-end flex-wrap">
            <input type="hidden" name="sort" value="<?php echo esc_attr($sortBy); ?>">
            <input type="hidden" name="dir"  value="<?php echo esc_attr(strtolower($sortDir)); ?>">
            <div class="flex-1 min-w-64">
                <label class="block text-xs font-medium text-gray-600 mb-1">
                    <?php echo $lang['fulltext_search_label'] ?? 'Volltextsuche (Titel, Autor, ISBN, Schlagworte, Beschreibung, Standort, Buchnummer&hellip;)'; ?>
                </label>
                <input type="text"
                       name="search"
                       value="<?php echo esc_attr($searchQuery); ?>"
                       placeholder="<?php echo esc_attr($lang['search_input_hint'] ?? 'Suchbegriff eingeben…'); ?>"
                       class="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-blue-500 focus:border-blue-500"
                       autofocus>
            </div>
            <div>
                <label class="block text-xs font-medium text-gray-600 mb-1"><?php echo $lang['per_page'] ?? 'Pro Seite'; ?></label>
                <select name="per_page" class="px-3 py-2 border border-gray-300 rounded-md text-sm">
                    <?php foreach ([25, 50, 100] as $n): ?>
                        <option value="<?php echo $n; ?>" <?php echo $perPage === $n ? 'selected' : ''; ?>><?php echo $n; ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="px-5 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm">
                <?php echo $lang['search'] ?? 'Suchen'; ?>
            </button>
            <?php if (!empty($searchQuery)): ?>
                <a href="<?php echo esc_attr(bookUrl(['search' => null, 'page' => null])); ?>"
                   class="px-4 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 text-sm">
                    &times; <?php echo $lang['reset_filter'] ?? 'Zurücksetzen'; ?>
                </a>
            <?php endif; ?>
        </form>

        <?php if (!empty($searchQuery)): ?>
            <p class="mt-2 text-xs text-blue-700">
                <?php echo $lang['search_label'] ?? 'Suche'; ?>: <strong><?php echo esc_html($searchQuery); ?></strong>
                &mdash; <?php echo number_format($totalBooks, 0, ',', '.'); ?> <?php echo $lang['hits'] ?? 'Treffer'; ?>
                <?php if ($useFulltext): ?>
                    <span class="text-gray-400">(<?php echo $lang['fulltext_label'] ?? 'Volltext'; ?>)</span>
                <?php endif; ?>
            </p>
        <?php endif; ?>
    </div>

    <!-- Table -->
    <?php if (!empty($books)): ?>
    <div class="bg-white rounded-lg shadow overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200 text-sm">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('book_number', $lang['book_abbr_nr'] ?? 'BuchNr'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('location', $lang['location'] ?? 'Standort'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('author_short', $lang['author_abbr'] ?? 'Kürzel'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('author', $lang['book_author'] ?? 'Autor'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('title', $lang['book_title'] ?? 'Titel'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">
                        <?php echo sortLink('subtitle', $lang['subtitle'] ?? 'Untertitel'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">
                        <?php echo sortLink('isbn', $lang['book_isbn'] ?? 'ISBN'); ?>
                    </th>
                    <th class="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        <?php echo sortLink('status', $lang['status'] ?? 'Status'); ?>
                    </th>
                    <th class="px-3 py-2 text-center text-xs font-medium text-gray-500 uppercase"><?php echo $lang['actions'] ?? 'Aktionen'; ?></th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-100">
                <?php foreach ($books as $book):
                    $available = (int)$book['available_count'];
                    $total     = (int)$book['copy_count'];
                    if ($total === 0) {
                        $statusClass = 'bg-gray-100 text-gray-500';
                        $statusText  = $lang['no_copies'] ?? 'kein Exemplar';
                    } elseif ($available === $total) {
                        $statusClass = 'bg-green-100 text-green-800';
                        $statusText  = $available . '/' . $total . ' ' . ($lang['available_of'] ?? 'verfügbar');
                    } elseif ($available === 0) {
                        $statusClass = 'bg-red-100 text-red-800';
                        $statusText  = $lang['all_on_loan'] ?? 'alle ausgeliehen';
                    } else {
                        $statusClass = 'bg-yellow-100 text-yellow-800';
                        $statusText  = $available . '/' . $total . ' ' . ($lang['available_of'] ?? 'verfügbar');
                    }
                ?>
                <tr class="hover:bg-gray-50">
                    <td class="px-3 py-2 font-mono text-gray-700 whitespace-nowrap">
                        <?php echo esc_html($book['book_number'] ?? '-'); ?>
                    </td>
                    <td class="px-3 py-2 text-gray-600 whitespace-nowrap">
                        <?php echo esc_html($book['locations'] ?? '-'); ?>
                    </td>
                    <td class="px-3 py-2 font-mono text-gray-700 whitespace-nowrap">
                        <?php echo esc_html($book['author_short'] ?? '-'); ?>
                    </td>
                    <td class="px-3 py-2 text-gray-700 whitespace-nowrap max-w-36 truncate" title="<?php echo esc_attr($book['author'] ?? ''); ?>">
                        <?php echo esc_html($book['author'] ?? '-'); ?>
                    </td>
                    <td class="px-3 py-2 font-medium text-gray-900 max-w-64 truncate" title="<?php echo esc_attr($book['title']); ?>">
                        <a href="books-edit-detail.php?id=<?php echo $book['id']; ?>" class="hover:text-blue-600 hover:underline">
                            <?php echo esc_html($book['title']); ?>
                        </a>
                    </td>
                    <td class="px-3 py-2 text-gray-500 max-w-48 truncate hidden lg:table-cell" title="<?php echo esc_attr($book['subtitle'] ?? ''); ?>">
                        <?php echo esc_html($book['subtitle'] ?? ''); ?>
                    </td>
                    <td class="px-3 py-2 font-mono text-gray-600 whitespace-nowrap hidden md:table-cell">
                        <?php echo esc_html($book['isbn'] ?? '-'); ?>
                    </td>
                    <td class="px-3 py-2 whitespace-nowrap">
                        <span class="px-2 py-0.5 rounded-full text-xs font-medium <?php echo $statusClass; ?>">
                            <?php echo esc_html($statusText); ?>
                        </span>
                    </td>
                    <td class="px-3 py-2 text-center whitespace-nowrap">
                        <a href="books-edit-detail.php?id=<?php echo $book['id']; ?>"
                           class="inline-flex items-center justify-center w-7 h-7 bg-blue-100 text-blue-700 rounded hover:bg-blue-200"
                           title="<?php echo esc_attr($lang['edit'] ?? 'Bearbeiten'); ?>">
                            <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                            </svg>
                        </a>
                        <a href="books-delete.php?id=<?php echo $book['id']; ?>"
                           class="inline-flex items-center justify-center w-7 h-7 bg-red-100 text-red-700 rounded hover:bg-red-200 ml-1"
                           title="<?php echo esc_attr($lang['delete'] ?? 'Löschen'); ?>"
                           onclick="return confirm('<?php echo esc_attr($lang['confirm_delete_book'] ?? 'Buch wirklich löschen?'); ?> &quot;<?php echo esc_attr($book['title']); ?>&quot;\n<?php echo esc_attr($lang['all_copies_deleted'] ?? 'Alle Exemplare werden ebenfalls gelöscht.'); ?>')">
                            <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                            </svg>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="bg-gray-50 px-4 py-3 border-t flex flex-wrap items-center justify-between gap-3">
            <p class="text-sm text-gray-600">
                <?php echo $lang['showing'] ?? 'Zeige'; ?>
                <?php echo number_format($offset + 1, 0, ',', '.'); ?>–<?php echo number_format(min($offset + $perPage, $totalBooks), 0, ',', '.'); ?>
                <?php echo $lang['of'] ?? 'von'; ?>
                <?php echo number_format($totalBooks, 0, ',', '.'); ?>
            </p>
            <div class="flex gap-1 flex-wrap">
                <?php if ($currentPage > 1): ?>
                    <a href="<?php echo esc_attr(bookUrl(['page' => $currentPage - 1])); ?>"
                       class="px-3 py-1.5 bg-white border rounded hover:bg-gray-50 text-sm">&larr;</a>
                <?php endif; ?>

                <?php
                $startPage = max(1, $currentPage - 2);
                $endPage   = min($totalPages, $currentPage + 2);
                if ($startPage > 1): ?>
                    <a href="<?php echo esc_attr(bookUrl(['page' => 1])); ?>"
                       class="px-3 py-1.5 bg-white border rounded hover:bg-gray-50 text-sm">1</a>
                    <?php if ($startPage > 2): ?>
                        <span class="px-2 py-1.5 text-gray-500 text-sm">&hellip;</span>
                    <?php endif; ?>
                <?php endif; ?>

                <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <a href="<?php echo esc_attr(bookUrl(['page' => $i])); ?>"
                       class="px-3 py-1.5 border rounded text-sm <?php echo $i === $currentPage ? 'bg-blue-600 text-white border-blue-600' : 'bg-white hover:bg-gray-50'; ?>">
                        <?php echo $i; ?>
                    </a>
                <?php endfor; ?>

                <?php if ($endPage < $totalPages): ?>
                    <?php if ($endPage < $totalPages - 1): ?>
                        <span class="px-2 py-1.5 text-gray-500 text-sm">&hellip;</span>
                    <?php endif; ?>
                    <a href="<?php echo esc_attr(bookUrl(['page' => $totalPages])); ?>"
                       class="px-3 py-1.5 bg-white border rounded hover:bg-gray-50 text-sm"><?php echo $totalPages; ?></a>
                <?php endif; ?>

                <?php if ($currentPage < $totalPages): ?>
                    <a href="<?php echo esc_attr(bookUrl(['page' => $currentPage + 1])); ?>"
                       class="px-3 py-1.5 bg-white border rounded hover:bg-gray-50 text-sm">&rarr;</a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php else: ?>
    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded">
        <p class="text-sm text-yellow-700">
            <?php echo !empty($searchQuery)
                ? (($lang['no_books_found'] ?? 'Keine Bücher gefunden für') . ' "' . esc_html($searchQuery) . '".')
                : ($lang['no_books_in_db'] ?? 'Keine Bücher in der Datenbank.'); ?>
        </p>
    </div>
    <?php endif; ?>

</div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
