<?php
/**
 * Books Edit Detail - Edit Single Book
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

// ─── AJAX: Cover-URL via ISBN.de ermitteln ─────────────────────────────────────
if (isset($_GET['action']) && $_GET['action'] === 'cover_lookup') {
    header('Content-Type: application/json; charset=utf-8');

    $isbn = preg_replace('/[^0-9X]/i', '', strtoupper(trim($_GET['isbn'] ?? '')));
    if (strlen($isbn) !== 10 && strlen($isbn) !== 13) {
        echo json_encode(['error' => 'Ungueltige ISBN']);
        exit;
    }

    function _fetchUrl(string $url, int $timeout = 8): string {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => $timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT      => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            CURLOPT_FOLLOWLOCATION => true,
        ]);
        $data = curl_exec($ch);
        curl_close($ch);
        return $data ?: '';
    }

    $html = _fetchUrl('https://www.isbn.de/buch/' . urlencode($isbn));
    $image = '';

    if ($html) {
        $dom = new DOMDocument();
        @$dom->loadHTML('<?xml encoding="UTF-8">' . $html);
        $xpath = new DOMXPath($dom);

        $jsonScripts = $xpath->query('//script[@type="application/ld+json"]');
        if ($jsonScripts->length > 0) {
            $data = json_decode($jsonScripts->item(0)->nodeValue, true);
            if ($data && ($data['@type'] ?? '') === 'Book' && !empty($data['image'])) {
                $image = $data['image'];
            }
        }
    }

    echo json_encode(['image' => $image]);
    exit;
}

$page_title   = 'Buch bearbeiten';
$current_user = get_logged_in_user();

// Get book ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    set_flash('Ungültige Buch-ID', 'error');
    header('Location: books-edit.php');
    exit;
}

$bookId = intval($_GET['id']);

// ─── Helper: berechne Signatur für ein Exemplar ────────────────────────────────
function calc_signature(?string $sig1, ?string $sig2, string $bookNumber): string {
    $parts = array_filter([$sig1, $sig2, $bookNumber], fn($v) => $v !== null && $v !== '');
    return implode(' ', $parts);
}

// ─── POST: Buchungsnummer eines Exemplars ändern ───────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_copy_number') {
    $copyId       = intval($_POST['copy_id']        ?? 0);
    $newNumber    = trim($_POST['new_book_number']  ?? '');

    try {
        if (!$copyId || $newNumber === '') throw new Exception('Ungültige Eingabe');

        // Gehört Exemplar zu diesem Buch?
        $copy = db_get_row("SELECT id, book_number FROM book_copies WHERE id = ? AND book_id = ?",
                            [$copyId, $bookId]);
        if (!$copy) throw new Exception('Exemplar nicht gefunden');

        // Bereits vergeben (andere Zeile)?
        if ($newNumber !== $copy['book_number']) {
            $taken = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ? AND id != ?",
                                       [$newNumber, $copyId]);
            if ($taken) throw new Exception("Buchungsnummer '{$newNumber}' ist bereits vergeben");
        }

        // Aktuelle Signatur-Teile ermitteln
        $book = db_get_row("SELECT author_short, subcategory_id FROM books WHERE id = ?", [$bookId]);
        $sig1 = '';
        if ($book['subcategory_id']) {
            $sub  = db_get_row("SELECT code FROM book_subcategories WHERE id = ?", [$book['subcategory_id']]);
            $sig1 = $sub['code'] ?? '';
        }
        $sig2 = $book['author_short'] ?? '';
        $sig  = calc_signature($sig1, $sig2, $newNumber);

        db_update('book_copies', [
            'book_number'      => $newNumber,
            'accession_number' => $newNumber,
            'signature'        => $sig ?: null,
        ], 'id = ?', [$copyId]);

        log_activity('copy_number_updated', 'book_copies', $copyId, json_encode([
            'old' => $copy['book_number'], 'new' => $newNumber, 'signature' => $sig
        ]));

        set_flash("Buchungsnummer auf '{$newNumber}' geaendert (Signatur: {$sig})", 'success');

    } catch (Exception $e) {
        set_flash($e->getMessage(), 'error');
    }

    header('Location: books-edit-detail.php?id=' . $bookId);
    exit;
}

// ─── POST: Buch aktualisieren ──────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_book') {
    try {
        $title = trim($_POST['title'] ?? '');
        if (!$title) throw new Exception('Titel ist ein Pflichtfeld');

        $isbn = preg_replace('/[^0-9X]/i', '', strtoupper(trim($_POST['isbn'] ?? '')));
        if ($isbn && strlen($isbn) !== 10 && strlen($isbn) !== 13) {
            throw new Exception('ISBN ungültig (muss 10 oder 13 Stellen haben)');
        }

        $year  = !empty($_POST['year'])  ? intval($_POST['year'])  : null;
        $pages = !empty($_POST['pages']) ? intval($_POST['pages']) : null;
        $price = !empty($_POST['price']) ? floatval(str_replace(',', '.', $_POST['price'])) : null;

        if ($year  && ($year  < 1000 || $year  > date('Y') + 10)) $year  = null;
        if ($pages && ($pages < 1    || $pages > 10000))           $pages = null;

        $subcatId    = !empty($_POST['subcategory_id']) ? intval($_POST['subcategory_id']) : null;
        $authorShort = trim($_POST['author_short'] ?? '') ?: null;

        db_update('books', [
            'isbn'                 => $isbn ?: null,
            'title'                => $title,
            'subtitle'             => trim($_POST['subtitle']             ?? '') ?: null,
            'title_sort'           => trim($_POST['title_sort']           ?? '') ?: null,
            'author'               => trim($_POST['author']               ?? '') ?: null,
            'author_short'         => $authorShort,
            'author_additional'    => trim($_POST['author_additional']    ?? '') ?: null,
            'editor'               => trim($_POST['editor']               ?? '') ?: null,
            'translator'           => trim($_POST['translator']           ?? '') ?: null,
            'illustrator'          => trim($_POST['illustrator']          ?? '') ?: null,
            'photographer'         => trim($_POST['photographer']         ?? '') ?: null,
            'graphic_artist'       => trim($_POST['graphic_artist']       ?? '') ?: null,
            'other_persons'        => trim($_POST['other_persons']        ?? '') ?: null,
            'publisher'            => trim($_POST['publisher']            ?? '') ?: null,
            'place_of_publication' => trim($_POST['place_of_publication'] ?? '') ?: null,
            'year'                 => $year,
            'edition'              => trim($_POST['edition']              ?? '') ?: null,
            'series'               => trim($_POST['series']               ?? '') ?: null,
            'series_volume'        => trim($_POST['series_volume']        ?? '') ?: null,
            'pages'                => $pages,
            'language'             => trim($_POST['language']             ?? 'Deutsch') ?: 'Deutsch',
            'description'          => trim($_POST['description']          ?? '') ?: null,
            'keywords'             => trim($_POST['keywords']             ?? '') ?: null,
            'subcategory_id'       => $subcatId,
            'price'                => $price,
            'image_url'            => trim($_POST['image_url']            ?? '') ?: null,
            'updated_at'           => date('Y-m-d H:i:s'),
        ], 'id = ?', [$bookId]);

        // Signaturen aller Exemplare neu berechnen
        $sig1 = '';
        if ($subcatId) {
            $sub  = db_get_row("SELECT code FROM book_subcategories WHERE id = ?", [$subcatId]);
            $sig1 = $sub['code'] ?? '';
        }
        $allCopies = db_get_results("SELECT id, book_number FROM book_copies WHERE book_id = ?", [$bookId]);
        foreach ($allCopies as $c) {
            $sig = calc_signature($sig1, $authorShort, $c['book_number']);
            db_update('book_copies', [
                'signature' => $sig ?: null,
                'location'  => $sig1 ?: null,
            ], 'id = ?', [$c['id']]);
        }

        log_activity('book_updated', 'books', $bookId, json_encode(['title' => $title]));

        set_flash('Buch erfolgreich aktualisiert', 'success');
        header('Location: books-edit-detail.php?id=' . $bookId);
        exit;

    } catch (Exception $e) {
        set_flash('Fehler beim Aktualisieren: ' . $e->getMessage(), 'error');
    }
}

// ─── DATA ──────────────────────────────────────────────────────────────────────
$book = db_get_row("SELECT * FROM books WHERE id = ?", [$bookId]);

if (!$book) {
    set_flash('Buch nicht gefunden', 'error');
    header('Location: books-edit.php');
    exit;
}

// Creator name
$book['created_by_first'] = null;
$book['created_by_last']  = null;
if ($book['created_by']) {
    $creator = db_get_row("SELECT first_name, last_name FROM users WHERE id = ?", [$book['created_by']]);
    $book['created_by_first'] = $creator['first_name'] ?? null;
    $book['created_by_last']  = $creator['last_name']  ?? null;
}

// Categories & subcategories
$categories = db_get_results("
    SELECT id, code, name FROM book_categories WHERE active = 1 ORDER BY sort_order, name
");
$subcategories = db_get_results("
    SELECT id, category_id, code, name FROM book_subcategories WHERE active = 1 ORDER BY sort_order, name
");

// Copies (vor Pre-Selection laden, damit Signatur als Fallback dient)
$copies = db_get_results("
    SELECT bc.*,
           l.id as loan_id,
           u.first_name as loaned_to_first, u.last_name as loaned_to_last,
           l.due_date
    FROM book_copies bc
    LEFT JOIN loans l ON bc.id = l.copy_id AND l.return_date IS NULL
    LEFT JOIN users u ON l.user_id = u.id
    WHERE bc.book_id = ?
    ORDER BY CAST(bc.book_number AS UNSIGNED)
", [$bookId]);

// Pre-Selection Kategorie/Untergruppe
// 1) aus subcategory_id des Buches
// 2) Fallback: erstes Wort der Exemplar-Signatur (z.B. "De" aus "De Jür 2705")
$preselectedCatId = '';
if ($book['subcategory_id']) {
    foreach ($subcategories as $sub) {
        if ((int)$sub['id'] === (int)$book['subcategory_id']) {
            $preselectedCatId = $sub['category_id'];
            break;
        }
    }
}
if (!$preselectedCatId) {
    foreach ($copies as $copy) {
        if (!empty($copy['signature'])) {
            $sig1code = explode(' ', trim($copy['signature']))[0];
            foreach ($subcategories as $sub) {
                if ($sub['code'] === $sig1code) {
                    $preselectedCatId       = $sub['category_id'];
                    $book['subcategory_id'] = $sub['id'];
                    break 2;
                }
            }
            break;
        }
    }
}

include __DIR__ . '/includes/header.php';
$flash = get_flash();

function sel(string $value, string $current): string {
    return $value === $current ? 'selected' : '';
}
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Back -->
    <div class="mb-6">
        <a href="books-edit.php" class="text-sm text-gray-600 hover:text-gray-900">← <?php echo $lang['back_to_overview'] ?? 'Zurück zur Übersicht'; ?></a>
    </div>

    <!-- Flash -->
    <?php if ($flash): ?>
        <?php $fc = ['success'=>'green','error'=>'red','warning'=>'yellow','info'=>'blue']; $c = $fc[$flash['type']] ?? 'blue'; ?>
        <div class="mb-6 bg-<?php echo $c; ?>-50 border-l-4 border-<?php echo $c; ?>-400 p-4 rounded">
            <p class="text-sm text-<?php echo $c; ?>-700"><?php echo esc_html($flash['message']); ?></p>
        </div>
    <?php endif; ?>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 items-start">

        <!-- ══════════════════════════════════════════════════════ -->
        <!-- MAIN FORM (2/3)                                        -->
        <!-- ══════════════════════════════════════════════════════ -->
        <div class="lg:col-span-2 space-y-5">
            <form method="POST" id="book-form" class="space-y-5">
                <input type="hidden" name="action"    value="update_book">
                <input type="hidden" name="image_url" id="f-image_url"
                       value="<?php echo esc_attr($book['image_url'] ?? ''); ?>">

                <!-- ── Titeldaten ── -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4"><?php echo $lang['title_data'] ?? 'Titeldaten'; ?></h3>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_isbn'] ?? 'ISBN'; ?></label>
                        <input type="text" name="isbn" id="f-isbn"
                               value="<?php echo esc_attr($book['isbn'] ?? ''); ?>"
                               placeholder="978-3-…"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg font-mono text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['book_title'] ?? 'Titel'; ?> <span class="text-red-500">*</span>
                        </label>
                        <input type="text" name="title" id="f-title" required
                               value="<?php echo esc_attr($book['title']); ?>"
                               oninput="suggestTitleSort()"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['subtitle_field'] ?? 'Untertitel'; ?></label>
                        <input type="text" name="subtitle"
                               value="<?php echo esc_attr($book['subtitle'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['title_sort_field'] ?? 'Sortierform'; ?>
                            <span class="text-xs text-gray-400 font-normal">(<?php echo $lang['title_sort_hint'] ?? 'Titel ohne Artikel'; ?>)</span>
                        </label>
                        <input type="text" name="title_sort" id="f-title_sort"
                               value="<?php echo esc_attr($book['title_sort'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>
                </div>

                        <!-- ── Beteiligte Personen ── -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4"><?php echo $lang['involved_persons'] ?? 'Beteiligte Personen'; ?></h3>

                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_author'] ?? 'Autor'; ?></label>
                            <input type="text" name="author" id="f-author"
                                   value="<?php echo esc_attr($book['author'] ?? ''); ?>"
                                   placeholder="<?php echo $lang['author_name_placeholder'] ?? 'Nachname Vorname'; ?>"
                                   oninput="suggestAuthorShort()"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                <?php echo $lang['author_abbr_field'] ?? 'Kürzel (Autor)'; ?>
                                <span class="text-xs text-gray-400 font-normal"></span>
                            </label>
                            <input type="text" name="author_short" id="f-author_short"
                                   value="<?php echo esc_attr($book['author_short'] ?? ''); ?>"
                                   placeholder="<?php echo $lang['author_abbr_placeholder'] ?? 'z.B. Böl'; ?>" maxlength="20"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm font-mono focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['additional_authors'] ?? 'Weitere Autoren'; ?></label>
                        <input type="text" name="author_additional"
                               value="<?php echo esc_attr($book['author_additional'] ?? ''); ?>"
                               placeholder="<?php echo $lang['comma_separated'] ?? 'kommagetrennt'; ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div>
                        <button type="button" onclick="toggleExtra()" id="btn-extra"
                                class="text-sm text-blue-600 hover:underline mb-3">
                            <?php
                            $hasExtra = array_filter([
                                $book['editor'], $book['translator'], $book['illustrator'],
                                $book['photographer'], $book['graphic_artist'], $book['other_persons']
                            ]);
                            echo $hasExtra
                                ? ($lang['hide_extra_fields'] ?? '− Zusatzfelder einklappen')
                                : ($lang['show_extra_fields'] ?? '+ Herausgeber, Übersetzer, Illustratoren…');
                            ?>
                        </button>
                        <div id="extra-persons" class="<?php echo $hasExtra ? '' : 'hidden '; ?>grid grid-cols-2 gap-4">
                            <?php foreach ([
                                ['editor',        $lang['editor_field']        ?? 'Herausgeber'],
                                ['translator',     $lang['translator_field']    ?? 'Übersetzer'],
                                ['illustrator',    $lang['illustrator_field']   ?? 'Illustrator'],
                                ['photographer',   $lang['photographer_field']  ?? 'Fotograf'],
                                ['graphic_artist', $lang['graphic_artist_field']?? 'Grafiker'],
                                ['other_persons',  $lang['other_persons_field'] ?? 'Sonstige'],
                            ] as [$field, $label]): ?>
                                <div>
                                    <label class="block text-xs font-medium text-gray-600 mb-1"><?php echo $label; ?></label>
                                    <input type="text" name="<?php echo $field; ?>"
                                           value="<?php echo esc_attr($book[$field] ?? ''); ?>"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <!-- ── Publikation ── -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4"><?php echo $lang['publication'] ?? 'Publikation'; ?></h3>

                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_publisher'] ?? 'Verlag'; ?></label>
                            <input type="text" name="publisher"
                                   value="<?php echo esc_attr($book['publisher'] ?? ''); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['place_of_publication'] ?? 'Erscheinungsort'; ?></label>
                            <input type="text" name="place_of_publication"
                                   value="<?php echo esc_attr($book['place_of_publication'] ?? ''); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <div class="grid grid-cols-3 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_year'] ?? 'Jahr'; ?></label>
                            <input type="number" name="year"
                                   value="<?php echo esc_attr($book['year'] ?? ''); ?>"
                                   min="1000" max="<?php echo date('Y') + 10; ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['edition_field'] ?? 'Auflage'; ?></label>
                            <input type="text" name="edition"
                                   value="<?php echo esc_attr($book['edition'] ?? ''); ?>"
                                   placeholder="<?php echo $lang['edition_placeholder'] ?? 'z.B. 3. Aufl.'; ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['page_count'] ?? 'Seiten'; ?></label>
                            <input type="number" name="pages"
                                   value="<?php echo esc_attr($book['pages'] ?? ''); ?>"
                                   min="1" max="10000"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <div class="grid grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_language'] ?? 'Sprache'; ?></label>
                            <select name="language"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                                <option value="Deutsch"     <?php echo sel('Deutsch',     $book['language'] ?? ''); ?>>Deutsch</option>
                                <option value="Englisch"    <?php echo sel('Englisch',    $book['language'] ?? ''); ?>>Englisch</option>
                                <option value="Französisch" <?php echo sel('Französisch', $book['language'] ?? ''); ?>>Französisch</option>
                                <option value="Spanisch"    <?php echo sel('Spanisch',    $book['language'] ?? ''); ?>>Spanisch</option>
                                <option value="Italienisch" <?php echo sel('Italienisch', $book['language'] ?? ''); ?>>Italienisch</option>
                                <option value="Andere"      <?php echo sel('Andere',      $book['language'] ?? ''); ?>>Andere</option>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['price_field'] ?? 'Preis (€)'; ?></label>
                            <input type="text" name="price"
                                   value="<?php echo esc_attr($book['price'] ?? ''); ?>"
                                   placeholder="<?php echo $lang['price_placeholder'] ?? 'z.B. 24.90'; ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['series_volume_field'] ?? 'Reihe / Band'; ?></label>
                            <div class="flex gap-1">
                                <input type="text" name="series"
                                       value="<?php echo esc_attr($book['series'] ?? ''); ?>"
                                       placeholder="<?php echo $lang['series'] ?? 'Reihe'; ?>"
                                       class="flex-1 min-w-0 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                                <input type="text" name="series_volume"
                                       value="<?php echo esc_attr($book['series_volume'] ?? ''); ?>"
                                       placeholder="<?php echo $lang['volume_abbr'] ?? 'Bd.'; ?>"
                                       class="w-16 px-2 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- ── Kategorie & Inhalt ── -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4"><?php echo $lang['category_and_content'] ?? 'Kategorie & Inhalt'; ?></h3>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_category'] ?? 'Kategorie'; ?></label>
                        <select id="f-category" onchange="filterSubcategories(this.value)"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                            <option value=""><?php echo $lang['choose_category'] ?? '— Kategorie wählen —'; ?></option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>"
                                        <?php echo (string)$cat['id'] === (string)$preselectedCatId ? 'selected' : ''; ?>>
                                    <?php echo esc_html($cat['code'] . ' — ' . $cat['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['subgroup_field'] ?? 'Untergruppe'; ?>
                            <span class="text-xs text-gray-400 font-normal"></span>
                        </label>
                        <select name="subcategory_id" id="f-subcategory_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                            <option value=""><?php echo $lang['choose_subgroup'] ?? '— Untergruppe wählen —'; ?></option>
                            <?php foreach ($subcategories as $sub): ?>
                                <option value="<?php echo $sub['id']; ?>"
                                        data-cat="<?php echo $sub['category_id']; ?>"
                                        data-code="<?php echo esc_attr($sub['code']); ?>"
                                        class="subcat-option"
                                        <?php echo (int)$sub['id'] === (int)($book['subcategory_id'] ?? 0) ? 'selected' : ''; ?>>
                                    <?php echo esc_html($sub['code'] . ' — ' . $sub['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['keywords'] ?? 'Schlagworte'; ?>
                            <span class="text-xs text-gray-400 font-normal">(<?php echo $lang['comma_separated'] ?? 'kommagetrennt'; ?>)</span>
                        </label>
                        <input type="text" name="keywords"
                               value="<?php echo esc_attr($book['keywords'] ?? ''); ?>"
                               placeholder="<?php echo $lang['keywords_placeholder'] ?? 'z.B. Roman, Österreich, 20. Jahrhundert'; ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['description'] ?? 'Inhaltsbeschreibung'; ?></label>
                        <textarea name="description" rows="5"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 resize-y"><?php echo esc_html($book['description'] ?? ''); ?></textarea>
                    </div>
                </div>

            </form>
        </div>

        <!-- ══════════════════════════════════════════════════════ -->
        <!-- SIDEBAR (1/3)                                          -->
        <!-- ══════════════════════════════════════════════════════ -->
        <div class="space-y-5">

            <!-- Coverbild -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 text-center">
                <h3 class="font-semibold text-gray-800 mb-3 text-left"><?php echo $lang['cover_image'] ?? 'Coverbild'; ?></h3>
                <?php if (!empty($book['image_url'])): ?>
                    <img id="cover-preview"
                         src="<?php echo esc_attr($book['image_url']); ?>"
                         alt="<?php echo $lang['cover_image'] ?? 'Cover'; ?>"
                         class="max-h-64 mx-auto rounded-lg shadow">
                <?php else: ?>
                    <div id="cover-preview-placeholder"
                         class="h-40 bg-gray-100 rounded-lg flex items-center justify-center text-gray-400 text-sm">
                        <?php echo $lang['no_cover'] ?? 'Kein Cover'; ?>
                    </div>
                    <img id="cover-preview" src="" alt="<?php echo $lang['cover_image'] ?? 'Cover'; ?>"
                         class="hidden max-h-64 mx-auto rounded-lg shadow">
                <?php endif; ?>
            </div>

            <!-- Cover-URL -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4">
                <h3 class="font-semibold text-gray-800 mb-3"><?php echo $lang['cover_url'] ?? 'Cover-URL'; ?></h3>
                <div class="flex gap-2">
                    <input type="url" id="f-image_url-visible"
                           value="<?php echo esc_attr($book['image_url'] ?? ''); ?>"
                           placeholder="https://…"
                           oninput="syncCoverUrl(this.value)"
                           class="flex-1 min-w-0 px-3 py-2 border border-gray-300 rounded-lg text-xs font-mono focus:ring-2 focus:ring-blue-500">
                    <button type="button" onclick="searchCover()"
                            class="shrink-0 px-3 py-2 bg-gray-100 text-gray-700 text-xs font-medium rounded-lg hover:bg-gray-200 transition-colors whitespace-nowrap">
                        <?php echo $lang['search'] ?? 'Suchen'; ?>
                    </button>
                </div>
                <p id="cover-search-status" class="mt-1 text-xs text-gray-400 hidden"></p>
            </div>

            <!-- Exemplare -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="font-semibold text-gray-800"><?php echo $lang['copies'] ?? 'Exemplare'; ?> (<?php echo count($copies); ?>)</h3>
                    <div class="flex items-center gap-3">
                        <?php if (!empty($copies)): ?>
                        <a href="books-label-print.php?book_id=<?php echo $book['id']; ?>"
                           class="text-sm text-gray-500 hover:text-gray-800" title="<?php echo $lang['print_all_labels'] ?? 'Alle Etiketten drucken'; ?>">
                            🏷️ <?php echo $lang['print_all_labels'] ?? 'Alle Etiketten'; ?>
                        </a>
                        <?php endif; ?>
                        <a href="books-add-copy.php?book_id=<?php echo $book['id']; ?>"
                           class="text-sm text-blue-600 hover:text-blue-800">+ <?php echo $lang['add_new_copy'] ?? 'Neu'; ?></a>
                    </div>
                </div>

                <?php if (!empty($copies)): ?>
                    <div class="space-y-3">
                        <?php foreach ($copies as $copy): ?>
                            <?php $avail = $copy['status'] === 'verfügbar'; ?>
                            <div class="border rounded-lg p-3 <?php echo $avail ? 'bg-green-50 border-green-200' : 'bg-yellow-50 border-yellow-200'; ?>">

                                <!-- Status-Badge -->
                                <div class="flex justify-between items-start mb-2">
                                    <span class="font-mono font-bold text-sm text-gray-800">
                                        <?php echo esc_html($copy['signature'] ?? '—'); ?>
                                    </span>
                                    <span class="px-2 py-0.5 text-xs font-medium rounded <?php echo $avail ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                        <?php echo esc_html($copy['status']); ?>
                                    </span>
                                </div>

                                <?php if ($copy['location']): ?>
                                    <div class="text-xs text-gray-500 mb-2">
                                        <?php echo esc_html($copy['location']); ?>
                                    </div>
                                <?php endif; ?>

                                <!-- Buchungsnummer bearbeiten -->
                                <form method="POST" class="flex items-center gap-1 mt-1">
                                    <input type="hidden" name="action"   value="update_copy_number">
                                    <input type="hidden" name="copy_id"  value="<?php echo $copy['id']; ?>">
                                    <label class="text-xs text-gray-500 shrink-0"><?php echo $lang['copy_number_abbr'] ?? 'Nr.'; ?></label>
                                    <input type="text"
                                           name="new_book_number"
                                           value="<?php echo esc_attr($copy['book_number']); ?>"
                                           class="flex-1 min-w-0 px-2 py-1 border border-gray-300 rounded text-xs font-mono focus:ring-1 focus:ring-blue-500"
                                           oninput="checkCopyNumber(this, '<?php echo (int)$copy['id']; ?>')"
                                           data-original="<?php echo esc_attr($copy['book_number']); ?>">
                                    <span id="num-status-<?php echo $copy['id']; ?>" class="text-sm shrink-0" title="">—</span>
                                    <button type="submit"
                                            class="shrink-0 px-2 py-1 bg-gray-200 text-gray-700 text-xs rounded hover:bg-gray-300 transition-colors">
                                        ✓
                                    </button>
                                    <a href="books-label-print.php?copy_id=<?php echo $copy['id']; ?>"
                                       class="shrink-0 px-2 py-1 bg-gray-100 text-gray-600 text-xs rounded hover:bg-gray-200 transition-colors"
                                       title="<?php echo $lang['print_label'] ?? 'Etikett drucken'; ?>">
                                        🏷️
                                    </a>
                                </form>

                                <?php if ($copy['loan_id']): ?>
                                    <div class="text-xs text-yellow-800 mt-2 pt-2 border-t border-yellow-200">
                                        <?php echo esc_html($copy['loaned_to_first'] . ' ' . $copy['loaned_to_last']); ?>
                                        · <?php echo $lang['due_date_prefix'] ?? 'fällig'; ?> <?php echo date('d.m.Y', strtotime($copy['due_date'])); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <p class="text-sm text-gray-500"><?php echo $lang['no_copies_available'] ?? 'Keine Exemplare vorhanden'; ?></p>
                <?php endif; ?>
            </div>

            <!-- Aktionen -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                <h3 class="font-semibold text-gray-800 mb-4"><?php echo $lang['actions'] ?? 'Aktionen'; ?></h3>
                <div class="space-y-2">
                    <a href="books-edit.php"
                       class="block w-full px-4 py-2 bg-gray-100 text-gray-700 text-center rounded-lg hover:bg-gray-200 text-sm transition-colors">
                        <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                    </a>
                    <a href="books-delete.php?id=<?php echo $book['id']; ?>"
                       onclick="return confirm('<?php echo $lang['confirm_delete_book'] ?? 'Buch wirklich loschen?'; ?>')"
                       class="block w-full px-4 py-2 bg-red-100 text-red-700 text-center rounded-lg hover:bg-red-200 text-sm transition-colors">
                        <?php echo $lang['delete_book'] ?? 'Buch loschen'; ?>
                    </a>
                    <button type="submit" form="book-form"
                            class="block w-full px-4 py-2 bg-blue-600 text-white font-semibold text-center rounded-lg hover:bg-blue-700 text-sm transition-colors">
                        <?php echo $lang['save_changes'] ?? 'Anderungen speichern'; ?>
                    </button>
                </div>
            </div>

            <!-- Metadaten -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                <h3 class="font-semibold text-gray-800 mb-3"><?php echo $lang['metadata'] ?? 'Metadaten'; ?></h3>
                <dl class="space-y-3 text-sm">
                    <div>
                        <dt class="text-gray-500"><?php echo $lang['book_id_field'] ?? 'Buch-ID'; ?></dt>
                        <dd class="font-mono text-gray-900">#<?php echo $book['id']; ?></dd>
                    </div>
                    <div>
                        <dt class="text-gray-500"><?php echo $lang['created_field'] ?? 'Erstellt'; ?></dt>
                        <dd class="text-gray-900">
                            <?php echo date('d.m.Y H:i', strtotime($book['created_at'])); ?>
                            <?php if ($book['created_by_first']): ?>
                                <br><span class="text-xs text-gray-400"><?php echo $lang['created_by_prefix'] ?? 'von'; ?> <?php echo esc_html($book['created_by_first'] . ' ' . $book['created_by_last']); ?></span>
                            <?php endif; ?>
                        </dd>
                    </div>
                    <?php if ($book['updated_at']): ?>
                        <div>
                            <dt class="text-gray-500"><?php echo $lang['last_modified'] ?? 'Letzte Anderung'; ?></dt>
                            <dd class="text-gray-900"><?php echo date('d.m.Y H:i', strtotime($book['updated_at'])); ?></dd>
                        </div>
                    <?php endif; ?>
                </dl>
            </div>

        </div><!-- /sidebar -->
    </div>
</div>
</div>

<script>
const BASE_URL = '<?php echo BASE_URL; ?>';
const LANG = {
    choose_category:    '<?php echo $lang['choose_category'] ?? '— Kategorie wahlen —'; ?>',
    choose_subgroup:    '<?php echo $lang['choose_subgroup'] ?? '— Untergruppe wahlen —'; ?>',
    no_subgroups:       '<?php echo $lang['no_subgroups'] ?? '— keine Untergruppen —'; ?>',
    show_extra_fields:  '<?php echo $lang['show_extra_fields'] ?? '+ Herausgeber, Ubersetzer, Illustratoren...'; ?>',
    hide_extra_fields:  '<?php echo $lang['hide_extra_fields'] ?? '− Zusatzfelder einklappen'; ?>',
    cover_search_enter_isbn: '<?php echo $lang['cover_search_enter_isbn'] ?? 'Bitte zuerst eine ISBN eingeben.'; ?>',
    cover_search_running:    '<?php echo $lang['cover_search_running'] ?? 'Suche lauft...'; ?>',
    cover_found:             '<?php echo $lang['cover_found'] ?? 'Cover gefunden.'; ?>',
    cover_not_found:         '<?php echo $lang['cover_not_found'] ?? 'Kein Cover gefunden.'; ?>',
    cover_search_error:      '<?php echo $lang['cover_search_error'] ?? 'Fehler bei der Abfrage.'; ?>',
    number_available:        '<?php echo $lang['number_available'] ?? 'Verfugbar'; ?>',
    number_taken:            '<?php echo $lang['number_taken'] ?? 'Bereits vergeben'; ?>',
};

// ── Kategorie-Filter ──────────────────────────────────────────────────────────
function filterSubcategories(catId) {
    const sel  = document.getElementById('f-subcategory_id');
    const opts = sel.querySelectorAll('.subcat-option');
    let visible = 0;
    opts.forEach(opt => {
        const show = !catId || opt.dataset.cat == catId;
        opt.style.display = show ? '' : 'none';
        if (show) visible++;
    });
    sel.options[0].textContent = catId
        ? (visible > 0 ? LANG.choose_subgroup : LANG.no_subgroups)
        : LANG.choose_subgroup;
    // deselect if hidden
    const cur = sel.options[sel.selectedIndex];
    if (cur && cur.style.display === 'none') sel.value = '';
}

// ── Auto-Kürzel ───────────────────────────────────────────────────────────────
function suggestAuthorShort() {
    const author  = document.getElementById('f-author').value.trim();
    const shortEl = document.getElementById('f-author_short');
    if (shortEl.value || !author) return;
    const parts = author.split(/[\s,]+/);
    if (parts[0]) shortEl.value = parts[0].substring(0, 3);
}

// ── Auto-Sortierform ──────────────────────────────────────────────────────────
function suggestTitleSort() {
    const title  = document.getElementById('f-title').value.trim();
    const sortEl = document.getElementById('f-title_sort');
    if (sortEl.value || !title) return;
    sortEl.value = title.replace(/^(Der|Die|Das|Ein|Eine|The|A|An|Le|La|Les|L')\s+/i, '');
}

// ── Zusatzfelder ──────────────────────────────────────────────────────────────
function toggleExtra() {
    const el  = document.getElementById('extra-persons');
    const btn = document.getElementById('btn-extra');
    el.classList.toggle('hidden');
    btn.textContent = el.classList.contains('hidden')
        ? LANG.show_extra_fields
        : LANG.hide_extra_fields;
}

// ── Cover URL sync ────────────────────────────────────────────────────────────
function syncCoverUrl(url) {
    document.getElementById('f-image_url').value = url;
    updateCoverPreview(url);
}

function updateCoverPreview(url) {
    const img  = document.getElementById('cover-preview');
    const ph   = document.getElementById('cover-preview-placeholder');
    if (url) {
        img.src = url;
        img.classList.remove('hidden');
        if (ph) ph.classList.add('hidden');
    } else {
        img.classList.add('hidden');
        if (ph) ph.classList.remove('hidden');
    }
}

// ── Cover-Suche via ISBN-Lookup ───────────────────────────────────────────────
async function searchCover() {
    const isbnEl = document.getElementById('f-isbn');
    const isbn   = (isbnEl ? isbnEl.value : '').replace(/[^0-9X]/gi, '');
    const status = document.getElementById('cover-search-status');

    if (!isbn) {
        status.textContent = LANG.cover_search_enter_isbn;
        status.classList.remove('hidden', 'text-green-600', 'text-red-500');
        status.classList.add('text-gray-400');
        return;
    }

    status.textContent = LANG.cover_search_running;
    status.classList.remove('hidden', 'text-green-600', 'text-red-500');
    status.classList.add('text-gray-400');

    try {
        const res  = await fetch(`${BASE_URL}/admin/books-edit-detail.php?action=cover_lookup&isbn=${encodeURIComponent(isbn)}`);
        const data = await res.json();

        if (data.image) {
            document.getElementById('f-image_url-visible').value = data.image;
            document.getElementById('f-image_url').value          = data.image;
            updateCoverPreview(data.image);
            status.textContent = LANG.cover_found;
            status.classList.replace('text-gray-400', 'text-green-600');
        } else {
            status.textContent = LANG.cover_not_found;
            status.classList.replace('text-gray-400', 'text-red-500');
        }
    } catch {
        status.textContent = LANG.cover_search_error;
        status.classList.replace('text-gray-400', 'text-red-500');
    }
}

// ── Buchungsnummer-Check (per Exemplar) ───────────────────────────────────────
const numCheckTimers = {};
async function checkCopyNumber(input, copyId) {
    const statusEl = document.getElementById('num-status-' + copyId);
    const val      = input.value.trim();
    const original = input.dataset.original;

    clearTimeout(numCheckTimers[copyId]);

    if (!val || val === original) {
        statusEl.textContent = '—';
        statusEl.title = '';
        return;
    }

    statusEl.textContent = '⏳';
    numCheckTimers[copyId] = setTimeout(async () => {
        try {
            const res  = await fetch(`${BASE_URL}/admin/books-add.php?action=check_number&number=${encodeURIComponent(val)}`);
            const data = await res.json();
            if (data.available) {
                statusEl.textContent = '✅';
                statusEl.title = LANG.number_available;
            } else {
                statusEl.textContent = '❌';
                statusEl.title = LANG.number_taken;
            }
        } catch {
            statusEl.textContent = '?';
        }
    }, 400);
}

// Init
filterSubcategories(document.getElementById('f-category').value);
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
