<?php
/**
 * Books Delete - Delete a book and all its copies
 * Shows confirmation page before deletion
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

// Require admin access
require_admin();

$lang = load_language();
$page_title = $lang['delete_book'] ?? 'Buch löschen';
$current_user = get_logged_in_user();

// Get book ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    set_flash($lang['invalid_book_id'] ?? 'Ungültige Buch-ID', 'error');
    header('Location: books-edit.php');
    exit;
}

$bookId = intval($_GET['id']);

// Handle deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_delete'])) {
    try {
        // Check if any copies are currently loaned out
        $activeLoans = db_get_var("
            SELECT COUNT(*)
            FROM loans l
            JOIN book_copies bc ON l.copy_id = bc.id
            WHERE bc.book_id = ? AND l.return_date IS NULL
        ", [$bookId]);

        if ($activeLoans > 0) {
            set_flash(($lang['cannot_delete_book'] ?? 'Buch kann nicht gelöscht werden:') . ' ' . $activeLoans . ' ' . ($lang['copies_still_loaned'] ?? 'Exemplar(e) sind noch ausgeliehen'), 'error');
            header('Location: books-edit-detail.php?id=' . $bookId);
            exit;
        }

        // Get book title for log
        $bookTitle = db_get_var("SELECT title FROM books WHERE id = ?", [$bookId]);

        // Delete in correct order (foreign key constraints)
        // 1. Delete loan history for this book's copies
        db_query("
            DELETE l FROM loans l
            JOIN book_copies bc ON l.copy_id = bc.id
            WHERE bc.book_id = ?
        ", [$bookId]);

        // 2. Delete all book copies
        db_query("DELETE FROM book_copies WHERE book_id = ?", [$bookId]);

        // 3. Delete the book itself
        db_query("DELETE FROM books WHERE id = ?", [$bookId]);

        // Log the deletion
        log_activity('book_deleted', 'books', $bookId, json_encode([
            'title' => $bookTitle
        ]));

        set_flash(($lang['delete_book'] ?? 'Buch') . ' "' . $bookTitle . '" ' . ($lang['msg_deleted_successfully'] ?? 'wurde erfolgreich gelöscht'), 'success');
        header('Location: books-edit.php');
        exit;

    } catch (Exception $e) {
        set_flash(($lang['error'] ?? 'Fehler') . ': ' . $e->getMessage(), 'error');
        header('Location: books-edit-detail.php?id=' . $bookId);
        exit;
    }
}

// Get book details
$book = db_get_row("SELECT * FROM books WHERE id = ?", [$bookId]);

if (!$book) {
    set_flash($lang['book_not_found'] ?? 'Buch nicht gefunden', 'error');
    header('Location: books-edit.php');
    exit;
}

// Get copies with book numbers
$copies = db_get_results("
    SELECT book_number, signature FROM book_copies WHERE book_id = ? ORDER BY CAST(book_number AS UNSIGNED)
", [$bookId]);
$copiesCount = count($copies);

// Check for active loans
$activeLoans = db_get_var("
    SELECT COUNT(*)
    FROM loans l
    JOIN book_copies bc ON l.copy_id = bc.id
    WHERE bc.book_id = ? AND l.return_date IS NULL
", [$bookId]);

// Debug: Check if query returns NULL instead of 0
if ($activeLoans === null) {
    $activeLoans = 0;
}

// Get loan history count
$loanHistoryCount = db_get_var("
    SELECT COUNT(*)
    FROM loans l
    JOIN book_copies bc ON l.copy_id = bc.id
    WHERE bc.book_id = ?
", [$bookId]);

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
    <div class="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Back Button -->
        <div class="mb-6">
            <a href="books-edit.php" class="inline-flex items-center text-sm text-gray-600 hover:text-gray-900">
                ← <?php echo $lang['back_to_overview'] ?? 'Zurück zur Übersicht'; ?>
            </a>
        </div>

        <!-- Warning Card -->
        <div class="bg-white rounded-lg shadow-lg border-t-4 border-red-500">
            <div class="px-6 py-4 border-b border-gray-200 bg-red-50">
                <div class="flex items-center">
                    <svg class="w-8 h-8 text-red-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                    </svg>
                    <h2 class="text-xl font-bold text-gray-900"><?php echo $lang['confirm_delete_book'] ?? 'Buch wirklich löschen?'; ?></h2>
                </div>
            </div>

            <div class="p-6 space-y-6">

                <!-- Book Info -->
                <div class="bg-gray-50 rounded-lg p-4">
                    <h3 class="font-bold text-lg text-gray-900 mb-2">
                        <?php echo esc_html($book['title']); ?>
                    </h3>
                    <dl class="grid grid-cols-2 gap-3 text-sm">
                        <?php if ($book['author']): ?>
                            <div>
                                <dt class="text-gray-600"><?php echo $lang['book_author'] ?? 'Autor'; ?></dt>
                                <dd class="font-medium text-gray-900"><?php echo esc_html($book['author']); ?></dd>
                            </div>
                        <?php endif; ?>
                        <?php if ($book['isbn']): ?>
                            <div>
                                <dt class="text-gray-600"><?php echo $lang['book_isbn'] ?? 'ISBN'; ?></dt>
                                <dd class="font-mono text-gray-900"><?php echo esc_html($book['isbn']); ?></dd>
                            </div>
                        <?php endif; ?>
                        <?php if ($book['publisher']): ?>
                            <div>
                                <dt class="text-gray-600"><?php echo $lang['book_publisher'] ?? 'Verlag'; ?></dt>
                                <dd class="text-gray-900"><?php echo esc_html($book['publisher']); ?></dd>
                            </div>
                        <?php endif; ?>
                        <?php if ($book['year']): ?>
                            <div>
                                <dt class="text-gray-600"><?php echo $lang['year_label'] ?? 'Jahr'; ?></dt>
                                <dd class="text-gray-900"><?php echo esc_html($book['year']); ?></dd>
                            </div>
                        <?php endif; ?>
                    </dl>
                    <?php if (!empty($copies)): ?>
                        <div class="mt-3 pt-3 border-t border-gray-200">
                            <p class="text-xs text-gray-500 mb-1">
                                <?php echo $lang['book_number_ids'] ?? 'Buchungsnummern'; ?>
                                (<?php echo $copiesCount; ?>
                                <?php echo $copiesCount !== 1
                                    ? ($lang['copies_count_plural'] ?? 'Exemplare')
                                    : ($lang['copies_count_label'] ?? 'Exemplar'); ?>)
                            </p>
                            <div class="flex flex-wrap gap-1">
                                <?php foreach ($copies as $copy): ?>
                                    <span class="px-2 py-0.5 bg-gray-200 text-gray-700 rounded font-mono text-xs"
                                          title="<?php echo $lang['signature'] ?? 'Signatur'; ?>: <?php echo esc_attr($copy['signature'] ?? '—'); ?>">
                                        <?php echo esc_html($copy['book_number']); ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Warning Messages -->
                <?php if ($activeLoans > 0): ?>
                    <div class="bg-red-50 border-l-4 border-red-500 p-4">
                        <div class="flex items-start">
                            <svg class="w-5 h-5 text-red-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                            </svg>
                            <div>
                                <p class="font-bold text-red-800"><?php echo $lang['cannot_delete_book'] ?? 'Buch kann nicht gelöscht werden!'; ?></p>
                                <p class="text-sm text-red-700 mt-1">
                                    <?php echo $activeLoans; ?> <?php echo $lang['copies_still_loaned'] ?? 'Exemplar(e) sind noch ausgeliehen. Bitte erst alle Exemplare zurückbuchen.'; ?>
                                </p>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4">
                        <div class="flex items-start">
                            <svg class="w-5 h-5 text-yellow-600 mt-0.5 mr-3" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                            </svg>
                            <div class="flex-1">
                                <p class="font-bold text-yellow-800"><?php echo $lang['warning_irreversible'] ?? 'Achtung: Diese Aktion kann nicht rückgängig gemacht werden!'; ?></p>
                                <ul class="text-sm text-yellow-700 mt-2 space-y-1">
                                    <li>• <?php echo $lang['book_permanently_deleted'] ?? 'Das Buch wird unwiderruflich gelöscht'; ?></li>
                                    <li>• <?php echo $lang['all_copies_deleted'] ?? 'Alle'; ?> <?php echo $copiesCount; ?> <?php echo $lang['copies_count_plural'] ?? 'Exemplare werden gelöscht'; ?></li>
                                    <?php if ($loanHistoryCount > 0): ?>
                                        <li>• <?php echo $lang['loan_history_deleted'] ?? 'Der Ausleih-Verlauf'; ?> (<?php echo $loanHistoryCount; ?> <?php echo $lang['loan_history_entries'] ?? 'Einträge'; ?>) <?php echo $lang['msg_deleted_successfully'] ?? 'wird gelöscht'; ?></li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Action Buttons -->
                <div class="flex gap-4 pt-4">
                    <?php if ($activeLoans == 0 || $activeLoans === null): ?>
                        <form method="POST" class="flex-1">
                            <input type="hidden" name="confirm_delete" value="1">
                            <button type="submit"
                                    class="w-full px-6 py-3 bg-red-600 text-white font-medium rounded-md hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500">
                                🗑️ <?php echo $lang['confirm_delete_final'] ?? 'Ja, Buch endgültig löschen'; ?>
                            </button>
                        </form>
                    <?php endif; ?>

                    <a href="books-edit-detail.php?id=<?php echo $book['id']; ?>"
                       class="flex-1 px-6 py-3 bg-gray-200 text-gray-800 font-medium rounded-md hover:bg-gray-300 text-center">
                        <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                    </a>
                </div>

                <!-- Alternative: Go back to list -->
                <div class="text-center">
                    <a href="books-edit.php" class="text-sm text-gray-600 hover:text-gray-900">
                        <?php echo $lang['back_to_book_list'] ?? 'Zurück zur Bücherliste'; ?>
                    </a>
                </div>

            </div>
        </div>

    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>
