<?php
/**
 * Buch erfassen — books-add.php
 * ISBN-Lookup (DNB + Google Books + ISBN.de), manuelle Erfassung,
 * Duplikat-Erkennung, Kategorie/Subkategorie, Buchungsnummer
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang = load_language();
$page_title   = $lang['admin_add_book'] ?? 'Buch erfassen';
$current_user = get_logged_in_user();

// ============================================================
// HILFSFUNKTIONEN (aus books-capture.php bewährt + verbessert)
// ============================================================

function bib_fetchUrl(string $url, int $timeout = 8): string {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
        CURLOPT_FOLLOWLOCATION => true,
    ]);
    $data = curl_exec($ch);
    curl_close($ch);
    return $data ?: '';
}

function bib_getGoogleBooks(string $isbn, int $timeout = 8): ?array {
    $url  = 'https://www.googleapis.com/books/v1/volumes?q=isbn:' . urlencode($isbn);
    $json = bib_fetchUrl($url, $timeout);
    $data = json_decode($json, true);

    if (empty($data['items'])) return null;

    $info = $data['items'][0]['volumeInfo'];
    $img  = $info['imageLinks']['thumbnail'] ?? '';
    $img  = str_replace('http://', 'https://', $img);

    $authors = $info['authors'] ?? [];

    return [
        'source'      => 'Google Books',
        'title'       => $info['title']       ?? '',
        'subtitle'    => $info['subtitle']     ?? '',
        'author'      => implode(', ', $authors),
        'publisher'   => $info['publisher']    ?? '',
        'year'        => substr($info['publishedDate'] ?? '', 0, 4),
        'pages'       => $info['pageCount']    ?? '',
        'description' => $info['description'] ?? '',
        'language'    => $info['language']     ?? '',
        'keywords'    => implode(', ', $info['categories'] ?? []),
        'image'       => $img,
        'cover_link'  => $info['infoLink']     ?? '',
        'price'       => '',
        'place_of_publication' => '',
    ];
}

function bib_getDNB(string $isbn, int $timeout = 8): ?array {
    // Offizielle SRU-Schnittstelle der Deutschen Nationalbibliothek
    // Kostenlos, ohne Registrierung, explizit zur Nutzung freigegeben
    // Doku: https://www.dnb.de/sru
    $url = 'https://services.dnb.de/sru/dnb?version=1.1&operation=searchRetrieve'
         . '&query=isbn%3D' . urlencode($isbn)
         . '&recordSchema=oai_dc&maximumRecords=1';

    $xml = bib_fetchUrl($url, $timeout);
    if (!$xml) return null;

    $doc = @simplexml_load_string($xml);
    if (!$doc) return null;

    $doc->registerXPathNamespace('srw',    'http://www.loc.gov/zing/srw/');
    $doc->registerXPathNamespace('dc',     'http://purl.org/dc/elements/1.1/');
    $doc->registerXPathNamespace('oai_dc', 'http://www.openarchives.org/OAI/2.0/oai_dc/');

    $records = $doc->xpath('//srw:record/srw:recordData/oai_dc:dc');
    if (empty($records)) return null;

    $r = $records[0];
    $r->registerXPathNamespace('dc', 'http://purl.org/dc/elements/1.1/');

    $get = function(string $field) use ($r): string {
        $vals = $r->xpath("dc:{$field}");
        return $vals ? trim((string)$vals[0]) : '';
    };
    $getAll = function(string $field) use ($r): array {
        $vals = $r->xpath("dc:{$field}");
        return array_map(fn($v) => trim((string)$v), $vals ?: []);
    };

    $title = $get('title');
    if (!$title) return null;

    // Verfasserangabe abschneiden: "Titel / Autor" → "Titel"  (DNB-Katalogformat)
    $title = trim(preg_replace('/\s+\/\s+.+$/', '', $title));

    // "Titel : Untertitel" aufteilen
    $subtitle = '';
    if (str_contains($title, ' : ')) {
        [$title, $subtitle] = explode(' : ', $title, 2);
        $title    = trim($title);
        $subtitle = trim($subtitle);
    }

    // Autoren (DNB-Format: "Nachname, Vorname [Verfasser]")
    // Rollentags wie [Verfasser], [Hrsg.], [Ill.] entfernen
    $creators = $getAll('creator');
    $creators = array_map(fn($c) => trim(preg_replace('/\s*\[[^\]]*\]/', '', $c)), $creators);
    $creators = array_values(array_filter($creators));
    $author   = implode('; ', $creators);

    // "Ort : Verlag" aufteilen
    $pubRaw    = $get('publisher');
    $publisher = $pubRaw;
    $place     = '';
    if (str_contains($pubRaw, ' : ')) {
        [$place, $publisher] = explode(' : ', $pubRaw, 2);
        $place     = trim($place);
        $publisher = trim($publisher);
    }

    $year = substr($get('date'), 0, 4);

    $langMap  = ['ger' => 'Deutsch', 'eng' => 'Englisch', 'fre' => 'Französisch',
                 'spa' => 'Spanisch', 'ita' => 'Italienisch'];
    $langCode = $get('language');
    $language = $langMap[$langCode] ?? $langCode;

    $subjects = $getAll('subject');
    $keywords = implode(', ', array_filter($subjects, fn($s) => strlen($s) > 2));

    return [
        'source'               => 'DNB',
        'title'                => $title,
        'subtitle'             => $subtitle,
        'author'               => $author,
        'publisher'            => $publisher,
        'place_of_publication' => $place,
        'year'                 => $year,
        'pages'                => '',
        'description'          => $get('description'),
        'language'             => $language,
        'keywords'             => $keywords,
        'image'                => '',
        'cover_link'           => '',
        'price'                => '',
    ];
}

function bib_scrapeIsbnDe(string $isbn, int $timeout = 8): ?array {
    $url  = 'https://www.isbn.de/buch/' . urlencode($isbn);
    $html = bib_fetchUrl($url, $timeout);
    if (!$html) return null;

    $dom = new DOMDocument();
    @$dom->loadHTML('<?xml encoding="UTF-8">' . $html);
    $xpath = new DOMXPath($dom);

    $jsonScripts = $xpath->query('//script[@type="application/ld+json"]');
    if ($jsonScripts->length === 0) return null;

    $data = json_decode($jsonScripts->item(0)->nodeValue, true);
    if (!$data || ($data['@type'] ?? '') !== 'Book') return null;

    $authors = [];
    if (isset($data['author'])) {
        if (isset($data['author']['name'])) {
            $authors[] = $data['author']['name'];
        } elseif (is_array($data['author'])) {
            foreach ($data['author'] as $a) {
                if (isset($a['name'])) $authors[] = $a['name'];
            }
        }
    }

    $price = $data['offers']['price'] ?? '';

    return [
        'source'               => 'ISBN.de',
        'title'                => $data['name']              ?? '',
        'subtitle'             => '',
        'author'               => implode(', ', $authors),
        'publisher'            => $data['publisher']['name'] ?? '',
        'place_of_publication' => '',
        'year'                 => substr($data['datePublished'] ?? '', 0, 4),
        'pages'                => $data['numberOfPages']     ?? '',
        'description'          => html_entity_decode(strip_tags($data['description'] ?? ''), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
        'language'             => '',
        'keywords'             => '',
        'image'                => $data['image']             ?? '',
        'cover_link'           => $data['url']               ?? $url,
        'price'                => $price,
    ];
}

function bib_mergeBookData(?array $google, ?array $isbnDe, ?array $dnb = null): array {
    $empty = ['title'=>'','subtitle'=>'','author'=>'','publisher'=>'','year'=>'',
              'pages'=>'','description'=>'','language'=>'','keywords'=>'',
              'image'=>'','cover_link'=>'','price'=>'','place_of_publication'=>''];

    if (!$google && !$isbnDe && !$dnb) return $empty;

    // Priorität Metadaten: DNB > Google > ISBN.de
    // Cover + Preis: Google > ISBN.de  (DNB liefert beides nicht)
    $base = $dnb ?? $google ?? $isbnDe;

    // Lücken aus allen Quellen auffüllen
    foreach ([$google, $isbnDe, $dnb] as $src) {
        if (!$src) continue;
        foreach ($empty as $key => $_) {
            if (empty($base[$key]) && !empty($src[$key])) {
                $base[$key] = $src[$key];
            }
        }
    }

    // Cover + Preis explizit aus Google/ISBN.de holen
    foreach ([$google, $isbnDe] as $src) {
        if (!$src) continue;
        if (empty($base['image'])      && !empty($src['image']))      $base['image']      = $src['image'];
        if (empty($base['cover_link']) && !empty($src['cover_link'])) $base['cover_link'] = $src['cover_link'];
        if (empty($base['price'])      && !empty($src['price']))      $base['price']      = $src['price'];
    }

    return array_merge($empty, $base);
}

// ISBN-API Settings aus DB lesen
function bib_getIsbnSettings(): array {
    return [
        'primary'  => db_get_var("SELECT setting_value FROM settings WHERE setting_key = 'isbn_api_primary'") ?: 'dnb',
        'second'   => db_get_var("SELECT setting_value FROM settings WHERE setting_key = 'isbn_api_second'") ?: 'google',
        'use_both' => (db_get_var("SELECT setting_value FROM settings WHERE setting_key = 'isbn_api_use_both'") ?: 'true') === 'true',
        'timeout'  => intval(db_get_var("SELECT setting_value FROM settings WHERE setting_key = 'isbn_api_timeout'") ?: 8),
    ];
}

// Signatur berechnen
function bib_build_signature(?int $subcatId, ?string $authorShort, string $bookNumber): ?string {
    $sig1 = '';
    if ($subcatId) {
        $sub  = db_get_row("SELECT code FROM book_subcategories WHERE id = ?", [$subcatId]);
        $sig1 = $sub['code'] ?? '';
    }
    $parts = array_filter([$sig1, $authorShort ?: '', $bookNumber]);
    return $parts ? implode(' ', $parts) : null;
}

// nächste freie Buchungsnummer ermitteln
function bib_nextBookNumber(): int {
    $stored = db_get_var("SELECT setting_value FROM settings WHERE setting_key = 'next_book_number'");
    if ($stored && intval($stored) > 1) return intval($stored);
    $max = db_get_var("
        SELECT MAX(CAST(REGEXP_REPLACE(book_number, '[^0-9]', '') AS UNSIGNED))
        FROM book_copies
        WHERE book_number REGEXP '^[0-9]+$'
    ");
    return $max ? intval($max) + 1 : 1;
}

// ============================================================
// AJAX: ISBN-Lookup
// ============================================================
if (isset($_GET['action']) && $_GET['action'] === 'isbn_lookup') {
    header('Content-Type: application/json; charset=utf-8');
    $isbn = preg_replace('/[^0-9X]/i', '', strtoupper(trim($_GET['isbn'] ?? '')));

    if (strlen($isbn) !== 10 && strlen($isbn) !== 13) {
        echo json_encode(['error' => 'Ungültige ISBN — bitte 10 oder 13 Stellen eingeben']);
        exit;
    }

    // Duplikat?
    $existing = db_get_row("
        SELECT b.id, b.title, b.author, b.subtitle, b.publisher, b.year,
               b.pages, b.language, b.description, b.keywords,
               b.image_url, b.cover_link, b.author_short, b.subcategory_id,
               COUNT(bc.id) as copy_count,
               (SELECT book_number FROM book_copies
                WHERE book_id = b.id AND copy_index = 1 LIMIT 1) as base_number
        FROM books b
        LEFT JOIN book_copies bc ON bc.book_id = b.id
        WHERE b.isbn = ?
        GROUP BY b.id
    ", [$isbn]);

    if ($existing) {
        // Basisnummer: Exemplar 1 (ohne Suffix), Fallback: nächste freie Nummer
        $baseNum  = $existing['base_number'] ?? bib_nextBookNumber();
        $suffix   = intval($existing['copy_count']) + 1;
        $maxTries = 100;
        do {
            $candidate = $baseNum . '_' . $suffix;
            $taken = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ?", [$candidate]);
            if ($taken) $suffix++;
            $maxTries--;
        } while ($taken && $maxTries > 0);

        echo json_encode([
            'duplicate'      => true,
            'book_id'        => $existing['id'],
            'title'          => $existing['title'],
            'subtitle'       => $existing['subtitle'],
            'author'         => $existing['author'],
            'author_short'   => $existing['author_short'],
            'publisher'      => $existing['publisher'],
            'year'           => $existing['year'],
            'pages'          => $existing['pages'],
            'language'       => $existing['language'],
            'description'    => $existing['description'],
            'keywords'       => $existing['keywords'],
            'image'          => $existing['image_url'],
            'cover_link'     => $existing['cover_link'],
            'subcategory_id' => $existing['subcategory_id'],
            'copy_count'     => $existing['copy_count'],
            'next_number'    => $candidate,
        ]);
        exit;
    }

    // Settings aus DB lesen
    $cfg = bib_getIsbnSettings();

    // Hilfsfunktion: API anhand Schlüssel aufrufen
    $callApi = function(string $key) use ($isbn, $cfg): ?array {
        return match($key) {
            'dnb'    => bib_getDNB($isbn, $cfg['timeout']),
            'google' => bib_getGoogleBooks($isbn, $cfg['timeout']),
            'isbnde' => bib_scrapeIsbnDe($isbn, $cfg['timeout']),
            default  => null,
        };
    };

    // Primäre API immer abfragen
    $primaryData = $callApi($cfg['primary']);

    // Zweite API: wenn use_both aktiv und nicht dieselbe wie primäre
    $secondaryData = null;
    if ($cfg['use_both'] && $cfg['second'] !== $cfg['primary']) {
        $secondaryData = $callApi($cfg['second']);
    }

    // Ergebnisse in benannte Slots aufteilen
    $google = null; $isbnDe = null; $dnb = null;
    foreach ([[$cfg['primary'], $primaryData], [$cfg['second'], $secondaryData]] as [$key, $data]) {
        if ($key === 'google') $google = $data;
        if ($key === 'isbnde') $isbnDe = $data;
        if ($key === 'dnb')    $dnb    = $data;
    }

    $merged = bib_mergeBookData($google, $isbnDe, $dnb);
    $merged['isbn']        = $isbn;
    $merged['next_number'] = bib_nextBookNumber();
    $merged['sources']     = array_values(array_filter([
        $dnb    ? 'DNB'          : null,
        $google ? 'Google Books' : null,
        $isbnDe ? 'ISBN.de'      : null,
    ]));

    echo json_encode($merged);
    exit;
}

// ============================================================
// POST: Buch speichern
// ============================================================
$saveError   = '';
$saveSuccess = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_book') {
    try {
        $title = trim($_POST['title'] ?? '');
        if (!$title) throw new Exception('Titel ist ein Pflichtfeld');

        $isbn = preg_replace('/[^0-9X]/i', '', strtoupper(trim($_POST['isbn'] ?? '')));
        if ($isbn && strlen($isbn) !== 10 && strlen($isbn) !== 13) {
            throw new Exception('ISBN ungültig (muss 10 oder 13 Stellen haben)');
        }

        $bookNumber = trim($_POST['book_number'] ?? '');
        if (!$bookNumber) throw new Exception('Buchungsnummer ist ein Pflichtfeld');

        $numExists = db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ?", [$bookNumber]);
        if ($numExists) throw new Exception("Buchungsnummer '$bookNumber' ist bereits vergeben");

        $year  = !empty($_POST['year'])  ? intval($_POST['year'])  : null;
        $pages = !empty($_POST['pages']) ? intval($_POST['pages']) : null;
        $price = !empty($_POST['price']) ? floatval(str_replace(',', '.', $_POST['price'])) : null;

        $subcatId = !empty($_POST['subcategory_id']) ? intval($_POST['subcategory_id']) : null;

        $existingBookId = !empty($_POST['existing_book_id']) ? intval($_POST['existing_book_id']) : 0;

        if ($existingBookId) {
            $copyIndex   = intval(db_get_var(
                "SELECT COUNT(*) FROM book_copies WHERE book_id = ?", [$existingBookId]
            )) + 1;
            $existingBook    = db_get_row(
                "SELECT author_short, subcategory_id FROM books WHERE id = ?", [$existingBookId]
            );
            $existingSubcatId = $existingBook['subcategory_id'] ? (int)$existingBook['subcategory_id'] : null;
            $subcatCode       = $existingSubcatId
                ? (db_get_var("SELECT code FROM book_subcategories WHERE id = ?", [$existingSubcatId]) ?: null)
                : null;
            $signature = bib_build_signature(
                $existingSubcatId,
                $existingBook['author_short'] ?? null,
                $bookNumber
            );
            db_insert('book_copies', [
                'book_id'          => $existingBookId,
                'book_number'      => $bookNumber,
                'accession_number' => $bookNumber,
                'copy_index'       => $copyIndex,
                'status'           => 'verfügbar',
                'location'         => $subcatCode,
                'acquired_at'      => !empty($_POST['acquired_at']) ? $_POST['acquired_at'] : date('Y-m-d'),
                'signature'        => $signature,
            ]);
            log_activity('copy_added', 'book_copies', $existingBookId, json_encode([
                'book_number' => $bookNumber, 'copy_index' => $copyIndex
            ]));
            $saveSuccess = $existingBookId;

        } else {

        $bookId = db_insert('books', [
            'isbn'                 => $isbn ?: null,
            'title'                => $title,
            'subtitle'             => trim($_POST['subtitle']      ?? '') ?: null,
            'title_sort'           => trim($_POST['title_sort']    ?? '') ?: null,
            'author'               => trim($_POST['author']        ?? '') ?: null,
            'author_short'         => trim($_POST['author_short']  ?? '') ?: null,
            'author_additional'    => trim($_POST['author_additional'] ?? '') ?: null,
            'editor'               => trim($_POST['editor']        ?? '') ?: null,
            'translator'           => trim($_POST['translator']    ?? '') ?: null,
            'illustrator'          => trim($_POST['illustrator']   ?? '') ?: null,
            'photographer'         => trim($_POST['photographer']  ?? '') ?: null,
            'graphic_artist'       => trim($_POST['graphic_artist']?? '') ?: null,
            'other_persons'        => trim($_POST['other_persons'] ?? '') ?: null,
            'publisher'            => trim($_POST['publisher']     ?? '') ?: null,
            'place_of_publication' => trim($_POST['place_of_publication'] ?? '') ?: null,
            'year'                 => $year,
            'edition'              => trim($_POST['edition']       ?? '') ?: null,
            'series'               => trim($_POST['series']        ?? '') ?: null,
            'series_volume'        => trim($_POST['series_volume'] ?? '') ?: null,
            'pages'                => $pages,
            'language'             => trim($_POST['language']      ?? 'Deutsch') ?: 'Deutsch',
            'description'          => trim($_POST['description']   ?? '') ?: null,
            'keywords'             => trim($_POST['keywords']      ?? '') ?: null,
            'subcategory_id'       => $subcatId,
            'price'                => $price,
            'image_url'            => trim($_POST['image_url']     ?? '') ?: null,
            'cover_link'           => trim($_POST['cover_link']    ?? '') ?: null,
            'created_by'           => $current_user['id'],
        ]);

        if (!$bookId) throw new Exception('Fehler beim Anlegen des Buches');

        $subcatCode = $subcatId
            ? (db_get_var("SELECT code FROM book_subcategories WHERE id = ?", [$subcatId]) ?: null)
            : null;
        $signature = bib_build_signature(
            $subcatId,
            trim($_POST['author_short'] ?? '') ?: null,
            $bookNumber
        );

        db_insert('book_copies', [
            'book_id'          => $bookId,
            'book_number'      => $bookNumber,
            'accession_number' => $bookNumber,
            'copy_index'       => 1,
            'status'           => 'verfügbar',
            'location'         => $subcatCode,
            'acquired_at'      => !empty($_POST['acquired_at']) ? $_POST['acquired_at'] : date('Y-m-d'),
            'signature'        => $signature,
        ]);

        $nextNum = intval($bookNumber) + 1;
        db_query("INSERT INTO settings (setting_key, setting_value) VALUES ('next_book_number', ?)
                  ON DUPLICATE KEY UPDATE setting_value = ?", [$nextNum, $nextNum]);

        log_activity('book_created', 'books', $bookId, json_encode(['title' => $title, 'isbn' => $isbn]));

        $saveSuccess = $bookId;

        }

    } catch (Exception $e) {
        $saveError = $e->getMessage();
    }
}

// ============================================================
// AJAX: Buchungsnummer-Eindeutigkeits-Check
// ============================================================
if (isset($_GET['action']) && $_GET['action'] === 'check_number') {
    header('Content-Type: application/json; charset=utf-8');
    $num = trim($_GET['number'] ?? '');
    $exists = $num ? db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ?", [$num]) : 0;
    echo json_encode(['available' => !$exists]);
    exit;
}

$categories = db_get_results("
    SELECT id, code, name FROM book_categories WHERE active = 1 ORDER BY sort_order, name
");
$subcategories = db_get_results("
    SELECT id, category_id, code, name FROM book_subcategories WHERE active = 1 ORDER BY sort_order, name
");

$nextNumber = bib_nextBookNumber();

include __DIR__ . '/includes/header.php';
?>

<div class="max-w-7xl mx-auto px-4 py-6">

    <div class="mb-4 text-sm text-gray-500">
        <a href="books-edit.php" class="hover:text-blue-600"><?php echo $lang['books'] ?? 'Bücher'; ?></a>
        <span class="mx-2">›</span>
        <span class="text-gray-800"><?php echo $lang['admin_add_book'] ?? 'Buch erfassen'; ?></span>
    </div>

    <?php if ($saveSuccess): ?>
    <div class="bg-green-50 border border-green-300 rounded-xl p-6 mb-6">
        <div class="flex items-start gap-4">
            <div class="text-3xl">✅</div>
            <div class="flex-1">
                <h2 class="text-lg font-bold text-green-800 mb-1"><?php echo $lang['book_saved_success'] ?? 'Buch erfolgreich gespeichert'; ?></h2>
                <p class="text-green-700 text-sm mb-4"><?php echo sprintf($lang['book_saved_id'] ?? 'Das Buch wurde mit ID #%s angelegt.', $saveSuccess); ?></p>
                <div class="flex flex-wrap gap-3">
                    <a href="books-add.php" class="px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700">
                        + <?php echo $lang['add_next_book'] ?? 'Nächstes Buch erfassen'; ?>
                    </a>
                    <a href="books-edit-detail.php?id=<?php echo $saveSuccess; ?>"
                       class="px-4 py-2 bg-white border border-gray-300 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-50">
                        <?php echo $lang['view_edit_book'] ?? 'Buch anzeigen / bearbeiten'; ?>
                    </a>
                    <a href="books-label-print.php?book_id=<?php echo $saveSuccess; ?>"
                       class="px-4 py-2 bg-white border border-gray-300 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-50">
                        🏷️ <?php echo $lang['print_label'] ?? 'Etikett drucken'; ?>
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($saveError): ?>
    <div class="bg-red-50 border border-red-300 rounded-xl p-4 mb-6">
        <p class="text-red-800 font-medium">⚠️ <?php echo esc_html($saveError); ?></p>
    </div>
    <?php endif; ?>

    <!-- ISBN-Suche -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
        <h2 class="text-lg font-bold text-gray-800 mb-4">📷 <?php echo $lang['isbn_scan_enter'] ?? 'ISBN scannen / eingeben'; ?></h2>
        <div class="flex gap-3">
            <input type="text" id="isbn-input" placeholder="<?php echo esc_attr($lang['isbn_placeholder'] ?? 'ISBN scannen oder eingeben…'); ?>"
                   class="flex-1 px-4 py-3 border border-gray-300 rounded-lg text-lg font-mono focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                   autofocus inputmode="numeric">
            <button onclick="lookupISBN()" id="btn-lookup"
                    class="px-6 py-3 bg-blue-600 text-white font-medium rounded-lg hover:bg-blue-700 transition-colors">
                🔍 <?php echo $lang['search'] ?? 'Suchen'; ?>
            </button>
            <button onclick="clearForm()"
                    class="px-4 py-3 bg-gray-100 text-gray-600 font-medium rounded-lg hover:bg-gray-200 transition-colors"
                    title="<?php echo esc_attr($lang['isbn_clear_form'] ?? 'Formular leeren'); ?>">✕</button>
        </div>

        <div id="lookup-status" class="mt-3 hidden">
            <div id="status-loading" class="hidden text-sm text-blue-600">⏳ <?php echo $lang['isbn_searching'] ?? 'Suche läuft…'; ?></div>
            <div id="status-sources" class="hidden text-sm text-green-600"></div>
            <div id="status-error"   class="hidden text-sm text-red-600"></div>
        </div>

        <div id="duplicate-banner" class="hidden mt-4 bg-blue-50 border border-blue-300 rounded-lg p-4">
            <p class="font-semibold text-blue-800">📚 <?php echo $lang['isbn_duplicate_notice'] ?? 'Dieses Buch ist bereits im Bestand — weiteres Exemplar wird hinzugefügt'; ?></p>
            <p id="duplicate-info" class="text-sm text-blue-700 mt-1"></p>
            <button onclick="clearDuplicateMode()" class="mt-2 text-xs text-blue-500 hover:underline">
                ✕ <?php echo $lang['isbn_cancel_clear'] ?? 'Abbrechen / Formular leeren'; ?>
            </button>
        </div>

        <div class="mt-3 flex flex-wrap gap-2">
            <span class="text-xs text-gray-400 self-center"><?php echo $lang['isbn_test_label'] ?? 'Test:'; ?></span>
            <?php foreach (['9783902404442','9783446210486','9783423345538','9783453605770','9783701182053'] as $t): ?>
                <button onclick="document.getElementById('isbn-input').value='<?php echo $t; ?>';lookupISBN()"
                        class="px-2 py-1 text-xs bg-gray-100 text-gray-500 rounded hover:bg-gray-200 font-mono">
                    <?php echo $t; ?>
                </button>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Erfassungsformular -->
    <form method="POST" id="book-form">
        <input type="hidden" name="action" value="save_book">
        <input type="hidden" name="existing_book_id" id="f-existing_book_id" value="">
        <input type="hidden" name="image_url"  id="f-image_url">
        <input type="hidden" name="cover_link" id="f-cover_link">

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

            <div class="space-y-5">
                <!-- Cover -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 text-center">
                    <div id="cover-wrap">
                        <div id="cover-placeholder" class="h-48 bg-gray-100 rounded-lg flex items-center justify-center text-gray-400 text-sm">
                            <?php echo $lang['no_cover'] ?? 'Kein Cover'; ?>
                        </div>
                        <img id="cover-img" src="" alt="<?php echo esc_attr($lang['cover_image'] ?? 'Cover'); ?>" class="hidden max-h-64 mx-auto rounded-lg shadow">
                    </div>
                    <a id="cover-link-btn" href="#" target="_blank"
                       class="hidden mt-3 inline-block text-xs text-blue-600 hover:underline">
                        ↗ <?php echo $lang['cover_source_link'] ?? 'Cover-Quelle öffnen'; ?>
                    </a>
                </div>

                <!-- Exemplar -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4">📦 <?php echo $lang['copy_section'] ?? 'Exemplar'; ?></h3>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['book_number_field'] ?? 'Buchungsnummer'; ?> <span class="text-red-500">*</span>
                        </label>
                        <div class="flex gap-2">
                            <input type="text" name="book_number" id="f-book_number"
                                   value="<?php echo $nextNumber; ?>"
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg font-mono text-lg font-bold focus:ring-2 focus:ring-blue-500"
                                   oninput="checkBookNumber(this.value)">
                            <div id="num-status" class="self-center text-lg" title="<?php echo esc_attr($lang['number_available'] ?? 'Verfügbar'); ?>">—</div>
                        </div>
                        <p class="text-xs text-gray-400 mt-1"><?php echo $lang['suggested'] ?? 'Vorgeschlagen'; ?>: <strong><?php echo $nextNumber; ?></strong></p>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_category'] ?? 'Kategorie'; ?></label>
                        <select id="f-category" onchange="filterSubcategories(this.value)"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                            <option value="">— <?php echo $lang['category_choose'] ?? 'Kategorie wählen'; ?> —</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo $cat['id']; ?>">
                                    <?php echo esc_html($cat['code'] . ' — ' . $cat['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['subcategory_label'] ?? 'Untergruppe'; ?></label>
                        <select name="subcategory_id" id="f-subcategory_id"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                            <option value="">— <?php echo $lang['subcategory_choose_first'] ?? 'erst Kategorie wählen'; ?> —</option>
                            <?php foreach ($subcategories as $sub): ?>
                                <option value="<?php echo $sub['id']; ?>"
                                        data-cat="<?php echo $sub['category_id']; ?>"
                                        class="subcat-option">
                                    <?php echo esc_html($sub['code'] . ' — ' . $sub['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['location'] ?? 'Standort'; ?></label>
                        <input type="text" name="location" placeholder="<?php echo esc_attr($lang['location_placeholder'] ?? 'z.B. Regal A3'); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['acquired_at'] ?? 'Zugangsdatum'; ?></label>
                        <input type="date" name="acquired_at" value="<?php echo date('Y-m-d'); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>
                </div>
            </div>

            <div class="lg:col-span-2 space-y-5">

                <!-- Titeldaten -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4">📖 <?php echo $lang['title_data_section'] ?? 'Titeldaten'; ?></h3>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_isbn'] ?? 'ISBN'; ?></label>
                        <input type="text" name="isbn" id="f-isbn" placeholder="<?php echo esc_attr($lang['isbn_auto_filled'] ?? 'Automatisch befüllt'); ?>"
                               class="w-full px-3 py-2 border border-gray-200 bg-gray-50 rounded-lg font-mono text-sm">
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['book_title'] ?? 'Titel'; ?> <span class="text-red-500">*</span>
                        </label>
                        <input type="text" name="title" id="f-title" required
                               oninput="suggestAuthorShort(); suggestTitleSort()"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['subtitle'] ?? 'Untertitel'; ?></label>
                        <input type="text" name="subtitle" id="f-subtitle"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['title_sort_label'] ?? 'Sortierform'; ?>
                            <span class="text-xs text-gray-400 font-normal">(<?php echo $lang['title_sort_help'] ?? 'Titel ohne Artikel für Sortierung'; ?>)</span>
                        </label>
                        <input type="text" name="title_sort" id="f-title_sort"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>
                </div>

                <!-- Autoren -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4">👤 <?php echo $lang['persons_section'] ?? 'Beteiligte Personen'; ?></h3>
                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_author'] ?? 'Autor'; ?></label>
                            <input type="text" name="author" id="f-author" placeholder="<?php echo esc_attr($lang['author_placeholder'] ?? 'Nachname Vorname'); ?>"
                                   oninput="suggestAuthorShort()"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">
                                <?php echo $lang['author_short_label'] ?? 'Kürzel'; ?> <span class="text-xs text-gray-400 font-normal"></span>
                            </label>
                            <input type="text" name="author_short" id="f-author_short"
                                   placeholder="<?php echo esc_attr($lang['author_short_placeholder'] ?? 'z.B. Böl'); ?>" maxlength="20"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm font-mono">
                        </div>
                    </div>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['additional_authors'] ?? 'Weitere Autoren'; ?></label>
                        <input type="text" name="author_additional" id="f-author_additional"
                               placeholder="<?php echo esc_attr($lang['comma_separated'] ?? 'kommagetrennt'); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>
                    <div>
                        <button type="button" onclick="toggleExtra()" id="btn-extra"
                                class="text-sm text-blue-600 hover:underline mb-3">
                            ＋ <?php echo $lang['show_extra_persons'] ?? 'Herausgeber, Übersetzer, Illustratoren…'; ?>
                        </button>
                        <div id="extra-persons" class="hidden grid grid-cols-2 gap-4">
                            <?php foreach ([
                                ['editor',        $lang['editor_label']        ?? 'Herausgeber'],
                                ['translator',    $lang['translator_label']    ?? 'Übersetzer'],
                                ['illustrator',   $lang['illustrator_label']   ?? 'Illustrator'],
                                ['photographer',  $lang['photographer_label']  ?? 'Fotograf'],
                                ['graphic_artist',$lang['graphic_artist_label']?? 'Grafiker'],
                                ['other_persons', $lang['other_persons_label'] ?? 'Sonstige'],
                            ] as [$name,$label]): ?>
                            <div>
                                <label class="block text-xs font-medium text-gray-600 mb-1"><?php echo esc_html($label); ?></label>
                                <input type="text" name="<?php echo $name; ?>" id="f-<?php echo $name; ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <!-- Publikation -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4">🏢 <?php echo $lang['publication_section'] ?? 'Publikation'; ?></h3>
                    <div class="grid grid-cols-2 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_publisher'] ?? 'Verlag'; ?></label>
                            <input type="text" name="publisher" id="f-publisher"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['place_of_publication_label'] ?? 'Erscheinungsort'; ?></label>
                            <input type="text" name="place_of_publication" id="f-place_of_publication"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                    </div>
                    <div class="grid grid-cols-3 gap-4 mb-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_year'] ?? 'Jahr'; ?></label>
                            <input type="number" name="year" id="f-year"
                                   min="1000" max="<?php echo date('Y') + 5; ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['edition_label'] ?? 'Auflage'; ?></label>
                            <input type="text" name="edition" id="f-edition" placeholder="<?php echo esc_attr($lang['edition_placeholder'] ?? 'z.B. 3. Aufl.'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['pages_label'] ?? 'Seiten'; ?></label>
                            <input type="number" name="pages" id="f-pages" min="1" max="10000"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                    </div>
                    <div class="grid grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['book_language'] ?? 'Sprache'; ?></label>
                            <select name="language" id="f-language"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                                <option value="Deutsch">Deutsch</option>
                                <option value="Englisch">Englisch</option>
                                <option value="Französisch">Französisch</option>
                                <option value="Spanisch">Spanisch</option>
                                <option value="Italienisch">Italienisch</option>
                                <option value="Andere"><?php echo $lang['lang_other'] ?? 'Andere'; ?></option>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['price_label'] ?? 'Preis (€)'; ?></label>
                            <input type="text" name="price" id="f-price" placeholder="<?php echo esc_attr($lang['price_placeholder'] ?? 'z.B. 24.90'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['series_volume_label'] ?? 'Reihe / Band'; ?></label>
                            <div class="flex gap-1">
                                <input type="text" name="series" id="f-series" placeholder="<?php echo esc_attr($lang['series_placeholder'] ?? 'Reihe'); ?>"
                                       class="flex-1 min-w-0 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                                <input type="text" name="series_volume" id="f-series_volume" placeholder="<?php echo esc_attr($lang['series_volume_placeholder'] ?? 'Bd.'); ?>"
                                       class="w-16 px-2 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Inhalt -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <h3 class="font-semibold text-gray-800 mb-4">📝 <?php echo $lang['content_section'] ?? 'Inhalt & Erschließung'; ?></h3>
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-1">
                            <?php echo $lang['keywords_label'] ?? 'Schlagworte'; ?>
                            <span class="text-xs text-gray-400 font-normal">(<?php echo $lang['comma_separated'] ?? 'kommagetrennt'; ?>)</span>
                        </label>
                        <input type="text" name="keywords" id="f-keywords"
                               placeholder="<?php echo esc_attr($lang['keywords_placeholder'] ?? 'z.B. Roman, Österreich, 20. Jahrhundert'); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['description_label'] ?? 'Inhaltsbeschreibung'; ?></label>
                        <textarea name="description" id="f-description" rows="5"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 text-sm resize-y"></textarea>
                    </div>
                </div>

                <!-- Speichern -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-5">
                    <div class="flex flex-col sm:flex-row gap-3">
                        <button type="submit"
                                class="flex-1 px-6 py-3 bg-green-600 text-white font-semibold rounded-lg hover:bg-green-700 transition-colors text-center">
                            💾 <?php echo $lang['save_book_btn'] ?? 'Buch speichern'; ?>
                        </button>
                        <a href="books-edit.php"
                           class="px-6 py-3 bg-gray-100 text-gray-700 font-medium rounded-lg hover:bg-gray-200 transition-colors text-center">
                            <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                        </a>
                    </div>
                    <p class="text-xs text-gray-400 mt-2" id="save-hint">
                        <?php echo $lang['save_hint_prefix'] ?? 'Das Buch wird mit Buchungsnummer'; ?> <strong id="preview-num"><?php echo $nextNumber; ?></strong>
                        <?php echo $lang['save_hint_suffix'] ?? 'als erstes Exemplar gespeichert.'; ?>
                    </p>
                </div>

            </div>
        </div>
    </form>
</div>

<script>
const SUBCATS  = <?php echo json_encode($subcategories); ?>;
const BASE_URL = '<?php echo BASE_URL; ?>';

// ---- ISBN Lookup ----
let isbnLookupRunning = false;
async function lookupISBN() {
    if (isbnLookupRunning) return;
    const isbn = document.getElementById('isbn-input').value.replace(/[^0-9X]/gi, '');
    if (!isbn) return;

    isbnLookupRunning = true;
    document.getElementById('btn-lookup').disabled = true;
    showStatus('loading');
    document.getElementById('duplicate-banner').classList.add('hidden');

    try {
        const controller = new AbortController();
        const tmr = setTimeout(() => controller.abort(), 30000);
        const res  = await fetch(`${BASE_URL}/admin/books-add.php?action=isbn_lookup&isbn=${encodeURIComponent(isbn)}`,
                                 { signal: controller.signal });
        clearTimeout(tmr);
        const data = await res.json();

        if (data.error) { showStatus('error', data.error); return; }

        if (data.duplicate) {
            fillForm(data);
            setDuplicateMode(data);
            showStatus('sources', `Weiteres Exemplar für: "${data.title}"`);
            return;
        }

        fillForm(data);
        showStatus('sources', 'Gefunden via: ' + (data.sources || []).join(', '));

    } catch (e) {
        showStatus('error', e.name === 'AbortError'
            ? 'Timeout — Server antwortet nicht'
            : 'Netzwerkfehler: ' + e.message);
    } finally {
        isbnLookupRunning = false;
        document.getElementById('btn-lookup').disabled = false;
    }
}

function fillForm(data) {
    const f = (id, val) => { const el = document.getElementById('f-' + id); if (el && val !== undefined && val !== '') el.value = val; };

    f('isbn',                 data.isbn);
    f('title',                data.title);
    f('subtitle',             data.subtitle);
    f('author',               data.author);
    f('publisher',            data.publisher);
    f('place_of_publication', data.place_of_publication);
    f('year',                 data.year);
    f('pages',                data.pages);
    f('description',          data.description);
    f('keywords',             data.keywords);
    f('price',                data.price);
    f('image_url',            data.image);
    f('cover_link',           data.cover_link);

    // Sprache mappen
    const langMap = { de: 'Deutsch', en: 'Englisch', fr: 'Französisch', es: 'Spanisch', it: 'Italienisch' };
    if (data.language) {
        const sel    = document.getElementById('f-language');
        const mapped = langMap[data.language.toLowerCase()] || data.language;
        for (let opt of sel.options) { if (opt.value === mapped) { sel.value = mapped; break; } }
    }

    if (data.next_number) {
        document.getElementById('f-book_number').value = data.next_number;
        document.getElementById('preview-num').textContent = data.next_number;
        checkBookNumber(data.next_number);
    }

    if (data.image) {
        const img = document.getElementById('cover-img');
        img.src = data.image;
        img.classList.remove('hidden');
        document.getElementById('cover-placeholder').classList.add('hidden');
    }
    if (data.cover_link) {
        const btn = document.getElementById('cover-link-btn');
        btn.href = data.cover_link;
        btn.classList.remove('hidden');
    }

    suggestAuthorShort();
    suggestTitleSort();
}

function clearForm() {
    document.getElementById('book-form').reset();
    ['f-isbn','f-image_url','f-cover_link'].forEach(id => {
        const el = document.getElementById(id); if (el) el.value = '';
    });
    document.getElementById('cover-img').classList.add('hidden');
    document.getElementById('cover-placeholder').classList.remove('hidden');
    document.getElementById('cover-link-btn').classList.add('hidden');
    document.getElementById('f-category').value = '';
    filterSubcategories('');
    showStatus('hide');
    document.getElementById('duplicate-banner').classList.add('hidden');
}

function setDuplicateMode(data) {
    document.getElementById('f-existing_book_id').value = data.book_id;
    document.getElementById('duplicate-info').textContent =
        `"${data.title}" von ${data.author || '—'} · aktuell ${data.copy_count} Exemplar(e) im Bestand`;
    document.getElementById('duplicate-banner').classList.remove('hidden');

    const readonlyIds = ['f-isbn','f-title','f-subtitle','f-title_sort','f-author','f-author_short',
        'f-author_additional','f-editor','f-translator','f-illustrator','f-photographer',
        'f-graphic_artist','f-other_persons','f-publisher','f-place_of_publication',
        'f-year','f-edition','f-pages','f-language','f-price','f-series','f-series_volume',
        'f-keywords','f-description'];
    readonlyIds.forEach(id => {
        const el = document.getElementById(id);
        if (el) { el.readOnly = true; el.classList.add('bg-gray-50','text-gray-500'); }
    });
    document.getElementById('f-category').disabled = true;
    document.getElementById('f-subcategory_id').disabled = true;
    document.getElementById('save-hint').innerHTML =
        `Speichert nur ein <strong>neues Exemplar</strong> mit Buchungsnummer <strong id="preview-num">${data.next_number}</strong>.`;
}

function clearDuplicateMode() {
    document.getElementById('f-existing_book_id').value = '';
    document.getElementById('duplicate-banner').classList.add('hidden');
    clearForm();
}

function showStatus(type, msg = '') {
    const wrap = document.getElementById('lookup-status');
    wrap.classList.remove('hidden');
    ['status-loading','status-sources','status-error'].forEach(id =>
        document.getElementById(id).classList.add('hidden'));

    if (type === 'hide')    { wrap.classList.add('hidden'); return; }
    if (type === 'loading') { document.getElementById('status-loading').classList.remove('hidden'); return; }
    if (type === 'sources') { const el = document.getElementById('status-sources'); el.textContent = '✓ ' + msg; el.classList.remove('hidden'); return; }
    if (type === 'error')   { const el = document.getElementById('status-error');   el.textContent = '✗ ' + msg; el.classList.remove('hidden'); return; }
}

function filterSubcategories(catId) {
    const sel  = document.getElementById('f-subcategory_id');
    const opts = sel.querySelectorAll('.subcat-option');
    sel.value  = '';
    let visible = 0;
    opts.forEach(opt => {
        const show = !catId || opt.dataset.cat === catId;
        opt.style.display = show ? '' : 'none';
        if (show) visible++;
    });
    sel.options[0].textContent = catId
        ? (visible > 0 ? '— <?php echo addslashes($lang['subcategory_choose'] ?? 'Untergruppe wählen'); ?> —' : '— <?php echo addslashes($lang['subcategory_no_subcats'] ?? 'keine Untergruppen'); ?> —')
        : '— <?php echo addslashes($lang['subcategory_choose_first'] ?? 'erst Kategorie wählen'); ?> —';
}

let numCheckTimer;
async function checkBookNumber(val) {
    document.getElementById('preview-num').textContent = val;
    clearTimeout(numCheckTimer);
    const el = document.getElementById('num-status');
    if (!val) { el.textContent = '—'; return; }
    el.textContent = '⏳';
    numCheckTimer = setTimeout(async () => {
        try {
            const res  = await fetch(`${BASE_URL}/admin/books-add.php?action=check_number&number=${encodeURIComponent(val)}`);
            const data = await res.json();
            el.textContent = data.available ? '✅' : '❌';
            el.title       = data.available ? '<?php echo addslashes($lang['number_available'] ?? 'Verfügbar'); ?>' : '<?php echo addslashes($lang['number_taken'] ?? 'Bereits vergeben'); ?>!';
        } catch { el.textContent = '?'; }
    }, 400);
}

function suggestAuthorShort() {
    const author  = document.getElementById('f-author').value.trim();
    const shortEl = document.getElementById('f-author_short');
    if (shortEl.value || !author) return;
    const parts = author.split(/[\s,]+/);
    if (parts[0]) shortEl.value = parts[0].substring(0, 3);
}

function suggestTitleSort() {
    const title  = document.getElementById('f-title').value.trim();
    const sortEl = document.getElementById('f-title_sort');
    if (sortEl.value || !title) return;
    sortEl.value = title.replace(/^(Der|Die|Das|Ein|Eine|The|A|An|Le|La|Les|L')\s+/i, '');
}

function toggleExtra() {
    const el  = document.getElementById('extra-persons');
    const btn = document.getElementById('btn-extra');
    const open = el.classList.toggle('hidden');
    btn.textContent = open ? '＋ <?php echo addslashes($lang['show_extra_persons'] ?? 'Herausgeber, Übersetzer, Illustratoren…'); ?>' : '－ <?php echo addslashes($lang['hide_extra_persons'] ?? 'Zusatzfelder einklappen'); ?>';
}

document.getElementById('isbn-input').addEventListener('keydown', e => {
    if (e.key === 'Enter') { e.preventDefault(); lookupISBN(); }
});
document.getElementById('f-book_number').addEventListener('input', function() {
    document.getElementById('preview-num').textContent = this.value;
});

checkBookNumber(document.getElementById('f-book_number').value);
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
