<?php
/**
 * Exemplar hinzufügen — books-add-copy.php
 * Fügt einem bestehenden Buch ein weiteres physisches Exemplar hinzu.
 */

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/session.php';

require_admin();

$lang = load_language();

$page_title   = $lang['add_copy'] ?? 'Exemplar hinzufügen';
$current_user = get_logged_in_user();

// ─── book_id validieren ────────────────────────────────────────────────────────
if (!isset($_GET['book_id']) || !is_numeric($_GET['book_id'])) {
    set_flash($lang['invalid_book_id'] ?? 'Ungültige Buch-ID', 'error');
    header('Location: books-edit.php');
    exit;
}

$bookId = intval($_GET['book_id']);

$book = db_get_row("SELECT * FROM books WHERE id = ?", [$bookId]);
if (!$book) {
    set_flash($lang['book_not_found'] ?? 'Buch nicht gefunden', 'error');
    header('Location: books-edit.php');
    exit;
}

// ─── Signatur-Helper ──────────────────────────────────────────────────────────
function build_signature(int $bookId, string $bookNumber): string {
    $b    = db_get_row("SELECT author_short, subcategory_id FROM books WHERE id = ?", [$bookId]);
    $sig1 = '';
    if ($b['subcategory_id']) {
        $sub  = db_get_row("SELECT code FROM book_subcategories WHERE id = ?", [$b['subcategory_id']]);
        $sig1 = $sub['code'] ?? '';
    }
    $sig2  = $b['author_short'] ?? '';
    $parts = array_filter([$sig1, $sig2, $bookNumber]);
    return implode(' ', $parts);
}

// ─── Nächste Buchungsnummer vorschlagen ────────────────────────────────────────
function next_copy_number(int $bookId): string {
    // Basisnummer = Exemplar mit copy_index = 1 (ohne Suffix)
    $base = db_get_var(
        "SELECT book_number FROM book_copies WHERE book_id = ? AND copy_index = 1 LIMIT 1",
        [$bookId]
    );

    if (!$base) {
        // Kein Exemplar vorhanden: globale nächste Nummer
        $max = db_get_var("
            SELECT MAX(CAST(REGEXP_REPLACE(book_number, '[^0-9]', '') AS UNSIGNED))
            FROM book_copies WHERE book_number REGEXP '^[0-9]+$'
        ");
        return (string)(($max ?: 0) + 1);
    }

    // Nächsten freien Suffix suchen (14613 → 14613_2 → 14613_3 …)
    $count    = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_id = ?", [$bookId]);
    $suffix   = $count + 1;
    $maxTries = 200;
    do {
        $candidate = $base . '_' . $suffix;
        $taken     = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ?", [$candidate]);
        if ($taken) $suffix++;
        $maxTries--;
    } while ($taken && $maxTries > 0);

    return $candidate;
}

// ─── POST: Exemplar speichern ─────────────────────────────────────────────────
$saveError = '';
$newCopyId = 0;
$savedBookNumber = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'add_copy') {
    try {
        $bookNumber = trim($_POST['book_number'] ?? '');
        if (!$bookNumber) throw new Exception($lang['book_number_required'] ?? 'Buchungsnummer ist ein Pflichtfeld');

        $taken = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_number = ?", [$bookNumber]);
        if ($taken) throw new Exception(($lang['book_number_field'] ?? 'Buchungsnummer') . " '{$bookNumber}' " . ($lang['number_taken'] ?? 'ist bereits vergeben'));

        $copyIndex  = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_id = ?", [$bookId]) + 1;
        $signature  = build_signature($bookId, $bookNumber) ?: null;

        $bk         = db_get_row("SELECT subcategory_id FROM books WHERE id = ?", [$bookId]);
        $subcatCode = $bk['subcategory_id']
            ? (db_get_var("SELECT code FROM book_subcategories WHERE id = ?", [$bk['subcategory_id']]) ?: null)
            : null;

        $newCopyId = db_insert('book_copies', [
            'book_id'          => $bookId,
            'book_number'      => $bookNumber,
            'accession_number' => $bookNumber,
            'copy_index'       => $copyIndex,
            'status'           => 'verfügbar',
            'location'         => $subcatCode,
            'condition_note'   => trim($_POST['condition_note'] ?? '') ?: null,
            'acquired_at'      => !empty($_POST['acquired_at']) ? $_POST['acquired_at'] : date('Y-m-d'),
            'signature'        => $signature,
        ]);

        log_activity('copy_added', 'book_copies', $bookId, json_encode([
            'book_number' => $bookNumber,
            'copy_index'  => $copyIndex,
            'signature'   => $signature,
        ]));

        $savedBookNumber = $bookNumber;

    } catch (Exception $e) {
        $saveError = $e->getMessage();
    }
}

// ─── Vorschlag für nächste Nummer ─────────────────────────────────────────────
$nextNumber = next_copy_number($bookId);
$copyCount  = (int) db_get_var("SELECT COUNT(*) FROM book_copies WHERE book_id = ?", [$bookId]);

// Signatur-Vorschau berechnen
$sigPreview = build_signature($bookId, $nextNumber);

include __DIR__ . '/includes/header.php';
?>

<div class="flex-1 overflow-y-auto">
<div class="max-w-xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

    <!-- Zurück -->
    <div class="mb-6">
        <a href="books-edit-detail.php?id=<?php echo $bookId; ?>"
           class="text-sm text-gray-600 hover:text-gray-900">
            ← <?php echo $lang['back'] ?? 'Zurück'; ?>
        </a>
    </div>

    <!-- Buch-Info -->
    <div class="bg-blue-50 border border-blue-200 rounded-xl p-4 mb-6">
        <p class="text-xs text-blue-500 font-medium mb-1"><?php echo $lang['book_label'] ?? 'Buch'; ?></p>
        <p class="font-bold text-blue-900"><?php echo esc_html($book['title']); ?></p>
        <?php if ($book['author']): ?>
            <p class="text-sm text-blue-700"><?php echo esc_html($book['author']); ?></p>
        <?php endif; ?>
        <p class="text-xs text-blue-500 mt-1">
            <?php echo $copyCount; ?>
            <?php echo $copyCount !== 1
                ? ($lang['copies_existing_plural'] ?? 'Exemplare vorhanden')
                : ($lang['copies_existing'] ?? 'Exemplar vorhanden'); ?>
        </p>
    </div>

    <!-- Erfolg -->
    <?php if ($newCopyId): ?>
        <div class="mb-6 bg-green-50 border border-green-300 rounded-xl p-5">
            <div class="flex items-start gap-3">
                <div class="text-2xl">&#10003;</div>
                <div class="flex-1">
                    <h3 class="font-bold text-green-800 mb-1"><?php echo $lang['copy_added_success'] ?? 'Exemplar erfolgreich hinzugefügt'; ?></h3>
                    <p class="text-green-700 text-sm mb-3">
                        <?php echo $lang['book_number_field'] ?? 'Buchungsnummer'; ?>
                        <strong class="font-mono"><?php echo esc_html($savedBookNumber); ?></strong>
                        <?php echo $lang['msg_saved_successfully'] ?? 'wurde angelegt.'; ?>
                    </p>
                    <div class="flex flex-wrap gap-2">
                        <a href="books-edit-detail.php?id=<?php echo $bookId; ?>"
                           class="px-3 py-1.5 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700">
                            <?php echo $lang['to_book'] ?? 'Zum Buch'; ?>
                        </a>
                        <a href="books-add-copy.php?book_id=<?php echo $bookId; ?>"
                           class="px-3 py-1.5 bg-white border border-gray-300 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-50">
                            + <?php echo $lang['add_another_copy'] ?? 'Weiteres Exemplar'; ?>
                        </a>
                        <a href="books-label-print.php?copy_id=<?php echo $newCopyId; ?>"
                           class="px-3 py-1.5 bg-white border border-gray-300 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-50">
                            <?php echo $lang['print_label'] ?? 'Etikett drucken'; ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- Fehler -->
    <?php if ($saveError): ?>
        <div class="mb-6 bg-red-50 border-l-4 border-red-400 p-4 rounded">
            <p class="text-sm text-red-700"><?php echo esc_html($saveError); ?></p>
        </div>
    <?php endif; ?>

    <!-- Formular -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
        <h2 class="font-semibold text-gray-800 mb-5"><?php echo $lang['new_copy_form'] ?? 'Neues Exemplar erfassen'; ?></h2>

        <form method="POST" class="space-y-5">
            <input type="hidden" name="action" value="add_copy">

            <!-- Buchungsnummer -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">
                    <?php echo $lang['book_number_field'] ?? 'Buchungsnummer'; ?> <span class="text-red-500">*</span>
                </label>
                <div class="flex gap-2 items-center">
                    <input type="text" name="book_number" id="f-book_number"
                           value="<?php echo esc_attr($nextNumber); ?>"
                           required
                           oninput="checkNumber(this.value)"
                           class="flex-1 px-3 py-2 border border-gray-300 rounded-lg font-mono text-lg font-bold focus:ring-2 focus:ring-blue-500">
                    <span id="num-status" class="text-xl shrink-0" title="">—</span>
                </div>
                <p class="mt-1 text-xs text-gray-400">
                    <?php echo $lang['suggested'] ?? 'Vorgeschlagen'; ?>: <strong><?php echo esc_html($nextNumber); ?></strong>
                    <?php if ($sigPreview): ?>
                        · <?php echo $lang['signature'] ?? 'Signatur'; ?>: <strong class="font-mono"><?php echo esc_html($sigPreview); ?></strong>
                    <?php endif; ?>
                </p>
            </div>

            <!-- Standort -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['location'] ?? 'Standort'; ?></label>
                <?php
                // Standort des ersten Exemplars als Vorschlag
                $defaultLocation = db_get_var("SELECT location FROM book_copies WHERE book_id = ? AND copy_index = 1 LIMIT 1", [$bookId]);
                ?>
                <input type="text" name="location"
                       value="<?php echo esc_attr($defaultLocation ?? ''); ?>"
                       placeholder="z.B. Regal A3"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Zugangsdatum -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['acquired_at'] ?? 'Zugangsdatum'; ?></label>
                <input type="date" name="acquired_at"
                       value="<?php echo date('Y-m-d'); ?>"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Zustand -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1"><?php echo $lang['condition_note'] ?? 'Zustand / Notiz'; ?></label>
                <input type="text" name="condition_note"
                       placeholder="optional, z.B. leichte Gebrauchsspuren"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-blue-500">
            </div>

            <!-- Buttons -->
            <div class="flex gap-3 pt-2">
                <button type="submit"
                        class="flex-1 px-5 py-2.5 bg-blue-600 text-white font-semibold rounded-lg hover:bg-blue-700 transition-colors text-center text-sm">
                    <?php echo $lang['save_copy'] ?? 'Exemplar speichern'; ?>
                </button>
                <a href="books-edit-detail.php?id=<?php echo $bookId; ?>"
                   class="px-5 py-2.5 bg-gray-100 text-gray-700 font-medium rounded-lg hover:bg-gray-200 transition-colors text-center text-sm">
                    <?php echo $lang['cancel'] ?? 'Abbrechen'; ?>
                </a>
            </div>
        </form>
    </div>

</div>
</div>

<script>
const BASE_URL = '<?php echo BASE_URL; ?>';
const langAvailable = <?php echo json_encode($lang['number_available'] ?? 'Verfügbar'); ?>;
const langTaken     = <?php echo json_encode($lang['number_taken'] ?? 'Bereits vergeben'); ?>;

let numCheckTimer;
async function checkNumber(val) {
    const el = document.getElementById('num-status');
    clearTimeout(numCheckTimer);
    if (!val) { el.textContent = '—'; el.title = ''; return; }
    el.textContent = '⏳';
    numCheckTimer = setTimeout(async () => {
        try {
            const res  = await fetch(`${BASE_URL}/admin/books-add.php?action=check_number&number=${encodeURIComponent(val)}`);
            const data = await res.json();
            el.textContent = data.available ? '✅' : '❌';
            el.title       = data.available ? langAvailable : langTaken;
        } catch { el.textContent = '?'; }
    }, 400);
}

// Beim Laden sofort prüfen
checkNumber(document.getElementById('f-book_number').value);
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
