<?php
/**
 * LDAP-Benutzersuche API
 * GET-Endpoint fuer Admin-seitige LDAP-Suche (z.B. fuer users-add.php)
 *
 * Parameter: q (Suchbegriff, mind. 2 Zeichen)
 * Gibt JSON-Array zurueck: [{username, email, first_name, last_name}]
 *
 * Nur fuer Administratoren zugaenglich.
 */

require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/db.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/session.php';
require_once __DIR__ . '/../../includes/auth.php';

header('Content-Type: application/json; charset=utf-8');

// Nur Admins
if (!is_admin()) {
    http_response_code(403);
    echo json_encode(['error' => 'Zugriff verweigert']);
    exit;
}

// LDAP muss aktiviert sein
if (getSetting('ldap_enabled') !== '1') {
    http_response_code(400);
    echo json_encode(['error' => 'LDAP ist nicht aktiviert']);
    exit;
}

$q = trim($_GET['q'] ?? '');
if (mb_strlen($q) < 2) {
    echo json_encode([]);
    exit;
}

// LDAP-Verbindungsparameter
if (!extension_loaded('ldap')) {
    http_response_code(500);
    echo json_encode(['error' => 'PHP LDAP-Extension nicht geladen']);
    exit;
}

$server        = getSetting('ldap_host');
$port          = (int)getSetting('ldap_port', '389');
$base_dn       = getSetting('ldap_base_dn');
$bind_dn       = getSetting('ldap_bind_dn');
$bind_password = getSetting('ldap_bind_password');
$usernameAttr  = getSetting('ldap_username_attribute', 'sAMAccountName');
$emailAttr     = getSetting('ldap_email_attribute', 'mail');
$firstAttr     = getSetting('ldap_firstname_attribute', 'givenName');
$lastAttr      = getSetting('ldap_lastname_attribute', 'sn');

if (!$server || !$base_dn) {
    http_response_code(500);
    echo json_encode(['error' => 'LDAP nicht konfiguriert (Host/BaseDN fehlt)']);
    exit;
}

try {
    $ldap = @ldap_connect($server, $port);
    if (!$ldap) {
        echo json_encode(['error' => 'Verbindung zum LDAP-Server fehlgeschlagen']);
        exit;
    }

    ldap_set_option($ldap, LDAP_OPT_PROTOCOL_VERSION, 3);
    ldap_set_option($ldap, LDAP_OPT_REFERRALS, 0);
    ldap_set_option($ldap, LDAP_OPT_NETWORK_TIMEOUT, 5);

    // Service-Account-Bind
    if (!@ldap_bind($ldap, $bind_dn, $bind_password)) {
        ldap_close($ldap);
        echo json_encode(['error' => 'LDAP-Bind fehlgeschlagen']);
        exit;
    }

    // Suchfilter: Nachname oder Vorname enthaelt Suchbegriff
    $escaped = ldap_escape($q, '', LDAP_ESCAPE_FILTER);
    $filter = '(&(objectClass=user)(|(sn=*' . $escaped . '*)(givenName=*' . $escaped . '*)))';

    $search = @ldap_search($ldap, $base_dn, $filter, [
        'dn', $usernameAttr, $emailAttr, $firstAttr, $lastAttr
    ], 0, 50); // max. 50 Ergebnisse

    $results = [];

    if ($search) {
        $entries = ldap_get_entries($ldap, $search);
        for ($i = 0; $i < $entries['count']; $i++) {
            $e = $entries[$i];
            $results[] = [
                'username'   => $e[strtolower($usernameAttr)][0] ?? '',
                'email'      => $e[strtolower($emailAttr)][0] ?? '',
                'first_name' => $e[strtolower($firstAttr)][0] ?? '',
                'last_name'  => $e[strtolower($lastAttr)][0] ?? '',
                'ldap_dn'    => $e['dn'] ?? '',
            ];
        }
    }

    ldap_close($ldap);

    echo json_encode($results);

} catch (Exception $e) {
    if (isset($ldap)) @ldap_close($ldap);
    error_log('LDAP Search Error: ' . $e->getMessage());
    echo json_encode(['error' => 'LDAP-Fehler: ' . $e->getMessage()]);
}
